/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

/**
 \file
 Implementation of methods in existingresults.h.
 */


using namespace std;

#include <iomanip>
#include <iostream>
#include <vector>

#include "shortratecontinuous.h"
#include "shortratediscrete.h"

#include "flexibletree/binomial.h"
#include "flexibletree/trinomial.h"

#include "parameter/constantzeroinitial.h"
#include "parameter/constantzeroinitialfinal.h"
#include "parameter/integrablebridge.h"
#include "parameter/integrableconstant.h"

#include "payoff/bullspread.h"
#include "payoff/butterfly.h"
#include "payoff/callcashunderlying.h"
#include "payoff/callphysicalunderlying.h"
#include "payoff/callphysical.h"
#include "payoff/putcashunderlying.h"
#include "payoff/putphysical.h"

#include "treeproduct/americanbuyeroptional.h"
#include "treeproduct/americanselleroptional.h"
#include "treeproduct/europeanbuyeroptional.h"
#include "treeproduct/europeanselleroptional.h"

void BoyleVorst1992()
{
    cout << "Boyle, P.P., Vorst, T., Option Replication in Discrete Time with Transaction Costs, The Journal of Finance, 1992, XLVII, 347-382" << endl;

    coefficient S = 100.0;
    ParameterIntegrableBridge sigma (ParameterIntegrableConstant (0.2));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateDiscrete r (ParameterIntegrableConstant (0.1));
    coefficient T = 1.0;

    vector<coefficient> costs;
    costs.push_back (0.0);
    costs.push_back (0.00125);
    costs.push_back (0.005);
    costs.push_back (0.02);

    cout << endl << "Table I: European Long Call Prices" << endl << endl;

    cout << "K \\ n" << setw (NUMWIDTH) << 6 << " " << setw (NUMWIDTH) << 13 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 250 << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma, kappa_sigma_square, r, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma, kappa_sigma_square, r, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductEuropeanSeller askproduct (T, option);

            cout << setw (4) << strike << " " << setw (NUMWIDTH) << tree6.price (askproduct) << " " << setw (NUMWIDTH) << tree13.price (askproduct) << " " << setw (NUMWIDTH) << tree52.price (askproduct) << " " << setw (NUMWIDTH) << tree250.price (askproduct) << endl;
        }
    }

    cout << endl << "Table IV: European Short Call Prices" << endl << endl;

    cout << "K \\ n" << setw (NUMWIDTH) << 6 << " " << setw (NUMWIDTH) << 13 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 250 << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma, kappa_sigma_square, r, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma, kappa_sigma_square, r, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductEuropeanBuyer bidproduct (T, option);

            cout << setw (4) << strike << " " << setw (NUMWIDTH) << tree6.price (bidproduct) << " " << setw (NUMWIDTH) << tree13.price (bidproduct) << " " << setw (NUMWIDTH) << tree52.price (bidproduct) << " " << setw (NUMWIDTH) << tree250.price (bidproduct) << endl;
        }
    }

    cout << endl << "Results differ due to different approaches (replication vs super-replication)." << endl;
}

void EdirsingheNaikUppal1993()
{

    cout << "Edirisinghe, C.; Naik, V. & Uppal, R. Optimal Replication of Options with Transactions Costs and Trading Restrictions The Journal of Financial and Quantitative Analysis, 1993, 28, 117-138" << endl << endl;

    coefficient S = 100.0;
    ParameterIntegrableBridge sigma5 (ParameterIntegrableConstant (0.05));
    ParameterIntegrableBridge sigma10 (ParameterIntegrableConstant (0.1));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateDiscrete r (ParameterIntegrableConstant (0.0));
    coefficient T = 1.0;

    PayoffCallPhysicalUnderlying payoff95 (95);
    PayoffCallPhysicalUnderlying payoff100 (100);
    PayoffCallPhysicalUnderlying payoff105 (105);
    PayoffBridge payoffcall95 (payoff95);
    PayoffBridge payoffcall100 (payoff100);
    PayoffBridge payoffcall105 (payoff105);
    TreeProductEuropeanSeller call95 (T, payoffcall95);
    TreeProductEuropeanSeller call100 (T, payoffcall100);
    TreeProductEuropeanSeller call105 (T, payoffcall105);

    cout << "Table 1: Cost of Replicating a Long Call Option" << endl << endl;
    cout << setw (NUMWIDTH) << "K = " << " " << setw (NUMWIDTH) << "95" << " " << setw (NUMWIDTH) << "95" << " " << setw (NUMWIDTH) << "100" << " " << setw (NUMWIDTH) << "100" << " " << setw (NUMWIDTH) << "105" << " " << setw (NUMWIDTH) << "105" << endl;
    cout << setw (NUMWIDTH) << "k \\ sigma" << " " << setw (NUMWIDTH) << "0.05" << " " << setw (NUMWIDTH) << "0.1" << " " << setw (NUMWIDTH) << "0.05" << " " << setw (NUMWIDTH) << "0.1" << " " << setw (NUMWIDTH) << "0.05" << " " << setw (NUMWIDTH) << "0.1" << endl;

    for (size_t k = 0; k < 6; k++)
    {
        ParameterConstantZeroInitialFinal lambda_inner (k/100.0, T);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree5 (S, sigma5, kappa_sigma_square, r, lambda, lambda, 8, T);
        FlexibleTreeBinomial tree10 (S, sigma10, kappa_sigma_square, r, lambda, lambda, 8, T);

        cout << setw (NUMWIDTH) << k/100.0 << " " << setw (NUMWIDTH) << tree5.price(call95) << " " << setw (NUMWIDTH) << tree10.price(call95) << " " << setw (NUMWIDTH) << tree5.price(call100) << " " << setw (NUMWIDTH) << tree10.price(call100) << " " << setw (NUMWIDTH) << tree5.price(call105) << " " << setw (NUMWIDTH) << tree10.price(call105) << endl;
    }

    cout << endl << "Table 2: Effect of Various Constraints on the Minimum Cost of Replicating Convex and Nonconvex Claims" << endl << endl;

    vector<coefficient> costs;
    costs.push_back (0.0);
    costs.push_back (0.025);
    costs.push_back (0.05);

    cout << setw(NUMWIDTH) << "T" << " " << setw(NUMWIDTH) << "k = 0" << " " << setw(NUMWIDTH) << "k = 0.025" << " " << setw(NUMWIDTH) << "k = 0.05" << endl;

    cout << "Call option (Long one call with K = 100)" << endl;

    PayoffCallCashUnderlying payoffcallcash100 (100);
    PayoffBridge payoffcallcash100bridge (payoff100);
    TreeProductEuropeanSeller call100cash (T, payoffcallcash100bridge);

    cout << setw(NUMWIDTH) << 8 << " ";
    for (size_t k = 0; k < costs.size(); k++)
    {
        ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
        ParameterBridge lambda (lambda_inner);
        FlexibleTreeBinomial tree8 (S, sigma10, kappa_sigma_square, r, lambda, lambda, 8, T);
        cout << setw (NUMWIDTH) << tree8.price (call100cash);
    }
    cout << endl << setw(NUMWIDTH) << 19 << " ";
    for (size_t k = 0; k < costs.size(); k++)
    {
        ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
        ParameterBridge lambda (lambda_inner);
        FlexibleTreeBinomial tree19 (S, sigma10, kappa_sigma_square, r, lambda, lambda, 19, T);
        cout << setw (NUMWIDTH) << tree19.price (call100);
    }
    cout << endl;

    cout << "Covered Call Position (Long one call with K = 97.5, short one call with K = 102.5)" << endl;

    PayoffBullSpread payoffbullspread (97.5, 102.5);
    TreeProductEuropeanSellerOptional bullproduct (T, payoffbullspread);

    cout << setw(NUMWIDTH) << 8 << " ";
    for (size_t k = 0; k < costs.size(); k++)
    {
        ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
        ParameterBridge lambda (lambda_inner);
        FlexibleTreeBinomial tree8 (S, sigma10, kappa_sigma_square, r, lambda, lambda, 8, T);
        cout << setw (NUMWIDTH) << tree8.price (bullproduct);
    }
    cout << endl << setw(NUMWIDTH) << 19 << " ";
    for (size_t k = 0; k < costs.size(); k++)
    {
        ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
        ParameterBridge lambda (lambda_inner);
        FlexibleTreeBinomial tree19 (S, sigma10, kappa_sigma_square, r, lambda, lambda, 19, T);
        cout << setw (NUMWIDTH) << tree19.price (bullproduct);
    }
    cout << endl;

    cout << "Butterfly Spread (Long one call each with K = 97.5, K = 102.5, short two calls with K = 100)" << endl;

    PayoffButterfly payoffbutterfly (100, 2.5);
    TreeProductEuropeanSellerOptional butterflyproduct (T, payoffbutterfly);

    cout << setw(NUMWIDTH) << 8 << " ";
    for (size_t k = 0; k < costs.size(); k++)
    {
        ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
        ParameterBridge lambda (lambda_inner);
        FlexibleTreeBinomial tree8 (S, sigma10, kappa_sigma_square, r, lambda, lambda, 8, T);
        cout << setw (NUMWIDTH) << tree8.price (butterflyproduct);
    }
    cout << endl << setw(NUMWIDTH) << 19 << " ";
    for (size_t k = 0; k < costs.size(); k++)
    {
        ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
        ParameterBridge lambda (lambda_inner);
        FlexibleTreeBinomial tree19 (S, sigma10, kappa_sigma_square, r, lambda, lambda, 19, T);
        cout << setw (NUMWIDTH) << tree19.price (butterflyproduct);
    }
    cout << endl;
}

void Palmer2001()
{
    cout << "Palmer, K., A note on the Boyle-Vorst discrete-time option pricing model with transactions costs, Mathematical Finance, 2001, 11, 357-363" << endl << endl;


    coefficient S = 100.0;
    ParameterIntegrableBridge sigma (ParameterIntegrableConstant (0.2));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateDiscrete r (ParameterIntegrableConstant (0.1));
    coefficient T = 1.0;

    vector<coefficient> costs;
    costs.push_back (0.0);
    costs.push_back (0.00125);
    costs.push_back (0.005);
    costs.push_back (0.02);

    cout << "K \\ n" << setw (NUMWIDTH) << 6 << " " << setw (NUMWIDTH) << 13 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 250 << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma, kappa_sigma_square, r, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma, kappa_sigma_square, r, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductEuropeanBuyer bidproduct (T, option);

            cout << setw (4) << strike << " " << setw (NUMWIDTH) << tree6.price (bidproduct) << " " << setw (NUMWIDTH) << tree13.price (bidproduct) << " " << setw (NUMWIDTH) << tree52.price (bidproduct) << " " << setw (NUMWIDTH) << tree250.price (bidproduct) << endl;
        }
    }

    cout << endl << "Results differ due to different approaches (replication vs super-replication)." << endl;

}

void PerrakisLefoll2004()
{
    cout << "Perrakis, S. & Lefoll, J. The American put under transactions costs Journal of Economic Dynamics and Control, 2004, 28, 915-935" << endl << endl;

    coefficient S = 100.0;
    ParameterIntegrableBridge sigma (ParameterIntegrableConstant (0.2));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateContinuous r (ParameterIntegrableConstant (0.1));
    coefficient T = 0.25;

    cout << "Table 1" << endl << endl;
    cout << setw(NUMWIDTH) << "X" << " " << setw(NUMWIDTH) << "Ask" << " " << setw(NUMWIDTH) << "Ask - E" << " " << setw(NUMWIDTH) << "Bid" << " " << setw(NUMWIDTH) << "Bid - E" << " " << setw(NUMWIDTH) << "Spread" << endl;

    ParameterConstantZeroInitial lambda_inner (0.005);
    ParameterBridge lambda (lambda_inner);

    for (size_t n = 20; n < 41; n+=20)
    {
        cout << "n = " << n << endl;

        FlexibleTreeBinomial tree (S, sigma, kappa_sigma_square, r, lambda, lambda, n, T);

        for (coefficient strike = 85; strike < 120; strike += 5)
        {
            PayoffPutPhysical put (strike);

            PayoffBridge option (put);

            TreeProductEuropeanBuyerOptional eurbidproduct (T, option);
            TreeProductAmericanBuyerOptional ambidproduct (T, option);
            TreeProductEuropeanSellerOptional euraskproduct (T, option);
            TreeProductAmericanSellerOptional amaskproduct (T, option);

            double amaskprice = tree.price (amaskproduct);
            double ambidprice = tree.price (ambidproduct);
            double amaskspread = amaskprice - tree.price (euraskproduct);
            double ambidspread = ambidprice - tree.price (eurbidproduct);

            cout << setw(NUMWIDTH) << strike << " " << setw(NUMWIDTH) << amaskprice << " " << setw(NUMWIDTH) << amaskspread << " " << setw(NUMWIDTH) << ambidprice << " " << setw(NUMWIDTH) << ambidspread << " " << setw(NUMWIDTH) << amaskprice - ambidprice << " " << endl;
        }
    }

    cout << endl << "Table 2" << endl << endl;
    cout << setw(NUMWIDTH) << "X" << " " << setw(NUMWIDTH) << "Ask" << " " << setw(NUMWIDTH) << "Ask - E" << " " << setw(NUMWIDTH) << "Bid" << " " << setw(NUMWIDTH) << "Bid - E" << " " << setw(NUMWIDTH) << "Spread" << endl;

    ParameterIntegrableBridge sigma_seller (ParameterIntegrableConstant (0.2*sqrt(1.005)));
    ParameterIntegrableBridge sigma_buyer (ParameterIntegrableConstant (0.2*sqrt(0.995)));

    for (size_t n = 64; n < 101; n+=36)
    {
        cout << "n = " << n << endl;

        ParameterConstantZeroInitial lambda_inner (0.005*sqrt(T/n));
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree_seller (S, sigma_seller, kappa_sigma_square, r, lambda, lambda, n, T);
        FlexibleTreeBinomial tree_buyer (S, sigma_buyer, kappa_sigma_square, r, lambda, lambda, n, T);

        for (coefficient strike = 85; strike < 120; strike += 5)
        {


            PayoffPutPhysical put (strike);
            PayoffBridge option (put);

            TreeProductEuropeanBuyerOptional eurbidproduct (T, option);
            TreeProductAmericanBuyerOptional ambidproduct (T, option);
            TreeProductEuropeanSellerOptional euraskproduct (T, option);
            TreeProductAmericanSellerOptional amaskproduct (T, option);

            double amaskprice = tree_seller.price (amaskproduct);
            double ambidprice = tree_buyer.price (ambidproduct);
            double amaskspread = amaskprice - tree_seller.price (euraskproduct);
            double ambidspread = ambidprice - tree_buyer.price (eurbidproduct);

            cout << setw(NUMWIDTH) << strike << " " << setw(NUMWIDTH) << amaskprice << " " << setw(NUMWIDTH) << amaskspread << " " << setw(NUMWIDTH) << ambidprice << " " << setw(NUMWIDTH) << ambidspread << " " << setw(NUMWIDTH) << amaskprice - ambidprice << " " << endl;
        }
    }
}

void TokarzZastawniak2006()
{
    cout << "Tokarz, K. & Zastawniak, T. American contingent claims under small proportional transaction costs Journal of Mathematical Economics, 2006, 43, 65-85" << endl << endl;

    coefficient S = 100.0;
    ParameterIntegrableBridge sigma (ParameterIntegrableConstant (0.2));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateDiscrete r (ParameterIntegrableConstant (0.1));
    coefficient T = 1.0;

    vector<coefficient> costs;
    costs.push_back (0.0);
    costs.push_back (0.00125);
    costs.push_back (0.005);

    cout << "Bid prices" << endl << endl;

    cout << "K \\ N" << setw(NUMWIDTH) << 6 << " " << setw(NUMWIDTH) << 13 << " " << setw(NUMWIDTH) << 52 << " " << setw(NUMWIDTH) << 250 << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma, kappa_sigma_square, r, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma, kappa_sigma_square, r, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductAmericanBuyer bidproduct (T, option);

            cout << setw (4) << strike << " " << setw (NUMWIDTH) << tree6.price (bidproduct) << " " << setw (NUMWIDTH) << tree13.price (bidproduct) << " " << setw (NUMWIDTH) << tree52.price (bidproduct) << " " << setw (NUMWIDTH) << tree250.price (bidproduct) << endl;
        }
    }

    cout << "Ask prices" << endl << endl;
    cout << "K \\ N" << setw(NUMWIDTH) << 6 << " " << setw(NUMWIDTH) << 13 << " " << setw(NUMWIDTH) << 52 << " " << setw(NUMWIDTH) << 250 << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma, kappa_sigma_square, r, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma, kappa_sigma_square, r, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductAmericanSeller askproduct (T, option);

            cout << setw (4) << strike << " " << setw (NUMWIDTH) << tree6.price (askproduct) << " " << setw (NUMWIDTH) << tree13.price (askproduct) << " " << setw (NUMWIDTH) << tree52.price (askproduct) << " " << setw (NUMWIDTH) << tree250.price (askproduct) << endl;
        }
    }

    cout << endl << "Results not exactly the same due to subtle difference in exercise rule." << endl;

}

void Roux2008()
{
    cout << "Roux, A. Options under transaction costs: Algorithms for pricing and hedging of European and American options under proportional transaction costs and different borrowing and lending rates VDM Verlag, 2008" << endl << endl;

    coefficient S = 100.0;
    ParameterIntegrableBridge sigma20 (ParameterIntegrableConstant (0.2));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateDiscrete r10 (ParameterIntegrableConstant (0.1));
    coefficient T = 1.0;

    vector<coefficient> costs;
    costs.push_back (0.0);
    costs.push_back (0.00125);
    costs.push_back (0.005);
    costs.push_back (0.0075);
    costs.push_back (0.02);

    cout << "Table 3.1: Ask prices of European call options" << endl << endl;
    cout << "K \\ N" << setw (NUMWIDTH) << 6 << " " << setw (NUMWIDTH) << 13 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 250 << " " << setw (NUMWIDTH) << 1000 << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 250, T);
        FlexibleTreeBinomial tree1000 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 1000, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductEuropeanSeller askproduct (T, option);

            cout << setw (4) << strike <<  " "
                 << setw (NUMWIDTH) << tree6.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree13.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree52.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree250.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree1000.price (askproduct) << endl;
        }
    }

    cout << endl << "Table 3.2: Bid prices of European call options" << endl << endl;
    cout << "K \\ N" << setw (NUMWIDTH) << 6 << " " << setw (NUMWIDTH) << 13 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 250 << " " << setw (NUMWIDTH) << 1000 << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 250, T);
        FlexibleTreeBinomial tree1000 (S, sigma20, kappa_sigma_square, r10, lambda, lambda, 1000, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductEuropeanBuyer bidproduct (T, option);
            TreeProductEuropeanSeller askproduct (T, option);

            cout << setw (4) << strike << " " << setw (NUMWIDTH) << tree6.price (bidproduct) << " "
                 << setw (NUMWIDTH) << tree13.price (bidproduct) << " "
                 << setw (NUMWIDTH) << tree52.price (bidproduct) << " "
                 << setw (NUMWIDTH) << tree250.price (bidproduct) << " "
                 << setw (NUMWIDTH) << tree1000.price (bidproduct) << endl;
        }
    }

    cout << endl << "Table 3.3: Ask prices of European call options" << endl << endl;
    cout << setw (NUMWIDTH) << "sigma \\ k" << " " << setw (NUMWIDTH) << "0" << " " << setw (NUMWIDTH) << "0.01" << " " << setw (NUMWIDTH) << "0.02" << " " << setw (NUMWIDTH) << "0.03" << " " << setw (NUMWIDTH) << "0.04" << " " << setw (NUMWIDTH) << "0.05" << endl;

    ShortRateDiscrete r0 (ParameterIntegrableConstant (0.0));

    for (size_t K = 95; K < 106; K+=5)
    {
        cout << "K = " << K << endl;
        PayoffCallPhysicalUnderlying payoff (K);
        PayoffBridge payoffcall (payoff);
        TreeProductEuropeanSeller callproduct (T, payoffcall);

        for (size_t vol = 5; vol < 11; vol +=5)
        {
            cout << setw(NUMWIDTH) << vol/100.0;
            ParameterIntegrableBridge sigma (ParameterIntegrableConstant (vol/100.0));

            for (size_t k = 0; k < 6; k++)
            {
                ParameterConstantZeroInitialFinal lambda_inner (k/100.0, T);
                ParameterBridge lambda (lambda_inner);

                FlexibleTreeBinomial tree (S, sigma, kappa_sigma_square, r0, lambda, lambda, 8, T);

                cout << " " << setw (NUMWIDTH) << tree.price(callproduct);
            }
            cout << endl;

        }
    }

    cout << endl << "Table 3.4: Ask prices of various European options with cash delivery" << endl << endl;

    ParameterIntegrableBridge sigma10 (ParameterIntegrableConstant (0.1));

    costs.clear();
    costs.push_back (0.0);
    costs.push_back (0.00125);
    costs.push_back (0.025);
    costs.push_back (0.0375);
    costs.push_back (0.05);

    cout << "T \\ k";

    for (size_t k = 0; k < costs.size(); k++)
        cout << " " << setw(NUMWIDTH) << costs[k];
    cout << endl;

    vector<size_t> steps;
    steps.push_back(8);
    steps.push_back(19);
    steps.push_back(52);
    steps.push_back(250);
//    steps.push_back(1000);

    cout << "Call option" << endl;

    PayoffCallCashUnderlying payoffcallcash100 (100);
    PayoffBridge payoffcallcash100bridge (payoffcallcash100);
    TreeProductEuropeanSeller call100cashask (T, payoffcallcash100bridge);


    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];
        for (size_t k = 0; k < costs.size(); k++)
        {
            ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
            ParameterBridge lambda (lambda_inner);
            FlexibleTreeBinomial tree (S, sigma10, kappa_sigma_square, r0, lambda, lambda, steps[n], T);
            cout << " " << setw (NUMWIDTH) << tree.price (call100cashask);
        }
        cout << endl;
    }

    cout << "Call spread" << endl;

    PayoffBullSpread payoffbullspread (97.5, 102.5);
    TreeProductEuropeanSellerOptional bullproduct (T, payoffbullspread);

    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];
        for (size_t k = 0; k < costs.size(); k++)
        {
            ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
            ParameterBridge lambda (lambda_inner);
            FlexibleTreeBinomial tree (S, sigma10, kappa_sigma_square, r0, lambda, lambda, steps[n], T);
            cout << " " << setw (NUMWIDTH) << tree.price (bullproduct);
        }
        cout << endl;
    }

    cout << "Butterfly spread" << endl;

    PayoffButterfly payoffbutterfly (100, 2.5);
    TreeProductEuropeanSellerOptional butterflyproduct (T, payoffbutterfly);

    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];
        for (size_t k = 0; k < costs.size(); k++)
        {
            ParameterConstantZeroInitialFinal lambda_inner (costs[k], T);
            ParameterBridge lambda (lambda_inner);
            FlexibleTreeBinomial tree (S, sigma10, kappa_sigma_square, r0, lambda, lambda, steps[n], T);
            cout << " " << setw (NUMWIDTH) << tree.price (butterflyproduct);
        }
        cout << endl;
    }

    cout << endl << "Algorithm segfaults for T = 1000 (most likely a memory issue)." << endl << endl;

    cout << endl << "Table 3.5: Prices of call options in trinomial model" << endl << endl;

    ShortRateContinuous r5 (ParameterIntegrableConstant (0.05));

    TreeProductEuropeanBuyer call100cashbid (T, payoffcallcash100bridge);

    steps.clear();
    steps.push_back(12);
    steps.push_back(52);
    steps.push_back(250);
    steps.push_back(500);

    cout << "Ask prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < 4; k++)
        cout << " " << setw (NUMWIDTH) << k/100.0;
    cout << endl;

    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < 4; k++)
        {
            ParameterConstant lambda_inner (k/100.0);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(call100cashask);
        }

        cout << endl;

    }

    cout << "Bid prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < 4; k++)
        cout << " " << setw (NUMWIDTH) << k/100.0;
    cout << endl;
    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < 4; k++)
        {
            ParameterConstant lambda_inner (k/100.0);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(call100cashbid);
        }

        cout << endl;

    }

    cout << endl << "Table 3.6: Prices of put options in trinomial model" << endl << endl;

    PayoffPutCashUnderlying payoffputcash110 (110);
    PayoffBridge payoffputcash110bridge (payoffputcash110);
    TreeProductEuropeanSeller put110cashask (T, payoffputcash110bridge);
    TreeProductEuropeanBuyer put110cashbid (T, payoffputcash110bridge);

    steps.clear();
    steps.push_back(12);
    steps.push_back(52);
    steps.push_back(250);
    steps.push_back(500);

    cout << "Ask prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < 4; k++)
        cout << " " << setw (NUMWIDTH) << k/100.0;
    cout << endl;

    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < 4; k++)
        {
            ParameterConstant lambda_inner (k/100.0);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(put110cashask);
        }

        cout << endl;

    }

    cout << "Bid prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < 4; k++)
        cout << " " << setw (NUMWIDTH) << k/100.0;
    cout << endl;
    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < 4; k++)
        {
            ParameterConstant lambda_inner (k/100.0);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(put110cashbid);
        }

        cout << endl;

    }

    cout << endl << "Table 3.7: Prices of butterfly spreads in trinomial model" << endl << endl;

    PayoffButterfly payoffbutterflywide (105, 10);
    PayoffBridge butterflywidebridge (payoffbutterflywide);
    TreeProductEuropeanSeller butterflywideask (T, butterflywidebridge);
    TreeProductEuropeanBuyer butterflywidebid (T, butterflywidebridge);

    steps.clear();
    steps.push_back(12);
    steps.push_back(52);

    cout << "Ask prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < 4; k++)
        cout << " " << setw (NUMWIDTH) << k/100.0;
    cout << endl;

    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < 4; k++)
        {
            ParameterConstant lambda_inner (k/100.0);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(butterflywideask);
        }

        cout << endl;

    }

    cout << "Bid prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < 4; k++)
        cout << " " << setw (NUMWIDTH) << k/100.0;
    cout << endl;
    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < 4; k++)
        {
            ParameterConstant lambda_inner (k/100.0);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(butterflywidebid);
        }

        cout << endl;

    }

    cout << endl << "Table 4.1: Ask prices of American call options" << endl << endl;

    costs.clear();
    costs.push_back (0.0);
    costs.push_back (0.00125);
    costs.push_back (0.005);
    costs.push_back (0.0075);

    steps.clear();
    steps.push_back(6);
    steps.push_back(13);
    steps.push_back(52);
    steps.push_back(250);
    steps.push_back(1000);

    cout << "K \\ T";
    for (size_t k = 0; k < steps.size(); k++)
        cout << " " << setw(NUMWIDTH) << steps[k];
    cout << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);
            TreeProductAmericanSeller askproduct (T, option);

            cout << setw (5) << strike;
            for (size_t n = 0; n < steps.size(); n++)
            {
                FlexibleTreeBinomial tree (S, sigma20, kappa_sigma_square, r10, lambda, lambda, steps[n], T);
                cout << " " << setw (NUMWIDTH) << tree.price (askproduct);

            }
            cout << endl;
        }
    }

    cout << endl << "Results not exactly the same due to subtle difference in exercise rule." << endl;

    cout << endl << "Table 4.2: Bid prices of American call options" << endl << endl;

    steps.clear();
    steps.push_back(6);
    steps.push_back(13);
    steps.push_back(52);
    steps.push_back(250);

    cout << "K \\ T";
    for (size_t k = 0; k < steps.size(); k++)
        cout << " " << setw(NUMWIDTH) << steps[k];
    cout << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);
            TreeProductAmericanBuyer bidproduct (T, option);

            cout << setw (5) << strike;
            for (size_t n = 0; n < steps.size(); n++)
            {
                FlexibleTreeBinomial tree (S, sigma20, kappa_sigma_square, r10, lambda, lambda, steps[n], T);
                cout << " " << setw (NUMWIDTH) << tree.price (bidproduct);

            }
            cout << endl;
        }
    }

    cout << endl << "Results not exactly the same due to subtle difference in exercise rule." << endl;

    cout << endl << "Table 4.3: Ask and bid prices of American put options" << endl << endl;

    ShortRateContinuous rc10 (ParameterIntegrableConstant (0.1));
    T = 0.25;

    cout << setw(4) << " " << " " << setw(NUMWIDTH) << "Ask" << " " << setw(NUMWIDTH) << "Bid" << " " << setw(NUMWIDTH) << "Ask" << " " << setw(NUMWIDTH) << "Bid" << " " << setw(NUMWIDTH) << "Ask" << " " << setw(NUMWIDTH) << "Bid" << endl;
    cout << setw(4) << "K \\ T";
    for (size_t n = 0; n < steps.size(); n++)
        cout << " " << setw(NUMWIDTH) << steps[n] << " " << setw(NUMWIDTH) << steps[n];
    cout << endl;

    ParameterConstantZeroInitial lambda_inner005 (0.005);
    ParameterBridge lambda005 (lambda_inner005);

    steps.clear();
    steps.push_back(20);
    steps.push_back(40);
    steps.push_back(100);

    for (coefficient strike = 85; strike < 120; strike += 5)
    {
        cout << setw(4) << strike;

        for (size_t n = 0; n < steps.size(); n++)
        {
            FlexibleTreeBinomial tree (S, sigma20, kappa_sigma_square, rc10, lambda005, lambda005, steps[n], T);

            PayoffPutPhysical put (strike);
            PayoffBridge option (put);

            TreeProductAmericanBuyerOptional ambidproduct (T, option);
            TreeProductAmericanSellerOptional amaskproduct (T, option);

            cout << " " << setw(NUMWIDTH) << tree.price(amaskproduct) << " " << setw(NUMWIDTH) << tree.price(ambidproduct);
        }
        cout << endl;
    }

    cout << endl << "Table 4.4: Ask and bid prices of American put options" << endl << endl;

    steps.clear();
    steps.push_back(64);
    steps.push_back(100);
    steps.push_back(250);

    costs.clear();
    for (size_t n = 0; n < steps.size(); n++)
        costs[n] = 0.005*sqrt(T/steps[n]);

    cout << setw(4) << "T = ";
    for (size_t n = 0; n < steps.size(); n++)
        cout << " " << setw(NUMWIDTH) << steps[n] << " " << setw(NUMWIDTH) << steps[n];
    cout << endl;
    cout << setw(4) << "k = ";
    for (size_t n = 0; n < steps.size(); n++)
        cout << " " << setw(NUMWIDTH) << costs[n] << " " << setw(NUMWIDTH) << costs[n];
    cout << endl;

    cout << setw(4) << "K" << " " << setw(NUMWIDTH) << "Ask" << " " << setw(NUMWIDTH) << "Bid" << " " << setw(NUMWIDTH) << "Ask" << " " << setw(NUMWIDTH) << "Bid" << " " << setw(NUMWIDTH) << "Ask" << " " << setw(NUMWIDTH) << "Bid" << endl;

    for (coefficient strike = 85; strike < 120; strike += 5)
    {
        cout << setw(4) << strike;

        for (size_t n = 0; n < steps.size(); n++)
        {
            ParameterConstantZeroInitial lambda_inner (costs[n]);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeBinomial tree (S, sigma20, kappa_sigma_square, rc10, lambda, lambda, steps[n], T);

            PayoffPutPhysical put (strike);
            PayoffBridge option (put);

            TreeProductAmericanBuyerOptional ambidproduct (T, option);
            TreeProductAmericanSellerOptional amaskproduct (T, option);

            cout << " " << setw(NUMWIDTH) << tree.price(amaskproduct) << " " << setw(NUMWIDTH) << tree.price(ambidproduct);
        }
        cout << endl;
    }

    cout << endl << "Table 4.5: Prices of call options in trinomial model" << endl << endl;

    T=1;

    PayoffCallPhysical  payoffcallphys100 (100);
    PayoffBridge payoffcallphys100bridge (payoffcallphys100);

    TreeProductAmericanSellerOptional call100amask (T, payoffcallphys100bridge);
    TreeProductAmericanBuyerOptional call100ambid (T, payoffcallphys100bridge);

    steps.clear();
    steps.push_back(6);
    steps.push_back(12);
    steps.push_back(52);
    steps.push_back(250);
    steps.push_back(500);

    costs.clear();
    costs.push_back(0);
    costs.push_back(0.005);
    costs.push_back(0.0075);
    costs.push_back(0.01);

    cout << "Ask prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < costs.size(); k++)
        cout << " " << setw (NUMWIDTH) << costs[k];
    cout << endl;

    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < costs.size(); k++)
        {
            ParameterConstant lambda_inner (costs[k]);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(call100amask);
        }

        cout << endl;

    }

    steps.clear();
    steps.push_back(6);
    steps.push_back(12);

    cout << "Bid prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < costs.size(); k++)
        cout << " " << setw (NUMWIDTH) << costs[k];
    cout << endl;
    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < costs.size(); k++)
        {
            ParameterConstant lambda_inner (costs[k]);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(call100ambid);
        }

        cout << endl;

    }

    cout << endl << "Table 4.6: Prices of put options in trinomial model" << endl << endl;

    PayoffPutPhysical payoffputphys110 (110);
    PayoffBridge payoffputphys110bridge (payoffputphys110);
    TreeProductAmericanSellerOptional put110physask (T, payoffputphys110bridge);
    TreeProductAmericanBuyerOptional put110physbid (T, payoffputphys110bridge);

    steps.clear();
    steps.push_back(12);
    steps.push_back(52);
    steps.push_back(250);
    steps.push_back(500);

    cout << "Ask prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < costs.size(); k++)
        cout << " " << setw (NUMWIDTH) << costs[k];
    cout << endl;

    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < costs.size(); k++)
        {
            ParameterConstant lambda_inner (costs[k]);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(put110physask);
        }

        cout << endl;

    }

    steps.clear();
    steps.push_back(6);
    steps.push_back(12);

    cout << "Bid prices (credit rate = 5%)" << endl << endl;
    cout << "T \\ k";
    for (size_t k = 0; k < costs.size(); k++)
        cout << " " << setw (NUMWIDTH) << costs[k];
    cout << endl;
    for (size_t n = 0; n < steps.size(); n++)
    {
        cout << setw(5) << steps[n];

        for (size_t k = 0; k < costs.size(); k++)
        {
            ParameterConstant lambda_inner (costs[k]);
            ParameterBridge lambda (lambda_inner);

            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, r5, lambda, lambda, steps[n], T);

            cout << " " << setw (NUMWIDTH) << tree.price(put110physbid);
        }

        cout << endl;

    }

}

void RouxTokarzZastawniak2008()
{
    cout << "Roux, A.; Tokarz, K. & Zastawniak, T. Options under Proportional Transaction Costs: An Algorithmic Approach to Pricing and Hedging Acta Applicandae Mathematicae, 2008, 103, 201-219" << endl << endl;

    cout << "Table 1: Bid and ask prices of European call options" << endl << endl;

    cout << "Bid prices" << endl << endl;
    cout << "K \\ N" << setw (NUMWIDTH) << 6 << " " << setw (NUMWIDTH) << 13 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 250 << endl;

    coefficient S = 100.0;
    ParameterIntegrableBridge sigma (ParameterIntegrableConstant (0.2));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateDiscrete r (ParameterIntegrableConstant (0.1));
    coefficient T = 1.0;

    vector<coefficient> costs;
    costs.push_back (0.0);
    costs.push_back (0.00125);
    costs.push_back (0.005);
    costs.push_back (0.02);

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma, kappa_sigma_square, r, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma, kappa_sigma_square, r, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductEuropeanBuyer askproduct (T, option);

            cout << setw (4) << strike <<  " "
                 << setw (NUMWIDTH) << tree6.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree13.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree52.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree250.price (askproduct) << endl;
        }
    }

    cout << endl << "Ask prices" << endl << endl;
    cout << "K \\ N" << setw (NUMWIDTH) << 6 << " " << setw (NUMWIDTH) << 13 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 250 << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;
        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeBinomial tree6 (S, sigma, kappa_sigma_square, r, lambda, lambda, 6, T);
        FlexibleTreeBinomial tree13 (S, sigma, kappa_sigma_square, r, lambda, lambda, 13, T);
        FlexibleTreeBinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeBinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        for (coefficient strike = 80; strike < 130; strike +=10)
        {
            PayoffCallPhysicalUnderlying payoff (strike);
            PayoffBridge option (payoff);

            TreeProductEuropeanSeller askproduct (T, option);

            cout << setw (4) << strike <<  " "
                 << setw (NUMWIDTH) << tree6.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree13.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree52.price (askproduct) << " "
                 << setw (NUMWIDTH) << tree250.price (askproduct) << endl;
        }
    }

    cout << endl << "Table 2: Ask prices of European options with various payoffs under transaction costs" << endl << endl;

    ParameterIntegrableBridge sigma10 (ParameterIntegrableConstant (0.1));
    ShortRateDiscrete r0 (ParameterIntegrableConstant (0.0));

    costs.clear();
    costs.push_back (0.0);
    costs.push_back (0.025);
    costs.push_back (0.05);

    cout << setw(NUMWIDTH) << "k \\ T";

    vector<size_t> steps;
    steps.push_back(8);
    steps.push_back(19);
    steps.push_back(52);
    steps.push_back(250);
    steps.push_back(1000);

    for (size_t k = 0; k < steps.size(); k++)
        cout << " " << setw(NUMWIDTH) << steps[k];
    cout << endl;

    cout << "Call" << endl;

    PayoffCallCashUnderlying payoffcallcash100 (100);
    PayoffBridge payoffcallcash100bridge (payoffcallcash100);
    TreeProductEuropeanSeller call100cashask (T, payoffcallcash100bridge);


    for (size_t n = 0; n < costs.size(); n++)
    {
        cout << setw(NUMWIDTH) << costs[n];
        for (size_t k = 0; k < steps.size(); k++)
        {
            ParameterConstantZeroInitialFinal lambda_inner (costs[n], T);
            ParameterBridge lambda (lambda_inner);
            FlexibleTreeBinomial tree (S, sigma10, kappa_sigma_square, r0, lambda, lambda, steps[k], T);
            cout << " " << setw (NUMWIDTH) << tree.price (call100cashask);
        }
        cout << endl;
    }

    cout << "Bull spread" << endl;

    PayoffBullSpread payoffbullspread (97.5, 102.5);
    TreeProductEuropeanSellerOptional bullproduct (T, payoffbullspread);

    for (size_t n = 0; n < costs.size(); n++)
    {
        cout << setw(NUMWIDTH) << costs[n];
        for (size_t k = 0; k < steps.size(); k++)
        {
            ParameterConstantZeroInitialFinal lambda_inner (costs[n], T);
            ParameterBridge lambda (lambda_inner);
            FlexibleTreeBinomial tree (S, sigma10, kappa_sigma_square, r0, lambda, lambda, steps[k], T);
            cout << " " << setw (NUMWIDTH) << tree.price (bullproduct);
        }
        cout << endl;
    }

    cout << "Butterfly" << endl;

    PayoffButterfly payoffbutterfly (100, 2.5);
    TreeProductEuropeanSellerOptional butterflyproduct (T, payoffbutterfly);

    for (size_t n = 0; n < costs.size(); n++)
    {
        cout << setw(NUMWIDTH) << costs[n];
        for (size_t k = 0; k < steps.size(); k++)
        {
            ParameterConstantZeroInitialFinal lambda_inner (costs[n], T);
            ParameterBridge lambda (lambda_inner);
            FlexibleTreeBinomial tree (S, sigma10, kappa_sigma_square, r0, lambda, lambda, steps[k], T);
            cout << " " << setw (NUMWIDTH) << tree.price (butterflyproduct);
        }
        cout << endl;
    }

    cout << endl << "Table 3: Bid and ask prices of options in the trinomial model" << endl;
    cout << setw(NUMWIDTH) << " " << " " << setw (NUMWIDTH) << "Ask" << " " << setw (NUMWIDTH) << "Bid" << " " << setw (NUMWIDTH) << "Ask" << " " << setw (NUMWIDTH) << "Bid" << " " << setw (NUMWIDTH) << "Ask" << " " << setw (NUMWIDTH) << "Bid" << endl;

    cout << setw(NUMWIDTH) << "k \\ T" << " " << setw (NUMWIDTH) << 12 << " " << setw (NUMWIDTH) << 12 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 52 << " " << setw (NUMWIDTH) << 250 << " " << setw (NUMWIDTH) << 250 << endl;


    costs.clear();
    costs.push_back (0.0);
    costs.push_back (0.01);
    costs.push_back (0.02);
    costs.push_back (0.03);

    PayoffCallCashUnderlying call2 (100);

    cout << "Call option" << endl;
    TreeProductEuropeanSeller callaskproduct (T, call2);
    TreeProductEuropeanBuyer callbidproduct (T, call2);

    for (size_t k = 0; k < costs.size(); k++)
    {
        ParameterConstant lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeTrinomial tree12 (S, sigma, kappa_sigma_square, r, lambda, lambda, 12, T);
        FlexibleTreeTrinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeTrinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        cout << setw(NUMWIDTH) << costs[k] << " " << setw (NUMWIDTH) << tree12.price (callaskproduct) << " " << setw (NUMWIDTH) << tree12.price (callbidproduct) << " " << setw (NUMWIDTH) << tree52.price (callaskproduct) << " " << setw (NUMWIDTH) << tree52.price (callbidproduct) << " " << setw (NUMWIDTH) << tree250.price (callaskproduct) << " " << setw (NUMWIDTH) << tree250.price (callbidproduct) << endl;
    }

    cout << "Bull spread" << endl;

     PayoffBullSpread bull (95, 105);
    TreeProductEuropeanSeller bullaskproduct (T, bull);
    TreeProductEuropeanBuyer bullbidproduct (T, bull);

    for (size_t k = 0; k < costs.size(); k++)
    {
        ParameterConstant lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        FlexibleTreeTrinomial tree12 (S, sigma, kappa_sigma_square, r, lambda, lambda, 12, T);
        FlexibleTreeTrinomial tree52 (S, sigma, kappa_sigma_square, r, lambda, lambda, 52, T);
        FlexibleTreeTrinomial tree250 (S, sigma, kappa_sigma_square, r, lambda, lambda, 250, T);

        cout << setw(NUMWIDTH) << costs[k] << " " << setw (NUMWIDTH) << tree12.price (bullaskproduct) << " " << setw (NUMWIDTH) << tree12.price (bullbidproduct) << " " << setw (NUMWIDTH) << tree52.price (bullaskproduct) << " " << setw (NUMWIDTH) << tree52.price (bullbidproduct) << " " << setw (NUMWIDTH) << tree250.price (bullaskproduct) << " " << setw (NUMWIDTH) << tree250.price (bullbidproduct) << endl;
    }
}

void RouxZastawniak2009()
{
    cout << "Roux, A. & Zastawniak, T. American Options under Proportional Transaction Costs: Pricing, Hedging and Stopping Algorithms for Long and Short Positions Acta Applicandae Mathematicae, 2009, 106, 199-228" << endl << endl;

    cout << endl << "Table 1: American put prices in the binomial model" << endl << endl;

    coefficient S = 100.0;
    ParameterIntegrableBridge sigma20 (ParameterIntegrableConstant (0.2));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateContinuous rc10 (ParameterIntegrableConstant (0.1));
    coefficient T = 0.25;

    vector<size_t> steps;
    steps.push_back(20);
    steps.push_back(40);
    steps.push_back(100);
    steps.push_back(250);
    steps.push_back(1000);

    cout << setw(4) << "N = ";
    for (size_t n = 0; n < steps.size(); n++)
        cout << " " << setw(NUMWIDTH) << steps[n];
    cout << endl;

    vector<coefficient> costs;
    costs.push_back(0);
    costs.push_back(0.0025);
    costs.push_back(0.005);
    costs.push_back(0.01);
    costs.push_back(0.02);

    PayoffPutPhysical put (100);
    PayoffBridge option (put);
    TreeProductAmericanSellerOptional amaskproduct (T, option);
    TreeProductAmericanBuyerOptional ambidproduct (T, option);

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;

        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        cout << setw(4) << "ask";
        for (size_t n = 0; n < steps.size(); n++)
        {
            FlexibleTreeBinomial tree (S, sigma20, kappa_sigma_square, rc10, lambda, lambda, steps[n], T);
            cout << " " << setw(NUMWIDTH) << tree.price(amaskproduct);
        }

        cout << endl;

        cout << setw(4) << "bid";
        for (size_t n = 0; n < steps.size(); n++)
        {
            FlexibleTreeBinomial tree (S, sigma20, kappa_sigma_square, rc10, lambda, lambda, steps[n], T);
            cout << " " << setw(NUMWIDTH) << tree.price(ambidproduct);
        }

        cout << endl;
    }

    cout << endl << "Table 2: American bull spread prices in the binomial model" << endl << endl;

    cout << setw(4) << "N = ";
    for (size_t n = 0; n < steps.size(); n++)
        cout << " " << setw(NUMWIDTH) << steps[n];
    cout << endl;

    PayoffBullSpread bullspread (95,105);
    PayoffBridge bullspreadbridge (bullspread);
    TreeProductAmericanSellerOptional amaskbull (T, bullspreadbridge);
    TreeProductAmericanBuyerOptional ambidbull (T, bullspreadbridge);

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;

        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        cout << setw(4) << "ask";
        for (size_t n = 0; n < steps.size(); n++)
        {
            FlexibleTreeBinomial tree (S, sigma20, kappa_sigma_square, rc10, lambda, lambda, steps[n], T);
            cout << " " << setw(NUMWIDTH) << tree.price(amaskbull);
        }

        cout << endl;

        cout << setw(4) << "bid";
        for (size_t n = 0; n < steps.size(); n++)
        {
            FlexibleTreeBinomial tree (S, sigma20, kappa_sigma_square, rc10, lambda, lambda, steps[n], T);
            cout << " " << setw(NUMWIDTH) << tree.price(ambidbull);
        }

        cout << endl;
    }

    cout << endl << "Table 3: American bull spread prices in the trinomial model" << endl << endl;

    cout << setw(4) << "N = ";
    for (size_t n = 0; n < steps.size(); n++)
        cout << " " << setw(NUMWIDTH) << steps[n];
    cout << endl;

    for (size_t k = 0; k < costs.size(); k++)
    {
        cout << "k = " << costs[k] << endl;

        ParameterConstantZeroInitial lambda_inner (costs[k]);
        ParameterBridge lambda (lambda_inner);

        cout << setw(4) << "ask";
        for (size_t n = 0; n < steps.size(); n++)
        {
            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, rc10, lambda, lambda, steps[n], T);
            cout << " " << setw(NUMWIDTH) << tree.price(amaskbull);
        }

        cout << endl;

        cout << setw(4) << "bid";
        for (size_t n = 0; n < steps.size(); n++)
        {
            FlexibleTreeTrinomial tree (S, sigma20, kappa_sigma_square, rc10, lambda, lambda, steps[n], T);
            cout << " " << setw(NUMWIDTH) << tree.price(ambidbull);
        }

        cout << endl;
    }
}




// void RouxZastawniakButterfly()
// {
// 	coefficient S = 10.0;
// 	size_t N = 250;
// 	coefficient T = 0.004*N;
//
// 	PayoffBridge option (PayoffButterfly (11,1));
//
// 	size_t jumps = (size_t) (log (11/S) / (0.1*sqrt (T/N)));
//
// 	ParameterIntegrableBridge sigma (ParameterIntegrableConstant (0.1));
// 	ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
// 	ShortRateDiscrete r (ParameterIntegrableConstant (0.0));
//
// 	vector<size_t> path;
// 	path.push_back (0);
//
// //	srand (30000);
// //	srand (6000);
//
// 	for (size_t k = 0; k < jumps; k++)
// 	{
// 		for (size_t l = 0; l < N/jumps; l++)
// 			path.push_back (path.back() + (rand() %3));
// 		path.push_back (path.back());
// 	}
//
// 	path.resize (N + 1);
//
// 	for (size_t k = path.size(); k <= N; k++)
// 		path.push_back (path.back());
//
// 	TreeProductAmericanSeller askproduct (T, option);
//
// 	ParameterConstant lambda_inner (0.005);
// 	ParameterBridge lambda (lambda_inner);
//
// 	FlexibleTreeTrinomial tree (S, sigma, kappa_sigma_square, r, lambda, lambda, N, T);
//
// 	StatisticsGathererPath<Spot> spot = tree.spot_path (path);
// 	StatisticsGathererPath<coefficient> stopping_time = tree.exercise (path, askproduct);
// 	StatisticsGathererPath<Portfolio> strategy = tree.hedge (path, askproduct);
//
// 	coefficient total_stopping = 0.0;
//
// 	for (size_t k = 0 ; k <= tree.N(); k++)
// 	{
// 		total_stopping += stopping_time[k];
// 		cout << setw (NUMWIDTH) << k << " " << setw (NUMWIDTH) << spot[k].underlying() << " " << setw (NUMWIDTH) << option (spot[k]).cash() << " " << setw (NUMWIDTH) << total_stopping << " " << setw (NUMWIDTH) << strategy[k].cash() << " " << setw (NUMWIDTH) << strategy[k].shares() << endl;
// 	}
//
// 	// TreeProductAmericanBuyer bidproduct (T, option);
// 	// cout << "Price for buyer: " << tree.price (bidproduct);
// }

// void RouxZastawniakBullSpread()
// {
// 	coefficient S0 = 20.0;
// 	size_t N = 250;
// 	coefficient T = 0.004*N;
//
// 	PayoffBridge option (PayoffPutPhysical (20));
//
// 	size_t jumps = (size_t) (fabs(log (11/S0)) / (0.1*sqrt (T/N)));
//
// 	ParameterIntegrableBridge sigma (ParameterIntegrableConstant (0.1));
// 	ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.05));
// 	ShortRateDiscrete r (ParameterIntegrableConstant (0.0));
//
// 	vector<size_t> path;
// 	path.push_back (0);
//
// 	srand (0);
//
// 	//18000
//
// 	for (size_t k = 0; k <= jumps; k++)
// 	{
// 		path.push_back (path.back());
// 		for (size_t l = 0; l <= N/jumps/3; l++)
// 			path.push_back (path.back() + (rand() %2));
// 	}
//
// 	path.resize (N/3);
//
// 	for (size_t k = 0; k <= 2*jumps; k++)
// 	{
// 		for (size_t l = 0; l <= N/jumps/3; l++)
// 			path.push_back (path.back() + (rand() %2));
// 		path.push_back (path.back()+1);
// 	}
//
// 	path.resize (N + 1);
//
// 	for (size_t k = path.size(); k <= N; k++)
// 	{
// 		path.push_back (path.back()+1);
// 		path.push_back (path.back() + (rand() %2));
// 	}
//
// 	TreeProductAmericanSeller askproduct (T, option);
// 	TreeProductAmericanBuyer bidproduct (T, option);
//
// 	ParameterConstant lambda_inner (0.005);
// 	ParameterBridge lambda (lambda_inner);
//
// 	FlexibleTreeBinomial tree (S0, sigma, kappa_sigma_square, r, lambda, lambda, N, T);
//
// 	StatisticsGathererPath<Spot> spot (path);
// 	StatisticsGathererPath<PiecewiseLinear> future_hedge (path);
//
// 	vector< path_type > path_extended = tree.with_siblings (path);
// 	StatisticsGathererSome<PiecewiseLinear> seller_current_hedge (path_extended);
// 	StatisticsGathererSome<coefficient> value (path_extended);
// 	StatisticsGathererSome<coefficient> x (path_extended);
// 	StatisticsGathererPath<coefficient> seller_exercise (path);
// 	StatisticsGathererPath<coefficient> S (path);
// 	StatisticsGathererPath<coefficient> y (path);
// 	StatisticsGathererSome<coefficient> p (path_extended);
// 	StatisticsGathererPath<Portfolio> seller_strategy (path);
// 	tree.pricing_function (askproduct, spot, seller_current_hedge, future_hedge);
// 	tree.EMM_exercise_function (askproduct, spot, seller_current_hedge, future_hedge, value, seller_exercise, x, y, S, p, seller_strategy, crossing (Line (0,0), seller_current_hedge[0][0]));
//
// 	StatisticsGathererInitial<PiecewiseLinear> buyer_current_hedge;
// 	StatisticsGathererPath<Portfolio> buyer_strategy (path);
// 	StatisticsGathererPath<int> buyer_exercise (path);
// 	tree.pricing_function (bidproduct, spot, buyer_current_hedge, future_hedge);
// 	tree.hedging_function (bidproduct, spot, buyer_current_hedge, future_hedge, buyer_strategy, crossing (Line (0,0), buyer_current_hedge));
// 	tree.exercise_function (bidproduct, spot, buyer_strategy, buyer_exercise);
//
// 	coefficient seller_total_stopping = 0.0;
// 	int buyer_exercised = 0;
//
// 	for (size_t k = 0 ; k <= tree.N(); k++)
// 	{
// 		seller_total_stopping += seller_exercise[k]*(1-seller_total_stopping);
// 		buyer_exercised += buyer_exercise[k];
// 		coefficient payoff = option (spot[k]).cash() + option (spot[k]).shares()*spot[k].underlying();
// 		cout << setw (NUMWIDTH) << k << " " << setw (NUMWIDTH) << spot[k].underlying() << " " << setw (NUMWIDTH) << (greaterthan(payoff,0) ? payoff:0) << " " << setw (NUMWIDTH) << seller_total_stopping << " " << setw (NUMWIDTH) << seller_strategy[k].cash() << " " << setw (NUMWIDTH) << seller_strategy[k].shares() << " " << setw (NUMWIDTH) << (buyer_exercised ? "nan" : "0") <<  " " << setw (NUMWIDTH) << (buyer_exercised ? "1" : "nan") << " " << setw (NUMWIDTH) << buyer_strategy[k].cash() << " " << setw (NUMWIDTH) << buyer_strategy[k].shares() << endl;
// 	}
// }

void LohneRudloff2011()
{
    cout << "Lohne, A. & Rudloff, B. An algorithm for calculating the set of superhedging portfolios and strategies in markets with transaction costs 2011" << endl << endl;

    cout << "Table 1: set-valued and scalar sub- and superhedging portfolios of European call options" << endl << endl;

    coefficient S = 100.0;
    ParameterIntegrableBridge sigma (ParameterIntegrableConstant (0.2));
    ParameterIntegrableBridge kappa_sigma_square (ParameterIntegrableConstant (0.0));
    ShortRateDiscrete r (ParameterIntegrableConstant (0.1));
    coefficient T = 1.0;

    coefficient costs = 0.00125;

                        vector<size_t> steps;
    steps.push_back(6);
    steps.push_back(13);
    steps.push_back(52);
    steps.push_back(250);
    steps.push_back(1000);
    steps.push_back(1800);

    cout << setw (NUMWIDTH) << "n = ";
    for (size_t n = 0; n < steps.size(); n++)
        cout << " " << setw (NUMWIDTH) << steps[n];
    cout << endl;

       ParameterConstant lambda_inner (costs);
        ParameterBridge lambda (lambda_inner);

    PayoffCallPhysicalUnderlying call (80);
    PayoffBridge callbridge (call);
             TreeProductEuropeanSeller askproduct (T, callbridge);
            TreeProductEuropeanBuyer bidproduct (T, callbridge);

   vector<coefficient> bid1, bid2, bid, ask1, ask2, ask;

    for (size_t n = 0; n < steps.size(); n++)
    {
        FlexibleTreeBinomial tree (S, sigma, kappa_sigma_square, r, lambda, lambda, steps[n], T);

         StatisticsGatherer<Spot> spot;
        StatisticsGathererInitial<PiecewiseLinear> hedge;
        StatisticsGatherer<PiecewiseLinear> future_hedge;

        tree.pricing_function (bidproduct, spot, hedge, future_hedge);
        PiecewiseLinear bidhedge = hedge;
        bid2.push_back(-bidhedge.intersection(1));
        bid1.push_back(-bidhedge(-bid2[n])/r.discount(0,T));
        bid.push_back(bidproduct.current_price(bidhedge));

         tree.pricing_function (askproduct, spot, hedge, future_hedge);
        PiecewiseLinear askhedge = hedge;
        ask2.push_back(askhedge.intersection(1));
        ask1.push_back(askhedge(ask2[n])/r.discount(0,T));
        ask.push_back(askproduct.current_price(askhedge));
    }

    cout << "lambda = " << costs << " for all t" << endl;
    cout << setw (NUMWIDTH) << "bid-x";
    for (size_t n = 0; n < steps.size(); n++)
         cout << " " << setw (NUMWIDTH) << bid1[n];
    cout << endl;
    cout << setw (NUMWIDTH) << "bid-y";
    for (size_t n = 0; n < steps.size(); n++)
         cout << " " << setw (NUMWIDTH) << bid2[n];
    cout << endl;
    cout << setw (NUMWIDTH) << "bid";
    for (size_t n = 0; n < steps.size(); n++)
         cout << " " << setw (NUMWIDTH) << bid[n];
    cout << endl;
    cout << setw (NUMWIDTH) << "ask-x";
    for (size_t n = 0; n < steps.size(); n++)
         cout << " " << setw (NUMWIDTH) << ask1[n];
    cout << endl;
    cout << setw (NUMWIDTH) << "ask-y";
    for (size_t n = 0; n < steps.size(); n++)
         cout << " " << setw (NUMWIDTH) << ask2[n];
    cout << endl;
    cout << setw (NUMWIDTH) << "ask";
    for (size_t n = 0; n < steps.size(); n++)
         cout << " " << setw (NUMWIDTH) << ask[n];
    cout << endl;

    cout << "No transaction costs at time 0" << endl;
    bid.clear();
    ask.clear();
       ParameterConstantZeroInitial lambda_inner0 (costs);
        ParameterBridge lambda0 (lambda_inner0);
    for (size_t n = 0; n < steps.size(); n++)
    {
        FlexibleTreeBinomial tree (S, sigma, kappa_sigma_square, r, lambda0, lambda0, steps[n], T);

        bid.push_back(tree.price(bidproduct));
        ask.push_back(tree.price(askproduct));
    }
     cout << setw (NUMWIDTH) << "bid";
    for (size_t n = 0; n < steps.size(); n++)
         cout << " " << setw (NUMWIDTH) << bid[n];
    cout << endl;
   cout << setw (NUMWIDTH) << "ask";
    for (size_t n = 0; n < steps.size(); n++)
         cout << " " << setw (NUMWIDTH) << ask[n];
    cout << endl;

}

void outputheader()
{
    cout << endl;
    for (size_t k = 0; k < 80; k++)
        cout << "=";
    cout << endl;
}

void existingresults()
{
//    outputheader();
//    BoyleVorst1992();

//   outputheader();EdirsingheNaikUppal1993();

    //  outputheader();
    // Palmer2001();

    // outputheader();
    // PerrakisLefoll2004();

    // outputheader();
    // TokarzZastawniak2006();

    //outputheader();    Roux2008();

    //outputheader(); RouxTokarzZastawniak2008();

    //outputheader(); RouxZastawniak2009();

    outputheader();
    LohneRudloff2011();
}
