/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "binomial.h"

#include "../spotmulambda.h"
#include "../config.h"

FlexibleTreeBinomial::FlexibleTreeBinomial (const coefficient& S, const ParameterIntegrableBridge& sigma, const ParameterIntegrableBridge& kappa_sigma_square, const ShortRate& r, const ParameterBridge& mu, const ParameterBridge& lambda, const size_t N, const coefficient& T)
		: FlexibleTree (S, sigma, kappa_sigma_square, r, mu, lambda, N, T)
{}

size_t FlexibleTreeBinomial::scenarios () const
{
	size_t scenarios = 1;
	for (size_t k = 0; k < N(); k++)
		scenarios *= 2;
	return scenarios;
}

size_t FlexibleTreeBinomial::nodes (const size_t n) const
{
#ifdef USE_ASSERT
		assert (n <= N());
#endif
	return n + 1;
}

void FlexibleTreeBinomial::successors_function (const size_t n, const size_t k, path_type& successors) const
{
#ifdef USE_ASSERT
	assert (n < N());
	assert (k < nodes (n));
#endif

	successors.resize (2);
	successors[0] = k;
	successors[1] = k+1;
}

path_type FlexibleTreeBinomial::path (const size_t scenario) const
{
	path_type path;
	path.push_back (0);

	size_t code = scenario;
	size_t position = 0;

	for (size_t k = 0; k < N(); k++)
	{
		position += code % 2;
		code /= 2;
		path.push_back (position);
	}
	return path;
}

size_t FlexibleTreeBinomial::scenario (const path_type& path) const
{
	size_t scenario = 0;

	for (size_t k = N(); k > 1; k--)
	{
		scenario += (path[k]-path[k-1]);
		if (scenario>0)
			scenario *= 2;
	}
	scenario += path[1];

	return scenario;
}

void FlexibleTreeBinomial::spot_function (const size_t n, std::vector<Spot>& spot) const
{
#ifdef USE_ASSERT
	assert (n <= N());
#endif
	size_t nodenum = n + 1;
	spot.clear();

	if (n == 0)
		spot.push_back (SpotMuLambda (exp (logS()), mu (0), lambda (0)));
	else
	{

		coefficient underlying = logS() + 2*jump (n-1);
		for (size_t l = 0; l < n; l++)
			underlying += jump (l) + drift (l);

		for (size_t k = 0; k < nodenum; k++)
		{
			underlying -= 2*jump (n-1);
			spot.push_back (SpotMuLambda (exp (underlying), mu (n), lambda (n)));
		}
	}
}

void FlexibleTreeBinomial::spot_path_function (StatisticsGathererPath<Spot>& gather_spot) const
{
	coefficient underlying = logS();
	gather_spot.dump_result (0, 0, SpotMuLambda ( (coefficient) exp (underlying), mu (0), lambda (0)));

	for (size_t n = 0; n<N(); n++)
	{
		underlying += drift (n);
		if (gather_spot.node (n+1) == gather_spot.node (n))
			underlying += jump (n);
		else
			underlying -= jump (n);

		gather_spot.dump_result (n+1, gather_spot.node (n+1), SpotMuLambda ( (coefficient) exp (underlying), mu (n+1), lambda (n+1)));
	}
}
