/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef FLEXIBLE_TREE_BINOMIAL_H
#define FLEXIBLE_TREE_BINOMIAL_H

#include "flexibletree.h"

/**
  The friction-free stock price follows the flexible binomial model of Chang, L.-B. & Palmer, K. "Smooth convergence in the binomial model", Finance and Stochastics, 2007, 11, 91-105, i.e. the underlying stock price \f$ S\left(\frac{kT}{N}\right) \f$ at time step \f$ k \f$ takes the form
 \f[ S\left(\frac{kT}{N}\right) = S\left(\frac{(k-1)T}{N}\right)\exp\left\{X_k\sqrt{\int_{\frac{(k-1)T}{N}}^{\frac{kT}{N}}\sigma^2(t)dt} + \int_{\frac{(k-1)T}{N}}^{\frac{kT}{N}}\kappa(t)\sigma^2(t)dt\right\}, \f]
 where \f$ X_k \in\left\{1,-1\right\} \f$.
 @author Alet Roux <alet.roux@york.ac.uk>
*/
///Flexible binomial tree with proportional transaction costs.
class FlexibleTreeBinomial : public FlexibleTree
{

public:
	/**
	 * Constructor.
	 * @param S Initial stock price
	 * @param sigma Stock price volatility
	 * @param kappa_sigma_square Stock price drift
	 * @param r Short rate
	 * @param mu Bid price premium
	 * @param lambda Ask price premium
	 * @param N Number of steps
	 * @param T Maturity date
	 */
	FlexibleTreeBinomial (const coefficient& S, const ParameterIntegrableBridge& sigma, const ParameterIntegrableBridge& kappa_sigma_square, const ShortRate& r, const ParameterBridge& mu, const ParameterBridge& lambda, const size_t N, const coefficient& T);

	size_t scenarios () const;
	size_t nodes (const size_t n) const;
	void successors_function (const size_t n, const size_t k, path_type& successors) const;

	path_type path (const size_t scenario) const;
	size_t scenario (const path_type& path) const;

	void spot_function (const size_t n, std::vector<Spot>& spot) const;
	void spot_path_function (StatisticsGathererPath<Spot>& gather_spot) const;
};

#endif // BINOMIAL_H
