/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "flexibletree.h"

#include "../config.h"


FlexibleTree::FlexibleTree (const coefficient& S, const ParameterIntegrableBridge& sigma, const ParameterIntegrableBridge& kappa_sigma_square, const ShortRate& r, const ParameterBridge& mu, const ParameterBridge& lambda, const size_t N, const coefficient& T)
    : _N (N), _T (T), _dt (T/N), _logS (log (S))
{
    coefficient t1, t2;
    size_t k;

    _mu.push_back (mu (0.0));
    _lambda.push_back (lambda (0.0));

    for (k = 0, t1 = 0.0, t2 = dt(); k < N; k++, t1 = t2, t2 += dt())
    {
        _jump.push_back (sigma.root_integral_square (t1, t2));
        _drift.push_back (kappa_sigma_square.integral (t1, t2));

        _accumulate.push_back (r.accumulate (t1, t2));
        _discount.push_back (1/_accumulate[k]);

        _mu.push_back (mu (t2));
        _lambda.push_back (lambda (t2));
    }
}

void FlexibleTree::pricing_function (const TreeProduct& product, StatisticsGatherer<Spot>& gather_spot, StatisticsGatherer<PiecewiseLinear>& gather_current_hedge, StatisticsGatherer<PiecewiseLinear>& gather_future_hedge) const
{
#ifdef USE_ASSERT
    assert (T() == product.expiry());
#endif

    //The spot price
    std::vector<Spot> spot;
    spot_function (N(),spot);

    //Number of nodes at time N();
    size_t nodenum = nodes (N());

    //Vector of hedging portfolios at time n, indexed by node number.
    std::vector<PiecewiseLinear> future_hedge;
    future_hedge.reserve (nodenum);

    //Set up final hedging values
    for (size_t k = 0; k < nodenum; k++)
    {
#ifdef VERBOSE_FLEXIBLETREE_PRICING_FUNCTION
        std::cout << std::endl << "Time step: " << T() << ", Node: " << k << std::endl;
        std::cout << "Spot: " << spot[k] << std::endl;
#endif

        gather_spot.dump_result (N(), k, spot[k]);

        future_hedge.push_back (product.final_hedge (spot[k]));
        gather_current_hedge.dump_result (N(), k, future_hedge[k]);

#ifdef VERBOSE_FLEXIBLETREE_PRICING_FUNCTION
        std::cout << "Hedge: " << future_hedge[k] << std::endl;
#endif
    }

    //Vector of hedging portfolios at time n-1, indexed by node number.
    std::vector<PiecewiseLinear> hedge;
    hedge.reserve (nodes (N()-1));

    //The successors of any given node
    path_type successors;

    //Given everything at time n, we calculate things for time n-1
    size_t n;

    //The real time at n-1
    coefficient t;

    for (n = N(), t=T()-dt(); n > 0; n--, t= (n-1.0) /N() *T())
    {

        nodenum = nodes (n-1);
        hedge.resize (nodenum);
        spot_function (n-1,spot);

#ifdef VERBOSE_FLEXIBLETREE_PRICING_FUNCTION
        std::cout << std::endl << "Time step " << n-1 << " at " << t << " (jump=" << jump (n-1) << ", drift=" << drift (n-1) << ", discount=" << discount (n-1) << ", mu=" << mu (n-1) << ", lambda=" << lambda (n-1) << ")." << std::endl;
#endif

        for (size_t k = 0; k < nodenum; k++)
        {
            gather_spot.dump_result (n-1, k, spot[k]);
#ifdef VERBOSE_FLEXIBLETREE_PRICING_FUNCTION
            std::cout << std::endl << "Time step: " << n-1 << ", Node: " << k << std::endl;
            std::cout << "Spot: " << spot[k] << std::endl;
#endif

            successors_function (n-1,k,successors);
#ifdef VERBOSE_FLEXIBLETREE_PRICING_FUNCTION
            for (size_t l = 0; l < successors.size(); l++)
                std::cout << "Successor " << successors[l] << ": " << future_hedge[successors[l]] << std::endl;
#endif

            hedge[k] = future_hedge[successors.front() ];
            for (size_t l = 1; l < successors.size(); l++)
                hedge[k] = maximum (hedge[k], future_hedge[successors[l]]);

            hedge[k] *= discount (n-1);
            gather_future_hedge.dump_result (n-1, k, hedge[k]);

            hedge[k] = product.interim_hedge (t, spot[k], hedge[k]);
            gather_current_hedge.dump_result (n-1, k, hedge[k]);

#ifdef VERBOSE_FLEXIBLETREE_PRICING_FUNCTION
            std::cout << "Hedge: " << hedge[k] << std::endl;
#endif
        }
        future_hedge.clear();
        swap (hedge, future_hedge);
    }
}

// StatisticsGathererPath<Portfolio> FlexibleTree::hedge (const path_type& course, const TreeProduct& product, const coefficient shares) const
// {
//     StatisticsGathererPath<Spot> gather_spot (course);
//     StatisticsGathererInitial<PiecewiseLinear> gather_current_hedge;
//     StatisticsGathererPath<PiecewiseLinear> gather_future_hedge (course);
//     StatisticsGathererPath<Portfolio> strategy (course);
//
//     pricing_function (product, gather_spot, gather_current_hedge, gather_future_hedge);
//     hedging_function (product, gather_spot, gather_current_hedge, gather_future_hedge, strategy, shares);
//
//     return strategy;
// }
//
void FlexibleTree::hedging_function (const TreeProduct& product, const StatisticsGathererPath<Spot>& gather_spot, const PiecewiseLinear& current_hedge, const StatisticsGathererPath<PiecewiseLinear>& gather_future_hedge, StatisticsGathererPath<Portfolio>& strategy, const coefficient shares) const
{
    Portfolio portfolio (shares, current_hedge (shares));
    strategy.dump_result (0, strategy.node (0), portfolio);

    //At each step we are considering hedging between n dt() and (n+1)dt().
    for (size_t n = 0; n < N(); n++)
    {
        Spot spot = gather_spot[n];
        PiecewiseLinear future_hedge = gather_future_hedge[n];

#ifdef VERBOSE_FLEXIBLETREE_HEDGING_FUNCTION
        std::cout << "Time step " << n << " with spot " << spot << "." << std::endl;
        std::cout << "Current portfolio: " << portfolio << std::endl;
        std::cout << "Future hedge: " << future_hedge << std::endl;
#endif

        portfolio = product.next_portfolio (n*dt(), spot, future_hedge, portfolio);
#ifdef VERBOSE_FLEXIBLETREE_HEDGING_FUNCTION
        std::cout << "New portfolio: " << portfolio << std::endl;
#endif

        strategy.dump_result (n+1, strategy.node (n+1), portfolio);

        portfolio.accumulate (accumulate (n));

#ifdef VERBOSE_FLEXIBLETREE_HEDGING_FUNCTION
        std::cout << "New portfolio accumulates to: " << portfolio << std::endl;
#endif
    }
}

// StatisticsGathererPath<int> FlexibleTree::exercise (const path_type& course, const TreeProductBuyer& product, const coefficient shares) const
// {
//     StatisticsGathererPath<Spot> gather_spot (course);
//     StatisticsGathererInitial<PiecewiseLinear> gather_current_hedge;
//     StatisticsGathererPath<PiecewiseLinear> gather_future_hedge (course);
//     StatisticsGathererPath<Portfolio> strategy (course);
//     StatisticsGathererPath<int> decision (course);
//
//     pricing_function (product, gather_spot, gather_current_hedge, gather_future_hedge);
//     hedging_function (product, gather_spot, gather_current_hedge, gather_future_hedge, strategy, shares);
//     exercise_function (product, gather_spot, strategy, decision);
//
//     return decision;
// }
//
void FlexibleTree::exercise_function (const TreeProductAmericanBuyer& product, const StatisticsGathererPath<Spot>& gather_spot, const StatisticsGathererPath<Portfolio>& strategy, StatisticsGathererPath<int>& decision) const
{
    Portfolio portfolio = strategy[0];
    Spot spot = gather_spot[0];
    decision.dump_result (0, 0, product.exercise (0, spot, portfolio));

    size_t n = 0;

#ifdef VERBOSE_FLEXIBLETREE_EXERCISE_FUNCTION
    std::cout << "Time step " << n << " with spot " << spot << "." << std::endl;
    std::cout << "Current portfolio: " << portfolio << std::endl;
    std::cout << "Exercise decision: " << ( (decision[n] == 1) ? "Yes" : "No") << std::endl;
#endif

    while ( (decision[n] == 0) && (n < N()))
    {
        n++;

        spot = gather_spot[n];
        portfolio = strategy[n];
        portfolio.accumulate (accumulate (n-1));

#ifdef VERBOSE_FLEXIBLETREE_EXERCISE_FUNCTION
        std::cout << "Time step " << n << " with spot " << spot << "." << std::endl;
        std::cout << "Current portfolio: " << portfolio << std::endl;
#endif

        decision.dump_result (n, decision.node (n), product.exercise (n*dt(), spot, portfolio));

#ifdef VERBOSE_FLEXIBLETREE_EXERCISE_FUNCTION
        std::cout << "Exercise decision: " << ( (decision[n] == 1) ? "Yes" : "No") << std::endl;
#endif
    }
}

// coefficient FlexibleTree::EMM (const path_type& course, const TreeProductSeller& product, const coefficient shares) const
// {
//     std::vector< path_type > course_extended = with_siblings (course);
//
//     StatisticsGathererPath<Spot> gather_spot (course);
//     StatisticsGathererSome<PiecewiseLinear> gather_current_hedge (course_extended);
//     StatisticsGathererPath<PiecewiseLinear> gather_future_hedge (course);
//
//     pricing_function (product, gather_spot, gather_current_hedge, gather_future_hedge);
//
//     StatisticsGathererSome<coefficient> value (course_extended);
//     StatisticsGathererSome<coefficient> x (course_extended);
//     StatisticsGathererPath<coefficient> chi (course);
//     StatisticsGathererPath<coefficient> S (course);
//     StatisticsGathererPath<coefficient> y (course);
//     StatisticsGathererSome<coefficient> p (course_extended);
//     StatisticsGatherer<Portfolio> strategy;
//
//     EMM_exercise_function (product, gather_spot, gather_current_hedge, gather_future_hedge, value, chi, x, y, S, p, strategy, shares);
//
//     coefficient probability = 1.0;
//     for (size_t n = 1; n <= N(); n++)
//         probability *= p[n][course[n]];
//
//     return probability;
// }
//
// StatisticsGathererPath<coefficient> FlexibleTree::exercise (const path_type& course, const TreeProductSeller& product, const coefficient shares) const
// {
//     std::vector< path_type > course_extended = with_siblings (course);
//
//     StatisticsGathererPath<Spot> gather_spot (course);
//     StatisticsGathererSome<PiecewiseLinear> gather_current_hedge (course_extended);
//     StatisticsGathererPath<PiecewiseLinear> gather_future_hedge (course);
//
//     pricing_function (product, gather_spot, gather_current_hedge, gather_future_hedge);
//
//     StatisticsGathererSome<coefficient> value (course_extended);
//     StatisticsGathererSome<coefficient> x (course_extended);
//     StatisticsGathererPath<coefficient> chi (course);
//     StatisticsGathererPath<coefficient> S (course);
//     StatisticsGathererPath<coefficient> y (course);
//     StatisticsGathererSome<coefficient> p (course_extended);
//     StatisticsGatherer<Portfolio> strategy;
//
//     EMM_exercise_function (product, gather_spot, gather_current_hedge, gather_future_hedge, value, chi, x, y, S, p, strategy, shares);
//
//     coefficient total_chi = 0.0;
//     for (size_t n = 1; n <= N(); n++)
//     {
//         total_chi += chi[n-1];
//         chi.dump_result (n, course[n], chi[n]*(1-total_chi));
//     }
//
//     return chi;
// }
//
void FlexibleTree::EMM_exercise_function (const TreeProductAmericanSeller& product, const StatisticsGathererPath<Spot>& gather_spot, const StatisticsGathererSome<PiecewiseLinear>& gather_current_hedge, const StatisticsGathererPath<PiecewiseLinear>& gather_future_hedge, StatisticsGathererSome<coefficient>& value, StatisticsGathererPath<coefficient>& chi, StatisticsGathererSome<coefficient>& x, StatisticsGathererPath<coefficient>& y, StatisticsGathererPath<coefficient>& S, StatisticsGathererSome<coefficient>& p, StatisticsGatherer<Portfolio>& strategy, const coefficient shares) const
{
    Portfolio current (shares, gather_current_hedge[0][0] (shares));
    p.dump_result (0,0,1.0);
    x.dump_result (0, 0, -gather_current_hedge[0][0].slope (gather_current_hedge[0][0].piece (current.shares())));
    strategy.dump_result (0, 0, current);
    value.dump_result (0, 0, current.value(x[0][0]));

    coefficient temp_y, temp_S, temp_chi;

    coefficient t = 0;
    size_t n, node;
    for (n = 0; n < N(); n++, t += dt())
    {
        node = gather_future_hedge.node (n);

        //which of the siblings is the root node?
        size_t index = 0;
        while (gather_current_hedge.node (n,index) != node)
            index++;

        current = product.next_portfolio(t, gather_spot[n], gather_future_hedge[n], current);

        temp_chi = product.random_exercise (t, gather_spot[n], gather_current_hedge[n][index], gather_future_hedge[n], x[n][index], temp_y, temp_S, current);

        chi.dump_result (n, node, temp_chi);
        S.dump_result (n, node, temp_S);
        y.dump_result (n, node, temp_y);

#ifdef VERBOSE_FLEXIBLETREE_EMM_EXERCISE_FUNCTION
        std::cout << "Node " << node << " at time step " << n << " with spot " << gather_spot[n] << "." << std::endl;
        std::cout << "x[" << n << "][" << index << "]: " << x[n][index] << std::endl;
        std::cout << "  S[" << n << "]: " << S[n] << ", chi[" << n << "]: " << chi[n] << std::endl;
        std::cout << "  y[" << n << "]: " << y[n] << ", 1-chi[" << n << "]: " << 1-chi[n] << std::endl;
#endif

        PiecewiseLinear future_hedge = gather_future_hedge[n];

        future_hedge *= accumulate (n);

        coefficient place = current.shares();
        untangle_maximum (future_hedge, gather_current_hedge[n+1], y[n]*accumulate (n), place, p[n+1], x[n+1]);

#ifdef VERBOSE_FLEXIBLETREE_EMM_EXERCISE_FUNCTION
        std::cout << "y[" << n << "]*accumulate(" << n << "): " << y[n]*accumulate (n) << std::endl;
#endif

        for (size_t k = 0; k < gather_current_hedge[n+1].size(); k++)
        {
            strategy.dump_result (n+1, gather_current_hedge.node (n+1,k), current);
            value.dump_result (n+1, gather_current_hedge.node (n+1,k), gather_current_hedge[n+1][k] (place) + place*x[n+1][k]);

#ifdef VERBOSE_FLEXIBLETREE_EMM_EXERCISE_FUNCTION
            std::cout << "  x[" << n+1 << "][" << k << "]: " << x[n+1][k] << ", p[" << n+1 << "][" << k << "]: " << p[n+1][k] << std::endl;
#endif
        }

        current.accumulate(accumulate(n));
    }

    n = N();
    node = gather_future_hedge.node (n);

    //which is the root node?
    size_t index = 0;
    while (gather_current_hedge.node (n,index) != gather_future_hedge.node (n))
        index++;

    temp_chi = product.random_exercise (T(), gather_spot[n], gather_current_hedge[n][index], gather_future_hedge[n], x[n][index], temp_y, temp_S, current);
    chi.dump_result (n, node, temp_chi);
    S.dump_result (n, node, temp_S);
    y.dump_result (n, node, temp_y);

#ifdef VERBOSE_FLEXIBLETREE_EMM_EXERCISE_FUNCTION
    std::cout << "Node " << gather_future_hedge.node (n) << " at time step " << n << " with spot " << gather_spot[n] << "." << std::endl;
    std::cout << "x[" << n << "][" << index << "]: " << x[n][index] << std::endl;
    std::cout << "temp_chi: " << temp_chi << std::endl;
    std::cout << "  S[" << n << "]: " << S[n] << ", chi[" << n << "]: " << chi[n] << std::endl;
    std::cout << "  y[" << n << "]: " << y[n] << ", 1-chi[" << n << "]: " << 1-chi[n] << std::endl;
#endif
}


