/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef FLEXIBLETREE_H
#define FLEXIBLETREE_H

#include "../parameter/bridge.h"
#include "../parameter/integrablebridge.h"
#include "../statisticsgatherer/some.h"
#include "../treeproduct/americanbuyer.h"
#include "../treeproduct/americanseller.h"

typedef std::vector<size_t> path_type;

/**
 Flexible tree with proportional transaction costs.
 @author Alet Roux <alet.roux@york.ac.uk>

 The value \f$ S\left(\frac{kT}{N}\right) \f$ of the friction-free stock price at time step \f$ k \f$ takes the form
 \f[ S\left(\frac{kT}{N}\right) = S\left(\frac{(k-1)T}{N}\right)\exp\left\{X_k\sqrt{\int_{\frac{(k-1)T}{N}}^{\frac{kT}{N}}\sigma^2(t)dt} + \int_{\frac{(k-1)T}{N}}^{\frac{kT}{N}}\kappa(t)\sigma^2(t)dt\right\} \f]
 for some discrete random variable \f$ X_k \f$, with \f$ S(0) = S \f$. The bid price at time \f$ t \f$ is then given by \f$ S(t)(1-\mu(t)) \f$ and the ask price by \f$ S(t)(1+\lambda(t)) \f$.
*/
class FlexibleTree
{

public:
	/**
	 * Constructor.
	 * @param S Initial stock price
	 * @param sigma Stock price volatility
	 * @param kappa_sigma_square Stock price drift
	 * @param r Short rate
	 * @param mu Bid price premium
	 * @param lambda Ask price premium
	 * @param N Number of steps
	 * @param T Maturity date
	 */
	FlexibleTree (const coefficient& S, const ParameterIntegrableBridge& sigma, const ParameterIntegrableBridge& kappa_sigma_square, const ShortRate& r, const ParameterBridge& mu, const ParameterBridge& lambda, const size_t N, const coefficient& T);

	///Price at time 0 of \a product for an investor with initial wealth \a wealth
	coefficient price (const TreeProduct& product, const Portfolio& wealth = Portfolio (0,0)) const
	{
		StatisticsGatherer<PiecewiseLinear> dummyhedge;
		StatisticsGatherer<Spot> dummyspot;
		StatisticsGathererInitial<PiecewiseLinear> initial;

		pricing_function (product, dummyspot, initial, dummyhedge);

		return product.current_price (initial, wealth);
	}

	/**
	 * Service method. Pricing function at time 0 of \a product that allows gathering of statistics for use outside this class. Application of \a product.current_price() to the first value of \a gather_future_hedge yields the initial price.
	*/
	void pricing_function (const TreeProduct& product, StatisticsGatherer<Spot>& gather_spot, StatisticsGatherer<PiecewiseLinear>& gather_current_hedge, StatisticsGatherer<PiecewiseLinear>& gather_future_hedge) const;

	/**
	 * Computes superreplicating strategy for an option.
	 * @param scenario Scenario in which to superreplicate
	 * @param product Objective to superreplicate
	 * @param shares Initial share holding. The cash holding is adjusted to reflect the cost of superreplication
	 * @return Sequence of N()+1 portfolios where the first portfolio is the initial hedging potfolio (with share holding \a wealth.shares()) and the \f$ k \f$-th portfolio is held between (k-1)dt() and k dt()
	 */
	inline StatisticsGathererPath<Portfolio> hedge (const size_t scenario, const TreeProduct& product, const coefficient shares = 0.0) const
	{
		return hedge (path (scenario), product, shares);
	}

	/**
	 * Computes superreplicating strategy for an option.
	 * @param path Scenario in which to superreplicate
	 * @param product Objective to superreplicate
	 * @param shares Initial share holding. The cash holding is adjusted to reflect the cost of superreplication
	 * @return Sequence of N()+1 portfolios where the first portfolio is the initial hedging potfolio (with share holding \a wealth.shares()) and the \f$ k \f$-th portfolio is held between (k-1)dt() and k dt()
	 */
	StatisticsGathererPath<Portfolio> hedge (const path_type& path, const TreeProduct& product, const coefficient shares = 0.0) const;

	/**
	 * Service method. Computation of superreplicating strategy for an option. The parameters \a current_hedge, \a gather_spot and \a gather_future_hedge must have been part of the output of a run of #pricing_function (where \a current_hedge is the initial hedging function). The parameters \a gather_spot, \a portfolios and \a gather_future_hedge must pertain to the same scenario (this is not checked).
	 */
	void hedging_function (const TreeProduct& product, const StatisticsGathererPath<Spot>& gather_spot, const PiecewiseLinear& current_hedge,  const StatisticsGathererPath<PiecewiseLinear>& gather_future_hedge, StatisticsGathererPath<Portfolio>& strategy, const coefficient shares = 0.0) const;

//	///Conditional transition probabilities associated with optimal equivalent martingale measure
//	inline coefficient EMM (const size_t scenario, const TreeProductSeller& product, const coefficient shares = 0.0) const
//	{
//		return EMM (path(scenario), product, shares);
//	}

//	///Conditional transition probabilities associated with optimal equivalent martingale measure
//	coefficient EMM (const path_type& course, const TreeProductSeller& product, const coefficient shares = 0.0) const;

	/**
	 *    Service method. Computation of optimal equivalent martingale measure for \a product, with optimal superreplication strategy as side effect. The parameters \a gather_current_hedge and \a gather_future_hedge must have been part of the output of a run of #pricing_function. The parameters \a spot, \a gather_current_hedge, \a gather_future_hedge, \a value, \a chi, \a x, \a y, \a S, \a p and \a strategy must pertain to the same scenario (enriched by adding relevant successor nodes): this is not checked.
	 * The processses \f$ \chi \f$ and \f$ p \f$ in the interval \f$[0,1]\f$ and \f$ x \f$, \f$ y \f$ and \f$ S \f$ in the bid-ask interval has the property that \f$ x[n][k] = \chi[n]S[n]+(1-\chi[n])y[n] \f$ for all \f$ n \f$ and nodes \f$ k \f$ and \f$ y[n]accumulate(n) = \sum_{k=1}^mp[n+1][m]x[n+1][m] \f$ for all \f$n\f$ before N() and where the sum is taken over successors.
	 */
	void EMM_exercise_function (const TreeProductAmericanSeller& product, const StatisticsGathererPath<Spot>& gather_spot, const StatisticsGathererSome<PiecewiseLinear>& gather_current_hedge, const StatisticsGathererPath<PiecewiseLinear>& gather_future_hedge, StatisticsGathererSome<coefficient>& value, StatisticsGathererPath<coefficient>& chi, StatisticsGathererSome<coefficient>& x, StatisticsGathererPath<coefficient>& y, StatisticsGathererPath<coefficient>& S, StatisticsGathererSome<coefficient>& p, StatisticsGatherer<Portfolio>& strategy, const coefficient shares = 0.0) const;

//	/*
//	 * Stopping time for the buyer of an option.
//	 * @param scenario Scenario in which to superreplicate
//	 * @param product Objective to superreplicate
//	 * @param shares Initial share holding. The cash holding is adjusted to reflect the cost of superreplication
//	 * @return Time at which to stop. Either in interval [0,T()] or INFINITY.
//	 */
//	inline coefficient stopping_time (const size_t scenario, const TreeProductBuyer& product, const coefficient shares = 0.0) const
//	{
//		return stopping_time (path(scenario),product,shares);
//	}

//	/*
//	 * Stopping time for the buyer of an option.
//	 * @param path Scenario in which to superreplicate
//	 * @param product Objective to superreplicate
//	 * @param shares Initial share holding. The cash holding is adjusted to reflect the cost of superreplication
//	 * @return Time at which to stop. Either in interval [0,T()] or INFINITY.
//	 */
//	coefficient stopping_time (const path_type course, const TreeProductBuyer& product, const coefficient shares = 0.0) const
//	{
//		StatisticsGathererPath<int> decision = exercise (course, product, shares);

//		for (size_t n = 0; n <= N(); n++)
//			if (decision[n])
//				return n*dt();

//		return INFINITY;
//	}

//	/*
//	 * Stopping time for the buyer of an option as sequence of binary exercise decisions
//	 * @param scenario Scenario in which to superreplicate
//	 * @param product Objective to superreplicate
//	 * @param shares Initial share holding. The cash holding is adjusted to reflect the cost of superreplication
//	 * @return Sequence of N()+1 numbers in set \f$ \{0,1\} \f$ where the \f$ k \f$-th number is 1 if exercise should take place at time \f$ k \f$ and 0 otherwise. If the option is not to be exercised at all in this scenario, then all numbers are 0.
//	 */
//	inline StatisticsGathererPath<int> exercise (const size_t scenario, const TreeProductBuyer& product, const coefficient shares = 0.0) const
//	{
//		return exercise (path(scenario), product, shares);
//	}

//	/*
//	 * Stopping time for the buyer of an option as sequence of binary exercise decisions
//	 * @param course Scenario in which to superreplicate
//	 * @param product Objective to superreplicate
//	 * @param shares Initial share holding. The cash holding is adjusted to reflect the cost of superreplication
//	 * @return Sequence of N()+1 numbers in set \f$ \{0,1\} \f$ where the \f$ k \f$-th number is 1 if exercise should take place at time \f$ k \f$ and 0 otherwise. If the option is not to be exercised at all in this scenario, then all numbers are 0.
//	 */
//	StatisticsGathererPath<int> exercise (const path_type& course, const TreeProductExercisable& product, const coefficient shares = 0.0) const;

//	/*
//	 * Stopping time for the seller of an option as sequence of randomised exercise decisions
//	 * @param scenario Scenario in which to superreplicate
//	 * @param product Objective to superreplicate
//	 * @param shares Initial share holding. The cash holding is adjusted to reflect the cost of superreplication
//	 * @return Sequence of N()+1 numbers in set \f$ [0,1] \f$ where the \f$ k \f$-th number is the probability that the option is exercised at time \f$ k \f$. If there is a possibility that the option may not to be exercised at all in this scenario, then the numbers will not add up to 1.
//	 */
//	inline StatisticsGathererPath<coefficient> exercise (const size_t scenario, const TreeProductSeller& product, const coefficient shares = 0.0) const
//	{
//		return exercise (path(scenario), product, shares);
//	}

//	/*
//	 * Stopping time for the seller of an option as sequence of randomised exercise decisions
//	 * @param course Scenario in which to superreplicate
//	 * @param product Objective to superreplicate
//	 * @param shares Initial share holding. The cash holding is adjusted to reflect the cost of superreplication
//	 * @return Sequence of N()+1 numbers in set \f$ [0,1] \f$ where the \f$ k \f$-th number is the probability that the option is exercised at time \f$ k \f$. If there is a possibility that the option may not to be exercised at all in this scenario, then the numbers will not add up to 1.
//	 */
//	StatisticsGathererPath<coefficient> exercise (const path_type& course, const TreeProductSeller& product, const coefficient shares = 0.0) const;

	/**
	 * Service method. Computes stopping time for the buyer of an option. The input \a strategy must superreplicate \a product for the buyer, and must correspond to the same scenario as \a gather_spot (this is not checked).
	 */
	void exercise_function (const TreeProductAmericanBuyer& product, const StatisticsGathererPath< Spot >& gather_spot, const StatisticsGathererPath< Portfolio >& strategy, StatisticsGathererPath< int >& decision) const;

	/**
	 * Number of scenarios. Scenarios are numbered from 0 to scenarios()-1.
	 */
	virtual size_t scenarios () const = 0;

	/**
	 * Number of nodes at time step \a n. Nodes at time step \a n are numbered from 0 to nodes(n)-1.
	 */
	virtual size_t nodes (const size_t n) const = 0;

	///Nodes at time step \a n + 1 that succeed node \a k at time step \a n.
	path_type successors (const size_t n, const size_t k) const
	{
		path_type successors;
		successors_function (n, k, successors);
		return successors;
	}

	/**
	 * Service method. Nodes at time step \a n + 1 that succeed node \a k at time step \a n.
	 */
	virtual void successors_function (const size_t n, const size_t k, path_type& successors) const = 0;

	/**
	 * Turns scenario code into path. Each scenario is given a unique positive number, which corresponds uniquely to the path of the stock price in that scenario.
	 * @param scenario Index number of scenario
	 * @return (N()+1)-dimensional std::vector of node numbers, with the k-th entry being the node number at time k dt().
	 */
	virtual path_type path (const size_t scenario) const = 0;

	/**
	 * Turns path into scenario code. Each scenario is given a unique positive number, which corresponds uniquely to the path of the stock price in that scenario.
	 * @param path (N()+1)-dimensional std::vector of node numbers, with the k-th entry being the node number at time k dt()
	 * @return Index number of scenario
	 */
	virtual size_t scenario (const path_type& path) const = 0;

	/**
	 * Extends stock price path to include siblings of those nodes already on the path.
	 * @param path An (N()+1)-dimensional std::vector of node numbers, with the k-th entry being the node number at time k dt().
	 * @return A std::vector of node std::vectors, with the (t,k)-th entry being a node at time t dt().
	 */
	std::vector<path_type > with_siblings (const path_type& path) const
	{
		std::vector<path_type > siblings;

		siblings.push_back (path_type (1,path[0]));

		for (size_t t = 1; t < path.size(); t++)
			siblings.push_back (successors (t-1,path[t-1]));

		return siblings;
	}

	///Spot prices at time step \a n, indexed by node.
	std::vector<Spot> spot (const size_t n) const
	{
		std::vector<Spot> spot;
		spot_function (n, spot);
		return spot;
	}

	///Service method. Spot prices at time step \a n, indexed by node.
	virtual void spot_function (const size_t n, std::vector<Spot>& spot) const = 0;

	///Stock price path for given \a scenario
	inline StatisticsGathererPath<Spot> spot_path (const size_t scenario) const	{	return spot_path (path(scenario));	}

	///Stock price path for given \a scenario
	StatisticsGathererPath<Spot> spot_path (const path_type& course) const
	{
		StatisticsGathererPath<Spot> gather_spot (course);
		spot_path_function (gather_spot);
		return gather_spot;
	}

	/**
	 * Service method. Computes stock price path for scenario to which \a gather_spot has been initialised.
	 */
	virtual void spot_path_function (StatisticsGathererPath<Spot>& gather_spot) const = 0;

	///Logarithm of initial stock price
	coefficient logS () const	{	return _logS;	}

	///Maturity date
	coefficient T () const
	{
		return _T;
	}

	///Number of steps
	size_t N () const	{	return _N;	}

	///Time between steps
	coefficient dt () const	{	return _dt;	}

	/**
	 * Jump over one step from \a n dt() to (\a n + 1)dt(). The formula is \f$ \sqrt{\int_{t_1}^{t_2}\sigma^2(s)ds} \f$ where \f$t_1\f$ is \a n dt() and \f$t_2\f$ is (\a n +1)dt().
	 */
	coefficient jump (const size_t n) const	{	return _jump[n];	}

	/**
	 * Drift over one step from \a n dt() to (\a n + 1)dt(). The formula is \f$ \int_{t_1}^{t_2}\kappa(t)\sigma^2(t)dt \f$, where \f$t_1\f$ is \a n dt() and \f$t_2\f$ is (\a n +1)dt().
	 */
	coefficient drift (const size_t n) const	{	return _drift[n];	}

	///Discount factor over one step from (\a n + 1) dt() to \a n dt()
	coefficient discount (const size_t n) const	{	return _discount[n];	}

	///Interest accumulation factor over one step from \a n dt() to (\a n +1) dt()
	coefficient accumulate (const size_t n) const	{	return _accumulate[n];	}

	///Bid price discount at step \a n.
	coefficient mu (const size_t n) const	{	return _mu[n];	}

	///Ask price premium at step \a n.
	coefficient lambda (const size_t n) const	{	return _lambda[n];	}

private:
	size_t _N;

	coefficient _T, _dt, _logS;

	std::vector<coefficient> _jump, _drift, _discount, _accumulate, _mu, _lambda;

};

#endif // FLEXIBLETREE_H
