/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "trinomial.h"

#include "../config.h"
#include "../spotmulambda.h"

FlexibleTreeTrinomial::FlexibleTreeTrinomial (const coefficient& S, const ParameterIntegrableBridge& sigma, const ParameterIntegrableBridge& kappa_sigma_square, const ShortRate& r, const ParameterBridge& mu, const ParameterBridge& lambda, const size_t N, const coefficient& T)
		: FlexibleTree (S, sigma, kappa_sigma_square, r, mu, lambda, N, T)
{}

size_t FlexibleTreeTrinomial::scenarios () const
{
	size_t scenarios = 1;
	for (size_t k = 0; k < N(); k++)
		scenarios *= 3;
	return scenarios;
}

size_t FlexibleTreeTrinomial::nodes (const size_t n) const
{
#ifdef USE_ASSERT
	assert (n <= N());
#endif
	return 2*n + 1;
}

void FlexibleTreeTrinomial::successors_function (const size_t n, const size_t k, path_type& successors) const
{
#ifdef USE_ASSERT
	assert (n < N());
	assert (k < nodes (n));
#endif
	successors.resize (3);

	successors[0] = k;
	successors[1] = k+1;
	successors[2] = k+2;
}

path_type FlexibleTreeTrinomial::path (const size_t scenario) const
{
	path_type path;
	path.push_back (0);

	size_t code = scenario;
	size_t position = 0;

	for (size_t k = 0; k < N(); k++)
	{
		position += code % 3;
		code /= 3;
		path.push_back (position);
	}
	return path;
}

size_t FlexibleTreeTrinomial::scenario (const path_type& path) const
{
	size_t scenario = 0;

	for (size_t k = N(); k > 0; k--)
	{
		if (scenario > 0)
			scenario *= 3;
		scenario += path[k]-path[k-1];
	}

	return scenario;
}

void FlexibleTreeTrinomial::spot_function (const size_t n, std::vector<Spot>& spot) const
{
#ifdef USE_ASSERT
	assert (n <= N());
#endif

	spot.clear();

	if (n == 0)
		spot.push_back (SpotMuLambda (exp (logS()), mu (0), lambda (0)));
	else
	{
		coefficient underlying = logS() + jump (n-1);
		for (size_t l = 0; l < n; l++)
			underlying += jump (l) + drift (l);

		size_t nodenum = 2*n + 1;

		for (size_t k = 0; k < nodenum; k++)
		{
			underlying -= jump (n-1);
			spot.push_back (SpotMuLambda ( (coefficient) exp (underlying), mu (n), lambda (n)));
		}
	}
}

void FlexibleTreeTrinomial::spot_path_function (StatisticsGathererPath<Spot>& gather_spot) const
{
	coefficient underlying = logS();
	gather_spot.dump_result (0, gather_spot.node (0), SpotMuLambda ( (coefficient) exp (underlying), mu (0), lambda (0)));

	for (size_t n = 0; n<N(); n++)
	{
		underlying += drift (n);
		if (gather_spot.node (n+1) == gather_spot.node (n))
			underlying += jump (n);
		else if (gather_spot.node (n+1) == gather_spot.node (n) + 2)
			underlying -= jump (n);

		gather_spot.dump_result (n+1, gather_spot.node (n+1), SpotMuLambda ( (coefficient) exp (underlying), mu (n+1), lambda (n+1)));
	}
}
