/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef NUMBERS_H
#define NUMBERS_H

/**
 \file
 Numerical conventions.
 */

#include <math.h>

#define EPSILON 	1e-10

#define NUMWIDTH 10

///General number type
typedef long double coefficient;
//typedef unsigned int size_t;

/// Tests whether \a x1 is strictly bigger than \a x2. Fudges it a little bit to help with numerics
inline int greaterthan (const coefficient x1, const coefficient x2)
{
	if (x1 >= EPSILON + x2)
		return 1;

	return 0;
}

/// Tests whether \a x1 is strictly smaller than \a x2. Fudges it a little bit to help with numerics
inline int lessthan (const coefficient x1, const coefficient x2)
{
	return greaterthan (x2, x1);
}

/// Tests whether \a x1 is greater than or equal to \a x2
inline int greaterequal (const coefficient x1, const coefficient x2)
{
	if (x1 >= -EPSILON + x2)
		return 1;

	return 0;
}

/// Tests whether \a x1 is less than or equal to \a x2
inline int lessequal (const coefficient x1, const coefficient x2)
{
	return greaterequal (x2,x1);
}

/// Tests whether \a x1 is equal to \a x2
inline int equal (const coefficient x1, const coefficient x2)
{
	return (greaterequal (x1,x2)) && lessequal (x1,x2);
}

/// Puts \a number equal to 0 if it is close enough to 0.
inline void fixzero (coefficient& number)
{
	if (!isnan (number) && equal (number, 0.0))
		number = 0.0;
}

#endif
