/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef PARAMETERINTEGRABLEBRIDGE_H
#define PARAMETERINTEGRABLEBRIDGE_H

#include "integrable.h"

///Wrapper class for #ParameterIntegrable.
/**
  Inspired by the Parameters class in Mark Joshi "Design Patterns and Derivatives Pricing", Cambridge University Press, 2004.
 @author Alet Roux <alet.roux@york.ac.uk>
*/
class ParameterIntegrableBridge
{
public:
	ParameterIntegrableBridge (const ParameterIntegrableBridge& original)
	{
		_parameter = original._parameter->clone();
	}

	ParameterIntegrableBridge (const ParameterIntegrable& parameter)
	{
		_parameter = parameter.clone();
	}

	///Assignment operator
	ParameterIntegrableBridge& operator= (const ParameterIntegrableBridge& original)
	{
		if (this != &original)
		{
			delete _parameter;
			_parameter = original._parameter->clone();
		}

		return *this;
	}

	///Integral of parameter over [\a time1, \a time2]
	coefficient integral (const coefficient time1, const coefficient time2) const
	{
		return _parameter->integral (time1, time2);
	}

	///Integral of squared parameter over [\a time1, \a time2]
	coefficient integral_square (const coefficient time1, const coefficient time2) const
	{
		return _parameter->integral_square (time1, time2);
	}

	///Square root of integral of squared parameter over [\a time1, \a time2]
	coefficient root_integral_square (const coefficient time1, const coefficient time2) const
	{
		return _parameter->root_integral_square (time1, time2);
	}

	///Current value of parameter
	inline coefficient operator() (const coefficient time) const
	{
		return _parameter->operator() (time);
	}

	~ParameterIntegrableBridge()
	{
		delete _parameter;
	}

private:
	ParameterIntegrable* _parameter;
};

#endif // PARAMETERINTEGRABLEBRIDGE_H
