/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

/**
 \file
 #Line and associated utility functions.
 */

#ifndef LINE_H
#define LINE_H

#include <ostream>

#include "../numbers.h"

///Straight line in two dimensions.
/**
 The line follows the equation y = slope()x + intercept().
 @author Alet Roux <alet.roux@york.ac.uk>
*/
class Line
{
public:
	///Constructor
	Line (const coefficient slope, const coefficient intercept)
			: _slope (slope), _intercept (intercept)
	{
		check();
	}

	///Slope of line
	inline coefficient slope() const;

	///Intercept with vertical axis
	inline coefficient intercept() const;

	/** Shifts line.
	 * @param x Number of units to shift to the right (to the left if negative)
	 * @param y Number of units to shift upwards (downwards if negative)
	 */
	inline void shift (const coefficient x, const coefficient y);

	///Negates both slope() and intersection().
	inline void negate();

	/** Adjust line to have slope \a slope, while leaving its value in \a x unchanged.
	 * @param slope New slope
	 * @param x Point at which value of function should be left unchanged.
	 */
	inline void swivel (const coefficient slope, const coefficient x);

	///Value of line for given horizontal value \a x
	inline coefficient operator() (const coefficient x) const;

	/// Multiplies both slope() and intercept() by \a value.
	inline Line& operator*= (const coefficient value);

	/// Checks whether slope or intercept is zero
	inline void check ();

private:
	coefficient _slope, _intercept;
};

coefficient Line::slope() const
{
	return _slope;
}

coefficient Line::intercept() const
{
	return _intercept;
}

void Line::negate()
{
	_slope = -_slope;
	_intercept = -_intercept;
}

void Line::shift (const coefficient x, const coefficient y)
{
	_intercept += y - slope() *x;
	check();
}

void Line::swivel (const coefficient slope, const coefficient x)
{
	_intercept += x* (_slope-slope);
	_slope = slope;
	check();
}

coefficient Line::operator() (const coefficient x) const
{
	return intercept() + x*slope();
}

Line& Line::operator*= (const coefficient value)
{
	_intercept *= value;
	_slope *= value;
	check();
	return *this;
}

void Line::check()
{
	fixzero (_slope);
	fixzero (_intercept);
}

/// Prints \a line in format line.slope()*x + line.intercept()
inline std::ostream& operator<< (std::ostream& output, const Line& line)
{
	output << line.slope() << "*x";
	if (isnan (line.intercept()) || (line.intercept() > 0))
		output << "+" << line.intercept();
	else if (line.intercept() < 0)
		output << line.intercept();
	return output;
}

/// Tests if \a line1 and \a line2 are parallel. Fudges it a little bit to help with numerics
inline int parallel (const Line &line1, const Line &line2)
{
	return equal (line1.slope(),line2.slope());
};

/// Finds crossing point of two lines. Returns NAN if the lines are parallel.
inline coefficient crossing (const Line &line1, const Line &line2)
{
	if (parallel (line1, line2))
		return NAN;

	if (equal (line1.intercept(),line2.intercept()))
		return 0.0;

	return (line2.intercept()-line1.intercept()) / (line1.slope()-line2.slope());
};

#endif // LINE_H
