/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "piecewiselinear.h"
#include "../config.h"

#include <iostream>
using namespace std;

/**
 \file
 Implementation of class and utility functions in piecewiselinear.h.
 */

void PiecewiseLinear::check (const int objective)
{
#ifdef VERBOSE_PIECEWISELINEAR_CHECK
	cout << endl << "Before checking (from left to right):" << endl;
	for (size_t k = 0; k < size(); k++)
		cout << (*this) [k] << endl;
#endif

//	coefficient next = INFINITY;

	_lines.back().check();

	for (line_array::iterator kline = _lines.end()-1; kline > _lines.begin(); kline--)
	{
#ifdef VERBOSE_PIECEWISELINEAR_CHECK
		cout << "Line " << kline-_lines.begin() << " (" << *kline << ")";
#endif

		(kline-1)->check();

		//checks that lines are not parallel. If not, checks whether intersections are in decreasing order.
		if (parallel (* (kline-1), *kline))
		{
#ifdef VERBOSE_PIECEWISELINEAR_CHECK
			cout << " parallel to line " << kline-1-_lines.begin() << " (" << * (kline-1) << "). ";
#endif

			int test = (kline->intercept() > (kline-1)->intercept());

			if (objective == PIECEWISELINEAR_CHECK_MINIMUM)
				test = 1-test;

			if (test)
			{
#ifdef VERBOSE_PIECEWISELINEAR_CHECK
                cout << "Deleting " << kline-1-_lines.begin() << "." << endl;
#endif
				kline = _lines.erase (kline-1) + 1;
			}
			else
			{
#ifdef VERBOSE_PIECEWISELINEAR_CHECK
                cout << "Deleting " << kline-_lines.begin() << "." << endl;
#endif
				kline = _lines.erase (kline);
			}
		}
		else
		{
			coefficient next;
			if (kline < _lines.end()-1)
				next = crossing (*kline,* (kline+1));
			else
				next = INFINITY;
			coefficient previous = crossing (* (kline-1),*kline);

#ifdef VERBOSE_PIECEWISELINEAR_CHECK
			cout << " on [" << previous << "," << next << "]. ";
#endif

			if (greaterthan (next, previous))
			{
#ifdef VERBOSE_PIECEWISELINEAR_CHECK
				cout << "It is kept." << endl;
#endif
			}
			else
			{
#ifdef VERBOSE_PIECEWISELINEAR_CHECK
				cout << "It is deleted." << endl;
#endif
				kline = _lines.erase (kline) + 1;
			}
		}
	}

	_intersections.clear();
	_intersections.push_back (-INFINITY);
	for (size_t k = 0; k < size()-1; k++)
		_intersections.push_back (crossing ( (*this) [k], (*this) [k+1]));
	_intersections.push_back (INFINITY);

#ifdef VERBOSE_PIECEWISELINEAR_CHECK
	cout << "After checking" << endl;
	cout << *this << endl;
#endif
}

void PiecewiseLinear::restrict_slope (const coefficient lowest, const coefficient highest)
{
#ifdef VERBOSE_PIECEWISELINEAR_RESTRICT_SLOPE
	cout << endl << "Before restriction to [" << lowest << "," << highest << "]:" << endl << *this << endl;
#endif

#ifdef USE_ASSERT
	assert (slope (size()-1) >= lowest);
	assert (slope (0) <= highest);
#endif

	for (size_t k = 0; k < size(); k++)
	{
#ifdef VERBOSE_PIECEWISELINEAR_RESTRICT_SLOPE
		cout << "Line " << k << " (" << _lines[k] << ") ";
#endif

		if (slope (k) < lowest)
		{
			_lines[k].swivel (lowest, intersection (k+1));
#ifdef VERBOSE_PIECEWISELINEAR_RESTRICT_SLOPE
			cout << "is too flat. It becomes " << _lines[k] << "." << endl;
#endif
		}
		else
			if (slope (k) > highest)
			{
				_lines[k].swivel (highest, intersection (k));
#ifdef VERBOSE_PIECEWISELINEAR_RESTRICT_SLOPE
				cout << "is too steep. It becomes " << _lines[k] << "." << endl;
#endif
			}
			else
			{
#ifdef VERBOSE_PIECEWISELINEAR_RESTRICT_SLOPE
				cout << "is within limits." << endl;
#endif
			}
	}

	check (PIECEWISELINEAR_CHECK_MINIMUM);

#ifdef VERBOSE_PIECEWISELINEAR_RESTRICT_SLOPE
	cout << "After restriction to [" << lowest << "," << highest << "]:" << endl << *this << endl;
#endif
}

ostream& operator<< (ostream& output, const PiecewiseLinear& line)
{
	if (line.size() == 1)
		return output << line[0];

	output << "x<" << line.intersection (1) << " ? " << line[0] << " : ";

	for (size_t k = 1; k < line.size()-1; k++)
		output << "x<" << line.intersection (k+1) << " ? " << line[k] << " : ";

	output << line.intersection (line.size()-1) << "<=x" << " ? " << line[line.size()-1] << " : 1/0";

	return output;
}

coefficient crossing (const Line &line, const PiecewiseLinear &function, const coefficient x, const int direction)
{
	size_t k = function.piece (x);

#ifdef VERBOSE_PIECEWISELINEAR_CROSSING
	cout << "The relevant line piece is number " << k << " (" << function[k] << ")." << endl;
	cout << "Searching from " << x << " to the ";
	if (direction == PIECEWISELINEAR_CROSSING_LEFT)
		cout << "left";
	else
		cout << "right";
	cout << "." << endl;
#endif

	coefficient z;

	switch (direction)
	{
	case PIECEWISELINEAR_CROSSING_LEFT:
		k++;
		while (k > 0)
		{
			k--;
			z = crossing (line, function[k]);

#ifdef VERBOSE_PIECEWISELINEAR_CROSSING
			cout << "Line piece " << k << " is " << function[k] << " on [" << function.intersection (k) << "," << function.intersection (k+1) << "]." << endl;
			cout << "Candidate crossing point is " << z << "." << endl;
#endif

			if (!isnan (z) && lessequal (z,x) && greaterequal (z,function.intersection (k)) && lessequal (z,function.intersection (k+1)))
				return z;
		}
		break;

	case PIECEWISELINEAR_CROSSING_RIGHT:
		while (k < function.size())
		{
			z = crossing (line, function[k]);

#ifdef VERBOSE_PIECEWISELINEAR_CROSSING
			cout << "Line piece " << k << " is " << function[k] << " on [" << function.intersection (k) << "," << function.intersection (k+1) << "]." << endl;
			cout << "Candidate crossing point is " << z << "." << endl;
#endif

			if (!isnan (z) && lessequal (x,z) && greaterequal (z,function.intersection (k)) && lessequal (z,function.intersection (k+1)))
				return z;
			k++;
		}
		break;

	default:
		return NAN;
	}

	return NAN;
}

PiecewiseLinear maximum (PiecewiseLinear& first, PiecewiseLinear& second)
{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
	cout << "Before computing the maximum:" << endl;
	cout << "First function: " << first << endl;
	cout << "Second function: " << second << endl;
#endif

	line_array max;

	//To keep track of where we are
	PiecewiseLinear *winner, *loser;

	size_t kfirst = 0;
	size_t ksecond = 0;
	size_t *kwinner, *kloser;

	size_t addfirst = 0;
	size_t addsecond = 0;
	size_t *addwinner, *addloser;

	coefficient lpoint = min (first.intersection (1), second.intersection (1));
	coefficient test = crossing (first[0], second[0]);

#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
	if (isnan (test))
		cout << "Left-hand line pieces (" << first[0] << ", " << second[0] << ") are parallel." << endl;
	else
	{
		cout << "Left-hand line pieces (" << first[0] << ", " << second[0] << ") crossing at " << test << "." << endl;
		cout << "Af the first intersection point " << lpoint << " their values are " << first[0] (lpoint) << " and " << second[0] (lpoint) << "." << endl;
	}
#endif

	//to start off with
	if ( ( (second.slope (0) > first.slope (0)) && greaterthan (test,lpoint))
			|| ( (second.slope (0) > first.slope (0)) && lessequal (test,lpoint))
			|| (isnan (test) && (second.intercept (0) < first.intercept (0))))
	{
		winner = &first;
		kwinner = &kfirst;
		addwinner = &addfirst;
		loser = &second;
		kloser = &ksecond;
		addloser = &addsecond;
	}
	else
	{
		winner = &second;
		kwinner = &ksecond;
		addwinner = &addsecond;
		loser = &first;
		kloser = &kfirst;
		addloser = &addfirst;
	}

	if (loser->size() > 1)
	{
		if (lessthan (loser->intersection (1), winner->intersection (1)) && lessthan (loser->intersection (1), test))
			*addloser=1;
//		else if (equal (loser->slope (0), winner->slope (0)))
//			*addloser = 1;
	}

	do
	{
		//what should have been added at the end of the previous step
		*kwinner += *addwinner;
		*kloser += *addloser;
		*addloser = 0;
		*addwinner = 0;

		//first add current piece
		if ( (max.size() == 0) || !parallel (max.back(), (*winner) [*kwinner]))
			max.push_back ( (*winner) [*kwinner]);

#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
		cout << "Running maximum:" << endl;
		for (size_t k = 0; k < max.size(); k++)
			cout << max[k] << endl;
#endif

		//left- and right-hand endpoints of intervals under which the current lines are defined
		coefficient rwinner = winner->intersection (*kwinner+1);
		coefficient lwinner = winner->intersection (*kwinner);
		coefficient rloser = loser->intersection (*kloser+1);
		coefficient lloser = loser->intersection (*kloser);

#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
		cout << "Current line pieces under consideration:" << endl;
		cout << "\"winner\": " << *kwinner << " of " << winner->size()-1 << " (" << (*winner) [*kwinner] << ") on [" << lwinner << "," << rwinner << "]" << endl;
		cout << "\"loser\": " << *kloser << " of " << loser->size()-1 << " (" << (*loser) [*kloser] << ") on [" << lloser << "," << rloser << "]" << endl;
		cout << "Verdict: ";
#endif

		//domains of two line pieces are disjoint: winner has passed
		if (lessthan (rwinner,lloser) || (equal (rwinner,lloser) && greaterthan ( (*winner) [*kwinner] (rwinner), (*loser) [*kloser] (lloser))))
		{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
			cout << "Domains of line pieces disjoint (winner has passed)." << endl;
#endif
			*addwinner=1;
		}
		//domains of two line pieces are disjoint: loser has passed
		else if (greaterthan (lwinner,rloser) || (equal (rloser,lwinner) && greaterthan ( (*winner) [*kwinner] (lwinner), (*loser) [*kloser] (rloser))))
		{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
			cout << "Domains of line pieces disjoint (loser has passed)." << endl;
#endif
			*addloser=1;
		}
		//domains of two line pieces intersect in more than one point
		else
		{
			//possibility of an upcrossing
			if (greaterthan (loser->slope (*kloser), winner->slope (*kwinner)))
			{
				//should we switch sides?
				int doswitch = 0;

				//next intersection of the two current line pieces
				test = crossing (max.back(), (*loser) [*kloser]);

#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
				cout << "intersection at " << test;
#endif

				//intersection falls in interior of domain of both winner and loser`
				if (greaterthan (test,lwinner) && greaterthan (test,lloser) && lessthan (test,rloser) && lessthan (test,rwinner))
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << " (in interior)" << endl;
#endif
					doswitch = 1;
					if (rwinner < rloser)
						*addwinner=1;
				}
				//intersection falls on left hand side of loser and interior of winner
				else if (equal (test,lloser) && lessthan (lwinner,lloser))
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << " (on left-hand point of loser)." << endl;
#endif
					doswitch=1;
				}
				//intersection falls on left hand side of winner and interior of loser
				else if (equal (test,lwinner) && lessthan (lloser,lwinner))
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << " (on left-hand point of winner)." << endl;
#endif
					doswitch=1;
				}
				//intersection falls on right hand side of loser and interior of winner
				else if (equal (test,rloser) && lessthan (rloser,rwinner))
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << " (on right-hand point of loser)." << endl;
#endif
					if (loser->slope (*kloser+1) > winner->slope (*kwinner))
						doswitch = 1;
					*addloser=1;
				}
				//intersection falls on right hand side of winner and interior of loser
				else if (equal (test,rwinner) && lessthan (rwinner,rloser))
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << " (on right-hand point of winner)." << endl;
#endif
					if (loser->slope (*kloser) > winner->slope (*kwinner+1))
						doswitch = 1;
					*addwinner=1;
				}
				//intersection on mutual right-hand side
				else if (isfinite (test) && equal (test,rwinner) && equal (test,rloser))
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << " (on mutual right-hand side)." << endl;
#endif
					if (loser->slope (*kloser+1) > winner->slope (*kwinner+1))
						doswitch = 1;
					*addwinner=1;
					*addloser=1;
				}
				//intersection on mutual left-hand side
				else if (isfinite (test) && equal (test,lwinner) && equal (test,lloser))
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << " (on mutual left-hand side)." << endl;
#endif
					doswitch = 1;
					//*addwinner=1;
					//*addloser=1;
				}
				else
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << " (not relevant)." << endl;
#endif
					if (rloser < rwinner)
						*addloser=1;
					else if (isfinite (rwinner))
						*addwinner=1;
				}

#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
				cout << "Decision: ";
#endif

				if (doswitch)
				{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
					cout << "switching winner and loser." << endl;
#endif

					if (winner == &second)
					{
						winner = &first;
						kwinner = &kfirst;
						addwinner = &addfirst;
						loser = &second;
						kloser = &ksecond;
						addloser = &addsecond;
					}
					else
					{
						winner = &second;
						kwinner = &ksecond;
						addwinner = &addsecond;
						loser = &first;
						kloser = &kfirst;
						addloser = &addfirst;
					}
				}
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
				else
					cout << "not switching winner and loser." << endl;
#endif
			}
			//domains of line pieces intersect, but not an upcrossing
			else
			{
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
				cout << "domains intersect, but loser still loses." << endl;
#endif
				if (lessequal (rloser,rwinner))
					*addloser=1;
				else if (isfinite (rwinner))
					*addwinner=1;
			}
		}
	}
	while ( (*kwinner < winner->size()-1) || (*kloser < loser->size()-1));

	//add final piece if necessary (this may be needed if there was a switch on the final segment)
	if (!parallel (max.back(), (*winner) [*kwinner]))
		max.push_back ( (*winner) [*kwinner]);

	PiecewiseLinear maximum (max);
#ifdef VERBOSE_PIECEWISELINEAR_MAXIMUM
	cout << "Maximum: " << maximum << endl;
#endif

#ifdef USE_ASSERT
	assert (maximum >= first);
	assert (maximum >= second);
#endif

	return maximum;
}

void print_convex_conjugate (const PiecewiseLinear& line)
{
	if (line.size() == 1)
	{
		cout << "x == " << -line.slope (0) << " ? " << line.intercept (0) << " : 1/0";
		return;
	}

#ifdef USE_ASSERT
	assert (line.convex());
#endif

	cout << "x<= " << -line.slope (0) << "? ";

	for (size_t k = 0; k < line.size()-1; k++)
		cout << -line.slope (k+1) << "<=x ? " << Line (line.intersection (k+1), line.intersection (k+1) *line.slope (k) + line.intercept (k)) << " : ";

	cout << "1/0 : 1/0";
}

void untangle_maximum (const PiecewiseLinear& f, const vector<PiecewiseLinear>& g, const coefficient y, coefficient& w, vector<coefficient>& p, vector<coefficient>& x)
{
#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
	cout << "Maximum to be untangled: " << endl << f << endl;
	cout << "input w: " << w << ", input y: " << y << endl;
#endif

	p.reserve (g.size());
	x.reserve (g.size());

	size_t left = 0;
	size_t right;

	if (f.size() == 1)
	{
		right = left;
		w = 0;
	}
	else
	{
		while ( (left < f.size()-1) && lessthan (f.slope (left+1),-y))
			left++;

		if ( (left < f.size()-1) && equal (-y, f.slope (left+1)))
		{
			left++;
			right = left;
		}
		else
			right = left+1;

		if (right > left)
			w = f.intersection (right);
		else if (lessequal (f.intersection (left), w) && lessequal (w,f.intersection (left+1)))
			w = w;
		else
			w = 0.5* isinf (f.intersection (left)) ? (f.intersection (1)-1) : f.intersection (left) + 0.5* isinf (f.intersection (left+1)) ? (f.intersection (f.size()-1) +1) : f.intersection (left+1);
	}

#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
	cout << "w: " << w << ", y: " << y;
	if (left == right)
		cout << " (slope " << left << ")";
	else
		cout << " (slope " << left << ": " << f.slope (left) << ", slope " << right << ": " << f.slope (right) << ")";
	cout << endl;
#endif

#ifdef USE_ASSERT
	assert (right < f.size());
#endif
	//We have -y = q f.slope(left) + (1-q)f.slope(right) for some q in [0,1]
	//The tricky question is: Which line pieces do these slopes belong to?
	coefficient q;
	if (left == right)
		q = 0.5;
	else
		q = (-y-f.slope (right)) / (f.slope (left) - f.slope (right));

	//Total weight assigned. In case the slope of g-functions are the same, we simply assign equal weights, and standardize later if needed.
	coefficient weight = 0.0;

	for (size_t k = 0; k < g.size(); k++)
	{
#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
		cout << "g[" << k << "]: " << g[k] << endl;
		//  cout << "Convex conjugate of g[" << k << "]: " << endl;
		//   print_convex_conjugate (g[k]);
		//  cout << endl;
		cout << "Result: ";
#endif

		//this is so that we can later check whether it has been changed
		p[k]=0.0;

		size_t k_right = g[k].piece (w);

		if (equal (g[k] (w),f (w)))
		{
			size_t k_left;
			if (equal (w, g[k].intersection (k_right)))
				k_left = k_right - 1;
			else
				k_left = k_right;

#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
			if (k_left == k_right)
				cout << "slope " << k_left;
			else
				cout << "slope " << k_left << ": " << g[k].slope (k_left) << ", slope " << right << ": " << g[k].slope (k_right);
			cout << ".";
#endif

			if (equal (g[k].slope (k_left), f.slope (left)))
			{
#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
				cout << " LHS contributes.";
#endif
				x[k] = -g[k].slope (k_left);
				p[k] = q;
				weight += q;
			}

			if (equal (g[k].slope (k_right), f.slope (right)))
			{
#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
				cout << " RHS contributes.";
#endif
				//need to check whether left-hand side contributes too
				if (equal (p[k],0.0))
				{
					x[k]=-g[k].slope (k_right);
					p[k]=1-q;
				}
				else
				{
					x[k]=y;
					p[k]=1;
				}
				weight += 1-q;
			}
		}

		//this happens if g[k](w) < f (w) *OR* if g[k] (w) == f (w) but the slopes of g[k] and f do not match.
		if (equal (p[k],0.0))
		{
			x[k] = -g[k].slope (k_right);
#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
			cout << " No contribution.";
#endif
		}


#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
		cout << " x[" << k << "]: " << x[k] << ", p[" << k << "]: " << p[k] << endl;
#endif
	}

	if (!equal (weight,1.0))
	{
#ifdef VERBOSE_PIECEWISELINEAR_UNTANGLE_MAXIMUM
		cout << "Total weight assigned " << weight << ". Standardizing..." << endl;
#endif
		for (size_t k = 0; k < g.size(); k++)
			p[k] /= weight;
	}
}


