/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/


#ifndef PIECEWISELINEAR_H
#define PIECEWISELINEAR_H

/**
 \file
#PiecewiseLinear and utility functions.
*/

#include <vector>

#include "line.h"

///STL container used to hold lines
typedef std::vector<Line> line_array;

//#include <iostream>

///Removes parallel lines by using maximum as objective (makes epigraph as small as possible).
#define PIECEWISELINEAR_CHECK_MAXIMUM 1

///Removes parallel lines by using minimum as objective (makes epigraph as large as possible).
#define PIECEWISELINEAR_CHECK_MINIMUM 2

///Piecewise linear (affine) function.
/**
A #PiecewiseLinear function is a collection of Line objects, arranged from left to right on the horizontal axis.
 @author Alet Roux <alet.roux@york.ac.uk>
*/
class PiecewiseLinear
{
public:
	///Default constructor, encapsulates Line
	PiecewiseLinear (const Line &line = Line (NAN, NAN))
	{
		_lines.push_back (line);
		check();
	}

	///Constructs PiecewiseLinear from \a lines, checks for errors in input data and precomputes intersections between lines.
	PiecewiseLinear (const line_array &lines)
			: _lines (lines)
	{
		check();
	}

	///A copy of the \a k-th line.
	Line operator[] (const size_t k) const
	{
		return _lines[k];
	}

	///Evaluates piecewise linear function at \a x
	coefficient operator() (const coefficient x) const
	{
		return _lines[piece (x) ] (x);
	}

	/**
	 * Finds line piece on which \a x appears. If \a x == intersection(k) for some \f$ k \f$ then we choose piece(x) to be \f$ k \f$.
	 */
	size_t piece (const coefficient x) const
	{
		size_t k = 0;
		while (greaterequal (x,intersection (k+1)))
			k++;
		return k;
	}

	///Multiplies all lines and intercepts by \a value
	PiecewiseLinear& operator*= (const coefficient value)
	{
		for (size_t k = 0; k < size(); k++)
			_lines[k] *= value;
		return *this;
	}

	///Assignment operator
	PiecewiseLinear& operator= (const line_array &lines)
	{
		_lines = lines;
		check();
		return *this;
	}

	///Comparison
	int operator>= (const PiecewiseLinear& function)
	{
		if (function.slope (0) < slope (0))
			return 0;

		if (function.slope (function.size()-1) > slope (size()-1))
			return 0;

		for (size_t k = 1; k < size(); k++)
			if (lessthan (operator() (intersection (k)),function (intersection (k))))
				return 0;

		for (size_t k = 1; k < function.size(); k++)
			if (lessthan (operator() (function.intersection (k)),function (function.intersection (k))))
				return 0;

		return 1;
	}

	///Number of line pieces
	size_t size() const
	{
		return _lines.size();
	}

	/** The \a k -th intersection. This is the left-hand point of line \a k, and the right-hand point of line \a k -1.
	 */
	inline coefficient intersection (const size_t k) const
	{
		return _intersections[k];
	}

	///The slope of line \a k
	coefficient slope (const size_t k) const
	{
		return _lines[k].slope();
	}

	///The intercept of line \a k with the vertical axis
	coefficient intercept (const size_t k) const
	{
		return _lines[k].intercept();
	}

	/** Shifts graph of function.
	 * @param x Number of units to shift to the right (to the left if negative)
	 * @param y Number of units to shift upwards (downwards if negative)
	 */
	void shift (const coefficient x, const coefficient y)
	{
		for (size_t k = 0; k < size(); k++)
			_lines[k].shift (x, y);

		for (size_t k = 1; k < size(); k++)
			_intersections[k] = _intersections[k] + x;
	}

	///Negates both slope and intersection.
	void negate()
	{
		for (size_t k = 0; k < size(); k++)
			_lines[k].negate();
	}

	/** Restricts slope of line pieces to be between \a lowest and \a highest. Has the effect of enlarging the epigraph.
	 * @param lowest Lowest of the gradients
	 * @param highest Highest of the gradients
	 */
	void restrict_slope (const coefficient lowest, const coefficient highest);

	///Tests whether this function is convex or not.
	int convex() const
	{
		for (size_t k = 1; k < size(); k++)
			if (lessthan (slope (k),slope (k-1)))
				return 0;
		return 1;
	}

protected:
	/** Performs sanity check on lines. Removes adjacent parallel lines with eye on \a objective, and precomputes array of intersections of lines.
	- If \a objective is CHECK_MAXIMUM, the objective is to maximise the value of the function (and minimize the size of the epigraph).
	- If \a objective is CHECK_MINIMUM, the objective is to minimise the value of the function (or to maximize the size of the epigraph).
	 */
	void check (const int objective = PIECEWISELINEAR_CHECK_MAXIMUM);

private:
	line_array _lines;

	std::vector<coefficient> _intersections;
};

///Outputs \a line in format that can be copied and pasted into Gnuplot (http://www.gnuplot.info/).
std::ostream& operator<< (std::ostream& output, const PiecewiseLinear& line);

///Search to the left for a crossing point between a line and a piecewise linear function
#define PIECEWISELINEAR_CROSSING_LEFT -1

///Search to the right for a crossing point between a line and a piecewise linear function
#define PIECEWISELINEAR_CROSSING_RIGHT 1

/// Finds a crossing point between \a line and \a function, starting at \a x and searching in the given direction. Returns NAN if it cannot be found. Intersections between lines and piecewise linear functions are not necessarily unique, and by no means guaranteed.
coefficient crossing (const Line &line, const PiecewiseLinear &function, const coefficient x = -INFINITY, const int direction = PIECEWISELINEAR_CROSSING_RIGHT);

/**
 * Maximum of two piecewise linear functions. The original functions are not touched, the absence of const is for technical reasons. It is safe to assign the returned function to either of the functions that was passed as a parameter.
 * @param first The first function
 * @param second The second function
 */
PiecewiseLinear maximum (PiecewiseLinear &first, PiecewiseLinear &second);

/**
 * Minimum of two piecewise linear functions. The original functions are not touched, the absence of const is for technical reasons. It is safe to assign the returned function to either of the functions that was passed as a parameter. This function is currently a wrapper around #maximum.
 * @param first The first function
 * @param second The second function
 */
inline PiecewiseLinear minimum (PiecewiseLinear &first, PiecewiseLinear &second)
{
	first.negate();
	second.negate();

	PiecewiseLinear minimum = maximum (first, second);

	first.negate();
	second.negate();
	minimum.negate();

	return minimum;
}

///Computes and prints convex conjugate of \a line.
void print_convex_conjugate (const PiecewiseLinear& line);

/**
 * Service method. Suppose given a function \a f that is the maximum of convex functions \f$ g_0,\ldots,g_n \f$ (not checked) and a number \a y such that \f$ f^L(w) \le -y \le f^R(w)\f$ for some number \a w, where superscripts \f$^L\f$ and \f$^R\f$ refer to left and right derivatives. This function returns this number \a w, numbers \f$ p_1,\ldots,p_n \f$ in the interval \f$[0,1]\f$ together with numbers \f$ x_0,\ldots,x_n \f$ such that \f$ f(w) = \sum_{k=0}^np_kg_k(w)\f$, \f$-y = \sum_{k=0}^np_kx_k \f$, \f$ f^L(w) \le -y \le f^R(w)\f$ and
\f$ g_k^L(w) \le -x_k \le g_k^R(w)\f$ for all \f$ k = 0,\ldots,n \f$. Any input value for \a w is considered as a first guess for \a w (but this input value is checked for correctness).
 */
void untangle_maximum (const PiecewiseLinear& f, const std::vector<PiecewiseLinear>& g, const coefficient y, coefficient& w, std::vector<coefficient>& p, std::vector<coefficient>& x);

#endif // PIECEWISELINEAR_H
