/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/
#ifndef PORTFOLIO_H
#define PORTFOLIO_H

#include "spot.h"
#include "shortrate.h"

/**
 \file
#Portfolio and utility function.
 */


/// Holdings of cash and stock.
class Portfolio
{
public:
	Portfolio (const coefficient shares = NAN, const coefficient cash = NAN)
			: _cash (cash),_shares (shares)
	{
		check();
	}

	///Amount of cash in the portfolio
	coefficient cash() const
	{
		return _cash;
	}

	///Number of shares in the portfolio
	coefficient shares() const
	{
		return _shares;
	}

	///Assignment operator
	Portfolio& operator= (const Portfolio& portfolio)
	{
		_cash = portfolio.cash();
		_shares = portfolio.shares();
		check();
		return *this;
	}

	///Negation operator
	Portfolio operator- () const
	{
		return Portfolio (-_shares, -_cash);
	}

	///Addition operator
	Portfolio& operator+= (const Portfolio& portfolio)
	{
		_cash += portfolio.cash();
		_shares += portfolio.shares();
		check();
		return *this;
	}

	///Multiplication operator
	Portfolio& operator*= (const coefficient scale)
	{
		_cash *= scale;
		_shares *= scale;
		check();
		return *this;
	}

	///Buys (if positive) or sell (if negative) \a x shares. Cash is adjusted in self-financing way by using \a spot.
	void buy (const coefficient s, const Spot& spot)
	{
		_shares += s;
		_cash -= (s > 0) ? s*spot.ask() : s*spot.bid();
		check();
	}

	///Cash accumulation at the rate \a R -1.
	void accumulate (const coefficient R)
	{
		_cash *= R;
		fixzero (_cash);
	}

	///Accumulates interest between \a t1 and \a t2 at the rate \a r
	void accumulate (const ShortRate& r, const coefficient t1, const coefficient t2)
	{
		accumulate (r.accumulate (t1,t2));
	}

	///computes notional value for a given notional share value
	coefficient value (const coefficient spot) const
	{
	  return cash() + spot*shares();
	}

private:
	///Checks whether values are negligible (but not zero), and if they are, fixes them.
	void check()
	{
		fixzero (_cash);
		fixzero (_shares);
	}

	coefficient _cash,_shares;
};

std::ostream& operator<< (std::ostream& output, const Portfolio& portfolio);

#endif // PORTFOLIO_H
