/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef STATISTICSGATHERER_PATH_H
#define STATISTICSGATHERER_PATH_H

#include <vector>
#include <ostream>
#include "initial.h"

///Collects paths of objects of interest.
template<class T>
class StatisticsGathererPath : public StatisticsGatherer<T>
{
public:
	/**
	 * Constructor.
	 * @param path Information is gathered for this sequence of nodes. The element \a path[t] is a node at the \f$ t \f$-th time step.
	 */
	StatisticsGathererPath (const std::vector< size_t >& path)
			: _path (path), _object (path.size())
	{}

	operator StatisticsGathererInitial<T>() const
	{
		StatisticsGathererInitial<T> gatherer;
		gatherer.dump_result (0, 0, _object[0]);

		return gatherer;
	}

	operator T() const
	{
		return _object[0];
	}

	virtual void dump_result (const size_t t, const size_t k, const T& object)
	{
		if (_path[t] == k)
			_object[t] = object;
	}

	///Node at time step \a t.
	size_t node (const size_t t) const
	{
		return _path[t];
	};

	T operator[] (const size_t t) const
	{
		return _object[t];
	};

	T& operator[] (const size_t t)
	{
		return _object[t];
	};

	///Number of steps
	size_t size() const
	{
		return _path.size();
	}

private:
	std::vector<size_t> _path;
	std::vector<T> _object;
};

template<class T>
inline std::ostream& operator<< (std::ostream& output, const StatisticsGathererPath<T>& function)
{
	for (size_t k = 0; k < function.size(); k++)
		output << "(" << k << "," << function.node (k) << "): " << function[k] << std::endl;
	return output;
}


#endif // STATISTICSGATHERER_PATH_H
