/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/
#ifndef TRADING_H
#define TRADING_H

/**
 \file
Utility functions for trading and solvency.
 */

#include "config.h"
#include "piecewiselinear/piecewiselinear.h"
#include "spot.h"
#include "portfolio.h"

///Returns a piecewise linear convex function whose epigraph is the solvency cone.
inline PiecewiseLinear solvency_cone (const Spot& spot)
{
	line_array newfunction;
	newfunction.push_back (Line (-spot.ask(),0));
	if (spot.ask() > spot.bid())
		newfunction.push_back (Line (-spot.bid(), 0));
	return PiecewiseLinear (newfunction);
}

///Returns a piecewise linear convex function whose epigraph is the portfolios from which one can afford \a portfolio at the price \a spot
inline PiecewiseLinear portfolios_that_can_afford (const Portfolio& portfolio, const Spot& spot)
{
	PiecewiseLinear current = solvency_cone ( spot );
	current.shift ( portfolio.shares(), portfolio.cash() );
	return current;
}

/**
 * Restricts a set of portfolios to ensure that they are solvent.
 * @param portfolios Piecewise linear function forming lower bound of collection of portfolios
 * @param spot Current spot price
 */
inline void ensure_solvent (PiecewiseLinear& portfolios, const Spot& spot)
{
	PiecewiseLinear reference = solvency_cone (spot);
	portfolios = maximum (portfolios, reference);
}

/**
 * Enlarges a set of portfolios to contain all solvent portfolios
 * @param portfolios Piecewise linear function forming lower bound of collection of portfolios
 * @param spot Current spot price
 */
inline void include_solvent_portfolios (PiecewiseLinear& portfolios, const Spot& spot)
{
	PiecewiseLinear reference = solvency_cone (spot);
	portfolios = minimum (portfolios, reference);
}

///Checks whether \a portfolio can be converted into a nonnegative amount of cash when the current share price is \a spot.
inline int soluble (const Portfolio& portfolio, const Spot& spot)
{
	return greaterequal (portfolio.cash()- (portfolio.shares() >0.0) ? portfolio.shares() *spot.bid() : portfolio.shares() *spot.ask() ,0.0);
}

/// Adjusts \a portfolio, which is assumed to be in the epigraph of portfolios.restrict(-\a spot.ask(), -\a spot.bid()) (this is not tested!), in a self-financing way so that it is in the epigraph of \a portfolios.
inline void untangle_restriction (Portfolio& portfolio, const PiecewiseLinear& portfolios, const Spot& spot)
{
#ifdef VERBOSE_UNTANGLE_RESTRICTION
		std::cout << "Unrestricted portfolios: " << portfolios << "." << std::endl;
		std::cout << "Spot: " << spot << std::endl;
#endif
	//This portfolio fell out of the epigraph in the process of slope restriction
	if (lessthan (portfolio.cash(), portfolios (portfolio.shares())))
	{
		size_t k = portfolios.piece (portfolio.shares());
		coefficient new_shares;

		//Slope too steep, move to the right.
		if ( (k == 0) || lessthan (portfolios.slope (k),-spot.ask()))
		{
			new_shares = crossing (Line (-spot.ask(),portfolio.cash() +spot.ask() *portfolio.shares()),portfolios,portfolio.shares(),PIECEWISELINEAR_CROSSING_RIGHT);
		}
		else
		{
			new_shares = crossing (Line (-spot.bid(),portfolio.cash() +spot.bid() *portfolio.shares()),portfolios,portfolio.shares(),PIECEWISELINEAR_CROSSING_LEFT);
		}
		portfolio.buy (new_shares - portfolio.shares(), spot);
#ifdef VERBOSE_UNTANGLE_RESTRICTION
		std::cout << "Portfolio becomes " << portfolio << "." << std::endl;
#endif
	}
	//portfolio hedges fine
	else
	{
#ifdef VERBOSE_UNTANGLE_RESTRICTION
		std::cout << "Portfolio remains unchanged." << std::endl;
#endif
	}
}

#endif
