/*
    American option pricer under proportional transaction costs
    Copyright (C) 2011 Alet Roux alet.roux@york.ac.uk

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef TREEPRODUCT_H
#define TREEPRODUCT_H

#include "../config.h"
//#include "../payoff/bridge.h"
#include "../trading.h"

///Virtual base class for path-independent European and American options.
class TreeProduct
{
public:
	TreeProduct ( const coefficient expiry)	: _expiry ( expiry )
	{
	#ifdef USE_ASSERT
		assert ( expiry > 0 );
	#endif
	}

  	/**
	 * Finds set of portfolios that hedges the option at its expiry time.
	 * @param spot Price of underlying asset
	 * @return Piecewise linear function that gives the lower boundary of set of hedging portfolios
	 */
	virtual PiecewiseLinear final_hedge (const Spot& spot) const = 0;

	/**
	 * Finds set of portfolios that hedges the payoff of the option at time \a t and in the future
	 * @param t Time
	 * @param spot Price of underlying asset
	 * @param future_hedge Piecewise linear function whose epigraph consist of portfolios that hedge the option at all future time steps
	 * @return Piecewise linear function that gives the lower boundary of set of hedging portfolios
	 */
	virtual PiecewiseLinear interim_hedge (const coefficient t, const Spot& spot, const PiecewiseLinear& future_hedge) const = 0;

	/**
	 * Computes current price of option, given the holdings of the investor
	 * @param current Lower boundary of set of hedging portfolios
	 * @param holdings Current holdings of investor
	 */
	virtual coefficient current_price (const PiecewiseLinear& current, const Portfolio& holdings = Portfolio (0,0)) const = 0;

	/**
	 * Given a portfolio that hedges at the current time, computes a portfolio into which the trader can enter that hedges at all future times
	 * @param t Time
	 * @param spot Price of underlying asset
	 * @param future_hedge Piecewise linear function whose epigraph consist of portfolios that hedge the option at all future time steps
	 * @param current_portfolio Current portfolio. Must correspond to a point in the epigraph of #interim_hedge(\a t, \a spot, \a future_hedge) (this is not checked).
	 * @return Portfolio that will hedge the product over the next time step.
	 */
	Portfolio next_portfolio (const coefficient t, const Spot& spot, const PiecewiseLinear& future_hedge, const Portfolio& current_portfolio) const
	{
#ifdef USE_ASSERT
		assert (t < expiry());
#endif
		Portfolio portfolio (current_portfolio);
		untangle_restriction (portfolio, future_hedge, spot);
		return portfolio;
	}

	///Expiry time
	coefficient expiry() const { return _expiry;}

private:
	coefficient _expiry;
};

inline coefficient buyers_price(const PiecewiseLinear& current, const Portfolio& holdings)
{
	return - current ( holdings.shares() ) - holdings.cash();
};

inline coefficient sellers_price(const PiecewiseLinear& current, const Portfolio& holdings)
{
	return current ( holdings.shares() ) - holdings.cash();
};

#endif // TREEPRODUCT_H
