package mutation_system_for_jason;

import jason.JasonException;
import jason.asSemantics.Event;
import jason.asSemantics.Intention;
import jason.asSemantics.Message;
import jason.asSemantics.TransitionSystem;
import jason.asSyntax.Atom;
import jason.asSyntax.ListTerm;
import jason.asSyntax.ListTermImpl;
import jason.asSyntax.Literal;
import jason.asSyntax.Plan;
import jason.asSyntax.PlanBody;
import jason.asSyntax.PlanBodyImpl;
import jason.asSyntax.PlanLibrary;
import jason.asSyntax.Rule;
import jason.asSyntax.Term;
import jason.asSyntax.Trigger;
import jason.asSyntax.PlanBody.BodyType;
import jason.asSyntax.Trigger.TEOperator;
import jason.asSyntax.Trigger.TEType;
import jason.infra.centralised.CentralisedAgArch;
import jason.infra.centralised.CentralisedEnvironment;
import jason.infra.centralised.CentralisedExecutionControl;
import jason.infra.centralised.RunCentralisedMAS;
import jason.mas2j.MAS2JProject;
import jason.runtime.MASConsoleGUI;

import java.awt.Frame;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import javax.swing.JFrame;

public class MutationSystem {
	
	private RunCentralisedMAS mas;
	private String[] args;
	
	private TestBed tb;
	
	private ArrayList<MutationOperator> mops;
	
	private JFrame myFrame;
	private MutationSystemPane mutationSystemPane;
	
	public MutationSystem(String[] args) throws JasonException {
		String[] args0 = {args[0]};
		this.args = args0;
		tb = new TestBed();
		initMAS();
		initMops();
		initComponents();
		mutationSystemPane.updateMOPs(mops);
		mutationSystemPane.updateAgents(getAgNamesFromInfra());
	}
	
	public static void main(String[] args) throws JasonException {
		new MutationSystem(args);
	}
	
	public void restartMAS() {
		finishMAS();
		try {
			initMAS();
		} catch (JasonException e) {}
	}
	
	private void initMAS() throws JasonException {
		try {
			Field lastPlanLabel = PlanLibrary.class.getDeclaredField("lastPlanLabel");
			lastPlanLabel.setAccessible(true);
			lastPlanLabel.set(null,new AtomicInteger(0));
		} catch (Exception e) {}
		mas = new RunCentralisedMAS();
		mas.init(args);
		create();
		runEmptyCycle();
		mas.start();
	}
	
	private void create() throws JasonException {
		createEnvironment();
		mas.createAgs();
		mas.createController();
	}
	
	private void createEnvironment() throws JasonException {
        try {
    		Field fEnv = mas.getClass().getDeclaredField("env");
    		fEnv.setAccessible(true);
    		
    		Field fProject = mas.getClass().getDeclaredField("project");
    		fProject.setAccessible(true);
    		MAS2JProject project = (MAS2JProject) fProject.get(mas);
    		
    		fEnv.set(mas, new CustomCentralisedEnvironment(project.getEnvClass(), mas));
        }catch(Exception e) {
        	e.printStackTrace();
        }
	}
	
	private void finishMAS() {
        try {
            System.out.flush();
            System.err.flush();

            if (MASConsoleGUI.hasConsole()) {
                MASConsoleGUI.get().close();
            }
            
    		Map<String, CentralisedAgArch> agents = mas.getAgs();
    		Set<String> keys = agents.keySet();
    		Iterator<String> agentNames = keys.iterator();
    		while(agentNames.hasNext()) {
    			String agentName = agentNames.next();
    			mas.getAg(agentName).stopAg();
    			mas.delAg(agentName);
    		}
            
    		CentralisedExecutionControl control = mas.getControllerInfraTier();
            if (control != null) {
            	control.stop();
            }
            
            CentralisedEnvironment env = mas.getEnvironmentInfraTier();
            if (env != null) {
            	env.stop();
            }
            
    		Frame[] frames = JFrame.getFrames();
    		int length = frames.length;
    		for(int i=0;i<length;i++) {
    			if(!frames[i].equals(myFrame)) {
    				frames[i].dispose();
    			}
    		}
    		
            mas = null;
        } catch (Exception e) {
            e.printStackTrace();
        }
	}
	
	private void initMops() {
		mops = new ArrayList<MutationOperator>();
		mops.add(new MutationOperator("Belief Deletion", "agent", 0));
		mops.add(new MutationOperator("Initial Goal Deletion", "agent", 1));
		mops.add(new MutationOperator("Plan Deletion", "agent", 2));
		mops.add(new MutationOperator("Rule Condition Deletion", "agent.belief", 3));
		mops.add(new MutationOperator("Triggering Event Operator Replacement", "agent.plan.trigger", 4));
		mops.add(new MutationOperator("Plan Context Deletion", "agent.plan.context", 5));
		mops.add(new MutationOperator("Plan Body Deletion", "agent.plan.body", 6));
		mops.add(new MutationOperator("Formula Deletion", "agent.plan.body", 7));
		mops.add(new MutationOperator("Formula Order Swap", "agent.plan.body", 8));
		mops.add(new MutationOperator("Formula Operator Replacement", "agent.plan.body.formula", 9));
		mops.add(new MutationOperator("Message Receiver Replacement", "agent.plan.body.formula.comm", 10));
		mops.add(new MutationOperator("Illocutionary Force Replacement", "agent.plan.body.formula.comm", 11));
		mops.add(new MutationOperator("Message Content Deletion", "agent.plan.body.formula.comm", 12));
	}
	
	private void initComponents() {
		mutationSystemPane = new MutationSystemPane(this);
		myFrame = new JFrame("muJason v0.5");
		myFrame.getContentPane().add(mutationSystemPane);
		myFrame.setVisible(true);
		myFrame.pack();
		myFrame.setResizable(false);
	}
	
    private List<String> getAgNamesFromInfra() {
        try {
        	Set<String> agentSet = mas.getControllerInfraTier().getRuntimeServices().getAgentsNames();
        	return new ArrayList<String>(agentSet);
        } catch (Exception e) {
        	e.printStackTrace();
        }
        return null;
    }
    
	private void runEmptyCycle() {
		Map<String, CentralisedAgArch> agents = mas.getAgs();
		Set<String> keys = agents.keySet();
		Iterator<String> agentNames = keys.iterator();
		while(agentNames.hasNext()) {
			String agentName = agentNames.next();
			mas.getAg(agentName).getTS().getC().addEvent(new Event(Trigger.parseTrigger("+emptybelief"), new Intention()));
		}
	}
	
    public void getSelectedAgFromInfra(String ag) {
    	TransitionSystem ts = mas.getAg(ag).getTS();
    	mutationSystemPane.updateBeliefs(ts.getAg().getBB().iterator());
    	mutationSystemPane.updateInitGoals(ts.getC().getEvents().iterator());
    	mutationSystemPane.updatePlans(ts.getAg().getPL().getPlans().iterator());
    }
    
    public void startMutation(MutationDomain mutD, ArrayList<MutationOperator> mops) {
    	Thread mutationThread = new Thread(new MutationExecutor(mutD, mops));
    	mutationThread.start();
    }
    
    private class MutationExecutor implements Runnable {
    	
    	private MutationDomain mutD;
    	private ArrayList<MutationOperator> mops;
    	private HashMap<Integer, HashMap<Integer, Boolean>> mutantNumbers;
		private int currentMop, mutantNumber;
    	
    	private MutationExecutor(MutationDomain mutD, ArrayList<MutationOperator> mops) {
    		this.mutD = mutD;
    		this.mops = mops;
    		mutantNumbers = new HashMap<Integer, HashMap<Integer, Boolean>>();

    		int size = mops.size();
    		for(int i=0;i<size;i++) {
    			mutantNumbers.put(i, new HashMap<Integer, Boolean>());
    		}
    	}
    	
    	public void run() {
    		mutationSystemPane.clearLog();
    		int testNumber = tb.getTestNumber();
    		for(int i=0;i<testNumber;i++) {
    			tb.deploy(i);
    			mutate();
    		}
    		printResult();
    	}
    	
    	private void printResult() {
    		new MutationResultSheet(mops, mutantNumbers);
    	}
    	
    	private void mutate() {
    		String agName = mutD.getAgName();
    		int endCycle = tb.getAgentTTL(tb.getCurrentTestID());
    		mutationSystemPane.log("Running agent "+agName+" in test "+(tb.getCurrentTestID())+"");
    		runOriginal(endCycle);
    		int mopSize = mops.size();
    		for (currentMop=0;currentMop<mopSize;currentMop++) {
    	    	Iterator<Literal> bels = mutD.getBels();
    	    	Iterator<Literal> initGoals = mutD.getInitGoals();
    	    	Iterator<String> plans = mutD.getPlans();
    			MutationOperator mop = mops.get(currentMop);
    			mutantNumber = 0;
    			switch(mop.getIndex()) {
    				case 0:
    					while(bels.hasNext()) {
    						Literal bel = bels.next();
    						mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with belief "+bel+" deleted");
    						delBel(agName, bel);
    						runMutant(endCycle);
    					}
    					break;
    				case 1:
    					while(initGoals.hasNext()) {
    						Literal goal = initGoals.next();
    						mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with goal !"+goal+" deleted");
    						delInitGoal(agName, goal);
    						runMutant(endCycle);
    					}
    					break;
    				case 2:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with plan @"+pLabel+" deleted");
							delPlan(agName, pLabel);
							runMutant(endCycle);
						}
						break;
    				case 3:
						while(bels.hasNext()) {
							Literal bel = bels.next();
							if(bel.isRule()) {
								if(!((Rule)bel).getBody().equals(Rule.LTrue)) {
									mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the condition of belief "+bel+" deleted");
									delRuleCondition(agName, bel);
									runMutant(endCycle);
								}
							}
						}
						break;
    				case 4:
						while(plans.hasNext()) {
							String pLabel = plans.next();							
							TEOperator teo = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getTrigger().getOperator();
							TEOperator newTeo = null;
							if(teo.equals(TEOperator.add))
								newTeo = TEOperator.del;
							else if(teo.equals(TEOperator.del))
								newTeo = TEOperator.add;
							mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the triggering event operator of plan @"+pLabel+" changed from "+teo.name()+" to "+newTeo.name());
							changeTEOperator(agName, pLabel, newTeo);
							runMutant(endCycle);
						}
						break;
    				case 5:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							if(mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getContext()!=null) {
								mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the context of plan @"+pLabel+" deleted");
								delPlanContext(agName, pLabel);
								runMutant(endCycle);
							}
						}
						break;
    				case 6:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							if(!mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody().isEmptyBody()) {
								mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the body of plan @"+pLabel+" deleted");
								delPlanBody(agName, pLabel);
								runMutant(endCycle);
							}
						}
						break;
    				case 7:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody().clonePB();
							int size = pb.getPlanSize();
							for(int i=0; i<size;i++) {
								mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with formulae "+getClonedFormula(pb, i)+" in plan @"+pLabel+" deleted");
								delPlanFormula(agName, pLabel, i);
								runMutant(endCycle);
							}
						}
						break;
    				case 8:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody().clonePB();
							int size = pb.getPlanSize();
							if(size>1) {
								for(int i=0;i<size-1;i++) {
									mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with formulae "+getClonedFormula(pb, i)+" and formulae "+getClonedFormula(pb, i+1)+" in plan @"+pLabel+" swaped");
									swapFormula(agName, pLabel, i);
									runMutant(endCycle);
								}
							}
						}
						break;
    				case 9:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody().clonePB();
							int size = pb.getPlanSize();
							for(int i=0; i<size;i++) {
								PlanBody formula = getClonedFormula(pb, i);
								BodyType type = formula.getBodyType();
								BodyType newType = null;
								if(type.equals(BodyType.achieve))
									newType = BodyType.achieveNF;
								else if(type.equals(BodyType.achieveNF))
									newType = BodyType.achieve;
								if(newType != null) {
									mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the type of formulae "+formula+" in plan @"+pLabel+" changed from "+type.name()+" to "+newType.name());
									changeBodyType(agName,pLabel,i,newType);
									runMutant(endCycle);
								}
								
								if(type.equals(BodyType.addBel)||type.equals(BodyType.delBel)||type.equals(BodyType.delAddBel)) {
									BodyType[] mentalNoteTypes = {BodyType.addBel, BodyType.delBel, BodyType.delAddBel};
									int length = mentalNoteTypes.length;
									for(int j=0;j<length;j++) {
										if(!type.equals(mentalNoteTypes[j])) {
											mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the type of formulae "+formula+" in plan @"+pLabel+" changed from "+type.name()+" to "+mentalNoteTypes[j].name());
											changeBodyType(agName, pLabel,i, mentalNoteTypes[j]);
											runMutant(endCycle);
										}
									}
								}
							}
						}
						break;
    				case 10:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody().clonePB();
							int size = pb.getPlanSize();
							for(int i=0; i<size;i++) {
								PlanBody formula = getClonedFormula(pb, i);
								if(formula.getBodyType().equals(BodyType.internalAction)) {
									Literal ia = Literal.parseLiteral(formula.toString());
									if(ia.getFunctor().equals(".send") || ia.getFunctor().equals(".broadcast")) {
										List<List<String>> powerSet = new ArrayList<List<String>>();
										powerSet(0, getAgNamesFromInfra(), powerSet, new ArrayList<String>());
								    	for (List<String> set: powerSet) {
								    		Term receiverTerm = ia.getTerm(0);
								    		if(receiverTerm.isList()) {
								    			ArrayList receiverList = new ArrayList((List)receiverTerm);
								    			if(!set.isEmpty()&&(!set.toString().equals(receiverList.toString()))) {
										    		mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the sender of formulae "+ia+" in plan @"+pLabel+" changed to "+set);
										    		changeReceiver(agName, pLabel, i, set);
										    		runMutant(endCycle);
								    			}
								    		} else {
								    			String receiver = receiverTerm.toString();
								    			if(!set.isEmpty()) {
								    				if((set.size()==1&&!set.contains(receiver))||set.size()!=1) {
											    		mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the sender of formulae "+ia+" in plan @"+pLabel+" changed to "+set);
											    		changeReceiver(agName, pLabel, i, set);
											    		runMutant(endCycle);
								    				}
								    			}
								    		}
								    	}
									}
								}
							}
						}
						break;
    				case 11:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody().clonePB();
							int size = pb.getPlanSize();
							for(int i=0; i<size;i++) {
								PlanBody formula = getClonedFormula(pb, i);
								if(formula.getBodyType().equals(BodyType.internalAction)) {
									Literal ia = Literal.parseLiteral(formula.toString());
									String[] performatives = Message.knownPerformatives;
									int length = performatives.length;
									String performative;
									if(ia.getFunctor().equals(".send")) {
										performative = ia.getTerm(1).toString();
										for(int j=0;j<length;j++) {
											if(!performative.equals(performatives[j])) {
												mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the performative of formulae "+ia+" in plan @"+pLabel+" changed to "+performatives[j]);
												changePerformative(agName, pLabel, i, performatives[j]);
												runMutant(endCycle);
											}
										}
									}else if(ia.getFunctor().equals(".broadcast")) {
										performative = ia.getTerm(0).toString();
										for(int j=0;j<length;j++) {
											if(!performative.equals(performatives[j])) {
												mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with the performative of formulae "+ia+" in plan @"+pLabel+" changed to "+performatives[j]);
												changePerformative(agName, pLabel, i, performatives[j]);
												runMutant(endCycle);
											}
										}
									}
								}
							}
						}   					
    					break;
    				case 12:
						while(plans.hasNext()) {
							String pLabel = plans.next();
							PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody().clonePB();
							int size = pb.getPlanSize();
							for(int i=0; i<size;i++) {
								PlanBody formula = getClonedFormula(pb, i);
								if(formula.getBodyType().equals(BodyType.internalAction)) {
									Literal ia = Literal.parseLiteral(formula.toString());
									if(ia.getFunctor().equals(".send")) {
										if(ia.getTerm(2).isList()) {
											ListTerm contentList = ((ListTerm)ia.getTerm(2));
											int contentSize = contentList.size();
											if(contentSize>1) {
												for(int j=0;j<contentSize;j++) {
													mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with item "+contentList.get(j)+" in the content list of formulae "+ia+" in plan @"+pLabel+" deleted");
													delMessageContent(agName, pLabel, i, j);
													runMutant(endCycle);
												}
											}
										}
									}else if(ia.getFunctor().equals(".broadcast")) {
										if(ia.getTerm(1).isList()) {
											ListTerm contentList = ((ListTerm)ia.getTerm(1));
											int contentSize = contentList.size();
											if(contentSize>1) {
												for(int j=0;j<contentSize;j++) {
													mutationSystemPane.log("Running the mutant of "+agName+" in test "+tb.getCurrentTestID()+" with item "+contentList.get(j)+" in the content list of formulae "+ia+" in plan @"+pLabel+" deleted");
													delMessageContent(agName, pLabel, i, j);
													runMutant(endCycle);
												}
											}
										}
									}
								}
							}
						}
						break;
    			}
    		}
    	}
    	
    	private void runOriginal(int endCycle) {
        	((ExecutionController)mas.getControllerInfraTier().getUserControl()).run(endCycle);
        	restartMAS();
    	}
    	
        private void runMutant(int endCycle) {
        	mutantNumber++;
        	if(tb.getCurrentTestID()==0) {
        		mutantNumbers.get(currentMop).put(mutantNumber, false);
        	}
        	((ExecutionController)mas.getControllerInfraTier().getUserControl()).run(endCycle);
        	if(tb.isMutantKilled(tb.getCurrentTestID())) {
        		if(!mutantNumbers.get(currentMop).get(mutantNumber))
        			mutantNumbers.get(currentMop).put(mutantNumber, true);
        		mutationSystemPane.logError("The above mutant can be killed by test "+tb.getCurrentTestID());
        	}
        	restartMAS();
        }
    }
    
    private void delBel(String agName, Literal bel) {
    	Iterator<Event> events = mas.getAg(agName).getTS().getC().getEvents().iterator();
    	while(events.hasNext()) {
    		Event e = events.next();
    		if(e.getTrigger().getOperator().equals(TEOperator.add)&&e.getTrigger().getLiteral().equals(bel)&&e.getTrigger().getType().equals(TEType.belief)) {
    			mas.getAg(agName).getTS().getC().removeEvent(e);
    			events.remove();
    		}
    	}
    	mas.getAg(agName).getTS().getAg().getBB().remove(bel);
    }

    private void delInitGoal(String agName, Literal initGoal) {
    	Iterator<Event> events = mas.getAg(agName).getTS().getC().getEvents().iterator();
    	while(events.hasNext()) {
    		Event e = events.next();
    		if(e.getTrigger().getOperator().equals(TEOperator.add)&&e.getTrigger().getLiteral().equals(initGoal)&&(e.getTrigger().getType().equals(TEType.achieve))) {
    			mas.getAg(agName).getTS().getC().removeEvent(e);
    			events.remove();
    		}
    	}
    }
    
    private void delPlan(String agName, String pLabel) {
    	mas.getAg(agName).getTS().getAg().getPL().remove(pLabel);
    }
    
    private void delPlanContext(String agName, String pLabel) {
    	mas.getAg(agName).getTS().getAg().getPL().get(pLabel).setContext(null);
    }
    
    private void delRuleCondition(String agName, Literal bel) {
    	Iterator<Literal> beliefs = mas.getAg(agName).getTS().getAg().getBB().iterator();
    	while(beliefs.hasNext()) {
    		Literal belief = beliefs.next();
    		if(belief.equals(bel)) {
    			try {
    				Field body = belief.getClass().getDeclaredField("body");
	    			body.setAccessible(true);
					body.set(belief, Rule.LTrue);
				} catch (Exception e) {}
    		}
    	}
    }
    
    private void delPlanBody(String agName, String pLabel) {
    	Plan plan = mas.getAg(agName).getTS().getAg().getPL().get(pLabel);
    	try {
			Field planBody = plan.getClass().getDeclaredField("body");
			planBody.setAccessible(true);
			planBody.set(plan, new PlanBodyImpl());
		} catch (Exception e) {}	
    }
    
    private void delPlanFormula(String agName, String pLabel, int index) {
    	mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody().removeBody(index);
    }
    
    private void swapFormula(String agName, String pLabel, int index) {
    	PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody();
    	PlanBody formula = getClonedFormula(pb, index);
    	pb.removeBody(index);
		pb.add(index+1, formula);
    }
    
    private void changeBodyType(String agName, String pLabel, int index, BodyType newType) {
    	PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody();
    	PlanBody formula = getClonedFormula(pb, index);
    	formula.setBodyType(newType);
    	pb.removeBody(index);
    	pb.add(index, formula);
    }
    
    private void changeTEOperator(String agName, String pLabel, TEOperator newOperator) {
    	mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getTrigger().setTrigOp(newOperator);
    }
    
    private void changeReceiver(String agName, String pLabel, int index, List<String> receivers) {
    	PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody();
    	PlanBody formula = getClonedFormula(pb, index);
    	Literal ia = (Literal)formula.getBodyTerm();
    	if(ia.getFunctor().equals(".send")) {
    		ia.setTerm(0, ListTermImpl.parseList(receivers.toString()));
    	} else if(ia.getFunctor().equals(".broadcast")) {
        	try {
    			Field functor = Atom.class.getDeclaredField("functor");
    			functor.setAccessible(true);
    			functor.set(ia, ".send");
    		} catch (Exception e) {}
    		ia.getTerms().add(0, ListTermImpl.parseList(receivers.toString()));
    	}
    	pb.removeBody(index);
    	pb.add(index, formula);
    }
    
    private void changePerformative(String agName, String pLabel, int index, String newPerformative) {
    	PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody();
    	PlanBody formula = getClonedFormula(pb, index);
    	Literal ia = (Literal)formula.getBodyTerm();
    	if(ia.getFunctor().equals(".send")) {
    		ia.setTerm(1, new Atom(newPerformative));
    	} else if(ia.getFunctor().equals(".broadcast")) {
    		ia.setTerm(0, new Atom(newPerformative));
    	}
    	pb.removeBody(index);
    	pb.add(index, formula);
    }
    
    private void delMessageContent(String agName, String pLabel, int index, int index2) {
    	PlanBody pb = mas.getAg(agName).getTS().getAg().getPL().get(pLabel).getBody();
    	PlanBody formula = getClonedFormula(pb, index);
    	Literal ia = (Literal)formula.getBodyTerm();
    	if(ia.getFunctor().equals(".send")) {
    		ListTerm contentList = (ListTerm)ia.getTerm(2);
    		contentList.remove(index2);
    	}else if(ia.getFunctor().equals(".broadcast")) {
    		ListTerm contentList = (ListTerm)ia.getTerm(1);
    		contentList.remove(index2);
    	}
    	pb.removeBody(index);
    	pb.add(index, formula);
    }
    
    public static PlanBody getClonedFormula(PlanBody pb, int index) {
    	PlanBody formula = pb.clonePB();
    	
	    for(int i=0;i<index;i++) {
	    	formula = formula.getBodyNext();
	    }
    	
		for(int i=1;i<formula.getPlanSize();) {
			formula.removeBody(i);
		}
		
    	return new PlanBodyImpl(formula.getBodyType(), formula.getBodyTerm());
    }
    
    public static void powerSet(int i, List<String> list, List<List<String>> powerSet, List<String> subset) {
    	if(i>list.size()-1) {
    		List<String> subset2 = new ArrayList<String>();
        	for (String agName: subset) {
        		subset2.add(agName);
        	}
    		powerSet.add(subset2);
    	} else {
    		subset.add(list.get(i));
    		powerSet(i+1,list,powerSet,subset);
    		subset.remove(list.get(i));
    		powerSet(i+1,list,powerSet,subset);
    	}
    }
}