package queso.constraints;


import queso.core.*;
import java.util.*;
import gnu.trove.*;

// needs to be adapted to deal with universal domains with holes, and generally negative domain elements using the offset.
// this adaptation done by adding the offset for each array indexing, and checking is_present when checking the number of subtrees.

class treenode
{
    final int variable;
    final boolean quant;
    
    final boolean leaf;     // One of these four is true; others false. Indicates what type of node.
    final boolean root;
    final boolean internal;
    final boolean listheader;
    
    final treenode [] subtrees;
    treenode prev;
    
    int subtreecounter=0;
    
    boolean intree=true;
    
    // for the linked lists going across the tree
    // linking the supports for varabove=valabove.
    treenode left;
    treenode right;
    
    final int varabove;  // also double as the var/val for the listheader nodes.   // make final
    final int valabove;  // this is the array index, not the actual CSP value.
    
    int nodenumber; // for printing out purposes only.
    
    final sqgac cons;
    
    treenode(boolean quant, int numvalues, sqgac cons)
    {
        // root constructor
        root=true;
        leaf=false;
        internal=false;
        listheader=false;
        
        variable=0;
        
        varabove=-1;
        valabove=0;
        this.quant=quant;
        
        subtrees=new treenode[numvalues];
        
        this.cons=cons;
    }
    
    treenode(boolean quant, int variable, int valabove, treenode prev, int numvalues, sqgac cons)
    {
        // internal node constructor
        root=false;
        leaf=false;
        internal=true;
        listheader=false;
        this.quant=quant;
        this.variable=variable;
        varabove=variable-1;
        this.valabove=valabove;
        
        this.prev=prev;
        
        subtrees=new treenode[numvalues];
        this.cons=cons;
    }
    
    treenode(int varabove, int valabove, treenode prev, sqgac cons)
    {
        // leaf node constructor
        root=false;
        leaf=true;
        internal=false;
        listheader=false;
        
        this.variable=-1;
        this.quant=false;
        
        this.varabove=varabove;
        this.valabove=valabove;   
        
        this.prev=prev;
        this.cons=cons;
        subtrees=null;
    }
    
    treenode(int var, int val, sqgac cons)
    {
        // list header constructor
        root=false;
        leaf=false;
        internal=false;
        listheader=true;
        
        this.quant=false;
        this.variable=-1;
        this.varabove=var;
        this.valabove=val;
        this.cons=cons;
        subtrees=null;
    }
    
    void prune_during_construction()
    {
        // prune this node
        // without storing backtrack information.
        // for use from the tree construction phase.
        // Does not handle the horizontal linked lists because these have not yet been put in place.
        
        assert prev.subtrees[valabove]==this;
        prev.subtrees[valabove]=null;
        prev.subtreecounter--;
        
        intree=false;  // this should not be necessary. This node is about to be GC'd!
    }
    
    void prune()
    {
        assert !root && !listheader;
        assert prev!=null;
        prev.subtrees[valabove]=null;
        prev.subtreecounter--;
        
        prune_subtree();
    }
    
    // should only be called by prune
    // removes crossways linked-list entries.
    private void prune_subtree()
    {
        assert !root && !listheader;
        assert intree;
        intree=false;
        
        cons.sqgac_bt.add_backtrack_item(this);
        
        // this should never be called twice for the same node,
        // so check some things.
        //assert prev.subtrees[valabove]==this;
        
        // descendents of the root node are not in a list.
        assert prev.root || left!=null;   // it should be in a list because it should not be the root node or a listheader.
        
        if(right==null && left!=null && left.listheader)
        {
            // this is the last remaining element of the list.
            // So we need to prune the associated var/val.
            //System.out.println("Adding "+cons.variables[varabove]+" and "+(valabove-cons.offsets[varabove])+" to the remove lists.");
            cons.remove_vars.add(varabove);
            cons.remove_vals.add(valabove-cons.offsets[varabove]);
        }
        
        if(right!=null)
        {
            assert left!=null;   // it is not allowed for a node to have a right pointer but no left pointer.
            right.left=left;
        }
        
        if(left!=null)
            left.right=right;
        
        if(!leaf)
        {
            for(int i=0; i<subtrees.length; i++)
            {
                if(subtrees[i]!=null)
                {
                    subtrees[i].prune_subtree();
                }
            }
        }
    }
    
    // now included in backtrack, no longer needed.
    void restore()
    {
        assert prev!=null;
        assert leaf || internal;
        assert prev.subtrees[valabove]==null || prev.subtrees[valabove]==this;
        assert !intree;
        intree=true;
        
        if(prev.subtrees[valabove]==null)
        {
            prev.subtreecounter++;
            prev.subtrees[valabove]=this;   // in case this node was detached from its parent, reattach.
        }
        
        // reattach the horizontal links.
        if(left!=null)
        {
            left.right=this;
        }
        
        if(right!=null)
        {
            right.left=this;
        }
    }
}

public class sqgac extends constraint implements fd_propagate, stateful
{
    final mid_domain [] variables;
    
    treenode superstrategy;  // i.e. the set of all strategies in one tree
    
    // the cross-linking lists, referenced by variable (local index) and value 
    treenode [][] supports;
    
    // backtrack object
    final sqgac_backtrack sqgac_bt;
    
    final TIntArrayList remove_vars;  // variables and values to be removed, set by node objects as they discover 
    final TIntArrayList remove_vals;  // that values are no longer supported because the list is empty.
    
    int [][] goods;
    // or
    
    // should not be public.
    public predicate_wrapper pred;
    
    final int [] offsets;
    
    public sqgac(mid_domain[] vars, qcsp prob, int [][] goods) 
    {
        super(prob);
        
        variables=vars;
        
        check_variables((variable[]) variables);
        
        remove_vars=new TIntArrayList();   // should be n*d initially.
        remove_vals=new TIntArrayList();
        
        sqgac_bt=new sqgac_backtrack(this);
        
        // should probably ensure that the goods are sorted.
        this.goods=goods;
        
        offsets=new int[vars.length];
        for(int i=0; i<vars.length; i++) offsets[i]=vars[i].offset();
    }
    
    public sqgac(mid_domain[] vars, qcsp prob, predicate_wrapper pred)
    {
        super(prob);
        
        variables=vars;
        
        remove_vars=new TIntArrayList();   // should be n*d initially.
        remove_vals=new TIntArrayList();
        
        sqgac_bt=new sqgac_backtrack(this);
        
        this.pred=pred;
        
        offsets=new int[vars.length];
        for(int i=0; i<vars.length; i++) offsets[i]=vars[i].offset();
    }
    
    public variable_iface [] variables()
    {
        return (variable_iface []) variables;
    }
    
    void linkup_horizontal(treenode node)
    {
        // call with a non-null argument
        assert node!=null && !node.listheader;
        if(node.leaf)
        {
            // base case.
            return;
        }
        else if(!node.root)  // skip over the first variable and universals.
        {
            // cross-link its values
            for(int val=variables[node.variable].lowerbound(); val<=variables[node.variable].upperbound(); val++)
            {
                treenode newnode=node.subtrees[val+offsets[node.variable]];
                if(newnode!=null)
                {
                    // add the crosslinking
                    //System.out.println("crosslinking "+variables[node.variable].name+"="+val);
                    assert newnode.left==null;
                    assert newnode.right==null;
                    //treenode temp=supports
                    newnode.left=supports[node.variable][val+offsets[node.variable]];
                    newnode.right=supports[node.variable][val+offsets[node.variable]].right;
                    newnode.left.right=newnode;
                    
                    if(newnode.right!=null)
                        newnode.right.left=newnode;
                    
                }
            }
        }
        
        for(int val=variables[node.variable].lowerbound(); val<=variables[node.variable].upperbound(); val++)
        {
            treenode newnode=node.subtrees[val+offsets[node.variable]];
        
            // now recurse
            if(newnode!=null) linkup_horizontal(newnode);
        }
        
        assert node.quant==variables[node.variable].quant();
    }
    
    treenode constructing_tree_backtrack(treenode curnode, int change_var)
    {
        // we always backtrack one from the SAT node. 
        
        curnode=curnode.prev;
        
        // The following could be optimised.
        // For example by having a 'safe' flag on existential
        // nodes which indicates that they can be safely backtracked
        // over because they have more than one subtree, so one subtree can be
        // safely removed.
        
        // optimization done - each node has a subtree counter.
        
        for(int var=variables.length-1; var!=change_var; var--)
        {
            assert !curnode.listheader && ( !(curnode.internal || curnode.root) || var==curnode.variable );
            
            if(curnode.quant)
            {
                assert checksubtreecounter(curnode);
                int numsubtrees=0; 
                for(int i=variables[var].lowerbound(); i<=variables[var].upperbound(); i++) 
                    if(variables[var].is_present(i)) numsubtrees++;
                
                if(curnode.subtreecounter<numsubtrees)
                {
                    // we can prune the tree at this point, and above up to 
                    // the next existential with more than one support, or
                    // the point where the two tuples diverge.
                    
                    treenode toprune=curnode;
                    curnode=curnode.prev;
                    
                    // PRUNE subtree here
                    toprune.prune_during_construction();
                }
                else
                {
                    curnode=curnode.prev;
                }
            }
            else
            {
                assert checksubtreecounter(curnode);
                if(curnode.subtreecounter==0)
                {
                    // need to prune curnode
                    treenode toprune=curnode;
                    curnode=curnode.prev;
                    
                    // PRUNE subtree here
                    toprune.prune_during_construction();
                }
                else
                {
                    curnode=curnode.prev;
                }
            }
        }
        return curnode;
    }
    
    boolean print_crosslinks=false;
    
    public String printtree_alt()
    {
        // alternative approach: make a big list of nodes and then iterate through and connect them up.
        ArrayList<treenode> nodes= new ArrayList<treenode>();
        
        printtree_alt_recurse(superstrategy, nodes);
        
        for(int i=0; i<supports.length; i++)
        {
            if(supports[i]!=null)
            {
                for(int j=0; j<supports[i].length; j++)
                {
                    if(supports[i][j]!=null)
                        nodes.add(supports[i][j]);
                }
            }
        }
        
        // now we have a complete list of nodes
        
        // add the node numbers
        for(int i=0; i<nodes.size(); i++)
        {
            nodes.get(i).nodenumber=i;
        }
        
        String st="digraph G {\n";
        for(treenode node : nodes)
        {
            
            if(node.root)
            {
                st+=node.nodenumber+" [label="+variables[node.variable].toString()+"];\n";
                // only need to link up to subtrees.
                for(int i=0; i<node.subtrees.length; i++)
                {
                    if(node.subtrees[i]!=null) st+=node.nodenumber+" -> "+node.subtrees[i].nodenumber+" [taillabel="+i+"];\n";
                }
            }
            else if(node.internal)
            {
                st+=node.nodenumber+" [label="+variables[node.variable].toString()+"];\n";
                // do sidelinks if there are any
                if(print_crosslinks)
                {
                if(node.left!=null) st+=node.nodenumber+" -> "+node.left.nodenumber+" [taillabel=left];\n";
                if(node.right!=null) st+=node.nodenumber+" -> "+node.right.nodenumber+" [taillabel=right];\n";
                }
                
                // prev
                //if(node.prev!=null) st+=node.nodenumber+" -> "+node.prev.nodenumber+" [taillabel=p];\n";
                
                // subtrees
                for(int i=0; i<node.subtrees.length; i++)
                {
                    if(node.subtrees[i]!=null) st+=node.nodenumber+" -> "+node.subtrees[i].nodenumber+" [taillabel="+i+"];\n";
                }
            }
            else if(node.leaf)
            {
                st+=node.nodenumber+" [label=SAT];\n";
                // do sidelinks if there are any
                if(print_crosslinks)
                {
                if(node.left!=null) st+=node.nodenumber+" -> "+node.left.nodenumber+" [taillabel=left];\n";
                if(node.right!=null) st+=node.nodenumber+" -> "+node.right.nodenumber+" [taillabel=right];\n";
                }
                // prev
                //if(node.prev!=null) st+=node.nodenumber+" -> "+node.prev.nodenumber+" [taillabel=p];\n";
            }
            else if(node.listheader)
            {
                if(print_crosslinks)
                {
                st+=node.nodenumber+" [label=\""+variables[node.varabove].toString()+"="+node.valabove+"\"];\n";
                if(node.right!=null) st+=node.nodenumber+" -> "+node.right.nodenumber+" [taillabel=right];\n";
                }
            }
            else
            {
                assert false;
            }
        }
        
        return st+"}\n";
    }
    
    void printtree_alt_recurse(treenode node, ArrayList<treenode> nodes)
    {
        nodes.add(node);
        
        if(node.root || node.internal)
        {
            for(int i=0; i<variables[node.variable].domsize(); i++)
            {
                if(node.subtrees[i]!=null)
                {
                    printtree_alt_recurse(node.subtrees[i], nodes);
                }
            }
        }
    }
    
    int nodenumber=1;  // to give a unique number to each node in the DOT output.
    
    boolean good_valid(int [] good)
    {
        for(int i=0; i<good.length; i++)
        {
            if(!variables[i].is_present(good[i]))
                return false;
        }
        return true;
    }
    
    boolean buildtree_goods()
    {
        // iteratively graft branches onto the tree.
        // takes nd^n time worst-case because we have to look at each element of an n d^n array.
        
        // 
        
        treenode curnode=superstrategy;
        
        // find the first valid good.
        
        int goodindex=0;
        int [] good=goods[0];
        while(goodindex<goods.length)
        {
            if(good_valid(good))
            {
                break;
            }
            else
            {
                goodindex++;
                good=goods[goodindex];
            }
        }
        
        if(goodindex==goods.length)
        {
            // no valid goods
            return false;
        }
        
        // descend for the first time to a leaf node.
        
        for(int var=0; var<variables.length; var++)
        {
            int val=goods[goodindex][var];
            
            assert curnode.variable==var;
            
            if(curnode.subtrees[val+offsets[var]]==null)
            {
                // make a new node
                if(var==(variables.length-1))  // if this is the last variable
                {
                    // make a leaf node.
                    curnode.subtrees[val+offsets[var]]=new treenode(var, val+offsets[var], curnode, this);  // varabove, valabove, prev
                    curnode.subtreecounter++;
                }
                else
                {
                    // make an internal node.
                    curnode.subtrees[val+offsets[var]]=new treenode(variables[var+1].quant(), var+1, val+offsets[var], curnode, variables[var+1].domsize(), this);
                    curnode.subtreecounter++;
                }
            }
            // the newly created node becomes the current node.
            
            curnode=curnode.subtrees[val+offsets[var]];
        }
        
        // converted up to here.
        
        goodindex++;
        int [] prevgood;
        
        for(; goodindex<goods.length; goodindex++)
        {    
            if(!good_valid(goods[goodindex]))
            {
                continue;
            }
            
            prevgood=good;
            good=goods[goodindex];
            
            // now compare the good with the previous good 
            // and backtrack up the tree to the first change point.
            
            // find first change point
            int change_var=-1;
            for(int j=0; j<good.length; j++)
            {
                if(prevgood[j]!=good[j])
                {
                    change_var=j;
                    break;
                }
            }
            
            assert change_var!=-1;
            
            // now backtrack in a loop to the first change point.
            curnode=constructing_tree_backtrack(curnode, change_var);
            
            // Have now backtracked to the first difference between this good and the last one.
            // so now continue to build the tree.
            
            for(int var=curnode.variable; var<good.length; var++)
            {
                int val=good[var];
                
                assert curnode.variable==var;
                
                if(curnode.subtrees[val+offsets[var]]==null)
                {
                    // make a new node
                    if(var==(variables.length-1))  // if this is the last variable
                    {
                        // make a leaf node
                        curnode.subtrees[val+offsets[var]]=new treenode(var, val+offsets[var], curnode, this);
                        curnode.subtreecounter++;
                    }
                    else
                    {
                        // make an internal node.
                        curnode.subtrees[val+offsets[var]]=new treenode(variables[var+1].quant(), var+1, val+offsets[var], curnode, variables[var+1].domsize(), this);
                        curnode.subtreecounter++;
                    }
                }
                // the newly created node becomes the current node.
                curnode=curnode.subtrees[val+offsets[var]];
            }
        }
        // now the tree has been built but we are at a leaf node and there may be
        // universals above which are not completely covered. Therefore backtrack as before
        curnode=constructing_tree_backtrack(curnode, 0);
        
        // delete the goods list
        goods=null;
        
        // Now add the crosslinking and make it display in the debugging output.
        
        // make the headers.
        supports=new treenode[variables.length][];
        for(int var=1; var<variables.length; var++)
        {
            // skips the first variable
            
            // observe the supports for values of existentials only.
            //if(!variables[var].quant)
            {
                supports[var]=new treenode[variables[var].domsize()];
                
                for(int val=0; val<variables[var].domsize(); val++)
                {
                    // new list header
                    supports[var][val]=new treenode(var, val, this);
                }
            }
        }
        
        // now we have the headers, traverse the tree and link up the appropriate nodes.
        
        linkup_horizontal(superstrategy);
        return true;
    }
    
    tuple next_valid_good(tuple good)
    {
        // given the predicate and a *valid good*, find the next valid good.
        // the argument is an in/out.
        
        do
        {
            // increment the tuple
            
            boolean success=false;
            for(int increment=good.vals.length-1; increment>=0; increment--)
            {
                good.vals[increment]++;
                while(!variables[increment].is_present(good.vals[increment]) && 
                            good.vals[increment]<=variables[increment].upperbound())
                    good.vals[increment]++;

                if(good.vals[increment]>variables[increment].upperbound())
                {
                    good.vals[increment]=variables[increment].lowerbound();
                    // and go on to the next inner value.
                }
                else
                {
                    success=true;
                    break;
                }
            }
            
            if(!success)
                return null;
        }
        while(!pred.predicate(good));
        
        assert good_valid(good.vals);
        return good;
    }
    
    boolean buildtree_predicate()
    {
        // iteratively graft branches onto the tree.
        // uses next_valid_good and a tuple.
        
        // make the first valid good.
        int [] valsarray=new int[variables.length];
        
        for(int i=0; i<valsarray.length; i++) 
            valsarray[i]=variables[i].lowerbound();
        
        tuple good=new tuple(valsarray);
        
        if(!pred.predicate(good))
            good=next_valid_good(good);
        
        if(good==null)
        {
            // no valid goods
            return false;
        }
        
        // descend for the first time to a leaf node.
        treenode curnode=superstrategy;
        
        for(int var=0; var<variables.length; var++)
        {
            int val=good.vals[var];
            
            assert curnode.variable==var;
            assert curnode.subtrees[val+offsets[var]]==null;
            
            // make a new node
            if(var==(variables.length-1))  // if this is the last variable
            {
                // make a leaf node.
                curnode.subtrees[val+offsets[var]]=new treenode(var, val+offsets[var], curnode, this);  // varabove, valabove, prev
                curnode.subtreecounter++;
            }
            else
            {
                // make an internal node.
                curnode.subtrees[val+offsets[var]]=new treenode(variables[var+1].quant(), var+1, val+offsets[var], curnode, variables[var+1].domsize(), this);
                curnode.subtreecounter++;
            }
            // the newly created node becomes the current node.
            
            curnode=curnode.subtrees[val+offsets[var]];
        }
        
        int [] prevgood=new int[variables.length];
        
        while(true)
        {
            // store the old one.
            System.arraycopy(good.vals, 0, prevgood, 0, variables.length);
            // get a new good.
            good=next_valid_good(good);
            if(good==null)
                break;
            
            // now compare the good with the previous good 
            // and backtrack up the tree to the first change point.
            
            // find first change point
            int change_var=-1;
            for(int j=0; j<variables.length; j++)
            {
                if(prevgood[j]!=good.vals[j])
                {
                    change_var=j;
                    break;
                }
            }
            
            assert change_var!=-1;
            
            // now backtrack in a loop to the first change point.
            curnode=constructing_tree_backtrack(curnode, change_var);
            
            // Have now backtracked to the first difference between this good and the last one.
            // so now continue to build the tree.
            
            for(int var=curnode.variable; var<variables.length; var++)
            {
                int val=good.vals[var];
                
                assert curnode.variable==var;
                assert curnode.subtrees[val+offsets[var]]==null;
                if(curnode.subtrees[val+offsets[var]]==null)
                {
                    // make a new node
                    if(var==(variables.length-1))  // if this is the last variable
                    {
                        // make a leaf node
                        curnode.subtrees[val+offsets[var]]=new treenode(var, val+offsets[var], curnode, this);
                        curnode.subtreecounter++;
                    }
                    else
                    {
                        // make an internal node.
                        curnode.subtrees[val+offsets[var]]=new treenode(variables[var+1].quant(), var+1, val+offsets[var], 
                                                        curnode, variables[var+1].domsize(), this);
                        curnode.subtreecounter++;
                    }
                }
                // the newly created node becomes the current node.
                curnode=curnode.subtrees[val+offsets[var]];
            }
        }
        // now the tree has been built but we are at a leaf node and there may be
        // universals above which are not completely covered. Therefore backtrack as before 
        curnode=constructing_tree_backtrack(curnode, 0);
        
        // Now add the crosslinking and make it display in the debugging output.
        
        // make the headers.
        supports=new treenode[variables.length][];
        for(int var=1; var<variables.length; var++)
        {
            // skips the first variable
            
            // observe the supports for values of existentials only.
            //if(!variables[var].quant)
            {
                supports[var]=new treenode[variables[var].domsize()];
                
                for(int val=0; val<variables[var].domsize(); val++)
                {
                    // new list header
                    supports[var][val]=new treenode(var, val, this);  // was val+offsets[curnode.variable]
                }
            }
        }
        
        // now we have the headers, traverse the tree and link up the appropriate nodes.
        
        linkup_horizontal(superstrategy);
        return true;
    }
    
    public boolean establish()
    {
        // build the tree
        
        superstrategy=new treenode(variables[0].quant(), variables[0].domsize(), this); // create the root node
        
        // Adding to wakeups.
        for(int i=0; i<variables.length; i++)
        {
            variables[i].add_wakeup(this);
        }
        
        if(goods!=null)
        { 
            if(!buildtree_goods()) return false;
        }
        else
        {
            if(!buildtree_predicate()) return false;
        }
        
        // fix the tree-builder to take account of current domains. BUG
        
        // examine the tree and remove any unsupported values.
        
        // First variable is a special case
        // First variable is pruned according to the existence of subtrees for each value.
        // This is also the only (possibly) universal to be pruned: others
        // are pruned by implication i.e. if they lose a subtree, then the node above gets pruned.
        // Whatever happens here and in find_support is not backtrackable
        
        mid_domain curvar=variables[0];
        
        for(int val=curvar.lowerbound(); val<=curvar.upperbound(); val++)
        {
            if(superstrategy.subtrees[val+offsets[0]]==null && curvar.is_present(val))
            {
                // no supporting subtree
                boolean flag=curvar.exclude(val, this);
                if(!flag)
                    return false;
            }
        }
        
        // All the other existential variables.
        for(int var=1; var<variables.length; var++)
        {
            curvar=variables[var];
            
            if(!curvar.quant())
            {    // if existential
                for(int val=curvar.lowerbound(); val<=curvar.upperbound(); val++)
                {
                    treenode listheader=supports[var][val+offsets[var]];
                    assert listheader.listheader;
                    
                    if(listheader.right==null && curvar.is_present(val))
                    {
                        boolean flag=curvar.exclude(val, this);
                        if(!flag)
                            return false;
                    }
                }
            }
        }
        
        return true;
    }
    
    public void add_backtrack_level()
    {
        // Intended to be called by the search algorithm whenever we go to a new node.
        
        sqgac_bt.add_backtrack_level();
    }
    
    public void backtrack()
    {
        // Intended to be called when the search algorithm backtracks.
        remove_vars.clear();
        remove_vals.clear();
        
        sqgac_bt.backtrack();
    }
    
    // Three possible outcomes:
    // 1: AC established, keep constraint
    // 2: AC established, discard constraint
    // 3: Fail
    
    public boolean propagate(int problem_var, int val)
    {   
        // problem_var is an index into the problem variable array.
        int var=-1; // to cause an error later.
        assert superstrategy!=null;
        for(int i=0; i<variables.length; i++)
        {
            if(problem_var==variables[i].id())
            {
                var=i;
                break;
            }
        }
        assert var>=0 && var<variables.length;
        
        // var, val is the pair removed.
        
        // pruning the tree is incremental so it doesn't matter that the removals are not collected
        // and all processed together.
        
        //System.out.println("SQGAC Propagate called for "+problem.variables.get(problem_var)+", "+val);
        
        if(var==0)
        {
            // special case for the root node.
            treenode curnode=superstrategy.subtrees[val+offsets[0]];
            if(curnode!=null)
            {
                assert curnode.internal;  // otherwise the constraint would be unary.
                curnode.prune();
            }
        }
        else
        {
            // iterate along the appropriate list and prune the tree.
            assert supports!=null;
            assert (val+offsets[var])>=0 && (val+offsets[var])<supports[var].length; 
            treenode listnode=supports[var][val+offsets[var]];
            
            listnode=listnode.right;
            
            while(listnode!=null)
            {
                assert listnode.internal || listnode.leaf;
                // if a node previously in this list has been removed from
                // the list during the current propagation call, then the .right pointer should have been
                // changed, so the removed node should not be in the list at this point.
                assert listnode.intree;
                
                if(listnode.intree)
                {
                    treenode curnode=listnode;
                    // if we have a universal above curnode or an existential with only one support
                    // then we need to traverse upwards.
                    
                    // If there is a universal directly above, then the caller has pruned a universal
                    // for some reason. Don't fail, just remove the branch.
                    
                    if(!curnode.prev.quant)
                    {
                        while(true)
                        {
                            if(curnode.root)
                            {
                                // we have reached the root node so break
                                break;
                            }
                            
                            treenode tempnode=curnode.prev;
                            
                            if(tempnode.quant)
                            {
                                // if it's a universal
                                // check that the value is present.
                                if(variables[curnode.varabove].is_present(curnode.valabove))
                                {
                                    curnode=tempnode; // since the universal has lost a support
                                }
                                else
                                {
                                    break;
                                }
                            }
                            else
                            {
                                // if it's an existential
                                assert checksubtreecounter(tempnode);
                                
                                if(tempnode.subtreecounter==1)   // curnode is the only subtree...
                                {
                                    curnode=tempnode;
                                }
                                else
                                {
                                    // the existential tempnode is supported for other values.
                                    // So we need to prune from curnode downwards.
                                    break;
                                }
                            }
                        }
                    }
                    
                    // now prune from curnode downwards.
                    if(curnode.root)
                    {
                        // if we have reached the root node, all support for everything must be removed.
                        // therefore fail.
                        // bug: are you sure?
                        return false;
                    }
                    else
                    {
                        // if we are pruning from directly beneath the root node, then
                        // we need to remove the respective value of the root node.
                        
                        // If this is the case, then the root must be an existential otherwise
                        // we would have ascended to it.
                        
                        if(curnode.prev.root)
                        {
                            /*if(curnode.prev.quant)
                            {
                                // what went wrong?
                                System.out.println("numvars "+variables.length+" var [o] id:"+variables[0].id()+"var[0]"+variables[0].lowerbound()+", "+variables[0].upperbound());
                                System.out.println(""+curnode.valabove+", "+curnode.varabove);
                            }*/
                            assert !curnode.prev.quant || variables[0].upperbound()==variables[0].lowerbound();
                            
                            remove_vars.add(curnode.varabove);
                            remove_vals.add(curnode.valabove-offsets[curnode.varabove]);
                        }
                        
                        // might also check the subtree count on 
                        // curnode.prev because we can fail if it is 1.

                        curnode.prune();
                    }
                }
                // go on to the next item in the list.
                listnode=listnode.right;
            }
        }
        // Now look at the removal lists and see if anything needs to be pruned.
        // Note that, because all the supports for var=val have been removed
        // by the process above, var=val will be in this removal list.
        
        //System.out.println(remove_vars+" "+remove_vals);
        for(int i=remove_vars.size()-1; i>=0; i--)
        {
            int thisvar=remove_vars.remove(i);
            int thisval=remove_vals.remove(i);
            if(variables[thisvar].is_present(thisval))
            {
                boolean flag=variables[thisvar].exclude(thisval, this);
                if(!flag)
                {
                    return false; // fail.
                }
            }
        }
        
        return true;
    }
    
    treenode entail_watch=null;
    
    public boolean entailed()
    {
        // Assuming the propagate function has been called appropriately
        // for all domain removals before calling entailed.
        // And the propagate function never returned false,
        // i.e. no conflicts are apparent, so if all vars are unit
        // then there can be no conflicts.
        assert superstrategy!=null;
        // this opt actually seems to gain not much.
        boolean allunit=true;
        for(int i=0; i<variables.length; i++)
        {
            if(!variables[i].unit())
            {
                allunit=false;
                break;
            }
        }
        if(allunit)
            return true;
        
        // return true if the constraint is entailed, false otherwise.
        // check the watch -- try to return false as quickly as possible.
        treenode this_watch=entail_watch; // optimization, bring it onto the stack.
        
        if(this_watch!=null && this_watch.intree)   // it is a node in the tree.
        {
            int varnum=this_watch.variable;
            mid_domain var=variables[varnum];
            for(int val=var.lowerbound(); val<=var.upperbound(); val++)
            {
                if(var.is_present(val) && this_watch.subtrees[val+offsets[varnum]]==null)
                {
                    //System.out.println("Level 2 watch effective");
                    return false;
                }
            }
        }
        
        // Since vars are labelled from the outer blocks,
        // it is most likely that entail_watch was cut out of the
        // tree, more likely than one of its children was pruned.
        
        // traverse the tree looking for an E node 
        // where not all values are supported.
        // This is expensive..
        
        return search_entail(superstrategy);
    }
    
    // op: flatten this recursion into a while loop.
    
    private boolean search_entail(treenode curnode)
    {
        // recursively search for an existential node which does not
        // have a full set of supports. if found, store and
        // return false.
        if(curnode==null)
        {
            System.out.println(this);
        }
        int varnum=curnode.variable;
        mid_domain var=variables[varnum];
        
        if(!curnode.quant)
        {
            // it's an existential
            for(int val=var.lowerbound(); val<=var.upperbound(); val++)
            {
                if(var.is_present(val) && curnode.subtrees[val+offsets[varnum]]==null)
                {
                    entail_watch=curnode;  // store the watch.
                    return false;
                }
            }
        }
        
        // curnode has a full set of descendents if it is existential.
        // And if it is universal...
        for(int val=var.lowerbound(); val<=var.upperbound(); val++)
        {
            if(var.is_present(val) && curnode.subtrees[val+offsets[varnum]]!=null)
            {
                treenode newnode=curnode.subtrees[val+offsets[varnum]];
                
                if(!newnode.leaf && !search_entail(newnode))
                    return false;
            }
        }
        return true;
    }
    
    /*possible optimized version, with while loop...
        treenode curnode=superstrategy;
        int curvar=curnode.varabove;
        
        int [] subtreeindices=new int[variables.length];
        boolean flag=true;
        // When descending, store the subtree index into the array
        // When ascending, get the subtree index and go to the next one.
        while(flag)
        {
            if(!curnode.quant)
            {
                // it's an existential or the root node
                int varnum=curnode.varabove;
                mid_domain var=variables[varnum];
                for(int val=var.lowerbound(); val<=var.upperbound(); val++)
                {
                    if(var.is_present(val) && this_watch.subtrees[val+offsets[varnum]]==null)
                    {
                        entail_watch=curnode;  // store the watch.
                        return false;
                    }
                }
            }
            else
            {
                // descend
                
            }
        }
        
        // entire tree traversed and no falsifying node found.
        return true;*/
    
    private boolean checksubtreecounter(treenode node)
    {
        int count=0;
        for(int i=0; i<node.subtrees.length; i++)
        {
            if(node.subtrees[i]!=null)
                count++;
        }
        return count==node.subtreecounter;
    }
    
    public String toString()
    {
        return pred.getClass().getSimpleName()+" over "+Arrays.asList(variables);
    }
}

/*final class sqgac_backtrack
{
    final TIntArrayList backtrack_levels;   // maps tree depth -> first index of relevant data in removals_vars and removals_vals
    int level;                // Current tree depth 0.. #vars
    
    final ArrayList<treenode> nodelist;
    
    final sqgac cons;
    
    sqgac_backtrack(sqgac cons)
    {
        backtrack_levels= new TIntArrayList();  // should really be n*d
        level=0;
        backtrack_levels.add(-1);  //  this is invariant
        
        this.cons=cons;
        nodelist= new ArrayList<treenode>();
    }
    
    void add_backtrack_level()
    {
        backtrack_levels.add(backtrack_levels.get(level));
        level++;
    }
    
    void backtrack()
    {
        // abstract backtrack function
        assert level>0;
        
        for(int i=backtrack_levels.get(level); i>backtrack_levels.get(level-1); i--)
        {
            treenode node=nodelist.remove(i);
        
            node.restore();
        }
        
        backtrack_levels.remove(level);
        
        level--;
    }
    
    final void add_backtrack_item(treenode node)
    {
        nodelist.add(node);
        int curpointer=backtrack_levels.get(level)+1;
        
        backtrack_levels.set(level, curpointer);
    }
}*/


final class sqgac_backtrack extends backtrack
{
    final ArrayList<treenode> nodelist;
    
    final sqgac cons;
    
    sqgac_backtrack(sqgac cons)
    {
        this.cons=cons;
        nodelist= new ArrayList<treenode>(1000);  // op for connect 3 4*4
    }
    
    public void backtrack_item()
    {
        int index=nodelist.size()-1;
        
        treenode node=nodelist.remove(index);
        
        // following cutnpasted from node.restore
        node.intree=true;
        
        if(node.prev.subtrees[node.valabove]==null)
        {
            node.prev.subtreecounter++;
            node.prev.subtrees[node.valabove]=node;   // in case this node was detached from its parent, reattach.
        }
        
        // reattach the horizontal links.
        if(node.left!=null)
        {
            node.left.right=node;
        }
        
        if(node.right!=null)
        {
            node.right.left=node;
        }
        
        //node.restore();
    }
    
    final void add_backtrack_item(treenode node)
    {
        nodelist.add(node);
        
        backtrack_increment();
    }
    
    // call add_backtrack_level() and backtrack() from the superclass.
}
