// various tuple-processing functions that are shared between different table constraints.
// in here, tuples are always type int[].
package queso.constraints;

import java.util.*;
import queso.core.*;

public class tuples
{
    public static int [][] pred2table(predicate_wrapper pred, mid_domain [] variables)
    {
        // convert a predicate to a table of satisfying tuples.
        int arity=variables.length;
        int [] tuplearray=new int[arity];
        for(int i=0; i<arity; i++)
        {
            tuplearray[i]=variables[i].lowerbound();
        }
        
        ArrayList<int []> table=new ArrayList<int []>();
        
        if(pred.predicate(new tuple(tuplearray)))
        {
            table.add((int [])tuplearray.clone());
        }
        
        while(increment(tuplearray, variables))
        {
            assert valid(tuplearray, variables);
            if(pred.predicate(new tuple(tuplearray)))
            {
                table.add((int [])tuplearray.clone());
            }
        }
        
        int [][] rettable=new int[table.size()][];
        for(int i=0; i<rettable.length; i++)
        {
            rettable[i]=(int []) table.get(i);
        }
        
        return rettable;
    }
    
    public static boolean increment(int [] tuplearray, mid_domain [] variables)
    {
        boolean iterate=true;
        int k=tuplearray.length-1;
        while(iterate)
        {
            iterate=false;
            
            while(k>=0 && tuplearray[k]==variables[k].upperbound())
            {   // symbol at position k cannot be increased.
                k--;
                iterate=true;
            }
            
            if(k<0) return false;
        }
        
        // increment the symbol at position k
        for(int val=tuplearray[k]+1; val<=variables[k].upperbound(); val++)
        {
            if(variables[k].is_present(val))
            {
                tuplearray[k]=val;
                break;
            }
        }
        
        // zero the rest
        for(int index=k+1; index<variables.length; index++)
        {
            tuplearray[index]=variables[index].lowerbound();
        }
        
        return true;
    }
    
    public static boolean valid(int [] tuple, mid_domain [] variables)
    {
        for(int i=0; i<tuple.length; i++)
        {
            if(!variables[i].is_present(tuple[i]))
            {
                return false;
            }
        }
        return true;
    }
    
    public static boolean contains(int [] tuple, pa partial)
    {
        for(int i=0; i<partial.vars.length; i++)
        {
            if(tuple[partial.vars[i]]!=partial.vals[i])
                return false;
        }
        return true;
    }
    
    public static int comparetuples(int [] t1, int [] t2)
    {
        // returns -1, 0 or 1 for <, =, >
        for(int i=0; i<t1.length; i++)
        {
            if(t1[i]<t2[i])
                return -1;

            else if(t1[i]>t2[i])
                return 1;
        }
        return 0;
    }
    
    public static int binarysearch(int [] tuple, int [][]tuplelist)
    {
        // find the tuple in the lex ordered tuplelist
        // return index or -1;
        int top=tuplelist.length-1;
        int bottom=0;
        int index=top>>1;
        while(true)
        {
            int [] current=tuplelist[index];
            
            int match=comparetuples(current, tuple);
            if(match==0)
            {
                return index;
            }
            if(match==-1)
            {   // i.e. the currentpa is less than the one we are searching for
                // increase the index.
                bottom=index+1;
            }
            else
            {   // i.e. the currentpa is greater than the one we are searching for
                // decrease the index.
                assert match==1;
                top=index-1;
            }
            index=bottom+( (top-bottom)>>1 );
            if(top<bottom)
                return -1;
        }
    }
    
    public static int [][] fliptable(int [][] table, mid_domain[] variables)
    {
        // find all the tuples not in the table and make an array
        // Assume d=2
        int [] tuple=new int[variables.length];
        for(int i=0; i<variables.length; i++)
        {
            tuple[i]=variables[i].lowerbound();
        }
        ArrayList<int []> ret=new ArrayList<int []>();
        
        if(binarysearch(tuple, table)==-1)
        {
            ret.add((int [])tuple.clone());
        }
        
        while(tuples.increment(tuple, variables))
        {
            if(binarysearch(tuple, table)==-1)
            {
                ret.add((int [])tuple.clone());
            }
        }
        
        int [][] retarray=new int[ret.size()][];
        for(int i=0; i<ret.size(); i++)
            retarray[i]=ret.get(i);
        
        return retarray;
    }
    
    public static String atos(int [][] t)
    {
        String st="{";
        for(int i=0; i<t.length-1; i++)
        {
            st+=atos(t[i])+", ";
        }
        if(t.length>0)
            st+=atos(t[t.length-1]);
        st+="}";
        return st;
    }
    
    public static String atos(int [] t)
    {
        String st="{";
        for(int i=0; i<t.length-1; i++)
        {
            st+=t[i]+", ";
        }
        if(t.length>0)
            st+=t[t.length-1];
        st+="}";
        return st;
    }
    
    // this one doesn't split into pa's only into var/val.
    public static ArrayList [][] splittuples(int [][] alltuples, mid_domain [] variables) //<pa, goodlistwrapper>
    {
        int arity=alltuples[0].length;
        ArrayList [][] goods=new ArrayList[arity][];
        for(int i=0; i<arity; i++)
        {
            goods[i]=new ArrayList[variables[i].domsize()];
            for(int val=variables[i].lowerbound(); val<=variables[i].upperbound(); val++)
            {
                if(variables[i].is_present(val)) goods[i][val+variables[i].offset()]=new ArrayList();
            }
        }
        
        for(int var=0; var<arity; var++)
        {
            // assumes tuples sorted in increasing lex order.
            //System.out.println("Variable "+var);
            
            for(int val=variables[var].lowerbound(); val<=variables[var].upperbound(); val++)
            {   
                if(variables[var].is_present(val))
                {
                    for(int tupleindex=0; tupleindex<alltuples.length; tupleindex++)
                    {
                        if(alltuples[tupleindex][var]==val)
                        {
                            // matching tuple
                            goods[var][val+variables[var].offset()].add(alltuples[tupleindex]);
                        }
                    }
                }
            }
        }
        return goods;
    }
    
    public static int [][] generateprobability(int arity, mid_domain [] variables, Random r, float probability) 
    {
        int [] t=new int[arity];
        for(int i=0; i<t.length; i++) t[i]=variables[i].lowerbound();
        ArrayList a= new ArrayList<int []>();
        int posstuples=0;
        while(true)
        {
            if(!tuples.increment(t, variables))
                break;
            posstuples++;
            if(r.nextFloat()<probability)
            {
                int [] s=(int [])t.clone();
                a.add(s);
            }
        }
        int [][]ret=new int[a.size()][];
        for(int i=0; i<ret.length; i++) ret[i]=(int []) a.get(i);
        //System.out.println("From "+posstuples+" possible tuples, generated "+ret.length+" tuples.");
        if(ret.length==0) ret=tuples.generateprobability(arity, variables, r, probability);
        return ret;
    }
    
    public static int [][] generatetuples(int arity, mid_domain [] variables, Random r, int numtuples) 
    {
        HashSet<tuple> tupleset=new HashSet<tuple>();
        
        while(tupleset.size()<numtuples)
        {
            // randomly generate a tuple
            int [] t=new int[arity];
            for(int i=0; i<arity; i++)
            {
                int lb=variables[i].lowerbound();
                int ub=variables[i].upperbound();
                int val=r.nextInt(ub-lb+1);
                while(!variables[i].is_present(val)) val=r.nextInt(ub-lb+1);
                
                t[i]=val;
            }
            tupleset.add(new tuple(t));
        }
        
        int [][]ret=new int[numtuples][];
        int counter=0;
        for(tuple t : tupleset)
        {
            ret[counter]=t.vals;
            counter++;
        }
        
        TupleComparator2 tc = new TupleComparator2() ;
        Arrays.sort( (Object []) ret, tc);
        
        return ret;
    }
    
    public static boolean in(int [] arr, int val)
    {
        for(int i=0; i<arr.length; i++)
        {
            if(arr[i]==val)
                return true;
        }
        return false;
    }
    
    public static boolean in(Object [] arr, Object val)
    {
        for(int i=0; i<arr.length; i++)
        {
            if(arr[i]==val)
                return true;
        }
        return false;
    }
}
