// A constraint which implements a disjunction

package queso.constraints;

//import gnu.trove.*;
import java.util.*;
//import gnu.math.*;
import java.io.*;
import queso.core.*;

// note on variables: numerical constraints can use both hash_var
// and bound_var types, but the other constraints can only use
// hash_var because they prune from the middle of the domains.


// the basic class, not directly used.
abstract class or_c extends constraint implements make_ac
{
    or_c(mid_domain [] variables, boolean [] negated, int xi, qcsp problem)
    // xi is the variable which is true iff the disjunction of all the others
    {
        super(problem);
        this.variables=variables;
        check_variables(variables);
        
        assert negated.length == variables.length;
        
        this.negated=negated;
        this.xi=xi;
    }
    
    final mid_domain [] variables;
    public final boolean [] negated;
    public final int xi;
    
    public boolean establish()
    {
        for(mid_domain temp : variables)
        {
            temp.add_wakeup(this);
        }
        
        return make_ac();
    }
    
    public variable_iface [] variables()
    {
        return (variable_iface []) variables;
    }
    
    int hintOne=-1;
    
    final boolean useHintOne=false;
    final boolean useHintOneInE=false;
    
    public boolean make_ac()
    {
        // variables are in quantification order
        
        // do a case split on xi
        
        int statexi=litstate(xi);
        
        if(statexi==0)
        {
            // scan the other variables for one which is set to 1 or universal
            // and fail if one is found, setting all existentials to 0.
            
            for(int i=0; i<variables.length; i++)
            {
                if(i!=xi)
                {
                    int ls=litstate(i);
                    if(ls==1 || ls==2)  // set to 1 or universal
                    {
                        return false;
                    }
                    else if(ls==3)
                    {
                        // existential literals are set to false
                        
                        boolean flag=pruning(i, true);
                        if(!flag)
                        {   // not sure why this would happen
                            return false;
                        }
                    }
                    
                    // else it is set to 0 and we are not interested.
                }
            }
            return true;
        }
        else if(statexi==1)
        {
            // this is the standard case for CNF QBF
            // so should we use the standard watching algorithm due to Rowley et al?

            // in the meantime lets implement a simple thing that finds literals again every time

            // trivially succeed if there is a 1.
            // we fail if all unset literals are universals


            // if there is only one existential, and no outer universals, set it.

            // could be implemented with 3 watches, 2 existentials and 1 outer universal.
            // While there are two existentials, we don't do anything.
            // When there's only one existential, we keep the outer universal up to date.
            // When there's no outer universal, we set the existential.
            // If there's no existential, fail.
            if(useHintOne && hintOne!=-1 && litstate(hintOne)==1)
            {
                // this optimizes the search for a 1 down a branch where it has
                // already been found.
                //System.out.println("Won something with hintOne!");
                return true;
            }
            
            // blank the watches
            int universal=-1;
            int existential=-1;
            
            // scan for 1's
            // This loop could become the setup for the watches,
            // with some other logic for when watches are already
            // there.
            for(int i=0; i<variables.length;i++)
            {
                if(i!=xi)
                {
                    int ls=litstate(i);
                    if(ls==1)
                    {
                        if(useHintOne) hintOne=i;

                        return true;
                    }
                    else if(ls==2)
                    {
                        // universal
                        if(universal==-1)
                        {   // this is the outermost one.
                            universal=i;
                        }
                    }
                    else if(ls==3)
                    {   // existential
                        if(existential==-1)
                        {   // outermost existential
                            if(universal!=-1)
                            {
                                // there is an outer universal, and an existential, so nothing can be done.
                                return true;
                            }
                            existential=i;
                        }
                        else
                        {   // next existential
                            // can return here because we have 2 existentials
                            // therefore we can't set anything because we don't know
                            // which existential to set, if either.
                            return true;
                        }
                    }
                }
                // else ls==0 and we don't care.
            }

            // now universal contains the outermost universal, 
            // existential1 and existential2 contain the outermost 2 existentials.

            if(existential==-1)
            {
                // there are no existentials or 1's therefore
                // there is no way to satisfy this clause
                return false;
            }

            assert universal==-1 || universal>existential;

            return pruning(existential, false);
        }
        else if(statexi==2)
        {
            // x_i  is universal.
            
            // the outer section of the constraint must evaluate to false,
            // and the inner section must have freedom i.e. no universals, no 1's
            // and at least one existential.
            
            // outer section
            // Actually a lot of this work should only be done at the root node.
            // Making sure the outer section is false, for example.
            for(int i=0; i<xi; i++)
            {
                int ls=litstate(i);
                if(ls==1 || ls==2)
                {   // a 1 or a universal
                    return false;
                }
                else if(ls==3)
                {   // existential
                    boolean flag=pruning(i, true);
                    if(!flag)
                        return false;
                }
                
                // else it's a 0 and we are not interested.
            }
            
            // inner section
            boolean foundexistential=false;
            for(int i=xi+1; i<variables.length; i++)
            {
                int ls=litstate(i);
                if(ls==1 || ls==2)
                {
                    return false;
                }
                else if(ls==3)
                {
                    foundexistential=true;
                }
            }
            
            return foundexistential;
        }
        else
        {
            assert statexi==3;
            // the disjuncts are unconstrained, but xi might need to be pruned.
            // e.g. if there is a 1 or an inner universal, or all disjuncts are 0.
            
            // if the inner section has a universal, we have to prune cixi=false.
            // then re-process because statexi==1 now.
            //System.out.println("in statexi==3 code.");
            
            if(useHintOneInE && hintOne!=-1 && litstate(hintOne)==1)
            {
                return pruning(xi, false);
            }
            
            boolean allfalse=true;
            for(int i=0; i<variables.length; i++)
            {
                if(i!=xi)
                {
                    int ls=litstate(i);
                    if(ls==1)
                    {
                        if(useHintOne) hintOne=i;
                        return pruning(xi, false);
                    }
                    else if(ls==2 && i>xi)
                    {
                        boolean flag=pruning(xi, false);
                        if(!flag)  // this should never happen.
                            return false;
                        
                        //System.out.println("Making recursive call.");
                        return make_ac();    // caution: recursive call.
                    }
                    else if(ls==3 || (ls==2 && i<xi))
                    {   // existential or inner universal
                        allfalse=false;
                    }
                    
                    // else litstate(i)==0 and we don't care about it.
                }
            }
            
            if(allfalse)
            {
                return pruning(xi, true);
            }
            
            return true;
        }
    }
    
    abstract boolean pruning(int variable, boolean value);
    abstract int litstate(int variable);
    
    public boolean entailed()
    {
        // return true iff the constraint cannot be falsified.
        // Assumes make_ac has been called after the last domain modification.
        int statexi=litstate(xi);
        
        if(statexi==0)
        {
            // check that all the other lits are false as well
            for(int i=0; i<variables.length; i++)
            {
                if(i!=xi)
                {
                    int l1=litstate(i);
                    if(l1!=0)
                    {
                        return false;
                    }
                }
            }
            return true;
        }
        else if(statexi==1)
        {
            // check that there exists a true lit.
            for(int i=0; i<variables.length; i++)
            {
                if(i!=xi)
                {
                    int l1=litstate(i);
                    if(l1==1)
                    {
                        return true;
                    }
                }
            }
            return false;
        }
        else
        {
            // are you sure?
            return false;
        }
    }
    
}

public class or_constraint extends or_c
{   
    public or_constraint(mid_domain [] variables, boolean [] negated, int xi, qcsp problem)
    {
        super(variables, negated, xi, problem);
        
        for(int i=0; i<variables.length; i++)
            assert variables[i].domsize()<=2: "variable "+variables[i]+" domain size "+variables[i].domsize();
    }
    
    boolean pruning(int variable, boolean value)
    {
        // prune the value from the literal
        
        if((negated[variable] && !value) || (!negated[variable] && value))
        {
            // then prune true from the variable.
            return variables[variable].exclude(1, this);
        }
        else
        {
            return variables[variable].exclude(0, this);
        }
    }
    
    int litstate(int i)
    {
        // variable of index i, with negated[i], makes a literal
        // get the state of the literal
        // 0, 1, a=2 and e=3
        
        mid_domain var=variables[i];
        
        if(var.is_present(0))
        {
            if(var.is_present(1))
            {
                if(var.quant())
                {
                    return 2;
                }
                else
                {
                    return 3;
                }
            }
            else
            {
                if(negated[i])
                {   // set to 0 and negated
                    return 1;
                }
                else
                {
                    return 0;
                }
            }
        }
        
        //assert var.is_present(1) : "why an empty domain in a constraint propagator?";
        else if(negated[i])
        {   // set to 1 and negated
            return 0;
        }
        else
        {
            return 1;
        }
    }
    
    public String toString()
    {
        String st="";
        for(int i=0; i<variables.length; i++)
        {
            if(i!=xi)
            {
                if(negated[i])
                {
                    st+="neg:"+variables[i]+" or ";
                }
                else
                {
                    st+=variables[i]+" or ";
                }
            }
        }
        st+=" <=> "+(negated[xi]?"neg:":"")+variables[xi];
        return st;
    }
}

class or_constraint_test
{
    public static void main(String[] args)
    {
        or_constraint_test o1=new or_constraint_test();
        
        for(int i=0; i<100000; i++)
        {
            o1.test_random_prob();
            System.out.println("Done test "+i);
        }
    }
    
    void test_random_prob()
    {
        mid_domain [] vars_disjunction= new mid_domain[10];
        mid_domain [] vars_sqgac= new mid_domain[10];
        
        qcsp prob_disjunction = new qcsp();
        qcsp prob_sqgac = new qcsp();
        
        boolean [] negated= new boolean[10];
        
        // random test.
        
        for(int i=0; i<10; i++)
        {
            if(Math.random()>0.5)
            {
                negated[i]=true;
            }
            else
            {
                negated[i]=false;
            }
        }
        
        for(int i=0; i<10; i++)
        {
            if(Math.random()>0.5)
            {
                // universal
                vars_disjunction[i]=new universal(2, prob_disjunction, "x"+(i+1));
                vars_sqgac[i]=new universal(2, prob_sqgac, "x"+(i+1));
            }
            else
            {
                // existential
                vars_disjunction[i]=new existential(2, prob_disjunction, "x"+(i+1));
                vars_sqgac[i]=new existential(2, prob_sqgac, "x"+(i+1));
            }
        }
        
        int xi = (int) (10*Math.random());
        assert xi>=0 && xi<=9;
        
        or_constraint c1= new or_constraint(vars_disjunction, negated, xi, prob_disjunction);
        
        predicate_wrapper pred=new or_predicate(negated, xi);
        sqgac c2= new sqgac(vars_sqgac, prob_sqgac, pred);
        
        System.out.println(""+c1);
        
        //System.out.println(""+Math.random());
        //System.out.println(""+Math.random());
        //System.out.println(""+Math.random());
        
        // first check equivalence of the first pass
        
        assert same_domains(prob_disjunction, prob_sqgac) : "something wrong in constructing the problems";
        
        boolean flag1=prob_disjunction.establish();
        boolean flag2=prob_sqgac.establish();
        
        assert flag1==flag2 : "found "+flag1+" for or_constraint and "+flag2+" for sqgac";
        
        if(flag1 && !same_domains(prob_disjunction, prob_sqgac))   // if not false, compare doms.
        {
            System.out.println("different domains");
            prob_disjunction.printdomains();
            prob_sqgac.printdomains();
            assert false;
        }
        
        if(!flag1)
        {
            return;
        }
        
        prob_sqgac.printdomains();
        for(int descents=0; descents<5; descents++)
        {
            System.out.println("New descent.");
            // now make some random assignments to a leaf node.
            int numassigned=0;
            while(numassigned<9)
            {
                // first check that an assignment is possible
                boolean assignmentposs=false;
                for(int i=0; i<9; i++)
                {
                    if(litstate(vars_sqgac[i], negated[i])==2 || litstate(vars_sqgac[i], negated[i])==3)
                    {
                        assignmentposs=true;
                        break;
                    }
                }
                
                if(!assignmentposs)
                    break;
                
                // pick an assignment to make
                int var=(int)(10*Math.random());
                int val=(int)(2*Math.random());
                
                boolean outer=false;
                for(int i=var-1; i>=0; i--) 
                    if((vars_sqgac[var].quant() && litstate(vars_sqgac[i], negated[i])==3) ||
                       (!vars_sqgac[var].quant() && litstate(vars_sqgac[i], negated[i])==2)) outer=true;
                
                while(litstate(vars_sqgac[var], negated[var])<2 || outer)
                {
                    var=(int)(10*Math.random());
                    outer=false;
                    for(int i=var-1; i>=0; i--) 
                        if((vars_sqgac[var].quant() && litstate(vars_sqgac[i], negated[i])==3) ||
                            (!vars_sqgac[var].quant() && litstate(vars_sqgac[i], negated[i])==2)) outer=true;
                }
                
                System.out.println("Test harness: Assigning variable "+prob_sqgac.variables.get(var)+" value "+val);
                
                /*try
                {
                    File output = new File("or_constraint_test.dot");
                    FileWriter out = new FileWriter(output);
                    out.write(c2.printtree_alt());
                    out.close();
                }
                catch(java.io.IOException e)
                {
                    System.out.println("Problem writing sqgac_test.dot");
                }*/
                
                numassigned++;
                prob_disjunction.add_backtrack_level();
                prob_sqgac.add_backtrack_level();
                
                ((mid_domain)prob_disjunction.variables.get(var)).instantiate(val);
                ((mid_domain)prob_sqgac.variables.get(var)).instantiate(val);
                
                flag1=prob_disjunction.propagate();
                flag2=prob_sqgac.propagate();
                
                assert flag1==flag2 : "found "+flag1+" for or_constraint and "+flag2+" for sqgac";
                
                if(flag1 && !same_domains(prob_disjunction, prob_sqgac))   // if not false, compare doms.
                {
                    System.out.println("different domains");
                    System.out.println("disjunction:");
                    prob_disjunction.printdomains();
                    System.out.println("sqgac:");
                    prob_sqgac.printdomains();
                    
                    /*try
                    {
                        File output = new File("or_constraint_test2.dot");
                        FileWriter out = new FileWriter(output);
                        out.write(c2.printtree_alt());
                        out.close();
                    }
                    catch(java.io.IOException e)
                    {
                        System.out.println("Problem writing sqgac_test.dot");
                    }*/
                    assert false;
                }
                
                if(!flag1)
                    break;  // no point making more assignments.
            }
            
            // backtrack before starting the next descent.
            for(int i=0; i<numassigned; i++)
            {
                prob_disjunction.backtrack();
                prob_sqgac.backtrack();
            }
        }
    }
    
    boolean same_domains(qcsp prob1, qcsp prob2)
    {
        if(prob1.variables.size()!=prob2.variables.size())
            return false;
        
        for(int i=0; i<prob1.variables.size(); i++)
        {
            mid_domain var1 = (mid_domain)prob1.variables.get(i);
            mid_domain var2 = (mid_domain)prob2.variables.get(i);
            
            if(var1.domsize()!=var2.domsize())
                return false;
            
            if(var1.lowerbound()!=var2.lowerbound())
                return false;
            
            if(var1.upperbound()!=var2.upperbound())
                return false;
            
            for(int j=var1.lowerbound(); j<=var1.upperbound(); j++)
            {
                if(var1.is_present(j)!=var2.is_present(j))
                    return false;
            }
        }
        return true;
    }
    
    int litstate(mid_domain var, boolean negated)
    {
        // variable of index i, with negated[i], makes a literal
        // get the state of the literal
        // 0, 1, a=2 and e=3
        
        if(var.is_present(0))
        {
            if(var.is_present(1))
            {
                if(var.quant())
                {
                    return 2;
                }
                else
                {
                    return 3;
                }
            }
            else
            {
                if(negated)
                {   // set to 0 and negated
                    return 1;
                }
                else
                {
                    return 0;
                }
            }
        }
        
        // assert var.is_present(1) : "why an empty domain in a constraint propagator?";
        
        else if(negated)
        {   // set to 1 and negated
            return 0;
        }
        else
        {
            return 1;
        }
    }
}

class or_watches_experiment
{
    public static void main(String[] args)
    {
        or_watches_experiment o1=new or_watches_experiment();
        o1.do_tests();
    }
    
    void do_tests()
    {
        // do a sequence of experiments with increasing length constraints.
        long [] results=new long[1000];
        
        // this is to get the hotspot compiler to kick in and 
        // inline litstate etc.
        r=new Random(12345);
        sw=new stopwatch("GMT");
        for(int i=0; i<500; i++)
        {
            test_random_prob(100);
        }
        
        for(int numvars=150; numvars<=150; numvars++)
        {
            r=new Random(12345);
            
            sw= new stopwatch("GMT");
            sw.start();
            for(int i=0; i<1000; i++)
            {
                test_random_prob(numvars);
            }
            sw.end();
            System.out.println("Test time: "+sw.elapsedMillis());
            
            results[numvars]=sw.elapsedMillis();
        }
        
        System.out.println(results.toString());
        
        /*
        sw= new stopwatch("GMT");
        sw.start();
        
        for(int i=0; i<50000; i++)
        {
            o1.test_random_prob();
            //System.out.println("Done test "+i);
        }
        
        sw.end();
        System.out.println("Test time: "+sw.elapsedMillis());*/
    }
    
    Random r; // random number generator.
    stopwatch sw;
    
    void test_random_prob(int numvars)
    {
        mid_domain [] vars_disjunction= new mid_domain[numvars];
        
        qcsp prob_disjunction = new qcsp();
        
        boolean [] negated= new boolean[numvars];
        
        // random test.
        
        for(int i=0; i<numvars; i++)
        {
            if(r.nextFloat()>0.5)
            {
                negated[i]=true;
            }
            else
            {
                negated[i]=false;
            }
        }
        
        for(int i=0; i<numvars; i++)
        {
            if(r.nextFloat()>2.0)
            {
                // universal
                vars_disjunction[i]=new universal(2, prob_disjunction, "x"+(i+1));
                
            }
            else
            {
                // existential
                vars_disjunction[i]=new existential(2, prob_disjunction, "x"+(i+1));
               
            }
        }
        
        int xi = (int) (numvars*r.nextFloat());
        assert xi>=0 && xi<numvars;
        
        or_constraint c1= new or_constraint(vars_disjunction, negated, xi, prob_disjunction);
        
        //System.out.println(""+c1);
        
        //System.out.println(""+Math.random());
        //System.out.println(""+Math.random());
        //System.out.println(""+Math.random());
        
        // first check equivalence of the first pass
        
        //sw.start();
        boolean flag1=prob_disjunction.establish();
        //sw.end();
        
        if(!flag1)
        {
            return;
        }
        
        for(int descents=0; descents<10; descents++)
        {
            //System.out.println("New descent.");
            // now make some random assignments to a leaf node.
            int numassigned=0;
            while(numassigned<numvars)
            {
                // first check that an assignment is possible
                boolean assignmentposs=false;
                for(int i=0; i<numvars; i++)
                {
                    // making an assumption about the internals of c1:
                    int a=litstate(vars_disjunction[i], negated[i]);
                    if(a==2 || a==3)
                    {
                        assignmentposs=true;
                        break;
                    }
                }
                
                if(!assignmentposs)
                    break;
                
                // pick an assignment to make
                int var=(int)(numvars*r.nextFloat());
                int val=(int)(2*r.nextFloat());
                
                boolean outer=false;
                for(int i=var-1; i>=0; i--) 
                    if((vars_disjunction[var].quant() && litstate(vars_disjunction[i], negated[i])==3) ||
                       (!vars_disjunction[var].quant() && litstate(vars_disjunction[i], negated[i])==2)) outer=true;
                
                while(litstate(vars_disjunction[var], negated[var])<2 || outer)
                {
                    var=(int)(numvars*r.nextFloat());
                    outer=false;
                    for(int i=var-1; i>=0; i--)
                        if((vars_disjunction[var].quant() && litstate(vars_disjunction[i], negated[i])==3) ||
                            (!vars_disjunction[var].quant() && litstate(vars_disjunction[i], negated[i])==2)) outer=true;
                }
                
                //System.out.println("Test harness: Assigning variable "+prob_disjunction.variables.get(var)+" value "+val);
                
                numassigned++;
                prob_disjunction.add_backtrack_level();
                
                ((mid_domain)prob_disjunction.variables.get(var)).instantiate(val);
                
                //sw.start();
                flag1=prob_disjunction.propagate();
                //sw.end();
                
                if(!flag1)
                    break;  // no point making more assignments.
            }
            
            // backtrack before starting the next descent.
            for(int i=0; i<numassigned; i++)
            {
                prob_disjunction.backtrack();
            }
        }
    }
    
    int litstate(mid_domain var, boolean negated)
    {
        // variable of index i, with negated[i], makes a literal
        // get the state of the literal
        // 0, 1, a=2 and e=3
        
        if(var.is_present(0))
        {
            if(var.is_present(1))
            {
                if(var.quant())
                {
                    return 2;
                }
                else
                {
                    return 3;
                }
            }
            else
            {
                if(negated)
                {   // set to 0 and negated
                    return 1;
                }
                else
                {
                    return 0;
                }
            }
        }
        
        // assert var.is_present(1) : "why an empty domain in a constraint propagator?";
        
        else if(negated)
        {   // set to 1 and negated
            return 0;
        }
        else
        {
            return 1;
        }
    }
}

