// A constraint which implements a product c1 x1 x2 .. xi-1 xi+1 .. xn = c2 xi

package queso.constraints;

//import gnu.trove.*;
import java.util.*;
import gnu.math.*;
import java.io.*;   // for test

import queso.core.*;

// note on variables: numerical constraints can use both mid_domain
// and bound_var types, but the other constraints can only use
// mid_domain because they prune from the middle of the domains.

public final class product_constraint extends constraint implements make_ac
{
    intnum_bounds[] variables;
    
    public product_constraint(intnum_bounds [] variables, int xi, IntNum c1, IntNum c2, qcsp problem)
    {
        super(problem);
        this.variables=variables;
        check_variables(variables);
        
        this.xi=xi;
        
        assert variables.length>=2;
        assert xi<variables.length && xi>=0;
        
        assert c1.compare(zero)!=0 && c2.compare(zero)>0;  // only c1 is allowed to be negative.
        this.c1=c1;
        this.c2=c2;
        
        /* If both negative.
        if(c1.compare(zero)<0 && c2.compare(zero)<0)
        {
            this.c1=(IntNum)zero.sub(c1);  // 0-c1
            this.c2=(IntNum)zero.sub(c2);  // 0-c2
        }
        else
        {
            this.c1=c1;
            this.c2=c2;
        }*/
    }
    
    static{
        zero=IntNum.zero();
        one=IntNum.one();
        minusone=IntNum.minusOne();
    }
    
    static final IntNum zero;
    static final IntNum one;
    static final IntNum minusone;
    
    final int xi;         // the index of the variable which is on the right hand side.
    final IntNum c1, c2;  // Only c1 (the product-side constant) is allowed to be negative.
    
    public String toString()
    {
        String st="";
        for(int i=0; i<variables.length; i++)
        {
            st+=(variables[i].quant()?"A":"E")+variables[i]+"["+variables[i].lowerbound_big()+","+variables[i].upperbound_big()+"] ";
        }
        
        st+=": "+c1.toString()+" ";
        
        for(int i=0; i<variables.length; i++)
        {
            if(i!=xi)
                st+=variables[i]+" ";
        }
        st+=" == "+c2+variables[xi];
        
        return st;
    }
    
    public variable_iface [] variables()
    {
        return (variable_iface []) variables;
    }
    
    public boolean establish()
    {
        for(intnum_bounds temp : variables)
        {
            temp.add_wakeup(this);
        }
        
        // Initialize the two (leftward and rightward) resolution caches.
        lxpositiveupper=new IntNum[variables.length];
        lxpositivelower=new IntNum[variables.length];
        lxnegativeupper=new IntNum[variables.length];
        lxnegativelower=new IntNum[variables.length];
        lxdenominator=new IntNum[variables.length];
        lxhaszero=new boolean[variables.length];
        lxemptydomain=new boolean[variables.length];
        lxdoneto=variables.length; // cache filled up to and including this index, from the right.
        
        rxpositiveupper=new IntNum[variables.length];
        rxpositivelower=new IntNum[variables.length];
        rxnegativeupper=new IntNum[variables.length];
        rxnegativelower=new IntNum[variables.length];
        rxdenominator=new IntNum[variables.length];
        rxhaszero=new boolean[variables.length];
        rxemptydomain=new boolean[variables.length];  // this should not really be needed.
        rxdoneto=-1; // cache filled up to and including this index, from the left.
        
        return make_ac();
    }
    
    // All numerical variables are integers
    
    // The two (leftward and rightward) resolution caches.
    IntNum[] lxpositiveupper;
    IntNum[] lxpositivelower;
    IntNum[] lxnegativeupper;
    IntNum[] lxnegativelower;
    IntNum[] lxdenominator;
    boolean[] lxhaszero;
    boolean[] lxemptydomain;
    int lxdoneto;
    
    IntNum[] rxpositiveupper;
    IntNum[] rxpositivelower;
    IntNum[] rxnegativeupper;
    IntNum[] rxnegativelower;
    IntNum[] rxdenominator;
    boolean[] rxhaszero;
    boolean[] rxemptydomain;
    int rxdoneto;
    
    public boolean make_ac()
    {
        // variables are in quantification order
        //System.out.println("product make_ac called.");
        //System.out.println(this);
        
        // reset the cache.
        lxdoneto=variables.length;  // cache filled up to and including this index, from the right.
        rxdoneto=-1;                // cache filled up to and including this index, from the left.
        
        // First propagate to x_i from the other variables
        // Resolving using the pair replacement rules.
        
        if(!prune_variable(xi))
            return false;
        
        // Now compute new bounds for all other variables x_j |ne i.
        
        // Vars that include 0 make it impossible to prune other inner 
        // xj vars, and all other vars if existential.
        
        // Vars set to 0 make it impossible to prune other xj vars.
        
        // first look for existentials which span 0.
        // If there are two, then we do no pruning at all.??
        
        // in this case, if another variable !=j can be freely set to 0, then
        // no pruning is possible on xj.
        
        // check for zeros
        for(int i=0; i<variables.length; i++)
        {
            if(i!=xi && variables[i].lowerbound_big().isZero() && variables[i].upperbound_big().isZero())
            {
                assert variables[xi].upperbound_big().isZero() && variables[xi].lowerbound_big().isZero();
                return true;  // xi should already have been set to 0 above (and have failed if that is impossible), 
                                // so it is OK to just return true here.
            }
        }
        
        // now check for variables containing zero, if the rhs contains zero, and is existential or outer.
        
        // A variable does not need to be pruned if:
        // 1. another existential variable contains 0, or an outer universal variable contains zero, AND
        // 2. the xi variable contains zero and is either existential or outer.
        
        int e1=-1;
        int e2=-1;
        int outermosta=-1;
        
        for(int i=0; i<variables.length; i++)
        {
            if(i!=xi && variables[i].containsZero())
            {
                if(!variables[i].quant())
                {
                    if(e1==-1)
                    {
                        e1=i;
                    }
                    else if(e2==-1)
                    {
                        e2=i;
                    }
                }
                else if(outermosta==-1)
                {
                    outermosta=i;
                }
            }
        }
        
        if(variables[xi].containsZero())
        {
            // sure about this??
            // actually if xi contains 0 and another value, and is universal, then the inner resolvent
            // must also contain 0 and another value, so there is no need to prune outer variables.
            /*if(variables[xi].quant())
            {
                // prune all variables outside xi and possibly some others.
                for(int i=0; i<xi; i++)
                {
                    if(!prune_variable(i))
                        return false;
                }
            }*/
            
            // what if xi is set to 0 and inner variables do not contain zero?? BUG
            
            // now only need to consider variables inside xi when xi is universal.
            
            if(e2!=-1)
            {   // if two existentials
                return true;
            }
            // we are now left with a single existential e1 or no existentials
            else if(e1!=-1)
            {
                if(outermosta!=-1 && outermosta<e1)
                {
                    // if one existential and one universal, and the universal
                    // is outside, then nothing more needs to be done.
                    return true;
                }
                
                // just need to prune e1 if xi is existential or outer. No existential or outer variable !=i spans zero.
                if(!variables[xi].quant() || xi<e1)
                {
                    return prune_variable(e1);
                }
                else
                {
                    return true;
                }
            }
            else if(outermosta!=-1)
            {   // now just universals to consider.
                // need to prune all variables outside outermosta
                if(!variables[xi].quant())
                {
                    for(int i=0; i<outermosta; i++)
                    {
                        if(i!=xi)
                        {
                            if(!prune_variable(i))
                                return false;
                        }
                    }
                    return true;
                }
                else
                {
                    // xi must be outer if it is universal.
                    for(int i=xi+1; i<outermosta; i++)
                    {
                        if(i!=xi)
                        {
                            if(!prune_variable(i))
                                return false;
                        }
                    }
                    return true;
                }
            }
            else
            {
                // need to prune all variables, but there are no spanzeros so that should make it easy.??
                // Even so, just using the same resolution as everything else.
                for(int i=0; i<variables.length; i++)
                {
                    if(i!=xi)
                    {
                        if(!prune_variable(i))
                            return false;
                    }
                }
                return true;
            }
        }
        else
        {
            // No possibility of avoiding work using the 0's, so prune all.
            for(int i=0; i<variables.length; i++)
            {
                if(i!=xi)
                {
                    if(!prune_variable(i)) 
                        return false;
                }
            }
            return true;
        }
        
        // Now do inner variance rule, analogue of the rule for sum constraint. 
        // Also deals with case where outermost spanzero is a universal.
        // Does this need to be done? I don't think so because the effect of an
        // inner universal will be caught by the resolution, and result in an
        // empty or 0 domain.
        
        // do we restart here if the domains have changed?
        // Probably better not to: see HC1. Goualard and Granvilliers.
        // But make sure that the constraint is requeued again if we do not restart.
    }
    
    public boolean entailed()
    {
        // if all vars unit
        for(int i=0; i<variables.length; i++)
        {
            if(!variables[i].unit())
            {
                return false;
            }
        }
        return true;
    }
    
    // If these two are not inlined, i'll kill some Sun engineers.
    private static final IntNum min(IntNum c1, IntNum c2)
    {
        if(c1.compare(c2)<0)
        {
            return c1;
        }
        else
        {
            return c2;
        }
    }
    
    private static final IntNum max(IntNum c1, IntNum c2)
    {
        if(c1.compare(c2)>0)
        {
            return c1;
        }
        else
        {
            return c2;
        }
    }
    
    ////////////////////////////////////////////////////////////////////////////////////////////////
    // The following methods are for pruning variables other than xi.
    
    private boolean prune_variable(int e1)
    {
        //System.out.println("Pruning variable "+variables[e1]);
        
        for(int i=lxdoneto-1; i>e1; i--)
        {   // compute the minimum number of resolvents.
            if(!res_leftward(i))
                return false;
        }
        
        IntNum lxpu;
        IntNum lxpl;
        IntNum lxnu;
        IntNum lxnl;
        if(c1.isNegative())
        {
            lxpu=(e1==variables.length-1?IntNum.zero():lxpositiveupper[e1+1]);
            lxpl=(e1==variables.length-1?IntNum.zero():lxpositivelower[e1+1]);
            lxnu=(e1==variables.length-1?c1:lxnegativeupper[e1+1]);
            lxnl=(e1==variables.length-1?c1:lxnegativelower[e1+1]);
        }
        else
        {
            lxpu=(e1==variables.length-1?c1:lxpositiveupper[e1+1]);
            lxpl=(e1==variables.length-1?c1:lxpositivelower[e1+1]);
            lxnu=(e1==variables.length-1?IntNum.zero():lxnegativeupper[e1+1]);
            lxnl=(e1==variables.length-1?IntNum.zero():lxnegativelower[e1+1]);
        }
        
        IntNum lxden=(e1==variables.length-1?IntNum.one():lxdenominator[e1+1]);
        boolean lxzero=(e1==variables.length-1?false:lxhaszero[e1+1]);
        boolean lxempty=(e1==variables.length-1?false:lxemptydomain[e1+1]);
        
        // zero means 'no value' in these numbers.
        
        for(int i=rxdoneto+1; i<e1; i++)
        {
            res_rightward(i);
        }
        
        IntNum rxpu=(e1==0?one:rxpositiveupper[e1-1]);
        IntNum rxpl=(e1==0?one:rxpositivelower[e1-1]);
        IntNum rxnu=(e1==0?zero:rxnegativeupper[e1-1]);
        IntNum rxnl=(e1==0?zero:rxnegativelower[e1-1]);
        IntNum rxden=(e1==0?one:rxdenominator[e1-1]);
        boolean rxzero=(e1==0?false:rxhaszero[e1-1]);
        boolean rxempty=(e1==0?false:rxemptydomain[e1-1]);
        
        combine_ee(lxpu, lxpl, lxnu, lxnl, lxzero, lxden, lxempty, rxpu, rxpl, rxnu, rxnl, rxzero, rxden, rxempty);
        
        // reset cache given that the domain of e1 just potentially changed.
        rxdoneto=e1-1;
        lxdoneto=e1+1;
        
        // revises bounds, including dividing the denominator by each bound
        if(e1==xi)
        {
            return revise_bounds_xi(positiveupper, positivelower, negativeupper, negativelower, haszero, denominator);
        }
        else
        {
            return revise_bounds(positiveupper, positivelower, negativeupper, negativelower, haszero, denominator, e1);
        }
    }
    
    private boolean res_leftward(int i)
    {
        // i is the index
        // compute the minimum number of resolvents.
        // remember to treat xi differently.
        
        // retrieve the bounds from the cache.
        assert lxdoneto==i+1;
        
        IntNum pu2;
        IntNum pl2;
        IntNum nu2;
        IntNum nl2;
        IntNum d2;
        boolean emptydomain2;
        boolean haszero2;
        
        if(lxdoneto<variables.length)
        {
            pu2=lxpositiveupper[lxdoneto];
            pl2=lxpositivelower[lxdoneto];
            nu2=lxnegativeupper[lxdoneto];
            nl2=lxnegativelower[lxdoneto];
            d2=lxdenominator[lxdoneto];
            emptydomain2=lxemptydomain[lxdoneto];
            haszero2=lxhaszero[lxdoneto];
        }
        else
        {
            assert lxdoneto==variables.length;
            if(c1.isNegative())
            {
                nu2=c1; nl2=c1;
                pu2=zero; pl2=zero;
            }
            else
            {
                pu2=c1; pl2=c1;
                nu2=zero; nl2=zero;
            }
            d2=one;
            haszero2=false;
            emptydomain2=false;
        }
        
        // now interrogate the variable i
        
        IntNum lb1=variables[i].lowerbound_big();
        IntNum ub1=variables[i].upperbound_big();
        IntNum pu1, pl1, nu1, nl1;
        boolean haszero1=variables[i].containsZero();
        
        if(!ub1.isZero() && !ub1.isNegative())
        {   // ub1 is positive.
            pu1=ub1;
            if(!lb1.isZero() && !lb1.isNegative())
            {   // lb1 is positive.
                pl1=lb1;
                nu1=nl1=zero;
                assert !haszero1;
            }
            else if(lb1.isZero())
            {
                assert haszero1;
                pl1=variables[i].positive_smallest_big();
                nu1=nl1=zero;
            }
            else
            {   // lb1 is negative. Variable spans zero.
                pl1=variables[i].positive_smallest_big();
                nu1=variables[i].negative_smallest_big();
                nl1=lb1;
            }
        }
        else if(ub1.isZero())
        {
            assert haszero1;
            pu1=pl1=zero;
            if(lb1.isNegative())
            {
                nu1=variables[i].negative_smallest_big();
                nl1=lb1;
            }
            else
            {
                nu1=nl1=zero;
            }
        }
        else
        {
            assert !haszero1;
            pu1=pl1=zero;
            nu1=ub1;
            nl1=lb1;
        }
        
        if(i==xi)
        {
            if(variables[i].quant())
            {
                combine_axi(pu1, pl1, nu1, nl1, haszero1, one, false,       // assumes it's not empty
                            pu2, pl2, nu2, nl2, haszero2, d2, emptydomain2);
                if(emptydomain) return false;
                // standard ae resolution might be made irrelevant by a 0 somewhere else,
                // but if axi resolution finds an empty domain, we need to fail.
            }
            else
            {
                combine_exi(pu1, pl1, nu1, nl1, haszero1, one, false,       // assumes it's not empty
                            pu2, pl2, nu2, nl2, haszero2, d2, emptydomain2);
            }
        }
        else
        {
            if(variables[i].quant())
            {
                combine_ae(pu1, pl1, nu1, nl1, haszero1, one, false,       // assumes it's not empty
                            pu2, pl2, nu2, nl2, haszero2, d2, emptydomain2);
            }
            else
            {
                combine_ee(pu1, pl1, nu1, nl1, haszero1, one, false,       // assumes it's not empty
                            pu2, pl2, nu2, nl2, haszero2, d2, emptydomain2);
            }
        }
        
        // fill in cache entry.
        lxpositiveupper[i]=positiveupper;
        lxpositivelower[i]=positivelower;
        lxnegativeupper[i]=negativeupper;
        lxnegativelower[i]=negativelower;
        lxhaszero[i]=haszero;
        lxdenominator[i]=denominator;
        lxemptydomain[i]=emptydomain;
        
        assert lxdoneto==i+1;
        lxdoneto=i;
        
        /*System.out.print("Resolving leftward for variable "+variables[i]+": [");
        if(!negativelower.isZero()) System.out.print(negativelower+", "+negativeupper+", ");
        if(haszero) System.out.print("0, ");
        if(!positiveupper.isZero()) System.out.print(positivelower+", "+positiveupper);
        System.out.println("], den "+denominator);*/
        return true;
    }
    
    private void res_rightward(int i)
    {
        // retrieve the bounds from the cache.
        assert rxdoneto==i-1;
        
        IntNum pu2;
        IntNum pl2;
        IntNum nu2;
        IntNum nl2;
        IntNum d2;
        boolean emptydomain2;
        boolean haszero2;
        
        if(rxdoneto>=0)
        {
            pu2=rxpositiveupper[rxdoneto];
            pl2=rxpositivelower[rxdoneto];
            nu2=rxnegativeupper[rxdoneto];
            nl2=rxnegativelower[rxdoneto];
            d2=rxdenominator[rxdoneto];
            emptydomain2=rxemptydomain[rxdoneto];
            haszero2=rxhaszero[rxdoneto];
        }
        else
        {
            assert rxdoneto==-1;
            pu2=pl2=one;
            nu2=nl2=zero;
            d2=one;
            haszero2=false;
            emptydomain2=false;
        }
        
        // now interrogate the variable i
        
        IntNum lb1=variables[i].lowerbound_big();
        IntNum ub1=variables[i].upperbound_big();
        IntNum pu1, pl1, nu1, nl1;
        boolean haszero1=variables[i].containsZero();
        
        if(!ub1.isZero() && !ub1.isNegative())
        {   // ub1 is positive.
            pu1=ub1;
            if(!lb1.isZero() && !lb1.isNegative())
            {   // lb1 is positive.
                pl1=lb1;
                nu1=nl1=zero;
                assert !haszero1;
            }
            else if(lb1.isZero())
            {
                assert haszero1;
                pl1=variables[i].positive_smallest_big();
                nu1=nl1=zero;
            }
            else
            {   // lb1 is negative. Variable spans zero.
                pl1=variables[i].positive_smallest_big();
                nu1=variables[i].negative_smallest_big();
                nl1=lb1;
            }
        }
        else if(ub1.isZero())
        {
            assert haszero1;
            pu1=pl1=zero;
            if(lb1.isNegative())
            {
                nu1=variables[i].negative_smallest_big();
                nl1=lb1;
            }
            else
            {
                nu1=nl1=zero;
            }
        }
        else
        {
            assert !haszero1;
            pu1=pl1=zero;
            nu1=ub1;
            nl1=lb1;
        }
        
        // now combine.
        
        if(i==xi)
        {
            combine_exi(pu1, pl1, nu1, nl1, haszero1, one, false,       // assumes it's not empty
                        pu2, pl2, nu2, nl2, haszero2, d2, emptydomain2);
        }
        else
        {
            combine_ee(pu1, pl1, nu1, nl1, haszero1, one, false,       // assumes it's not empty
                       pu2, pl2, nu2, nl2, haszero2, d2, emptydomain2);
        }
        check();
         // fill in cache entry.
        rxpositiveupper[i]=positiveupper;
        rxpositivelower[i]=positivelower;
        rxnegativeupper[i]=negativeupper;
        rxnegativelower[i]=negativelower;
        rxhaszero[i]=haszero;
        rxdenominator[i]=denominator;
        rxemptydomain[i]=emptydomain;
        
        assert rxdoneto==i-1;
        rxdoneto=i;
        
        /*System.out.print("Resolving rightward for variable "+variables[i]+": [");
        if(!negativelower.isZero()) System.out.print(negativelower+", "+negativeupper+", ");
        if(haszero) System.out.print("0, ");
        if(!positiveupper.isZero()) System.out.print(positivelower+", "+positiveupper);
        System.out.println("], den "+denominator);*/
    }
    
    // Only to be used to return data from the combine methods when pruning xi
    IntNum newupperbound;
    IntNum newlowerbound;
    
    // also used for non-xi
    boolean emptydomain;
    
    // only used when not pruning xi.
    IntNum positiveupper;
    IntNum positivelower;
    IntNum negativeupper;
    IntNum negativelower;
    IntNum denominator;
    boolean haszero;
    
    private void combine_ee(IntNum pu1, IntNum pl1, IntNum nu1, IntNum nl1, boolean haszero1, IntNum denom1, boolean emptydomain1, 
                    IntNum pu2, IntNum pl2, IntNum nu2, IntNum nl2, boolean haszero2, IntNum denom2, boolean emptydomain2)
    {
        denominator=IntNum.times(denom1, denom2);
        
        if(denominator.isZero())
        {
            // This indicates that xi is zero. Therefore be propagate through the
            // zero denominator
            positiveupper=zero;
            positivelower=zero;
            negativeupper=zero;
            negativelower=zero;
            haszero=false;
            emptydomain=true;  // still useful information in the domain.
            return;
        }
        else if(emptydomain1 || emptydomain2)
        {
            if((emptydomain1 && haszero2)
                || (emptydomain2 && haszero1))
            {
                positiveupper=zero;
                positivelower=zero;
                negativeupper=zero;
                negativelower=zero;
                haszero=true;
                denominator=one;
                emptydomain=false;
                return;
            }
            else
            {   // otherwise empty.
                positiveupper=zero;
                positivelower=zero;
                negativeupper=zero;
                negativelower=zero;
                haszero=false;
                denominator=one;
                emptydomain=true;
                return;
            }
        }
        
        if(!pu1.isZero() && !nl1.isZero())
        {
            // first variable spans zero.
            if(!pu2.isZero() && !nl2.isZero())
            {
                // both span zero. 
                // new pu
                IntNum candidate1=IntNum.times(pu1, pu2);
                IntNum candidate2=IntNum.times(nl1, nl2);
                positiveupper=max(candidate1, candidate2);
                
                // new pl
                candidate1=IntNum.times(pl1, pl2);
                candidate2=IntNum.times(nu1, nu2);
                positivelower=min(candidate1, candidate2);
                
                // new nu
                candidate1=IntNum.times(pl1, nu2);
                candidate2=IntNum.times(nu1, pl2);
                negativeupper=max(candidate1, candidate2);
                
                // new nl
                candidate1=IntNum.times(pu1, nl2);
                candidate2=IntNum.times(nl1, pu2);
                negativelower=min(candidate1, candidate2);
            }
            else if(!nl2.isZero())
            {   // second existential is negative.
                positiveupper=IntNum.times(nl1, nl2);
                positivelower=IntNum.times(nu1, nu2);
                negativeupper=IntNum.times(pl1, nu2);
                negativelower=IntNum.times(pu1, nl2);
            }
            else if(!pu2.isZero())
            {   // second existential is positive.
                positiveupper=IntNum.times(pu1, pu2);
                positivelower=IntNum.times(pl1, pl2);
                negativeupper=IntNum.times(nu1, pl2);
                negativelower=IntNum.times(nl1, pu2);
            }
            else
            {
                // second existential may have zero.
                positiveupper=positivelower=zero;
                negativeupper=negativelower=zero;
            }
        }
        else if(!nl1.isZero())
        {   // first variable is negative
            if(!pu2.isZero() && !nl2.isZero())
            {   // the second variable spans zero.
                positiveupper=IntNum.times(nl1, nl2);
                positivelower=IntNum.times(nu1, nu2);
                negativeupper=IntNum.times(nu1, pl2);
                negativelower=IntNum.times(nl1, pu2);
            }
            else if(!pu2.isZero())
            {   // second variable positive.
                negativeupper=IntNum.times(nu1, pl2);
                negativelower=IntNum.times(nl1, pu2);
                positivelower=positiveupper=zero;
            }
            else if(!nl2.isZero())
            {
                // both variables negative
                positivelower=IntNum.times(nu1, nu2);
                positiveupper=IntNum.times(nl1, nl2);
                negativeupper=negativelower=zero;
            }
            else
            {
                positivelower=positiveupper=zero;
                negativeupper=negativelower=zero;
            }
        }
        else if(!pu1.isZero())
        {
            // first variable is positive.
            if(!pu2.isZero() && !nl2.isZero())
            {   // the second variable spans zero.
                positiveupper=IntNum.times(pu1, pu2);
                positivelower=IntNum.times(pl1, pl2);
                negativeupper=IntNum.times(pl1, nu2);
                negativelower=IntNum.times(pu1, nl2);
            }
            else if(!pu2.isZero())
            {   // both variables positive.
                positiveupper=IntNum.times(pu1, pu2);
                positivelower=IntNum.times(pl1, pl2);
                negativelower=negativeupper=zero;
            }
            else if(!nl2.isZero())
            {
                // second variable negative
                negativelower=IntNum.times(pu1, nl2);
                negativeupper=IntNum.times(pl1, nu2);
                positiveupper=positivelower=zero;
            }
            else
            {
                positivelower=positiveupper=zero;
                negativeupper=negativelower=zero;
            }
        }
        else
        {
            positivelower=positiveupper=zero;
            negativeupper=negativelower=zero;
        }
        
        haszero=haszero1 || haszero2;
        
        if(positiveupper.compare(positivelower)<0)
        {
            positiveupper=positivelower=zero;
        }
        if(negativeupper.compare(negativelower)<0)
        {
            negativeupper=negativelower=zero;
        }
        
        emptydomain=(positiveupper.isZero() && negativelower.isZero() && !haszero);
        check();
    }
    
    private void combine_ae(IntNum pu1, IntNum pl1, IntNum nu1, IntNum nl1, boolean haszero1, IntNum denom1, boolean emptydomain1, 
                    IntNum pu2, IntNum pl2, IntNum nu2, IntNum nl2, boolean haszero2, IntNum denom2, boolean emptydomain2)
    {
        // ub1 lb1 etc represent the outer universal.
        // what if one or other has a zero denominator??
        denominator=IntNum.times(denom1, denom2);
        
        // The universal is not allowed to be set to 0. WHY?
        //assert !(pu1.isZero() && nl1.isZero() && haszero1); // universal not set 0.
        
        if(denominator.isZero())
        {
            // This indicates that xi is zero. Therefore be propagate through the
            // zero denominator
            positiveupper=zero;
            positivelower=zero;
            negativeupper=zero;
            negativelower=zero;
            haszero=false;
            emptydomain=true;  // still useful information in the domain.
            return;
        }
        else if(emptydomain2)
        {
            // since the universal is not allowed to be set to 0
            // an empty inner domain implies this resolvent should also
            // be empty.
            positiveupper=zero;
            positivelower=zero;
            negativeupper=zero;
            negativelower=zero;
            haszero=false;
            denominator=one;
            emptydomain=true;
            return;
        }
        else if(pu1.isZero() && nl1.isZero())
        {
            assert haszero1;
            positiveupper=positivelower=zero;
            negativeupper=negativelower=zero;
            haszero=true;
            denominator=one;
            emptydomain=false;
            return;
        }
        else if(!pu1.isZero() && !nl1.isZero())
        {
            if(!pu2.isZero() && !nl2.isZero())
            {
                // both span zero. 
                // new pu
                IntNum candidate1=IntNum.times(pl1, pu2);  // use the worst-case for the universal.
                IntNum candidate2=IntNum.times(nu1, nl2);
                positiveupper=min(candidate1, candidate2);  // choose min since universal gets to choose the sign.
                
                // new pl
                candidate1=IntNum.times(pu1, pl2);
                candidate2=IntNum.times(nl1, nu2);
                positivelower=max(candidate1, candidate2);
                
                // new nu
                candidate1=IntNum.times(pu1, nu2);
                candidate2=IntNum.times(nl1, pl2);
                negativeupper=min(candidate1, candidate2);
                
                // new nl
                candidate1=IntNum.times(pl1, nl2);
                candidate2=IntNum.times(nu1, pu2);
                negativelower=max(candidate1, candidate2);
            }
            else
            {
                // universal spans zero, existential does not.
                // Therefore universal can set the sign, therefore
                // empty the non-zero parts of the resolvent.
                positiveupper=positivelower=zero;
                negativeupper=negativelower=zero;
            }
        }
        else
        {   // universal does not span zero but it does have non-zero elements.
            if(!pu2.isZero() && !nl2.isZero())
            {
                // only the inner existential spans zero.
                if(pu1.isZero())
                {
                    // universal is negative.
                    positiveupper=IntNum.times(nu1, nl2);
                    positivelower=IntNum.times(nl1, nu2);
                    negativeupper=IntNum.times(nl1, pl2);
                    negativelower=IntNum.times(nu1, pu2);
                }
                else
                {
                    // universal is positive.
                    positiveupper=IntNum.times(pl1, pu2);
                    positivelower=IntNum.times(pu1, pl2);
                    negativeupper=IntNum.times(pu1, nu2);
                    negativelower=IntNum.times(pl1, nl2);
                }
            }
            else
            {
                // neither variable spans zero.
                if(pu1.isZero() && pu2.isZero())
                {
                    // both are negative.
                    positivelower=IntNum.times(nl1, nu2);
                    positiveupper=IntNum.times(nu1, nl2);
                    negativeupper=negativelower=zero;
                }
                else if(pu1.isZero() && !pu2.isZero())
                {
                    // universal is negative, existential is positive.
                    negativeupper=IntNum.times(nl1, pl2);
                    negativelower=IntNum.times(nu1, pu2);
                    positiveupper=positivelower=zero;
                }
                else if(!pu1.isZero() && pu2.isZero())
                {
                    // universal is positive, existential is negative.
                    negativeupper=IntNum.times(pu1, nu2);
                    negativelower=IntNum.times(pl1, nl2);
                    positiveupper=positivelower=zero;
                }
                else
                {
                    // both positive.
                    assert !pu1.isZero() && !pu2.isZero();
                    positivelower=IntNum.times(pu1, pl2);
                    positiveupper=IntNum.times(pl1, pu2);
                    negativeupper=negativelower=zero;
                }
            }
        }
        
        // a universal never uses its zero element when
        // pruning another xj variable because that
        // would mean no pruning.
        // However it may be used when related to the (outer)
        // xi variable! BUG.
        // Perhaps should separate the processing of 0's.
        haszero=haszero2;
        
        if(positiveupper.compare(positivelower)<0)
        {
            positiveupper=positivelower=zero;
        }
        if(negativeupper.compare(negativelower)<0)
        {
            negativeupper=negativelower=zero;
        }
        
        emptydomain=(positiveupper.isZero() && negativelower.isZero() && !haszero);
        check();
    }
    
    private void check()
    {
        assert !denominator.isNegative();
        assert positiveupper.compare(positivelower)>=0;
        assert !positiveupper.isNegative() && !positivelower.isNegative();
        assert negativeupper.compare(negativelower)>=0;
        assert (negativeupper.isNegative() || negativeupper.isZero()) && (negativelower.isNegative() || negativelower.isZero());
        assert (haszero || !positiveupper.isZero() || !negativelower.isZero())!=emptydomain;
    }
    
    private void combine_exi(IntNum pu1, IntNum pl1, IntNum nu1, IntNum nl1, boolean haszero1, IntNum denom1, boolean emptydomain1, 
                     IntNum pu2, IntNum pl2, IntNum nu2, IntNum nl2, boolean haszero2, IntNum denom2, boolean emptydomain2)
    {
        // The first set of arguments is the xi variable's bounds etc.
        // multiply by c2 (which is positive) then combine with 
        // multiply the bounds by c2
        
        assert denom1.isOne() && denom2.isOne();
        assert !emptydomain1;
        
        if(pu1.isZero() && nl1.isZero())
        {
            assert haszero1;
            // This indicates that xi is zero. Therefore propagate through the
            // zero denominator and clear the interval.
            positiveupper=zero;
            positivelower=zero;
            negativeupper=zero;
            negativelower=zero;
            denominator=zero;
            haszero=false;
            emptydomain=true;  // still useful information in the domain.
            return;
        }
        else if(pu1.isZero())
        {   // xi is negative.
            denominator=IntNum.times(c2, IntNum.times(nu1, nl1));
            if(!pu2.isZero() && !nl2.isZero())
            {
                // inner existential spans zero
                positiveupper=IntNum.times(nl1, nl2);
                positivelower=IntNum.times(nu1, nu2);
                negativeupper=IntNum.times(nu1, pl2);
                negativelower=IntNum.times(nl1, pu2);
            }
            else if(!nl2.isZero())
            {   // inner existential is negative.
                positiveupper=IntNum.times(nl1, nl2);
                positivelower=IntNum.times(nu1, nu2);
                negativeupper=negativelower=zero;
            }
            else if(!pu2.isZero())
            {
                // inner existential is positive.
                negativeupper=IntNum.times(nu1, pl2);
                negativelower=IntNum.times(nl1, pu2);
                positiveupper=positivelower=zero;
            }
            else
            {
                // inner existential may be zero
                positiveupper=zero;
                positivelower=zero;
                negativeupper=zero;
                negativelower=zero;
                denominator=one;
            }            
        }
        else if(nl1.isZero())
        {   // xi is positive.
            denominator=IntNum.times(c2, IntNum.times(pu1, pl1));
            if(!pu2.isZero() && !nl2.isZero())
            {
                // inner existential spans zero
                positiveupper=IntNum.times(pu1, pu2);
                positivelower=IntNum.times(pl1, pl2);
                negativeupper=IntNum.times(pl1, nu2);
                negativelower=IntNum.times(pu1, nl2);
            }
            else if(!nl2.isZero())
            {   // inner existential is negative.
                negativeupper=IntNum.times(pl1, nu2);
                negativelower=IntNum.times(pu1, nl2);
                positiveupper=positivelower=zero;
            }
            else if(!pu2.isZero())
            {
                // inner existential is positive.
                positiveupper=IntNum.times(pu1, pu2);
                positivelower=IntNum.times(pl1, pl2);
                negativeupper=negativelower=zero;
            }
            else
            {
                // inner existential may be zero
                positiveupper=zero;
                positivelower=zero;
                negativeupper=zero;
                negativelower=zero;
                denominator=one;
            }
        }
        else
        {   // xi spans zero
            IntNum posproduct=IntNum.times(pu1, pl1);
            IntNum negproduct=IntNum.times(nu1, nl1);
            denominator=IntNum.times(c2, IntNum.times(posproduct, negproduct));
            if(!pu2.isZero() && !nl2.isZero())
            {
                // inner existential spans zero
                // new pu
                IntNum candidate1=IntNum.times(negproduct, IntNum.times(pu1, pu2));
                IntNum candidate2=IntNum.times(posproduct, IntNum.times(nl1, nl2));
                positiveupper=max(candidate1, candidate2);
                
                // new pl
                candidate1=IntNum.times(negproduct, IntNum.times(pl1, pl2));
                candidate2=IntNum.times(posproduct, IntNum.times(nu1, nu2));
                positivelower=min(candidate1, candidate2);
                
                // new nu
                candidate1=IntNum.times(negproduct, IntNum.times(pl1, nu2));
                candidate2=IntNum.times(posproduct, IntNum.times(nu1, pl2));
                negativeupper=max(candidate1, candidate2);
                
                // new nl
                candidate1=IntNum.times(negproduct, IntNum.times(nl1, pu2));
                candidate2=IntNum.times(posproduct, IntNum.times(pu1, nl2));
                negativelower=min(candidate1, candidate2);
            }
            else if(!nl2.isZero())
            {   // inner existential is negative.
                positiveupper=IntNum.times(posproduct, IntNum.times(nl1, nl2));
                positivelower=IntNum.times(posproduct, IntNum.times(nu1, nu2));
                negativeupper=IntNum.times(negproduct, IntNum.times(pl1, nu2));
                negativelower=IntNum.times(negproduct, IntNum.times(pu1, nl2));
            }
            else if(!pu2.isZero())
            {
                // inner existential is positive.
                positiveupper=IntNum.times(negproduct, IntNum.times(pu1, pu2));
                positivelower=IntNum.times(negproduct, IntNum.times(pl1, pl2));
                negativeupper=IntNum.times(posproduct, IntNum.times(nu1, pl2));
                negativelower=IntNum.times(posproduct, IntNum.times(nl1, pu2));
            }
            else
            {
                // inner existential may be zero
                positiveupper=zero;
                positivelower=zero;
                negativeupper=zero;
                negativelower=zero;
                denominator=one;
            }
        }
        
        haszero=haszero2;
        
        // these comparisons are probably not necessary for existentials.
        if(positiveupper.compare(positivelower)<0)
        {
            positiveupper=positivelower=zero;
        }
        if(negativeupper.compare(negativelower)<0)
        {
            negativeupper=negativelower=zero;
        }
        
        emptydomain=(positiveupper.isZero() && negativelower.isZero() && !haszero);
        check();
    }
    
    
    private void combine_axi(IntNum pu1, IntNum pl1, IntNum nu1, IntNum nl1, boolean haszero1, IntNum denom1, boolean emptydomain1, 
                     IntNum pu2, IntNum pl2, IntNum nu2, IntNum nl2, boolean haszero2, IntNum denom2, boolean emptydomain2)
    {
        // The first set of arguments is the xi variable's bounds etc.
        // multiply by c2 (which is positive) then combine with 
        // multiply the bounds by c2
        
        assert denom1.isOne() && denom2.isOne();
        assert !emptydomain1;
        
        if(haszero1 && pu1.isZero() && nl1.isZero())
        {
            // This indicates that xi is zero. Therefore propagate through the
            // zero denominator and clear the interval.
            positiveupper=zero;
            positivelower=zero;
            negativeupper=zero;
            negativelower=zero;
            denominator=zero;
            haszero=false;
            emptydomain=true;  // still useful information in the domain.
            return;
        }
        else if(haszero1 && (!pu1.isZero() || !nl1.isZero()) && !( haszero2 && (!pu2.isZero() || !nl2.isZero())))
        {
            // xi is zero and other values, so the inner resolvent
            // must also have zero and other values to support these.
            // otherwise the domain is emptied. 
            // there is insufficient inner variance so empty
            // the resolvent.
            positiveupper=zero;
            positivelower=zero;
            negativeupper=zero;
            negativelower=zero;
            denominator=one;
            haszero=false;
            emptydomain=true;
            // Watch out for
            // combine_ee reintroducing a 0 into the emptied resolvent when resolving the rest.
            return;
        }
        else if(pu1.isZero())
        {   // xi is negative.
            denominator=IntNum.times(c2, IntNum.times(nu1, nl1));
            if(!pu2.isZero() && !nl2.isZero())
            {
                // inner existential spans zero
                positiveupper=IntNum.times(nu1, nl2);
                positivelower=IntNum.times(nl1, nu2);
                negativeupper=IntNum.times(nl1, pl2);
                negativelower=IntNum.times(nu1, pu2);
            }
            else if(!nl2.isZero())
            {   // inner existential is negative.
                positiveupper=IntNum.times(nu1, nl2);
                positivelower=IntNum.times(nl1, nu2);
                negativeupper=negativelower=zero;
            }
            else if(!pu2.isZero())
            {
                // inner existential is positive.
                negativeupper=IntNum.times(nl1, pl2);
                negativelower=IntNum.times(nu1, pu2);
                positiveupper=positivelower=zero;
            }
            else
            {
                // inner existential may be zero
                positiveupper=zero;
                positivelower=zero;
                negativeupper=zero;
                negativelower=zero;
                denominator=one;
            }
        }
        else if(nl1.isZero())
        {   // xi is positive.
            denominator=IntNum.times(c2, IntNum.times(pu1, pl1));
            if(!pu2.isZero() && !nl2.isZero())
            {
                // inner existential spans zero
                positiveupper=IntNum.times(pl1, pu2);
                positivelower=IntNum.times(pu1, pl2);
                negativeupper=IntNum.times(pu1, nu2);
                negativelower=IntNum.times(pl1, nl2);
            }
            else if(!nl2.isZero())
            {   // inner existential is negative.
                negativeupper=IntNum.times(pu1, nu2);
                negativelower=IntNum.times(pl1, nl2);
                positiveupper=positivelower=zero;
            }
            else if(!pu2.isZero())
            {
                // inner existential is positive.
                positiveupper=IntNum.times(pl1, pu2);
                positivelower=IntNum.times(pu1, pl2);
                negativeupper=negativelower=zero;
            }
            else
            {
                // inner existential may be zero
                positiveupper=zero;
                positivelower=zero;
                negativeupper=zero;
                negativelower=zero;
                denominator=one;
            }
        }
        else
        {   // xi spans zero
            IntNum posproduct=IntNum.times(pu1, pl1);
            IntNum negproduct=IntNum.times(nu1, nl1);
            denominator=IntNum.times(c2, IntNum.times(posproduct, negproduct));
            if(!pu2.isZero() && !nl2.isZero())
            {
                // inner existential spans zero
                // new pu
                IntNum candidate1=IntNum.times(negproduct, IntNum.times(pl1, pu2));
                IntNum candidate2=IntNum.times(posproduct, IntNum.times(nu1, nl2));
                positiveupper=max(candidate1, candidate2);
                
                // new pl
                candidate1=IntNum.times(negproduct, IntNum.times(pu1, pl2));
                candidate2=IntNum.times(posproduct, IntNum.times(nl1, nu2));
                positivelower=min(candidate1, candidate2);
                
                // new nu
                candidate1=IntNum.times(negproduct, IntNum.times(pu1, nu2));
                candidate2=IntNum.times(posproduct, IntNum.times(nl1, pl2));
                negativeupper=max(candidate1, candidate2);
                
                // new nl
                candidate1=IntNum.times(negproduct, IntNum.times(nu1, pu2));
                candidate2=IntNum.times(posproduct, IntNum.times(pl1, nl2));
                negativelower=min(candidate1, candidate2);
            }
            else if(!nl2.isZero())
            {   // inner existential is negative.
                positiveupper=IntNum.times(posproduct, IntNum.times(nu1, nl2));
                positivelower=IntNum.times(posproduct, IntNum.times(nl1, nu2));
                negativeupper=IntNum.times(negproduct, IntNum.times(pu1, nu2));
                negativelower=IntNum.times(negproduct, IntNum.times(pl1, nl2));
            }
            else if(!pu2.isZero())
            {
                // inner existential is positive.
                positiveupper=IntNum.times(negproduct, IntNum.times(pl1, pu2));
                positivelower=IntNum.times(negproduct, IntNum.times(pu1, pl2));
                negativeupper=IntNum.times(posproduct, IntNum.times(nl1, pl2));
                negativelower=IntNum.times(posproduct, IntNum.times(nu1, pu2));
            }
            else
            {
                // inner existential may be zero
                positiveupper=zero;
                positivelower=zero;
                negativeupper=zero;
                negativelower=zero;
                denominator=one;
            }
        }
        
        haszero=haszero2;
        
        if(positiveupper.compare(positivelower)<0)
        {
            positiveupper=positivelower=zero;
        }
        if(negativeupper.compare(negativelower)<0)
        {
            negativeupper=negativelower=zero;
        }
        
        emptydomain=(positiveupper.isZero() && negativelower.isZero() && !haszero);
        check();
    }
    
    private boolean revise_bounds_xi(IntNum pu, IntNum pl, IntNum nu, IntNum nl, boolean haszero, IntNum den)
    {
        // This one divides through by c2.
        // divisions need to be rounded inwards (i.e. towards the other bound, not towards 0.)
        assert denominator.isOne();
        //System.out.println("pu "+pu+" pl "+pl+" nu "+nu+" nl "+nl+" haszero "+haszero+" den "+den);
        if(pu.isZero() && nl.isZero() && !haszero)
        {   // if empty
            return false;
        }
        
        IntNum ub;
        IntNum lb;
        intnum_bounds var=variables[xi];
        
        // haszero indicates that there is a free zero somewhere.
        // We also need to know if there is a zero in an inner universal.
        boolean universalzero=false;
        for(int i=xi+1; i<variables.length; i++)
        {
            if(variables[i].quant() && variables[i].containsZero())
            {
                universalzero=true;
                break;
            }
        }
        
        if(universalzero)
        {
            // The only value here that is consistent with the inner zero is zero.
            ub=lb=zero;
        }
        else if(!pu.isZero() && !nl.isZero())
        {
            // spanzero
            pu=IntNum.quotient(pu, c2, IntNum.FLOOR);
            pl=IntNum.quotient(pl, c2, IntNum.CEILING);
            nu=IntNum.quotient(nu, c2, IntNum.FLOOR);
            nl=IntNum.quotient(nl, c2, IntNum.CEILING);
            
            if(pl.compare(var.upperbound_big())>0)
            {
                // positive section vanishes
                if(haszero && variables[xi].containsZero())
                    ub=zero;
                else
                    ub=nu;
            }
            else
            {
                ub=pu;
            }
            
            if(nu.compare(var.lowerbound_big())<0)
            {
                // negative section vanishes
                if(haszero && variables[xi].containsZero())
                    lb=zero;
                else
                    lb=pl;
            }
            else
            {
                lb=nl;
            }
        }
        else if(!pu.isZero())
        {
            // positive range.
            ub=IntNum.quotient(pu, c2, IntNum.FLOOR);
            lb=IntNum.quotient(pl, c2, IntNum.CEILING);
            /*if(haszero && ( lb.compare(var.upperbound_big())>0 ))
            {
                ub=lb=zero;
            }
            else
            {
                if(variables[xi].containsZero()) lb=zero; //what?
            }*/
            if(haszero)
                lb=zero;
        }
        else if(!nl.isZero())
        {
            // negative
            ub=IntNum.quotient(nu, c2, IntNum.FLOOR);
            lb=IntNum.quotient(nl, c2, IntNum.CEILING);
            /*if(haszero && ( ub.compare(var.lowerbound_big())<0 ))
            {
                ub=lb=zero;
            }
            else
            {
                if(variables[xi].containsZero()) ub=zero;
            }*/
            if(haszero)
                ub=zero;
        }
        else
        {
            assert haszero;
            ub=lb=zero;
        }
        
        //System.out.println("Setting xi to ["+lb+", "+ub+"]");
        
        if(var.upperbound_big().compare(ub)>0)
        {
            //System.out.println("Pruning upperbound");
            boolean flag=var.exclude_upper(ub, null);  // requeue this constraint
            if(!flag) return false;
        }
        
        if(var.lowerbound_big().compare(lb)<0)
        {
            //System.out.println("Pruning lowerbound");
            boolean flag=var.exclude_lower(lb, null);  // requeue
            if(!flag) return false;
        }
        
        //System.out.println("New bounds: ["+var.lowerbound_big()+", "+var.upperbound_big()+"]");
        return true;
    }
    
    private boolean revise_bounds(IntNum pu, IntNum pl, IntNum nu, IntNum nl, boolean haszero, IntNum den, int i)
    {
        // need to divide denominator by the bounds before applying.
        
        // for the time being, we are only going to prune from the top and bottom of the domain.
        intnum_bounds var=variables[i];
        assert !var.empty();
        
        assert !den.isNegative();
        IntNum ub;
        IntNum lb;
        if(den.isZero())
        {
            // this variable must be set to 0
            
            // new: before including 0 in the domain of this variable, check
            // if xi is existential or outer and includes 0. If it does not, then this variable can't be 0.
            if((!variables[xi].quant() || xi<i) && variables[xi].containsZero())
            {
                //System.out.println("xi Variable is existential or outer and contains zero, therefore pruning other variable to 0");
                ub=lb=zero;
            }
            else
            {
                assert false;
                //System.out.println("xi Variable is not suitable, therefore failing.");
                return false;
            }
        }
        else // den is positive.
        {
            boolean rhszero=variables[xi].containsZero() && (xi<i || !variables[xi].quant() || variables[xi].unit());
            
            /*System.out.println("rhszero:"+rhszero);
            System.out.println("contzero:"+variables[xi].containsZero());
            System.out.println("["+variables[xi].lowerbound_big()+", "+variables[xi].upperbound_big()+"]");
            System.out.println("["+((mid_domain)variables[xi]).lowerbound+", "+((mid_domain)variables[xi]).upperbound+"]");
            
            assert !rhszero || (variables[xi].upperbound_big().compare(zero)>=0 && variables[xi].lowerbound_big().compare(zero)<=0);*/
            
            // handle the case where the bounds are all zero and haszero is not, but there is an axi wipeout.
            
            if(!pu.isZero() && !nl.isZero())
            {   // we have a positive and negative range.
                // spanzero
                //System.out.println("Spanzero 1.");
                IntNum pu2=IntNum.quotient(den, pl, IntNum.FLOOR);
                IntNum pl2=IntNum.quotient(den, pu, IntNum.CEILING);
                IntNum nu2=IntNum.quotient(den, nl, IntNum.FLOOR);
                IntNum nl2=IntNum.quotient(den, nu, IntNum.CEILING);
                //System.out.println("["+nl2+", "+nu2+", "+pl2+", "+pu2+"]");
                
                if(pl2.compare(var.upperbound_big())>0)
                {
                    // positive section vanishes
                    if(rhszero && var.containsZero())
                        ub=zero;
                    else
                        ub=nu2;
                }
                else
                {
                    ub=pu2;
                }
                
                if(nu2.compare(var.lowerbound_big())<0)
                {
                    // negative section vanishes
                    if(rhszero && var.containsZero())
                        lb=zero;
                    else
                        lb=pl2;
                }
                else
                {
                    lb=nl2;
                }
            }
            else if(!pu.isZero())
            {   // just a positive range
                
                ub=IntNum.quotient(den, pl, IntNum.FLOOR);
                lb=IntNum.quotient(den, pu, IntNum.CEILING);
                
                if(rhszero && var.containsZero())
                {
                    if(lb.compare(var.upperbound_big())>0)
                    {
                        // the non-zero range lies outside the existing bounds of the variable.
                        ub=lb=zero;
                    }
                    else
                    {
                        if(var.containsZero()) lb=zero;
                    }
                }
            }
            else if(!nl.isZero())
            {   // just a negative range
                lb=IntNum.quotient(den, nu, IntNum.CEILING);
                ub=IntNum.quotient(den, nl, IntNum.FLOOR);
                
                if(rhszero && var.containsZero())
                {
                    if(ub.compare(var.lowerbound_big())<0)
                    {
                        ub=lb=zero;
                    }
                    else
                    {
                        if(var.containsZero()) ub=zero;
                    }
                }
            }
            else
            {
                // the range is empty. What to do?
                if(rhszero)
                {
                    //assert false : "should be dealt with by a zero denominator"; // Untrue.
                    ub=lb=zero;
                }
                else
                {
                    return false;
                }
            }
        }
        
        //System.out.println("Setting "+var+" to ["+lb+", "+ub+"]");
        
        if(var.upperbound_big().compare(ub)>0)
        {
            //System.out.println("Pruning upperbound");
            boolean flag=var.exclude_upper(ub, null);  // requeue
            if(!flag) return false;
        }
        if(var.lowerbound_big().compare(lb)<0)
        {
            //System.out.println("Pruning lowerbound");
            boolean flag=var.exclude_lower(lb, null);  // requeue
            if(!flag) return false;
        }
        
        return true;
    }
    
    /*boolean revise_bounds(IntNum lb, IntNum ub, boolean haszero, IntNum den, int i)
    {
        // need to divide denominator by the upper and lower bound before applying.
        
        // The upper and lower bounds are actually the nearest values to zero iff they span zero.
        // Otherwise they are regular upper and lower bounds.
        // Either way they should be used with the denominator to determine new bounds for variable i.
        
        // If haszero, then lb and ub are allowed to be zero, but normally not.
        
        // new: before including 0 in the domain of this variable, check
        // if xi is existential or outer and includes 0. If it does not, then this variable can't be 0.
        
        if(lb.isZero() || ub.isZero() || (lb.isNegative() && !ub.isNegative()) )
        {
            System.out.println("Resolvent spans zero when attempting to prune a non-xi variable.");
            return true;
            //assert false;
        }
        
        // These need to be rounded inwards (i.e. towards the other bound, not towards 0.)
        assert !den.isNegative();
        
        ub=IntNum.quotient(den, ub, IntNum.FLOOR);
        lb=IntNum.quotient(den, lb, IntNum.CEILING);
        
        // they may have swapped over.
        if(ub.compare(lb)<0)
        {
            IntNum temp=ub;
            ub=lb;
            lb=temp; 
        }
        
        System.out.println("Setting "+variables[i]+" to ["+lb+", "+ub+"]");
        
        intnum_bounds var=variables[i];
        if(var.upperbound_big().compare(ub)>0)
        {
            boolean flag=var.exclude_upper(ub, null);  // requeue
            if(!flag) return false;
        }
        if(var.lowerbound_big().compare(lb)<0)
        {
            boolean flag=var.exclude_lower(lb, null);  // requeue
            if(!flag) return false;
        }
        
        return true;
    }*/
}

class product_constraint_test2
{
    public static void main(String[] args)
    {
        mid_domain [] variables= new mid_domain[3];
        qcsp prob = new qcsp();
        
        variables[0]=new existential(-10, 10, prob, "a");
        //variables[1]=new existential(-10, 10, prob, "b");
        variables[1]=new existential(0, 3, prob, "c");
        variables[2]=new universal(0, 3, prob, "d");
        
        intnum_bounds [] v2={(intnum_bounds)variables[0], (intnum_bounds)variables[1], (intnum_bounds)variables[2]};
        
        product_constraint c1= new product_constraint(v2, 1, new IntNum(2), new IntNum(3), prob);
        c1.establish();
        IntNum ten= new IntNum(10);
        IntNum minusten= new IntNum(-10);
        IntNum three=new IntNum(3);
        System.out.println("10/3="+IntNum.quotient(ten, three));
        System.out.println("-10/3="+IntNum.quotient(minusten, three));
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        /*System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();
        
        System.out.println(c1.make_ac());
        
        prob.printdomains();*/
    }
}

class product_predicate implements predicate_wrapper
{
    product_predicate(int c1, int c2, int xi)
    {
        this.c1=c1;
        this.c2=c2;
        this.xi=xi;
    }
    
    int xi;
    int c1, c2;
    
    public boolean predicate(tuple tau)
    {
        int product=c1;
        for(int i=0; i<tau.vals.length; i++)
        {
            if(i!=xi)
            {
                product=product*tau.vals[i];
            }
        }
        
        int rhs=c2*tau.vals[xi];
        
        return product==rhs;
    }
}

class product_constraint_test
{
    public static void main(String[] args)
    {
        product_constraint_test o1=new product_constraint_test();
        
        for(int i=0; i<100000; i++)
        {
            o1.test_random_prob();
            System.out.println("Done test "+i);
        }
    }
    
    void test_random_prob()
    {
        final int numvars=6;
        intnum_bounds [] vars_product= new intnum_bounds[numvars];
        mid_domain [] vars_sqgac= new mid_domain[numvars];
        
        qcsp prob_product = new qcsp();
        qcsp prob_sqgac = new qcsp();
        
        // 5 types of universal to choose from
        
        // random test.
        
        for(int i=0; i<numvars; i++)
        {
            if(Math.random()>0.8)
            {
                // universal
                double temp=Math.random();
                int ub, lb;
                if(temp<0.2)
                {   // positive
                    ub=12; lb=3;
                }
                else if(temp<0.4)
                {   // includes 0.
                    ub=9; lb=0;
                }
                else if(temp<0.6)
                {   // spans 0
                    ub=6; lb=-3;
                }
                else if(temp<0.8)
                {   // includes 0 negative
                    ub=0; lb=-9;
                }
                else
                {
                    ub=-3; lb=-12;
                }
                vars_product[i]=new universal(lb, ub, prob_product, "x"+(i+1));
                vars_sqgac[i]=new universal(lb, ub, prob_sqgac, "x"+(i+1));
            }
            else
            {
                // existential
                double temp=Math.random();
                int ub, lb;
                if(temp<0.2)
                {   // positive
                    ub=12; lb=3;
                }
                else if(temp<0.4)
                {   // includes 0.
                    ub=9; lb=0;
                }
                else if(temp<0.6)
                {   // spans 0
                    ub=6; lb=-3;
                }
                else if(temp<0.8)
                {   // includes 0 negative
                    ub=0; lb=-9;
                }
                else
                {
                    ub=-3; lb=-12;
                }
                vars_product[i]=new existential(lb, ub, prob_product, "x"+(i+1));
                vars_sqgac[i]=new existential(lb, ub, prob_sqgac, "x"+(i+1));
            }
        }
        
        int xi = (int) (numvars*Math.random());
        assert xi>=0 && xi<=(numvars-1);
        
        product_constraint c1= new product_constraint(vars_product, xi, new IntNum(1), new IntNum(1), prob_product);
        
        predicate_wrapper pred=new product_predicate(1, 1, xi);
        sqgac c2= new sqgac(vars_sqgac, prob_sqgac, pred);
        
        System.out.println(""+c1);
        
        // first check equivalence of the first pass
        
        assert same_domains(prob_product, prob_sqgac) : "something wrong in constructing the problems";
        
        boolean flag1=prob_product.establish();
        boolean flag2=prob_sqgac.establish();
        
        assert !flag2 || flag1 : "Found "+flag2+" for sqgac and "+flag1+" for product";
        if(!flag2)
        {
            return;
        }
        
        flag1=prob_product.propagate();
        flag2=prob_sqgac.propagate();
        
        assert !flag2 || flag1 : "Found "+flag2+" for sqgac and "+flag1+" for product";
        
        if(flag1 && !same_domains(prob_product, prob_sqgac))   // if not false, compare doms.
        {
            System.out.println("different domains");
            System.out.println("Product:");
            prob_product.printdomains();
            System.out.println("SQGAC:");
            prob_sqgac.printdomains();
            assert false;
        }
        
        if(!flag2)
        {
            return;
        }
        
        prob_product.printdomains();
        for(int descents=0; descents<5; descents++)
        {
            System.out.println("New descent.");
            // now make some random assignments up to a leaf node.
            int numassigned=0;
            while(numassigned<9)
            {
                // first check that an assignment is possible
                boolean assignmentposs=false;
                for(int i=0; i<numvars; i++)
                {
                    // making an assumption about the internals of c1:
                    if(!vars_product[i].unit() && !vars_product[i].quant())
                    {
                        assignmentposs=true;
                        break;
                    }
                }
                
                if(!assignmentposs)
                    break;
                
                System.out.println("Assignment possible.");
                
                // pick an assignment to make
                int var=(int)(numvars*Math.random());
                //int val=(int)(2*Math.random());
                
                boolean outer=false;
                for(int i=var-1; i>=0; i--) 
                    if((vars_sqgac[var].quant() && !vars_product[i].quant() && !vars_product[i].unit()) ||
                       (!vars_sqgac[var].quant() && vars_product[i].quant() && !vars_product[i].unit())) outer=true;
                
                while(vars_product[var].unit() || outer)
                {
                    var=(int)(numvars*Math.random());
                    outer=false;
                    for(int i=var-1; i>=0; i--) 
                        if((vars_sqgac[var].quant() && !vars_product[i].quant() && !vars_product[i].unit()) ||
                            (!vars_sqgac[var].quant() && vars_product[i].quant() && !vars_product[i].unit())) outer=true;
                }
                
                int val=(int) (Math.random()*40-20);
                while(!vars_product[var].is_present(new IntNum(val)) || !vars_sqgac[var].is_present(val)) 
                {
                    val=(int) (Math.random()*40-20);
                }
                
                System.out.println("Test harness: Assigning variable "+prob_sqgac.variables.get(var)+" value "+val);
                
                /*try
                {
                    File output = new File("or_constraint_test.dot");
                    FileWriter out = new FileWriter(output);
                    out.write(c2.printtree_alt());
                    out.close();
                }
                catch(java.io.IOException e)
                {
                    System.out.println("Problem writing sqgac_test.dot");
                }*/
                
                numassigned++;
                prob_product.add_backtrack_level();
                prob_sqgac.add_backtrack_level();
                
                ((mid_domain)prob_product.variables.get(var)).instantiate(val);
                ((mid_domain)prob_sqgac.variables.get(var)).instantiate(val);
                
                for(int i=0; i<numvars; i++)
                {
                    assert !vars_product[i].empty();
                    assert !vars_sqgac[i].empty();
                }
                
                flag1=prob_product.propagate();
                flag2=prob_sqgac.propagate();
                
                if(flag2 && !flag1)
                {
                    System.out.println("product:");
                    prob_product.printdomains();
                    System.out.println("sqgac:");
                    prob_sqgac.printdomains();
                    
                    /*try
                    {
                        File output = new File("product_constraint_test2.dot");
                        FileWriter out = new FileWriter(output);
                        out.write(c2.printtree_alt());
                        out.close();
                    }
                    catch(java.io.IOException e)
                    {
                        System.out.println("Problem writing sqgac_test.dot");
                    }*/
                    
                    assert false : "found "+flag1+" for product_constraint and "+flag2+" for sqgac";
                }
                
                if(flag2 && !same_domains(prob_product, prob_sqgac))   // if not false, compare doms.
                {
                    System.out.println("different domains");
                    System.out.println("product:");
                    prob_product.printdomains();
                    System.out.println("sqgac:");
                    prob_sqgac.printdomains();
                    
                    /*try
                    {
                        File output = new File("or_constraint_test2.dot");
                        FileWriter out = new FileWriter(output);
                        out.write(c2.printtree_alt());
                        out.close();
                    }
                    catch(java.io.IOException e)
                    {
                        System.out.println("Problem writing sqgac_test.dot");
                    }*/
                    assert false;
                }
                
                if(!flag1 || !flag2)
                    break;  // no point making more assignments.
                
                
            }
            
            // backtrack before starting the next descent.
            for(int i=0; i<numassigned; i++)
            {
                prob_product.backtrack();
                prob_sqgac.backtrack();
            }
        }
    }
    
    boolean same_domains(qcsp prob1, qcsp prob2)
    {
        assert prob1.variables.size()==prob2.variables.size();  // would be silly otherwise.
        
        for(int i=0; i<prob1.variables.size(); i++)
        {
            mid_domain var1 = (mid_domain)prob1.variables.get(i);
            mid_domain var2 = (mid_domain)prob2.variables.get(i);
            
            //if(var1.domsize()!=var2.domsize())
            //    return false;
            
            if(var1.lowerbound()!=var2.lowerbound())
            {
                System.out.println("Difference in lowerbound:"+var1+" lb1 "+var1.lowerbound()+" lb2 "+var2.lowerbound());
                return false;
            }
            
            if(var1.upperbound()!=var2.upperbound())
            {
                System.out.println("Difference in upperbound:"+var1);
                return false;
            }
            
            //for(int j=var1.lowerbound(); j<=var1.upperbound(); j++)
            //{
            //    if(var1.is_present(j)!=var2.is_present(j))
            //    {
            //        System.out.println("Difference with value "+j+" variable "+var1);
            //        return false;
            //    }
            //}
        }
        return true;
    }
}


class product_constraint_test3
{
    // checks if solving a QCSP involving a single product constraint
    // produces the same result as propagating the SQGAC version.
    public static void main(String[] args)
    {
        product_constraint_test3 o1=new product_constraint_test3();
        
        for(int i=0; i<100000; i++)
        {
            o1.test_random_prob();
            System.out.println("Done test "+i);
        }
    }
    
    void test_random_prob()
    {
        final int numvars=6;
        intnum_bounds [] vars_product= new intnum_bounds[numvars];
        mid_domain [] vars_sqgac= new mid_domain[numvars];
        
        qcsp prob_product = new qcsp();
        qcsp prob_sqgac = new qcsp();
        
        // 5 types of universal to choose from
        // random test.
        
        for(int i=0; i<numvars; i++)
        {
            if(Math.random()>0.8)
            {
                // universal
                double temp=Math.random();
                int ub, lb;
                if(temp<0.2)
                {   // positive
                    ub=12; lb=3;
                }
                else if(temp<0.4)
                {   // includes 0.
                    ub=9; lb=0;
                }
                else if(temp<0.6)
                {   // spans 0
                    ub=6; lb=-3;
                }
                else if(temp<0.8)
                {   // includes 0 negative
                    ub=0; lb=-9;
                }
                else
                {
                    ub=-3; lb=-12;
                }
                vars_product[i]=new universal(lb, ub, prob_product, "x"+(i+1));
                vars_sqgac[i]=new universal(lb, ub, prob_sqgac, "x"+(i+1));
            }
            else
            {
                // existential
                double temp=Math.random();
                int ub, lb;
                if(temp<0.2)
                {   // positive
                    ub=12; lb=3;
                }
                else if(temp<0.4)
                {   // includes 0.
                    ub=9; lb=0;
                }
                else if(temp<0.6)
                {   // spans 0
                    ub=6; lb=-3;
                }
                else if(temp<0.8)
                {   // includes 0 negative
                    ub=0; lb=-9;
                }
                else
                {
                    ub=-3; lb=-12;
                }
                vars_product[i]=new existential(lb, ub, prob_product, "x"+(i+1));
                vars_sqgac[i]=new existential(lb, ub, prob_sqgac, "x"+(i+1));
            }
        }
        
        int xi = (int) (numvars*Math.random());
        assert xi>=0 && xi<=(numvars-1);
        
        product_constraint c1= new product_constraint(vars_product, xi, new IntNum(1), new IntNum(1), prob_product);
        
        predicate_wrapper pred=new product_predicate(1, 1, xi);
        sqgac c2= new sqgac(vars_sqgac, prob_sqgac, pred);
        
        boolean flag1=prob_product.establish();
        boolean flag2=prob_sqgac.establish();
        
        // remember sqgac is stronger
        
        System.out.println(c1);
        
        assert !flag2 || flag1 : "SQGAC found true and product found false";
        
        if(flag1) flag1=prob_product.search();
        
        System.out.println(c1);
        
        System.out.println("flag1="+flag1+", flag2="+flag2);
        assert flag1==flag2 : "Truth values do not match";
    }
}

class product_constraint_test4
{
    //same as product_constraint_test but it only checks if each bound is in the right segment i.e. positive, 0 or negative.
    public static void main(String[] args)
    {
        product_constraint_test4 o1=new product_constraint_test4();
        
        for(int i=0; i<100000; i++)
        {
            o1.test_random_prob();
            System.out.println("Done test "+i);
        }
    }
    
    void test_random_prob()
    {
        final int numvars=6;
        intnum_bounds [] vars_product= new intnum_bounds[numvars];
        mid_domain [] vars_sqgac= new mid_domain[numvars];
        
        qcsp prob_product = new qcsp();
        qcsp prob_sqgac = new qcsp();
        
        // 5 types of universal to choose from
        
        // random test.
        
        for(int i=0; i<numvars; i++)
        {
            if(Math.random()>0.8)
            {
                // universal
                double temp=Math.random();
                int ub, lb;
                if(temp<0.2)
                {   // positive
                    ub=12; lb=3;
                }
                else if(temp<0.4)
                {   // includes 0.
                    ub=9; lb=0;
                }
                else if(temp<0.6)
                {   // spans 0
                    ub=6; lb=-3;
                }
                else if(temp<0.8)
                {   // includes 0 negative
                    ub=0; lb=-9;
                }
                else
                {
                    ub=-3; lb=-12;
                }
                vars_product[i]=new universal(lb, ub, prob_product, "x"+(i+1));
                vars_sqgac[i]=new universal(lb, ub, prob_sqgac, "x"+(i+1));
                
            }
            else
            {
                // existential
                double temp=Math.random();
                int ub, lb;
                if(temp<0.2)
                {   // positive
                    ub=12; lb=3;
                }
                else if(temp<0.4)
                {   // includes 0.
                    ub=9; lb=0;
                }
                else if(temp<0.6)
                {   // spans 0
                    ub=6; lb=-3;
                }
                else if(temp<0.8)
                {   // includes 0 negative
                    ub=0; lb=-9;
                }
                else
                {
                    ub=-3; lb=-12;
                }
                vars_product[i]=new existential(lb, ub, prob_product, "x"+(i+1));
                vars_sqgac[i]=new existential(lb, ub, prob_sqgac, "x"+(i+1));
                
            }
        }
        
        int xi = (int) (numvars*Math.random());
        assert xi>=0 && xi<=(numvars-1);
        
        product_constraint c1= new product_constraint(vars_product, xi, new IntNum(1), new IntNum(1), prob_product);
        
        predicate_wrapper pred=new product_predicate(1, 1, xi);
        sqgac c2= new sqgac(vars_sqgac, prob_sqgac, pred);
        
        System.out.println(""+c1);
        
        // first check equivalence of the first pass
        
        assert same_domains(prob_product, prob_sqgac) : "something wrong in constructing the problems";
        
        boolean flag1=prob_product.establish();
        boolean flag2=prob_sqgac.establish();
        
        assert !flag2 || flag1 : "Found "+flag2+" for sqgac and "+flag1+" for product";
        if(!flag2)
        {
            return;
        }
        
        flag1=prob_product.propagate();
        flag2=prob_sqgac.propagate();
        
        assert !flag2 || flag1 : "Found "+flag2+" for sqgac and "+flag1+" for product";
        
        if(flag2 && !same_domains(prob_product, prob_sqgac))   // if not false, compare doms.
        {
            System.out.println("different domains");
            System.out.println("Product:");
            prob_product.printdomains();
            System.out.println("SQGAC:");
            prob_sqgac.printdomains();
            assert false;
        }
        
        if(!flag2)
        {
            return;
        }
        
        prob_product.printdomains();
        for(int descents=0; descents<5; descents++)
        {
            System.out.println("New descent.");
            // now make some random assignments up to a leaf node.
            int numassigned=0;
            while(numassigned<9)
            {
                // first check that an assignment is possible
                boolean assignmentposs=false;
                for(int i=0; i<numvars; i++)
                {
                    // making an assumption about the internals of c1:
                    if(!vars_product[i].unit() && !vars_product[i].quant())
                    {
                        assignmentposs=true;
                        break;
                    }
                }
                
                if(!assignmentposs)
                    break;
                
                System.out.println("Assignment possible.");
                
                // pick an assignment to make
                int var=(int)(numvars*Math.random());
                //int val=(int)(2*Math.random());
                
                boolean outer=false;
                for(int i=var-1; i>=0; i--) 
                    if((vars_sqgac[var].quant() && !vars_product[i].quant() && !vars_product[i].unit()) ||
                       (!vars_sqgac[var].quant() && vars_product[i].quant() && !vars_product[i].unit())) outer=true;
                
                while(vars_product[var].unit() || outer)
                {
                    var=(int)(numvars*Math.random());
                    outer=false;
                    for(int i=var-1; i>=0; i--) 
                        if((vars_sqgac[var].quant() && !vars_product[i].quant() && !vars_product[i].unit()) ||
                            (!vars_sqgac[var].quant() && vars_product[i].quant() && !vars_product[i].unit())) outer=true;
                }
                
                int val=(int) (Math.random()*40-20);
                while(!vars_product[var].is_present(new IntNum(val)) || !vars_sqgac[var].is_present(val)) 
                {
                    val=(int) (Math.random()*40-20);
                }
                
                System.out.println("Test harness: Assigning variable "+prob_sqgac.variables.get(var)+" value "+val);
                
                /*try
                {
                    File output = new File("or_constraint_test.dot");
                    FileWriter out = new FileWriter(output);
                    out.write(c2.printtree_alt());
                    out.close();
                }
                catch(java.io.IOException e)
                {
                    System.out.println("Problem writing sqgac_test.dot");
                }*/
                
                numassigned++;
                prob_product.add_backtrack_level();
                prob_sqgac.add_backtrack_level();
                
                prob_product.printdomains();
                
                ((mid_domain)prob_product.variables.get(var)).instantiate(val);
                ((mid_domain)prob_sqgac.variables.get(var)).instantiate(val);
                
                for(int i=0; i<numvars; i++)
                {
                    assert !vars_product[i].empty();
                    assert !vars_sqgac[i].empty();
                }
                
                flag1=prob_product.propagate();
                flag2=prob_sqgac.propagate();
                
                if(flag2 && !flag1)
                {
                    System.out.println("product:");
                    prob_product.printdomains();
                    System.out.println("sqgac:");
                    prob_sqgac.printdomains();
                    
                    /*try
                    {
                        File output = new File("product_constraint_test2.dot");
                        FileWriter out = new FileWriter(output);
                        out.write(c2.printtree_alt());
                        out.close();
                    }
                    catch(java.io.IOException e)
                    {
                        System.out.println("Problem writing sqgac_test.dot");
                    }*/
                    
                    assert false : "found "+flag1+" for product_constraint and "+flag2+" for sqgac";
                }
                
                if(flag2 && !same_domains(prob_product, prob_sqgac))   // if not false, compare doms.
                {
                    System.out.println("different domains");
                    System.out.println("product:");
                    prob_product.printdomains();
                    System.out.println("sqgac:");
                    prob_sqgac.printdomains();
                    
                    /*try
                    {
                        File output = new File("or_constraint_test2.dot");
                        FileWriter out = new FileWriter(output);
                        out.write(c2.printtree_alt());
                        out.close();
                    }
                    catch(java.io.IOException e)
                    {
                        System.out.println("Problem writing sqgac_test.dot");
                    }*/
                    assert false;
                }
                
                if(!flag1 || !flag2)
                    break;  // no point making more assignments.
                
                
            }
            
            // backtrack before starting the next descent.
            for(int i=0; i<numassigned; i++)
            {
                prob_product.backtrack();
                prob_sqgac.backtrack();
            }
        }
    }
    
    boolean same_domains(qcsp prob1, qcsp prob2)
    {
        assert prob1.variables.size()==prob2.variables.size();  // would be silly otherwise.
        
        for(int i=0; i<prob1.variables.size(); i++)
        {
            mid_domain var1 = (mid_domain)prob1.variables.get(i);
            mid_domain var2 = (mid_domain)prob2.variables.get(i);
            
            //if(var1.domsize()!=var2.domsize())
            //    return false;
            int sign1=(var1.lowerbound()>0?1:(var1.lowerbound()>-1?0:-1));
            int sign2=(var2.lowerbound()>0?1:(var2.lowerbound()>-1?0:-1));
            if(sign1!=sign2)
            {
                System.out.println("Difference in lowerbound:"+var1+" lb1 "+var1.lowerbound()+" lb2 "+var2.lowerbound());
                return false;
            }
            
            sign1=(var1.upperbound()>0?1:(var1.upperbound()>-1?0:-1));
            sign2=(var2.upperbound()>0?1:(var2.upperbound()>-1?0:-1));
            if(sign1!=sign2)
            {
                System.out.println("Difference in upperbound:"+var1+" ub1 "+var1.upperbound()+" ub2 "+var2.upperbound());
                return false;
            }
        }
        return true;
    }
}
