////////////////////////////////////////////////////////////////////////////////////
// Simple non-binary constraint for small domains
//

package queso.constraints;

import gnu.trove.*;
import java.util.*;
import gnu.math.*;
import queso.core.*;

public class table_constraint extends constraint implements make_ac
{
    final int[][] tables;
    
    final mid_domain [] variables;
    
    public predicate_wrapper pred;
    
    public table_constraint(mid_domain[] variables, int [][] tables, qcsp problem)
    {
        super(problem);
        this.variables=variables;
        this.tables=tables;
    }
    
    public table_constraint(mid_domain [] vars, qcsp prob, predicate_wrapper pred)
    {
        super(prob);
        this.variables=vars;
        tables=pred2table(pred);
        this.pred=pred;
    }
    
    private int [][] pred2table(predicate_wrapper pred)
    {
        // convert a predicate to a table of satisfying tuples.
        int [] tuplearray=new int[variables.length];
        for(int i=0; i<variables.length; i++)
        {
            tuplearray[i]=variables[i].lowerbound();
        }
        
        ArrayList<int []> table=new ArrayList<int []>();
        
        if(pred.predicate(new tuple(tuplearray)))
        {
            table.add((int [])tuplearray.clone());
        }
        
        while(increment(tuplearray))
        {
            if(pred.predicate(new tuple(tuplearray)))
            {
                table.add((int [])tuplearray.clone());
            }
        }
        
        int [][] rettable=new int[table.size()][];
        for(int i=0; i<rettable.length; i++)
        {
            rettable[i]=(int []) table.get(i);
        }
        
        return rettable;
    }
    
    private boolean increment(int [] tuplearray)
    {
        boolean iterate=true;
        int k=variables.length-1;
        while(iterate)
        {
            iterate=false;
            
            while(k>=0 && tuplearray[k]==variables[k].upperbound())
            {   // symbol at position k cannot be increased.
                k--;
                iterate=true;
            }
            
            if(k<0) return false;
        }
        
        // increment the symbol at position k
        for(int val=tuplearray[k]+1; val<=variables[k].upperbound(); val++)
        {
            if(variables[k].is_present(val))
            {
                tuplearray[k]=val;
                break;
            }
        }
        
        // zero the rest
        for(int index=k+1; index<variables.length; index++)
        {
            tuplearray[index]=variables[index].lowerbound();
        }
        
        return true;
    }
    
    public boolean establish()
    {
        for(int i=0; i<variables.length; i++)
        {
            variables[i].add_wakeup(this);
        }
        return make_ac();
    }
    
    public variable_iface [] variables()
    {
        return (variable_iface []) variables;
    }
    
    public boolean entailed()
    {
        for(int i=0; i<variables.length; i++)
        {
            if(!variables[i].unit())
                return false;
        }
        return true;
    }
    
    public boolean make_ac()
    {
        boolean iterate=true;
        int[] pavars=new int[variables.length];
        int[] pavals=new int[variables.length];
        //System.out.println("In make_ac for table_constraint");
        while(iterate)
        {
            iterate=false;
            
            for(int var=0; var<variables.length; var++)
            {
                for(int val=variables[var].lowerbound(); val<=variables[var].upperbound(); val++)
                {
                    if(variables[var].is_present(val))
                    {
                        // check the value is supported
                        pavars[0]=var;
                        pavals[0]=val;
                        if(!makepa(var+1, 1, pavars, pavals))
                        {
                            // no support for this value
                            if(!variables[var].exclude(val, this))
                            {
                                return false;
                            }
                            iterate=true;
                        }
                    }
                }
            }
        }
        return true;
    }
    
    private boolean makepa(int curvar, int paindex, int [] pavars, int [] pavals)
    {
        // pavars is full of -1's to begin with.
        if(curvar==variables.length)
        {
            return findtuple(paindex, pavars, pavals);
        }
        else if(variables[curvar].quant())
        {
            pavars[paindex]=curvar;
            for(int val=variables[curvar].lowerbound(); val<=variables[curvar].upperbound(); val++)
            {
                if(variables[curvar].is_present(val))
                {
                    pavals[paindex]=val;
                    boolean flag= makepa(curvar+1, paindex+1, pavars, pavals);
                    if(!flag) return false;
                }
            }
            return true;
        }
        else
        {
            return makepa(curvar+1, paindex, pavars, pavals);
        }
    }
    
    private boolean findtuple(int palength, int [] pavars, int [] pavals)
    {
        // int [][] tables
        for(int i=0; i<tables.length; i++)
        {
            int [] tau=tables[i];  // does tau match the pa and domains?
            boolean flag=true;
            for(int var=0; var<variables.length; var++)
            {
                if(!variables[var].is_present(tau[var]))
                {
                    flag=false;
                    break;
                }
                for(int j=0; j<palength; j++)
                {
                    assert pavars[j]!=-1;
                    if(pavars[j]==var && pavals[j]!=tau[var])
                    {
                        flag=false;
                        break;
                    }
                }
            }
            if(flag)
            {
                assert valid(new tuple(tables[i]));
                //System.out.println("Found support:"+(new tuple(tables[i])));
                return true;
            }
        }
        return false;
    }
    
    boolean valid(tuple tau)
    {
        if(tau==null)
            return false;   // surely a null tuple is not a valid tuple??
        
        for(int i=0; i<tau.vals.length; i++)
        {
            if(!variables[i].is_present(tau.vals[i]))
                return false;
        }
        return true;
    }
    
    public String toString()
    {
        String st="Table constraint:\n";
        for(int i=0; i<tables.length; i++)
        {
            System.out.println(variables[i].toString()+":");
            for(int j=0; j<tables[i].length; j++)
            {
                st+=tables[i][j]+", ";
            }
            st+="\n";
        }
        return st;
    }
}

///////////////////////////////////////////////////////////////////////////////
// test against gac-schema+predicate or positive

class table_constraint_test
{
    public static void main(String[] args)
    {
        table_constraint_test o1=new table_constraint_test();
        o1.do_tests();
    }
    
    void do_tests()
    {
        r=new Random(12345);
        sw=new stopwatch("GMT");
        
        for(int i=0; i<100000; i++)
        {
            test_random_prob();
            System.out.println("Done test "+(i+1));
        }
    }
    
    Random r; // random number generator.
    stopwatch sw;
    
    private int [] incrementtuple(int [] arg)
    {
        int [] ret=(int [])arg.clone();
        
        for(int i=ret.length-1; i>=0; i--)
        {
            if(ret[i]<5)
            {
                ret[i]++;
                for(int j=i+1; j<ret.length; j++)
                {
                    ret[j]=-5;
                }
                return ret;
            }
        }
        return null;
    }
    
    void test_random_prob()
    {
        // assume variable domains -5..5
        final int numvars=4;
        
        mid_domain [] vars_sqgac= new mid_domain[numvars];
        mid_domain [] vars_other= new mid_domain[numvars];
        
        qcsp prob_sqgac = new qcsp();
        qcsp prob_other = new qcsp();
        
        for(int i=0; i<numvars; i++)
        {
                if(r.nextFloat()>0.5)
                {
                    vars_other[i]=new universal(0, 1, prob_other, "bx"+(i+1));
                    vars_sqgac[i]=new universal(0, 1, prob_sqgac, "bx"+(i+1));
                }
                else
                {
                    vars_other[i]=new existential(0, 1, prob_other, "bx"+(i+1));
                    vars_sqgac[i]=new existential(0, 1, prob_sqgac, "bx"+(i+1));
                }
        }
        
        int [][] table2=tuples.generateprobability(4, vars_sqgac, r, (float)0.5);
        
        System.out.println("Table:"+tuples.atos(table2));
        
        predicate_wrapper pred=new table_predicate(table2);
        
        table_constraint c1= new table_constraint(vars_sqgac, prob_sqgac, pred);
        
        //THashMap goods=new THashMap();
        //gac_schema_positive c2=new gac_schema_positive(vars_other, prob_other, goods);
        //c2.splittuples(goods, table2);
        
        
        gac_schema_positive c2= new gac_schema_positive(vars_other, prob_other, pred);
        
        //prob_sqgac.pure_setup_universals();
        //prob_other.pure_setup_universals();
        
        // start the testing
        
        // first check equivalence of the first pass
        
        assert same_domains(prob_other, prob_sqgac) : "something wrong in constructing the problems";
        
        prob_sqgac.add_backtrack_level();
        prob_other.add_backtrack_level();
        
        System.out.println("Calling establish");
        boolean flag1=prob_other.establish();
        boolean flag2=prob_sqgac.establish();
        
        assert !flag2 || flag1 : "Found "+flag2+" for sqgac and "+flag1+" for other";
        if(!flag2)
        {
            return;
        }
        
        System.out.println("Calling propagate");
        flag1=prob_other.propagate();
        flag2=prob_sqgac.propagate();
        
        assert !flag2 || flag1 : "Found "+flag2+" for sqgac and "+flag1+" for other";
        
        if(flag2 && !same_domains(prob_other, prob_sqgac))   // if not false, compare doms.
        {
            System.out.println("different domains");
            System.out.println("other:");
            prob_other.printdomains();
            System.out.println("SQGAC:");
            prob_sqgac.printdomains();
            assert false;
        }
        
        if(!flag2)
        {
            return;
        }
        
        prob_other.printdomains();
        for(int descents=0; descents<5; descents++)
        {
            System.out.println("New descent.");
            // now make some random assignments up to a leaf node.
            int numassigned=0;
            while(numassigned<vars_other.length)
            {
                // first check that an assignment is possible
                boolean assignmentposs=false;
                for(int i=0; i<vars_other.length; i++)
                {
                    // making an assumption about the internals of c1:
                    if(!vars_other[i].unit())
                    {
                        assignmentposs=true;
                        break;
                    }
                }
                
                if(!assignmentposs)
                    break;
                
                System.out.println("Assignment possible.");
                
                // pick an assignment to make
                int var=(int)(numvars*r.nextFloat());
                
                boolean outer=false;
                for(int i=var-1; i>=0; i--) 
                    if((vars_other[var].quant() && !vars_other[i].quant() && !vars_other[i].unit()) ||
                       (!vars_other[var].quant() && vars_other[i].quant() && !vars_other[i].unit())) outer=true;
                
                while(vars_other[var].unit() || outer)
                {
                    var=(int)(numvars*r.nextFloat());
                    outer=false;
                    for(int i=var-1; i>=0; i--) 
                        if((vars_sqgac[var].quant() && !vars_other[i].quant() && !vars_other[i].unit()) ||
                            (!vars_sqgac[var].quant() && vars_other[i].quant() && !vars_other[i].unit())) outer=true;
                }
                
                int val=(int) (r.nextFloat()*40-20);
                
                while(!vars_other[var].is_present(val) || !vars_sqgac[var].is_present(val)) 
                {
                    val=(int) (r.nextFloat()*40-20);
                }
                
                System.out.println("Test harness: Assigning variable "+prob_sqgac.variables.get(var)+" value "+val);
                
                numassigned++;
                prob_other.add_backtrack_level();
                prob_sqgac.add_backtrack_level();
                
                ((mid_domain)prob_other.variables.get(var)).instantiate(val);
                ((mid_domain)prob_sqgac.variables.get(var)).instantiate(val);
                
                for(int i=0; i<numvars; i++)
                {
                    assert !vars_other[i].empty();
                    assert !vars_sqgac[i].empty();
                }
                
                flag1=prob_other.propagate();
                flag2=prob_sqgac.propagate();
                
                if(flag2 && !flag1)
                {
                    System.out.println("other:");
                    prob_other.printdomains();
                    System.out.println("sqgac:");
                    prob_sqgac.printdomains();
                    
                    assert false : "found "+flag1+" for other_constraint and "+flag2+" for sqgac";
                }
                
                if(flag2 && !same_domains(prob_other, prob_sqgac))   // if not false, compare doms.
                {
                    System.out.println("different domains");
                    System.out.println("other:");
                    prob_other.printdomains();
                    System.out.println("sqgac:");
                    prob_sqgac.printdomains();
                    
                    assert false;
                }
                
                if(!flag1 || !flag2)
                    break;  // no point making more assignments.
            }
            
            // backtrack before starting the next descent.
            for(int i=0; i<numassigned; i++)
            {
                prob_other.backtrack();
                prob_sqgac.backtrack();
            }
        }
    }
    
    boolean same_domains(qcsp prob1, qcsp prob2)
    {
        if(prob1.variables.size()!=prob2.variables.size())
            return false;
        
        for(int i=0; i<prob1.variables.size(); i++)
        {
            mid_domain var1 = (mid_domain)prob1.variables.get(i);
            mid_domain var2 = (mid_domain)prob2.variables.get(i);
            
            if(var1.domsize()!=var2.domsize())
                return false;
            
            if(var1.lowerbound()!=var2.lowerbound())
                return false;
            
            if(var1.upperbound()!=var2.upperbound())
                return false;
            
            for(int j=var1.lowerbound(); j<=var1.upperbound(); j++)
            {
                if(var1.is_present(j)!=var2.is_present(j))
                    return false;
            }
        }
        return true;
    }
}


