package savilerow.expression;
/*

    Savile Row http://savilerow.cs.st-andrews.ac.uk/
    Copyright (C) 2014-2024 Peter Nightingale
    
    This file is part of Savile Row.
    
    Savile Row is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Savile Row is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Savile Row.  If not, see <http://www.gnu.org/licenses/>.

*/


import java.io.BufferedWriter;
import java.io.IOException;
import java.util.ArrayList;

import savilerow.CmdFlags;
import savilerow.model.SymbolTable;

public class IfThenElse extends ASTNodeC
{
    public static final long serialVersionUID = 1L;
    public IfThenElse(ASTNode a, ASTNode b, ASTNode c) {
        super(a,b,c);
    }
    public IfThenElse(ASTNode[] ch) {
        super(ch);
        assert ch.length==3;
    }
    
    public ASTNode copy()
    {
        return new IfThenElse(getChildrenArray());
    }
    public boolean isNumerical() {
        return getChild(1).isNumerical() || getChild(2).isNumerical();
    }
    public boolean isRelation() {
        return getChild(1).isRelation() && getChild(2).isRelation();
    }
    
    public boolean toFlatten(boolean propagate){return false;}
    
    public ASTNode simplify()
    {
        if(getChild(0).isConstant()) {
            long cval=getChild(0).getValue();
            if(cval==0) {
                getChild(2).setParent(null);
                return getChild(2);
            }
            else {
                getChild(1).setParent(null);
                return getChild(1);
            }
        }
        return null;
    }
    
    public boolean typecheck(SymbolTable st) {
        for(int i=0; i<numChildren(); i++) {
            if(!getChild(i).typecheck(st)) return false;
            if(getChild(i).getDimension()>0) {
                CmdFlags.println("ERROR: Unexpected matrix in 'if': "+this);
                return false;
            }
            if(getChild(i).getCategory()>ASTNode.Quantifier) {
                CmdFlags.println("ERROR: 'if' cannot contain decision variables: "+this);
                return false;
            }
        }
        
        // Condition has to be boolean
        if(!getChild(0).isRelation()) {
            CmdFlags.println("ERROR: First argument of function 'if' must be boolean: "+this);
            return false;
        }
        if(! (getChild(1).isRelation() || getChild(1).isNumerical())) {
            CmdFlags.println("ERROR: Second argument of function 'if' must be integer or boolean: "+this);
            return false;
        }
        if(! (getChild(2).isRelation() || getChild(2).isNumerical())) {
            CmdFlags.println("ERROR: Third argument of function 'if' must be integer or boolean: "+this);
            return false;
        }
        
        return true;
    }
    
    public Intpair getBounds() {
        // The lower bound is the min of all lower bounds of children,
        // and similarly for the upper bound.
        Intpair a=getChild(1).getBounds();
        Intpair b=getChild(2).getBounds();
        if(b.lower<a.lower) {
            a.lower=b.lower;
        }
        if(b.upper>a.upper) {
            a.upper=b.upper;
        }
        return a;
    }
    
    public ArrayList<Intpair> getIntervalSetExp() {
        ArrayList<Intpair> intervals1=getChild(1).getIntervalSetExp();
        ArrayList<Intpair> intervals2=getChild(2).getIntervalSetExp();
        return Intpair.union(intervals1, intervals2);
    }
    
    public String toString() {
        return generic_to_string("if");
    }
}
