package savilerow.expression;
/*

    Savile Row http://savilerow.cs.st-andrews.ac.uk/
    Copyright (C) 2014-2024 Peter Nightingale
    
    This file is part of Savile Row.
    
    Savile Row is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Savile Row is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Savile Row.  If not, see <http://www.gnu.org/licenses/>.

*/


import java.io.BufferedWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;

import savilerow.*;
import savilerow.model.*;

// One child, a matrix type

public class Circuit extends ASTNodeC {
    public static final long serialVersionUID = 1L;
    public Circuit(ASTNode r) {
        super(r);
    }

    public ASTNode copy() {
        return new Circuit(getChild(0));
    }

    public boolean isRelation() { return true; }

    public boolean typecheck(SymbolTable st) {
        if (!getChild(0).typecheck(st)) {
            return false;
        }
        if (getChild(0).getDimension() != 1) {
            CmdFlags.println("ERROR: Expected one-dimensional matrix in circuit constraint: " + this);
            return false;
        }
        return true;
    }
    
    public ASTNode simplify() {
        ASTNode mat=getChildConst(0);
        
        if((mat instanceof CompoundMatrix || mat instanceof EmptyMatrix) && getChild(0).getCategory()==ASTNode.Constant) {
            if(mat instanceof EmptyMatrix) {
                return new BooleanConstant(true);
            }
            
            // check there are no sub-cycles. 
            ArrayList<Long> seq=new ArrayList<>();
            
            ArrayList<Intpair> idx=mat.getChild(0).getIntervalSet();
            long lb=idx.get(0).lower;
            
            long cur=lb;
            seq.add(cur);  //  start at the smallest index. 
            
            while(true) {
                cur=mat.getChild((int)(cur-lb+1)).getValue();
                if(seq.contains(cur)) {
                    break;
                }
                seq.add(cur);
            }
            
            return new BooleanConstant(seq.size() == mat.numChildren()-1);
        }
        return null;
    }
    
    public String toString() {
        return generic_to_string("circuit");
    }
    
    public void toFlatzinc(BufferedWriter b, boolean bool_context) throws IOException {
        if(CmdFlags.getOrtoolstrans() || CmdFlags.getChuffedtrans()) {
            if(CmdFlags.getOrtoolstrans()) {
                b.append("constraint ortools_circuit(");
            }
            else {
                b.append("constraint chuffed_circuit(");
            }
            getChild(0).toFlatzinc(b, false);
            b.append(",");
            b.append(String.valueOf(getChild(0).getChild(0).getBounds().lower));  // base index, usually 1.
            b.append(");");
        }
        else {
            if(CmdFlags.getGecodetrans()) {
                b.append("constraint gecode_circuit(");
            }
            else {
                b.append("constraint fzn_circuit(");
            }
            b.append(String.valueOf(getChild(0).getChild(0).getBounds().lower));  // base index, usually 1.
            b.append(",");
            getChild(0).toFlatzinc(b, false);
            b.append(");");
        }
    }
    public void toMinizinc(StringBuilder b, boolean bool_context) {
        b.append("circuit(");
        getChild(0).toMinizinc(b, false);  //  Check this includes the index domain.
        b.append(")");
    }
    
    ////////////////////////////////////////////////////////////////////////////
    //  JSON output for symmetry detection
    
    /*public void toJSON(StringBuilder bf) {
        toJSONHeader(bf, true);
        // children
        bf.append("\"Children\": [");
        if(getChild(0) instanceof CompoundMatrix && getChild(0).numChildren()==3) {
            //   Special case for binary != constraint.
            getChild(0).getChild(1).toJSON(bf);
            bf.append(", ");
            getChild(0).getChild(2).toJSON(bf);
        }
        else {
            // Same as toJSON method in ASTNode.
            for (int i = 0; i < numChildren(); i++) {
                bf.append("\n");
                getChild(i).toJSON(bf);
                // not last child
                if (i < numChildren() - 1) {
                    bf.append(",");
                }
            }
        }
        bf.append("]\n}");
    }
    
    public boolean childrenAreSymmetric() {
        return (getChild(0) instanceof CompoundMatrix && getChild(0).numChildren()==3);
    }
    
    public boolean isChildSymmetric(int childIndex) {
        // If not a binary != ct, then the matrix inside should be regarded as symmetric.
        return !(getChild(0) instanceof CompoundMatrix && getChild(0).numChildren()==3);
    }

    public boolean canChildBeConvertedToDifference(int childIndex) {
        return isMyOnlyOtherSiblingEqualZero(childIndex);
    }*/

}
