package savilerow;
/*

    Savile Row http://savilerow.cs.st-andrews.ac.uk/
    Copyright (C) 2014-2024 Peter Nightingale
    
    This file is part of Savile Row.
    
    Savile Row is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Savile Row is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Savile Row.  If not, see <http://www.gnu.org/licenses/>.

*/

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.io.*;

import savilerow.expression.*;
import savilerow.model.*;
import savilerow.treetransformer.*;
import savilerow.solver.*;

// Contains all tabulation methods
// Has a common instance of TabulationUtils shared by all tabulation methods. 

public class Tabulation 
{
    public static final String ANSI_RESET  = "\u001B[0m";
    public static final String ANSI_RED    = "\u001B[31m";
    public static final String ANSI_GREEN  = "\u001B[32m";
    
    private Model m;
    
    private TabulationUtils tu;
    
    private long nodelimit;
    
    public Tabulation(Model _m) {
        m=_m;
        tu=new TabulationUtils(m);
        nodelimit=CmdFlags.tabulate_nolimit ? Long.MAX_VALUE : 100000;
    }
    
    public void process(boolean prop) {
        //  First, MakeTable functions.
        processMakeTable(m.constraints);
        
        if(!prop) {
            // populate idToCons
            collectIdentifiers(m.constraints);
            
            if(CmdFlags.make_short_tab==3 || CmdFlags.make_short_tab==4) {
                CmdFlags.warning("-tabulate flag is deprecated and will be removed in a future release. Use -tabid or -tabid-light.");
                //  Tabulation of constraints. 
                //  First find sets of constraints with the same scope, attempt
                //  tabulation of the conjunction of them.
                HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist=buildScopesList(m.constraints);
                identicalScopes(m.constraints, scopeslist);
                
                //  The other three heuristics -- applied to each top-level constraint
                applyHeuristicsBool(m.constraints);
            }
            
            if(CmdFlags.tabid) {
                double tabstarttime=System.currentTimeMillis();
                
                //  New tabulation -- can extract boolean subexpressions 
                //  as well as numerical subexpressions. 
                
                // Always use long tables -- for now.
                int tmp=CmdFlags.make_short_tab;
                CmdFlags.make_short_tab=1;
                
                //  Identical scopes works as previously. 
                HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist=buildScopesList(m.constraints);
                identicalScopes(m.constraints, scopeslist);
                
                //  Apply heuristics to top-level constraints and boolean subexpressions.
                //  Also do a variation of identical scopes where a boolean subexpression has an
                //  identical scope to a top-level ct. 
                applyHeuristicsBool2(m.constraints, scopeslist);
                
                applyHeuristicsNumerical2(m.constraints, scopeslist);
                
                CmdFlags.make_short_tab=tmp;
                
                CmdFlags.tabtime=(((double) System.currentTimeMillis() - tabstarttime) / 1000.0);  // Record the time taken for tabulation. 
            }
            
            //  Might have projected out variables that appeared in only one table. 
            //  Get rid of any variables not used in any remaining constraint. 
            //  Only relevant when table minimise is uncommented in tabulate function. 
            //RemoveRedundantVars rrv=new RemoveRedundantVars();
            //rrv.transform(m);
            
            if(m.objective!=null) {
                // Propagation of table constraints (or dominance breaking if that is used) might mean the objective variable needs its bounds tightening.  
                tightenObjectiveVar();
            }
        }
    }
    
    //////////////////////////////////////////////////////////////////////////
    //  Deal with MakeTable functions
    //  Top-down descent of tree. 
    private void processMakeTable(ASTNode curnode) {
        if(curnode instanceof MakeTable) {
            if(CmdFlags.make_short_tab==0) {
                //  Option 0 given on command line. 
                //  Remove the MakeTable function
                ASTNode parent=curnode.getParent();
                int idx=curnode.getChildNo();
                curnode.getChild(0).setParent(null);
                parent.setChild(idx, curnode.getChild(0));
                
                //  Recursively process the replacement ASTNode. 
                processMakeTable(parent.getChild(idx));
            }
            else {
                //  Tabulate with no limits. Always returns a table constraint. 
                ASTNode newTable=tabulate(curnode.getChild(0), Long.MAX_VALUE, (CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4), "MakeTableFunction");
                
                curnode.getParent().setChild(curnode.getChildNo(), newTable);
            }
        }
        else if(curnode instanceof Table) {
            //  Check the table.
            int r=curnode.getChild(0).numChildren()-1;
            ASTNode tab=curnode.getChildConst(1);
            if(tab.numChildren()>1) {
                if(tab.getChild(1).getTupleLength()!=r) {
                    CmdFlags.errorExit("Table scope size does not match length of tuples.\n"+curnode);
                }
                if(!tab.isRegularMatrix()) {
                    CmdFlags.errorExit("Table constraint contains irregular matrix (i.e. length of tuples differs).\n"+curnode);
                }
            }
        }
        else {
            //  Recursive descent of the tree
            for(int i=0; i<curnode.numChildren(); i++) {
                processMakeTable(curnode.getChild(i));
            }
        }
    }
    
    //////////////////////////////////////////////////////////////////////////
    //  Identical scopes heuristic
    
    private void identicalScopes(ASTNode top, HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist) {
        //  Find entries in scopeslist where number of constraints > 1
        
        for(Map.Entry<ArrayList<ASTNode>,ArrayList<ASTNode>> p : scopeslist.entrySet()) {
            ArrayList<ASTNode> ctlist=p.getValue();
            if(ctlist.size()>1) {
                ASTNode totabulate=new And(ctlist);
                
                ASTNode newTable=tabulate(totabulate, nodelimit, (CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4), "IdenticalScopes", true);
                
                if(newTable!=null) {
                    replaceConstraintSet(ctlist, newTable);
                    // Update scopeslist
                    ctlist.clear();
                    ctlist.add(newTable);
                }
            }
        }
    }
    
    // Collect scopes of top-level constraints. 
    private HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> buildScopesList(ASTNode top) {
        HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist=new HashMap<>();
        
        // Populate scopeslist
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            for(int i=0; i<a.numChildren(); i++) {
                ASTNode ct=a.getChild(i);
                if(!(ct instanceof Table) && !(ct instanceof TableShort) && !(ct instanceof NegativeTable)
                    && !(ct instanceof Tag)) {
                    assert ct.isRelation();
                    
                    // Get the scope. 
                    ArrayList<ASTNode> scope=TabulationUtils.getVariablesOrdered(ct);
                    ASTNode.sortByAlpha(scope);  //  Sort alphabetically
                    
                    if(! scopeslist.containsKey(scope)) {
                        scopeslist.put(scope, new ArrayList<ASTNode>());
                    }
                    
                    scopeslist.get(scope).add(ct);
                }
            }
        }
        
        return scopeslist;
    }
    
    private void replaceConstraintSet(ArrayList<ASTNode> ctlist, ASTNode replacement) {
        ASTNode a=ctlist.get(0);
        
        a.getParent().setChild(a.getChildNo(), replacement);
        
        //  Clear the rest of the constraints.
        for(int i=1; i<ctlist.size(); i++) {
            a=ctlist.get(i);
            a.getParent().setChild(a.getChildNo(), new BooleanConstant(true));
        }
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Three single-constraint heuristics
    
    private void applyHeuristicsBool(ASTNode top) {
        //  Simply iterate through the top-level constraints. 
        
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            for(int i=0; i<a.numChildren(); i++) {
                ASTNode curnode=a.getChild(i);
                //  Checks are a bit paranoid, not all are necessary
                if(!(curnode instanceof And) && curnode.isRelation()
                    && !(curnode instanceof Table) && !(curnode instanceof TableShort) && !(curnode instanceof NegativeTable)
                    && !(curnode instanceof BooleanConstant)) {
                    //   Heuristics for applying the (short) table converter. 
                    //   Only applied in final tailoring process.
                    //   Top-level constraints only.  Boolean expressions nested within top-level constraints are not tabulated.  
                    String h=CmdFlags.tab_diag_2?heurAll(curnode):heuristic(curnode);
                    if(h!=null) {
                        ASTNode newTable=tabulate(curnode, nodelimit, (CmdFlags.make_short_tab==2 || CmdFlags.make_short_tab==4), h, true);
                        
                        if(newTable!=null) {
                            a.setChild(i, newTable);
                        }
                    }
                }
            }
        }
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Second version that can tabulate subexpressions as well as top-level
    //  cts.
    private void applyHeuristicsBool2(ASTNode top, HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist) {
        //  Breadth-first search in AST. 
        
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            
            ArrayDeque<ASTNode> deque=new ArrayDeque<>(a.getChildren());
            
            while(!deque.isEmpty()) {
                ASTNode curnode=deque.poll();
                
                //  Gecode and Minion support reified table so it's safe
                //  Chuffed and std flatzinc do not, and the reified table is
                //  expanded into a full-length table constraint with the reification var. 
                //  Sat encoding of reified table is weak.
                //  Apply product of domain size limit to chuffed+std flatzinc. 
                
                //  First check for identical scope with a top-level constraint
                //  (when curnode is not top-level)
                if(curnode.isRelation() && curnode.getDimension()==0
                    && !(curnode instanceof Table) && !(curnode instanceof TableShort) && !(curnode instanceof NegativeTable)
                    && !(curnode instanceof BooleanConstant)
                    && !curnode.getParent().inTopAnd()) {
                    
                    // Get the scope. 
                    ArrayList<ASTNode> scope=TabulationUtils.getVariablesOrdered(curnode);
                    ASTNode.sortByAlpha(scope);  //  Sort alphabetically
                    //  Check if there is a top-level with the same scope
                    //  and either there is more than one such top-level constraint, or
                    //  it does not contain curnode. 
                    if(scopeslist.containsKey(scope) && (scopeslist.get(scope).size()>1 || !contains(scopeslist.get(scope).get(0), curnode)) ) {
                        ASTNode toplevelcts=new And(scopeslist.get(scope));
                        
                        if(applyHeuristicsBool2Attempt(curnode, toplevelcts, "IdenticalScopesNested")) {
                            continue;
                        }
                    }
                }
                
                //  Check it's a candidate, then apply the three heuristics. 
                if(curnode.isRelation() && curnode.getDimension()==0
                    && !(curnode instanceof Table) && !(curnode instanceof TableShort) && !(curnode instanceof NegativeTable)
                    && !(curnode instanceof BooleanConstant)) {
                    String h=CmdFlags.tab_diag_2?heurAll(curnode):heuristic(curnode);
                    if(h!=null) {
                        if(applyHeuristicsBool2Attempt(curnode, null, curnode.getParent().inTopAnd()?h:(h+"(Nested)"))) {
                            continue;
                        }
                    }
                }
                
                //if(!(curnode.isRelation() && curnode.getDimension()==0) || !curnode.strongProp()) {
                    //  StrongProp check is an optimisation -- don't look into 
                    //  constraints that are already expected to get GAC. 
                    //  Unfortunately drops the 'identical scopes' heuristic. 
                    deque.addAll(curnode.getChildren());
                //}
            }
        }
    }
    
    private boolean checkSizeNestedTable(ASTNode curnode) {
        if(!CmdFlags.getMiniontrans() && !CmdFlags.getGecodetrans() && !CmdFlags.getSattrans() && !curnode.getParent().inTopAnd()) {
            // Not top-level, and solver does not support reified table. 
            // Limit product of domains. 
            ArrayList<ASTNode> scope=TabulationUtils.getVariablesOrdered(curnode);
            double domprod=1.0;
            for(int i=0; i<scope.size(); i++) {
                domprod=domprod*Intpair.numValues(scope.get(i).getIntervalSetExp());
            }
            if(domprod>((double)nodelimit)) {
                return false;
            }
        }
        return true;
    }
    
    //  When a bool heuristic fires, attempt tabulation either in place or extracted to a 
    //  variable (if not top-level, and not using Gecode or Minion). 
    
    private boolean applyHeuristicsBool2Attempt(ASTNode curnode, ASTNode context, String heuristic) {
        //  If targeting Minion or Gecode, reified table is supported so always do it in place with ordinary limit.
        //  Else, apply an extra limit because it will be extracted then full d^n table generated. 
        if(!checkSizeNestedTable(curnode)) {
            return false;
        }
        
        //  Attempt tabulation in-place
        //  Very conservative -- make sure everything is copied. 
        ASTNode totab=curnode.copy();
        if(context!=null) {
            totab=new And(context.copy(), totab);
        }
        
        ASTNode newTable=tabulate(totab, nodelimit, false, heuristic, curnode.getParent().inTopAnd());
        
        if(newTable!=null) {
            //   Replace. 
            curnode.getParent().setChild(curnode.getChildNo(), newTable);
            return true;
        }
        
        return false;
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Numerical expressions
    
    private void applyHeuristicsNumerical2(ASTNode top, HashMap<ArrayList<ASTNode>, ArrayList<ASTNode>> scopeslist) {
        //  Iterate through every numerical decision expression 
        ArrayList<ASTNode> newCts=new ArrayList<>();
        
        //  Cache mapping numerical expressions that have already been tabulated to aux variables
        HashMap<ASTNode, ASTNode> auxcache=new HashMap<>();
        
        if(top.getChild(0) instanceof And) {
            ASTNode a=top.getChild(0);
            
            for(int i=0; i<a.numChildren(); i++) {
                ASTNode outer_ct=a.getChild(i);
                
                //  Find numerical expressions inside outer_ct, looking inside containers if necessary
                ArrayDeque<ASTNode> deque=new ArrayDeque<>(outer_ct.getChildren());
                
                while(!deque.isEmpty()) {
                    ASTNode curnode=deque.poll();
                    
                    //  If curnode is a numerical expression, and it will be extracted by
                    //  general flattening, then it's a candidate for tabulation.
                    ASTNode par=curnode.getParent();
                    if(curnode.isNumerical() 
                        && !(curnode instanceof Identifier)
                        && !(curnode instanceof NumberConstant)
                        && !(curnode instanceof Mapping)
                        && curnode.getDimension()==0 
                        && curnode.toFlatten(false)
                        && !(par instanceof ToVariable || par instanceof Equals)
                        ) {
                        //  First check the cache -- have we already tabulated this numerical expression?
                        //  If so, use the aux variable we already made. 
                        if(auxcache.containsKey(curnode)) {
                            curnode.getParent().setChild(curnode.getChildNo(), auxcache.get(curnode));
                            continue;  //  Continue the while loop. 
                        }
                        
                        // Trial auxiliary variable and constraint. 
                        //System.out.println("Trying numerical expression:"+curnode);
                        //ASTNode tmpaux=m.global_symbols.newAuxHelper(curnode);
                        ArrayList<Intpair> auxvar_intervals=curnode.getIntervalSetExp();
                        ASTNode auxvar_dom=Intpair.makeDomain(auxvar_intervals, curnode.isRelation());
                        auxvar_dom=m.filt.constructDomain(curnode, auxvar_dom);  //  Look up stored (filtered) domain if there is one.
                        ASTNode tmpaux=m.global_symbols.newAuxiliaryVariable(auxvar_dom);
                        
                        ASTNode ct=new Equals(curnode.copy(), tmpaux);
                        
                        //  First do identical scopes heuristic. 
                        
                        ArrayList<ASTNode> scope=TabulationUtils.getVariablesOrdered(curnode);
                        ASTNode.sortByAlpha(scope);  //  Sort alphabetically
                        
                        //  Check if there is a top-level with the same scope
                        //  and either there is more than one such top-level constraint, or
                        //  it does not contain curnode. 
                        if(scopeslist.containsKey(scope) && (scopeslist.get(scope).size()>1 || !contains(scopeslist.get(scope).get(0), curnode)) ) {
                            ASTNode toplevelcts=new And(scopeslist.get(scope));
                            ASTNode newTable=tabulate(new And(ct, toplevelcts), nodelimit, false, "IdenticalScopes(Integer)", false);
                            
                            if(newTable!=null) {
                                newCts.add(newTable);
                                
                                //  Replace the numerical expression with the aux variable. 
                                curnode.getParent().setChild(curnode.getChildNo(), tmpaux);
                                
                                // Add to cache
                                auxcache.put(curnode, tmpaux);
                                continue;  // Continue looping through the queue. 
                            }
                        }
                        
                        // Temporarily replace curnode with tmpaux so that calling strongProp on outer_ct produces the appropriate answer. 
                        ASTNode p=curnode.getParent();
                        p.setChild(curnode.getChildNo(), tmpaux);
                        boolean outer_ct_strong=outer_ct.strongProp();
                        // Restore
                        curnode.setParent(null);
                        p.setChild(curnode.getChildNo(), curnode);
                        
                        //  Special case of strong prop heuristic, plus other two heuristics. 
                        boolean stprop=outer_ct_strong && !ct.strongProp();
                        String h=CmdFlags.tab_diag_2?heurAll(ct):heuristic(ct);
                        if(stprop || h!=null) {
                            String hname;
                            if(stprop) {
                                hname=(h==null?"":h+",")+"WeakPropagation1(Integer)";
                            }
                            else {
                                hname=h+"(Integer)";
                            }
                            ASTNode newTable=tabulate(ct, nodelimit, false, hname, false);
                            
                            if(newTable!=null) {
                                newCts.add(newTable);
                                
                                //  Replace the numerical expression with the aux variable. 
                                curnode.getParent().setChild(curnode.getChildNo(), tmpaux);
                                
                                // Add to cache
                                auxcache.put(curnode, tmpaux);
                                
                                // Continue the loop
                                continue;
                            }
                        }
                        
                        //  Tidy up by deleting the auxiliary variable. 
                        m.global_symbols.deleteSymbol(tmpaux.toString());
                        assert tmpaux.toString().equals("aux"+String.valueOf(m.global_symbols.auxvarcounter-1));
                        m.global_symbols.auxvarcounter--;
                    }
                    
                    //  Put children into the queue. 
                    deque.addAll(curnode.getChildren());
                }
            }
            // Add the new constraints at the top level. 
            if(newCts.size()>0) {
                top.getChild(0).setParent(null);
                top.setChild(0, new And(top.getChild(0), new And(newCts)));
            }
        }
    }
    
    //////////////////////////////////////////////////////////////////////////
    //
    //  Data for tracking the objective function. 
    
    ASTNode optVar;
    
    ASTNode optCt=null;
    
    HashMap<ASTNode, ArrayList<ASTNode>> idToCons=new HashMap<>();  // Collect constraints containing each identifier. 
    
    void findOptCt(ASTNode a) {
        if(a.equals(optVar)) {
            ASTNode p=a.getParent();
            while(! p.isRelation()) {
                p=p.getParent();
            }
            optCt=p;
            return;
        }
        
        for(int i=0; i<a.numChildren() && optCt==null; i++) {
            findOptCt(a.getChild(i));
        }
    }
    
    // Count occurrences of each variable, and collect all constraints mentioning each
    // variable that occurs in the optimisation function (sum). 
    void collectIdentifiers(ASTNode curnode) {
        if(curnode instanceof Identifier) {
            //  Find the constraint containing curnode.
            ASTNode p=curnode.getParent();
            while(! (p.isRelation() && p.getDimension()==0)) {
                p=p.getParent();
            }
            if(! idToCons.containsKey(curnode)) {
                idToCons.put(curnode, new ArrayList<ASTNode>());
            }
            idToCons.get(curnode).add(p);
        }
        else {
            for(int i=0; i<curnode.numChildren(); i++) {
                collectIdentifiers(curnode.getChild(i));
            }
        }
    }
    
    void tightenObjectiveVar() {
        if(m.objective instanceof Minimising && optCt instanceof LessEqual && optCt.getChild(1).equals(optVar)) {
            long bnd=optCt.getChild(0).getBounds().upper;
            if(bnd < optVar.getBounds().upper) {
                //  It should be simpler than this to reduce the upper bound. 
                ArrayList<Intpair> newBound=new ArrayList<>();
                newBound.add(new Intpair(Long.MIN_VALUE, bnd));
                m.global_symbols.setDomain(optVar.toString(), Intpair.makeDomain(Intpair.intersection(optVar.getIntervalSetExp(), newBound), optVar.isRelation()));
            }
        }
        else if(m.objective instanceof Maximising && optCt instanceof LessEqual && optCt.getChild(0).equals(optVar)) {
            long bnd=optCt.getChild(1).getBounds().lower;
            if(bnd > optVar.getBounds().lower) {
                ArrayList<Intpair> newBound=new ArrayList<>();
                newBound.add(new Intpair(bnd, Long.MAX_VALUE));
                m.global_symbols.setDomain(optVar.toString(), Intpair.makeDomain(Intpair.intersection(optVar.getIntervalSetExp(), newBound), optVar.isRelation()));
            }
        }
    }
    
    /////////////////////////////////////////////////////////////////////////
    // Evaluate the three single-constraint heuristics
    
    private String heuristic(ASTNode curnode) {
        ArrayList<ASTNode> varlist=TabulationUtils.getVariablesOrdered(curnode);
        ArrayList<ASTNode> varlistdups=TabulationUtils.getVariablesDup(curnode);
        
        //  Duplicate variables and within a reasonable size bound.
        //  Should perhaps look at domain size rather than # variables.
        if(varlist.size()<varlistdups.size()) {  // && varlist.size()<=10) {
            return "DuplicateVariables";
        }
        
        //  Tree size is much larger than the number of variables.
        //  Either the expression is highly complex or it contains a large 
        //  number of constants. 
        if(curnode.treesize() > 5*varlist.size()) {
            return "LargeAST";
        }
        
        // Constraint does not get GAC, and it overlaps with some other
        // constraint that does get GAC or similar. Uses the strongProp method. 
        if(!curnode.strongProp()) {  // && varlist.size()<=10) {
            ASTNode constraints=m.constraints.getChild(0);
            if(constraints instanceof And) {
                for(int i=0; i<constraints.numChildren(); i++) {
                    ASTNode c=constraints.getChild(i);
                    if(c.strongProp()) {
                        for(int varidx=0; varidx<varlist.size(); varidx++) {
                            if(c.contains(varlist.get(varidx))) {
                                return "WeakPropagation";
                            }
                        }
                    }
                }
            }
        }
        
        return null;
    }
    
    /////////////////////////////////////////////////////////////////////////
    // Evaluate all 3 heuristics -- build a string including them all
    
    private String heurAll(ASTNode curnode) {
        assert CmdFlags.tab_diag_2;
        ArrayList<ASTNode> varlist=TabulationUtils.getVariablesOrdered(curnode);
        ArrayList<ASTNode> varlistdups=TabulationUtils.getVariablesDup(curnode);
        
        String heur="";
        
        //  Duplicate variables and within a reasonable size bound.
        //  Should perhaps look at domain size rather than # variables.
        if(varlist.size()<varlistdups.size()) {  // && varlist.size()<=10) {
            heur+="DuplicateVariables,";
        }
        
        //  Tree size is much larger than the number of variables.
        //  Either the expression is highly complex or it contains a large 
        //  number of constants. 
        if(curnode.treesize() > 5*varlist.size()) {
            heur+="LargeAST,";
        }
        
        // Constraint does not get GAC, and it overlaps with some other
        // constraint that does get GAC or similar. Uses the strongProp method. 
        if(!curnode.strongProp()) {  // && varlist.size()<=10) {
            boolean bail=false;  // bail out
            ASTNode constraints=m.constraints.getChild(0);
            if(constraints instanceof And) {
                for(int i=0; i<constraints.numChildren() && !bail; i++) {
                    ASTNode c=constraints.getChild(i);
                    if(c.strongProp()) {
                        for(int varidx=0; varidx<varlist.size() && !bail; varidx++) {
                            if(c.contains(varlist.get(varidx))) {
                                heur+="WeakPropagation";
                                bail=true;
                            }
                        }
                    }
                }
            }
        }
        
        return heur.equals("")?null:heur;
    }
    
    //  Does outer contain inner (not a copy of inner)?
    public boolean contains(ASTNode outer, ASTNode inner) {
        if (outer==inner) {
            return true;
        }
        else {
            for (int i=0; i < outer.numChildren(); i++) {
                if (contains(outer.getChild(i), inner)) {
                    return true;
                }
            }
        }
        return false;
    }
    
    /////////////////////////////////////////////////////////////////////////
    //  Actually perform the tabulation of an expression.
    
    public ASTNode tabulate(ASTNode totab, long nodelimit, boolean shorttable, String heuristic) {
        return tabulate(totab, nodelimit, shorttable, heuristic, true);
    }
    
    public ASTNode tabulate(ASTNode totab, long nodelimit, boolean shorttable, String heuristic, boolean toplevel) {
        ASTNode a=tu.normalise(totab);
        ASTNode aCopyForCache=a.copy();
        
        if(CmdFlags.tabulate_diagnostics) {
            CmdFlags.println(ANSI_RED+heuristic+ANSI_RESET+"   Attempting tabulation: "+a);
        }
        
        //  Check the cache.
        TabulationUtils.RetPair ret = tu.tryCacheNormalised(totab, shorttable);
        if(ret.nodereplace != null) {
            if(CmdFlags.tabulate_diagnostics) {
                CmdFlags.println(ANSI_GREEN+"Tabulated"+ANSI_RESET+" by retrieving from cache.");
            }
            return ret.nodereplace.current_node;
        }
        
        //  Check fail cache
        if(tu.tryFailCache(ret.expstring)) {
            if(CmdFlags.tabulate_diagnostics) {
                CmdFlags.println(ANSI_RED+heuristic+ANSI_RESET+"   Not tabulated, found in fail cache.");
            }
            return null;
        }
        
        ArrayList<ASTNode> varlist=TabulationUtils.getVariablesOrdered(a);
        int arity=varlist.size();
        
        if(CmdFlags.tabid_light && arity>2 && 
            (CmdFlags.getChuffedtrans() || CmdFlags.getOrtoolstrans() || CmdFlags.getSattrans()) ) {
            if(CmdFlags.tabulate_diagnostics) {
                CmdFlags.println(ANSI_RED+heuristic+ANSI_RESET+"   Not tabulated, arity>2 and -tabid-light version with CDCL solver.");
            }
            return null;
        }
        
        if(arity>20) {
            if(CmdFlags.tabulate_diagnostics) {
                CmdFlags.println(ANSI_RED+heuristic+ANSI_RESET+"   Not tabulated, arity>20.");
            }
            return null;
        }
        
        //  Set up tuple limit for SAT backend. 
        //  Assumes Bacchus encoding, each additional tuple adds one binary clause per CSP variable.
        //  With arity 2 or 1 different encodings are used and the limits are different.
        //  For MDD encoding, clausebudget is used instead. 
        long tuplelim=1000000000;
        long clausebudget=1000000000;
        
        if(CmdFlags.tab_satsizelimit) {
            double margin=2.0;  //  How much the SAT encoding size will be allowed to increase. 
            
            //  Calculate the largest number of clauses that would be a useful limit on tabulation.
            long clauselim=1000000000;
            
            long domaintotal=0;
            for(int i=0; i<varlist.size(); i++) {
                domaintotal+=Intpair.numValues(varlist.get(i).getIntervalSetExp());
            }
            
            if(toplevel) {
                if(arity>2) {
                    if(!CmdFlags.sat_table_mdd) {
                        //  At most one tuple can be generated per search node: start with nodelimit
                        //  Number of tuples for Bacchus encoding approx arity*nTuples
                        clauselim = (long) Math.ceil( ((double) (domaintotal + (nodelimit*arity) +1)) / margin)+1;
                    }
                    //  If the encoding is MDD then there is no closed form for the number of clauses. 
                }
                else if(arity == 2) {
                    //  Encoding size of table (in clauses) is bounded by total number of domain values
                    clauselim = (long) Math.ceil( ((double) domaintotal) / margin)+1;
                }
                // If arity == 1 there is no clause limit because the generated table is absorbed into the domain. 
            }
            else {
                //  Based on encoding of nested table constraints. 
                if(arity>1) {
                    //  At most one tuple can be generated per search node: start with nodelimit 
                    //  Encoding has (arity+2)*nTuples + 1  clauses. 
                    clauselim = (long) Math.ceil( ((double) (nodelimit*(arity+2) +1)) / margin)+1;
                }
                else {
                    assert arity == 1;
                    clauselim = (long) Math.ceil( ((double)domaintotal) / margin)+1;
                }
            }
            
            if(arity>1 || !toplevel) {
                Intpair ses=estimateSATEncodingSize(a, clauselim, toplevel);  //  Get encoding size for a: (vars, clauses) up to clauselim
                
                //  Set clausebudget here -- to be used in case of MDD encoding.
                clausebudget=(long) Math.ceil(ses.upper*margin);
                
                //  If necessary, set tuplelim. It is necessary when the call to 
                //  estimateSATEncodingSize did not reach its limit clauselim, and
                //  for some encodings (not MDD and not unary-toplevel). 
                
                if(ses.upper<clauselim) {
                    if(toplevel) {
                        if(arity > 2) {
                            if(!CmdFlags.sat_table_mdd) {
                                double tuplelimd = Math.ceil(margin*(((double)ses.upper)-1-domaintotal)/((double)arity));
                                tuplelim = (tuplelimd>1000000000)? 1000000000 : (long)tuplelimd;
                            }
                        }
                        else if(arity == 2) {
                            // Support encoding of binary constraint. Clause budget is based on domain sizes of the variables.
                            if(domaintotal > ses.upper*margin) {
                                //System.out.println("Failed Tuple-limit check: binary constraint.");
                                return null;
                            }
                        }
                        //  arity-1 top-level, then it will be absorbed into the domain after tabulation. No tuple-limit.
                    }
                    else {
                        // Not top-level
                        if(arity>1) {
                            if(!CmdFlags.sat_table_mdd) {
                                double tuplelimd = Math.ceil(margin*((double)ses.upper-1.0)/((double)arity+2.0));
                                tuplelim = (tuplelimd>1000000000)? 1000000000 : (long)tuplelimd;
                            }
                        }
                        else {
                            // If not top-level, then put limit on.  Encoding has one clause per domain value. 
                            if(domaintotal > ses.upper*margin) {
                                //System.out.println("Failed Tuple-limit check: unary constraint.");
                                return null;
                            }
                        }
                    }
                }
            }
        }
        
        if(tuplelim<0) {
            tuplelim=0;
        }
        
        boolean saveInCache=true;  // It is not always correct to cache a tabulation -- do it by default but set this flag to false in cases where it would be potentially incorrect. 
        
        ASTNode dominanceVar=null;
        boolean dominanceMin=false;
        
        //  Table minimise
        /*boolean unroll=false;
        
        ArrayList<ASTNode> varsdup=TabulationUtils.getVariablesDup(a);
        
        if(toplevel) {
            // Remove variables that occur only within the set of constraints. Don't remove the objective variable.
            long combos=1; //  Product of the domains of the variables eliminated so far. 
            
            for(int j=0; j<varlist.size(); j++) {
                ASTNode var=varlist.get(j);
                
                // Count var. 
                int varcount=0;
                for(int k=0; k<varsdup.size(); k++) {
                    if(varsdup.get(k).equals(var)) {
                        varcount++;
                    }
                }
                
                combos=combos*Intpair.numValues(varlist.get(j).getIntervalSetExp());
                if(combos>10000) {
                    break;
                }
                
                if(idToCons.containsKey(var) && idToCons.get(var).size()==varcount) {   //  All occurrences of var are in the collection of constraints to tabulate. 
                    //  Shadowing the decision variable -- does this work correctly? Yes it does. 
                    a=new ExistsExpression(varlist.get(j), ((Identifier) varlist.get(j)).getDomain(), a);
                    unroll=true;
                    saveInCache=false;  // Cannot cache -- would need to do the same idToCons check when retrieving from cache. 
                }
            }
        }
        
        //  Dominance
        if(m.objective!=null) {
            min=(m.objective instanceof Minimising);
            
            if(optCt==null) {
                // Find the objective constraint. 
                optVar=m.objective.getChild(0);
                findOptCt(m.constraints);
                
                if( (optCt instanceof ToVariable && optCt.getChild(1).equals(optVar) && optCt.getChild(0) instanceof WeightedSum)
                    || (optCt instanceof Equals && optCt.getChild(1).equals(optVar) && optCt.getChild(0) instanceof WeightedSum)
                    || (optCt instanceof Equals && optCt.getChild(0).equals(optVar) && optCt.getChild(1) instanceof WeightedSum)
                    || (optCt instanceof LessEqual && optCt.getChild(1).equals(optVar) && optCt.getChild(0) instanceof WeightedSum && min) 
                    || (optCt instanceof LessEqual && optCt.getChild(0).equals(optVar) && optCt.getChild(1) instanceof WeightedSum && !min) ) {
                    
                    ASTNode wsum;
                    
                    if(optCt.getChild(0) instanceof WeightedSum) {
                        wsum=optCt.getChild(0);
                    }
                    else {
                        wsum=optCt.getChild(1);
                    }
                    optTerms=wsum.getChildren();
                    
                    optTermsPolarity=new boolean[optTerms.size()];
                    for(int i=0; i<optTerms.size(); i++) {
                        // true iff polarity is positive. 
                        optTermsPolarity[i]=((WeightedSum)wsum).getWeight(i)>0;
                    }
                }
            }
            
            //  Make sure it's not optCt we are tabulating by checking for presence of optVar. 
            if(optTerms!=null && !(a.contains(optVar))) {
                //  Check for a variable overlapping between optTerms and the constraint. 
                
                for(int i=0; i<optTerms.size(); i++) {
                    ASTNode term=optTerms.get(i);
                    if(a.contains(term)) {
                        // Count term. 
                        int varcount=0;
                        for(int k=0; k<varsdup.size(); k++) {
                            if(varsdup.get(k).equals(term)) {
                                varcount++;
                            }
                        }
                        
                        if(idToCons.containsKey(term) && idToCons.get(term).size()==varcount+1) {   //  All occurrences of term are in this constraint or the objective constraint. 
                            dominanceVar=term;
                            dominanceMin=(optTermsPolarity[i] && min) || (!optTermsPolarity[i] && !min);
                            
                            saveInCache=false;  // cannot cache result, relies on information from outside a.
                            break;
                        }
                        
                    }
                }
                
            }
        }
        
        // issue with cache -- must not cache a constraint with dominance applied because 
        // of the checks that the dominance variable is part of the objective function 
        // and it occurs in no other constraints. 
        
        if(unroll) {
            TransformQuantifiedExpression tqe=new TransformQuantifiedExpression(m);
            a=tqe.transform(a);
        }*/
        //  End of table minimise section. 
        
        //  First normalise  before generating table.
        ASTNode newTable;
        long nodecount=-1;
        
        if(!shorttable) {
            if(CmdFlags.tab_minion) {
                newTable = tabulateMinion(a, nodelimit, tuplelim);
            }
            else {
                if(dominanceVar!=null) {
                    newTable = tu.makeTableLongDominance(a, nodelimit, tuplelim, dominanceVar, dominanceMin);
                }
                else {
                    newTable = tu.makeTableLong(a, nodelimit, tuplelim);
                }
                nodecount=tu.nodecount;
            }
        }
        else {
            //  Make short table constraint
            if(nodelimit==Long.MAX_VALUE) {
                //  No limit.
                newTable = tu.makeTableShort(a, nodelimit, nodelimit, nodelimit);
            }
            else {
                newTable = tu.makeTableShort(a, 10000, 100000, 100000);
            }
        }
        
        // One last SAT encoding size check. 
        if(CmdFlags.tab_satsizelimit && toplevel && arity>2 && CmdFlags.sat_table_mdd && newTable!=null) {
            //  Encode the table constraint to get the number of clauses. 
            Intpair tabses=estimateSATEncodingSize(newTable, clausebudget+1, toplevel);  //  Get encoding size for a: (vars, clauses) up to clauselim
            if(tabses.upper > clausebudget) {
                System.out.println("Failed SAT size limit check.");
                newTable=null;
            }
        }
        
        if(newTable==null) {
            tu.saveToFailCache(ret.expstring);
        }
        else {
            // Save in the cache
            if(saveInCache) {
                tu.saveToCacheNormalised(ret.expstring, aCopyForCache, newTable);
            }
            
            if(CmdFlags.tabulate_diagnostics) {
                CmdFlags.println(ANSI_GREEN+"Tabulated"+ANSI_RESET+" in nodes:"+nodecount);
            }
            
            // Propagate
            if(toplevel) {
                if(newTable instanceof Table) {
                    ((Table)newTable).propagate();
                }
                else {
                    ((TableShort)newTable).propagate();
                }
            }
        }
        
        return newTable;
    }
    
    public Intpair estimateSATEncodingSize(ASTNode in, long clauselim, boolean toplevel) {
        //  Temp. set the back-end to SAT.
        CmdFlags.SolEnum bak_soltype=CmdFlags.soltype;
        CmdFlags.soltype=CmdFlags.SolEnum.SAT;
        
        ASTNode a=in.copy();
        
        ArrayList<ASTNode> varlist=TabulationUtils.getVariablesOrdered(a);
        
        ModelContainer mc=createModelForExpression(a, varlist, toplevel);
        Model m1=mc.m;
        
        HashSet<String> varset=new HashSet<String>();
        for(ASTNode id : varlist) {
            varset.add(id.toString());
        }
        
        boolean tmp_outready = CmdFlags.getOutputReady();
        boolean tmp_afteragg = CmdFlags.getAfterAggregate();
        
        //  Remove clause limit until after primary variables encoded. 
        long tmp_clauselim=CmdFlags.getCNFLimit();
        CmdFlags.setCNFLimit(0);
        
        //  Remove variables by assignment -- set to false in this method
        //  because nested constraints can be promoted to top-level in m1, and
        //  if those constraints are equalities then deleting variables will
        //  distort the size of the encoding. 
        boolean tmp_deletevars = CmdFlags.getUseDeleteVars();
        CmdFlags.setUseDeleteVars(false);
        
        // Rest of instancePreFlattening2 (after tabulation)
        
        //  Use mappers to avoid unnecessary integer aux variables when targeting SAT. 
        //  A mapper behaves as an integer variable for SAT encoding. 
        TransformSumToShift tsts = new TransformSumToShift(m1);
        m1.transform(tsts);
        
        TransformProductToMult tptm = new TransformProductToMult(m1);
        m1.transform(tptm);
        
        // Some reformulations will be affected by order, so normalise.
        TransformNormalise tnr = new TransformNormalise(m1);
        m1.transform(tnr);
        
        //  We are always in the output tailoring pass when doing tabulation. 
        mc.instanceFlattening(false);
        
        // From satPrepOutput:
        mc.decomposeSatEncodingFlat();
        
        TransformToFlatGecode tfg=new TransformToFlatGecode(m1);
        m1.transform(tfg);
        
        m1.simplify();  // Fix problem with unit vars when outputting sat. 
        
        // Mark variables for direct or order encoding.
        TransformCollectSATDirect tcsd=new TransformCollectSATDirect(m1);
        tcsd.transform(m1.constraints);
        
        // Pick apart the setupSAT method of Model. 
        m1.satModel=new SatNull(m1.global_symbols);
        
        // Set flags to avoid encoding any introduced variables -- only the original variables. 
        categoryentry itr = m1.global_symbols.getCategoryFirst();
        while (itr != null) {
            if(! varset.contains(itr.name)) {
                itr.already_written=true;
            }
            itr=itr.next;
        }
        
        try {
            m1.satModel.generateVariableEncoding(tcsd.getVarsInConstraints(), false);
        }
        catch(IOException e) {
            CmdFlags.errorExit(String.valueOf(e));
        }
        
        // Read the counters. 
        long nvars1=m1.satModel.getNumVars();
        long nclauses1=m1.satModel.getNumClauses();
        
        //  Fiddle the clause limit.
        CmdFlags.setCNFLimit(clauselim+nclauses1);
        
        //  Encode any variables introduced in flattening. 
        itr = m1.global_symbols.getCategoryFirst();
        while (itr != null) {
            if(! varset.contains(itr.name)) {
                itr.already_written=false;
            }
            itr=itr.next;
        }
        
        boolean clauseout=false;
        try {
            m1.satModel.generateVariableEncoding(tcsd.getVarsInConstraints(), false);
        }
        catch(IOException e) {
            clauseout=true;
        }
        
        if(!clauseout) {
            TransformSATEncoding tse=new TransformSATEncoding(m1);
            m1.constraints=tse.transform(m1.constraints);  // Avoid the branching on list and other things.
            
            m1.toSAT();  // Encode constraints. 
        }
        
        //System.out.println("Numbers for orig vars: "+nvars1+" cl:"+nclauses1);
        
        //  Get the numbers.
        nvars1=m1.satModel.getNumVars()-nvars1;
        nclauses1=m1.satModel.getNumClauses()-nclauses1;
        //System.out.println("Numbers for after vars: "+nvars1+" cl:"+nclauses1);
        // Restore the control flags to their original state (flags should probably eventually be in the model.)
        CmdFlags.setOutputReady(tmp_outready);
        CmdFlags.setAfterAggregate(tmp_afteragg);
        CmdFlags.setUseDeleteVars(tmp_deletevars);
        CmdFlags.setCNFLimit(tmp_clauselim);
        
        CmdFlags.soltype=bak_soltype;
        
        return new Intpair(nvars1,nclauses1);
    }
    
    
    public ModelContainer createModelForExpression(ASTNode a, ArrayList<ASTNode> varlist, boolean toplevel) {
        //  Make a new Model object containing only a.
        
        Model m1=new Model();
        SymbolTable st1=new SymbolTable();
        m1.setup(new Top(new BooleanConstant(true)), st1, null, null, m.heuristic, null, null);
        
        //  Collect variable refs etc before switching expression over to new model. 
        ArrayList<ASTNode> vardoms=tu.getDomains(varlist);
        
        ArrayList<ASTNode> cmlist=TabulationUtils.getConstMatrixRefs(a);
        
        TransformFixSTRef t1=new TransformFixSTRef(m1);
        t1.transform(a);
        
        //  Make a new variable in m1 for each variable in varlist. 
        for(int i=0; i<varlist.size(); i++) {
            st1.newVariable(varlist.get(i).toString(), vardoms.get(i), ASTNode.Decision);
        }
        
        m1.branchingon=new CompoundMatrix(varlist);
        t1.transform(m1.branchingon);
        
        //  Copy over any necessary constant matrices. 
        for(int i=0; i<cmlist.size(); i++) {
            m1.cmstore.newConstantMatrix(cmlist.get(i).toString(), m.cmstore.getConstantMatrix(cmlist.get(i).toString()).copy());
        }
        
        if(a.isRelation() && toplevel) {
            m1.constraints.setChild(0, a);
        }
        else {
            ASTNode auxvar=m1.global_symbols.newAuxHelper(a);
            ASTNode flatcon=new ToVariable(a, auxvar);
            m1.constraints.setChild(0, flatcon);
        }
        
        return new ModelContainer(m1, new ArrayList<ASTNode>());
    }
    
    public ASTNode tabulateMinion(ASTNode in, long nodelim, long tuplelim) {
        //  Flatten for Minion and tabulate using Minion's find all solutions. 
        ASTNode a=in.copy();
        
        ArrayList<ASTNode> varlist=TabulationUtils.getVariablesOrdered(a);
        
        ModelContainer mc=createModelForExpression(a, varlist, true);
        Model m1=mc.m;
        
        //  Remove mappers if they have been introduced for SAT backend. 
        if(CmdFlags.getSattrans()) {
            TransformRemoveMappers trm=new TransformRemoveMappers();
            m1.transform(trm);
        }
        
        HashSet<String> varset=new HashSet<String>();
        for(ASTNode id : varlist) {
            varset.add(id.toString());
        }
        
        boolean tmp_outready = CmdFlags.getOutputReady();
        boolean tmp_afteragg = CmdFlags.getAfterAggregate();
        
        //  Adjust config of Savile Row and Minion flags for this special run. 
        //  Adjust the flags for special run of Minion.
        //  Save the real flags. 
        ArrayList<String> tmp_solveropts=CmdFlags.solverflags;
        
        ArrayList<String> solveropts=new ArrayList<String>();
        solveropts.add("-findallsols");
        if(tuplelim<Long.MAX_VALUE) {
            solveropts.add("-sollimit");
            solveropts.add(String.valueOf((tuplelim>0) ? tuplelim : tuplelim+1));
        }
        if(nodelim<Long.MAX_VALUE) {
            solveropts.add("-nodelimit");
            solveropts.add(String.valueOf(nodelim));
        }
        solveropts.add("-X-tabulation");  //  Use special search procedure with progress checks. 
        //solveropts.add("-valorder");   ///  randomise to make the BT-FREE check more accurate. Makes tabulation time even worse on HTS, 
        //solveropts.add("random");      //   does not completely solve issue with BT-FREE check on HTS. 
        CmdFlags.solverflags=solveropts;
        
        String tmp_preprocess=CmdFlags.getPreprocess();
        CmdFlags.preprocess="GAC";
        
        boolean tmp_delvars=CmdFlags.getUseDeleteVars();
        CmdFlags.setUseDeleteVars(false);    //  Do not delete assigned variables during tailoring -- changes semantics of generated constraint. 
        
        boolean tmp_rrv=CmdFlags.getRemoveRedundantVars();
        CmdFlags.setRemoveRedundantVars(false);
        
        boolean tmp_auxnon=CmdFlags.getAuxNonFunctional();
        CmdFlags.setAuxNonFunctional(false);
        
        // Rest of instancePreFlattening2 (after tabulation) -- for Minion flattening. 
        
        TransformSafeElementOne tseo=new TransformSafeElementOne(m1);
        m1.transform(tseo);
        
        // Some reformulations will be affected by order, so normalise.
        TransformNormalise tnr = new TransformNormalise(m1);
        m1.transform(tnr);
        
        //  Continue flattening as if in the domain filtering pass. 
        mc.instanceFlattening(true);
        
        //  Get rid of sum equal for Minion.
        TransformSumEq t5 = new TransformSumEq(true);
        m1.transform(t5);
        
        m1.simplify();
        
        //  Default solve with Minion
        
        //  Clear the solution table file. 
        (new File(CmdFlags.getMinionSolsTempFile())).delete();
        
        try {
            BufferedWriter out;
            out = new BufferedWriter(new FileWriter(CmdFlags.minionfile));
            m1.toMinion(out, false);
            out.close();
        } catch (IOException e) {
            System.out.println("Could not open file for Minion output.");
            CmdFlags.exit();
        }
        
        MinionSolver min = new MinionSolver();
        
        ArrayList<String> stdout_lines=null;
        try {
            stdout_lines=min.runMinion(CmdFlags.getMinion(), CmdFlags.minionfile, m1, false, -1);
        } catch (Exception e) {
            System.out.println("Could not run Minion for tabulation.");
            CmdFlags.exit();
        }
        
        String lastline=stdout_lines.get(stdout_lines.size()-1).trim();
        
        ArrayList<ASTNode> table=null;
        
        if(lastline.equals("STOP-PC")) {
            System.out.println("Stopped by progress-check in Minion");
        }
        else if(lastline.equals("STOP-NC")) {
            System.out.println("Stopped by node limit in Minion");
        }
        else if(lastline.equals("STOP-SC")) {
            System.out.println("Stopped by solution limit in Minion (1)");
        }
        else if(CmdFlags.tab_stop_btfree && lastline.equals("STOP-BTFREE")) {
            System.out.println("Stopped by a backtrack-free search in Minion");
        }
        else {
            ///  Read in the solutions.   Same as makeTable method in MinionSearch. 
            BufferedReader minsolfile=null; 
            try { 
                minsolfile=new BufferedReader(new FileReader(CmdFlags.getMinionSolsTempFile()));
            } catch (IOException e) {
                System.out.println("Could not open Minion solution file.");
                CmdFlags.exit();
            }
            
            table=new ArrayList<ASTNode>();
            try {
                String s=minsolfile.readLine();
                while(s!=null) {
                    String[] vals=s.split("\\s");  // Split by space into individual values.
                    ArrayList<ASTNode> tup=new ArrayList<ASTNode>();
                    for(int i=0; i<vals.length; i++) {
                        tup.add(NumberConstant.make(Long.valueOf(vals[i])));
                    }
                    
                    assert tup.size()==varlist.size();
                    table.add(CompoundMatrix.make(tup));
                    
                    s=minsolfile.readLine();
                }
                if(table.size()>tuplelim) {
                    table=null;
                    System.out.println("Stopped by solution limit in Minion (2)");
                }
            }
            catch(IOException e) {
                return null;
            }
            
            try {
                File f=new File(CmdFlags.getMinionSolsTempFile());
                f.delete();
            } catch (Exception x) {
            }
        }
        
        //  Put everything back. 
        CmdFlags.setOutputReady(tmp_outready);
        CmdFlags.setAfterAggregate(tmp_afteragg);
        CmdFlags.solverflags=tmp_solveropts;
        CmdFlags.preprocess=tmp_preprocess;
        CmdFlags.setUseDeleteVars(tmp_delvars);
        CmdFlags.setRemoveRedundantVars(tmp_rrv);
        CmdFlags.setAuxNonFunctional(tmp_auxnon);
        
        //  Clear the solution table file. 
        (new File(CmdFlags.getMinionSolsTempFile())).delete();
        
        if(table!=null) {
            /*System.out.println("Replacing: "+in);
            System.out.println("           "+new Table(m, CompoundMatrix.make(TabulationUtils.getVariablesOrdered(in)), CompoundMatrix.make(table)));
            
            ASTNode othertab  = tu.makeTableLong(in.copy(), Long.MAX_VALUE, Long.MAX_VALUE);
            
            System.out.println("Other: ");
            System.out.println("           "+othertab);
            
            //System.out.println(othertab.getChildConst(1));
            
            assert othertab.getChildConst(1).equals(CompoundMatrix.make(table));*/
            
            ASTNode tab=CompoundMatrix.make(table);
            tab=m.cmstore.newConstantMatrixDedup(tab);
            
            ASTNode newTable=new Table(m, CompoundMatrix.make(TabulationUtils.getVariablesOrdered(in)), tab);
            return newTable;
        }
        else {
            return null;
        }
    }
    
}
