/*
Copyright 2001 Periklis Sochos

This file is part of PascalFCv1.0.

PascalFCv1.0 is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

PascalFCv1.0 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with PascalFCv1.0; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package pfc.bin.ui;

import pfc.bin.engine.PascalFC;

import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import javax.swing.SwingUtilities;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.filechooser.*;


/**
    The class providing the interface
    for the PFCI system
 */
public class PFCInterface implements ActionListener {
    private PascalFC pfc = new PascalFC();
    private File loadedFile = null;
    private Timer timer;
    private boolean stopExecution = false;
    
    // The variable below allows locating
    // resourses residing in the same dir as
    // the class, it must match the package
    // statement at the top
    private final String jpack = "pfc/bin/ui/"; 
    
    // The SwingWorker for the tedious task of
    // reading the output of a program's execution
    SwingWorker worker;
    
    // GUI Components
    private JFrame f;
    private JPanel p;
    private JMenuBar mb;
    private JMenu mFile;
    private JMenuItem mOpen;
    private JMenuItem mExit;
    private JMenu mEdit;
    private JMenuItem mSelectAll;
    private JMenuItem mCut;
    private JMenuItem mCopy;
    private JMenuItem mPaste;
    private JMenu mShow;
    private JMenuItem mProgramFile;
    private JMenuItem mListFile;
    private JMenuItem mPmdFile;
    private JMenu mRun;
    private JMenuItem mRunFile;
    private JMenuItem mStop;
    private ButtonGroup mGroup;
    private JRadioButtonMenuItem mFair;
    private JRadioButtonMenuItem mUnFair;
    private JMenu mAbout;
    private JMenuItem mAboutPascalFC;
    private JToolBar tb;
    private JButton tOpen;
    private JButton tRun; 
    private JButton tStop;
    private JScrollPane sp;
    private JTextArea ta;
    private JPanel bp;
    private JLabel l;
    private JProgressBar pb;
        
    public PFCInterface() {
        // Component Initialization
		// Menu Bar		
        mb = new JMenuBar();			
        populateMenuBar(mb); 
		// Output Screen
        ta = new JTextArea(23, 34);
        ta.setFont(new Font("SanSerif", Font.PLAIN, 10));
        sp = new JScrollPane(ta);
		// Tool Bar
        tb = new JToolBar();
        tb.setFloatable(false);
        populateToolBar(tb);	
        // Progress Bar
        pb = new JProgressBar(0, 100);
        pb.setVisible(false);				
		// Status Bar
        l = new JLabel("Click Open to load a pfc file");
        // Add Progress and Status Bar to the bottom panel
        bp = new JPanel(new BorderLayout());
        bp.add(pb, BorderLayout.NORTH);
        bp.add(l, BorderLayout.SOUTH);

		// Set up a main panel for the components
        p = new JPanel(new BorderLayout());		

		// Add components to main panel
        p.add(tb, BorderLayout.NORTH);
        p.add(sp, BorderLayout.CENTER);
        p.add(bp, BorderLayout.SOUTH);

		// Set up a main frame
        f = new JFrame("Pascal FC v1.0");
        // Don't allow the window to close when the user clicks the
        // X button unless you explicitly call the exit method
        f.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);

		// Add the menu bar to the frame
        f.setJMenuBar(mb);

		// Add the main pane to the frame
        f.setContentPane(p);

        f.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                exitProgram();
            }
        });

        f.pack();
        f.setVisible(true);				               
    }
    
	public void actionPerformed(ActionEvent e) {
		// Try for menus
		try {
			JMenuItem i = (JMenuItem)(e.getSource());           
            
			if(i.getText() == "Open...") {
                openFile();
			}
			if(i.getText() == "Exit") {
				exitProgram();
			}
			if(i.getText() == "Select all") {
				selectAllScreen();
			}
			if(i.getText() == "Cut") {
				cutText();
			}
			if(i.getText() == "Copy") {
				copyText();
			}
			if(i.getText() == "Paste") {
				pasteText();
			}
			if(i.getText() == "Program File") {
                showFile("programFile");
			}
			if(i.getText() == "List File") {
				showFile("listFile");
			}
			if(i.getText() == "Pmd File") {
				showFile("pmdFile");
			}
			if(i.getText() == "Run pfc file") {
                runFile();
			}
			if(i.getText() == "Stop execution") {
                stopExecution();
			}
			if(i.getText() == "fair interpeter (-f)"
                && loadedFile != null) {
                showMode("fair");
			}
			if(i.getText() == "unfair interpeter (-uf)"
                && loadedFile != null) {
                showMode("unfair");
			}
			if(i.getText() == "About PascalFC...") {                
				showAbout();
			}            
		// Catch toolbar buttons
		} catch(Exception ex) {
			JButton b = (JButton)(e.getSource());
			ImageIcon icon = (ImageIcon)b.getIcon();
			
			if(icon.toString() == jpack + "images/Open.gif") {
                openFile();
            }
			if(icon.toString() == jpack + "images/Run.gif") {
                runFile();                                                                    
			}    
			if(icon.toString() == jpack + "images/Stop.gif") {
                stopExecution();                                                                    
			}    
		}        
	}        
    
    private void runFile() {
        String mode = "";                        
        
        // Check if the given file exists
        // and send it for execution
        if(!loadedFile.exists()) {
            errorMessage("Specified file does not exist!", 
                         "File Error");           
            return;
        }
        else {
            disableRunEnableStop();
            disableShowMenuButtons();
                    
            if(mFair.isSelected()) mode = "-f";
            else mode = "-uf";                        
            pfc.executeFile(loadedFile, mode);
        }        

        // Read the chars returned from the 
        // controller and append them to the text area,
        // return control to the interface by using the
        // SwingWorker class
        worker = new SwingWorker() {
            public Object construct() {         
                while(!pfc.isEndOfFile()) {              
                        ta.append(getNextChar().toString());
                        scrollDown();            
                }

                return null;                            
            }
            public void finished() {
                enableRunDisableStop();
                enableShowMenuButtons();                   
            }            
        };
        worker.start();                               
        
        scrollDown();
        
        // Add visual separator
        ta.append("\n\n********************************************\n\n");   
    }    

    private Character getNextChar() {
        try {
            if (stopExecution) {
                throw new InterruptedException();
            } else {
                return new Character(pfc.getChar());
            }
        }
        catch (InterruptedException e) {
            // The finished() method of the SwingWorker
            // is automatically invoked afterwards.
            stopExecution = false;
            pfc.stopExecution();                                       
            ta.append("\n\nProgram terminated by user");                    
            
            return new Character(' ');
        }
    }

    private void stopExecution() {      
        stopExecution = true;
    }
    
    private void openFile() {
        JFileChooser fc;
        
        if(loadedFile == null)
            fc = new JFileChooser();
        else
            fc = new JFileChooser(loadedFile.getAbsolutePath());
            
        PFCFileFilter ff = new PFCFileFilter();
                
        fc.addChoosableFileFilter(ff);
        fc.setAcceptAllFileFilterUsed(false);        
        fc.setFileFilter(ff);
        
        int returnVal = fc.showOpenDialog(this.p);
                
        if (returnVal == JFileChooser.APPROVE_OPTION) {
            // A pfc file is loaded
            enableRunButtons();
            
            loadedFile = fc.getSelectedFile();
       
            String mode = "";
            if(mFair.isSelected())mode = "fair";
            else mode = "unfair";         
            l.setText("Loaded: " + loadedFile.getName()
                + ",  Click Run to execute (" + mode + ")");
        }
    }    
    
	private void exitProgram() {
        // If the user has loaded a program  and the
        // program is not currently running then
        // confirm before exit
        if(loadedFile != null && !mStop.isArmed()) {
            int n = JOptionPane.showConfirmDialog(
                f,
                "A file is loaded for execution, exit anyway?",
                "Exit Program",
                JOptionPane.YES_NO_OPTION);
            if(n == JOptionPane.YES_OPTION) {
                System.exit(0);
            }
            else {
                return;
            }
        } else {
            // Check if any process is running 
            // (is the Stop button armed? is a file loaded?)
            // and confirm before exit        
            if(loadedFile != null && mStop.isArmed()) {
                int n = JOptionPane.showConfirmDialog(
                    f,
                    "A program is running, exit anyway?",
                    "Exit Program",
                    JOptionPane.YES_NO_OPTION);
                if(n == JOptionPane.YES_OPTION) {
                    stopExecution();
                    // Wait until the program has been stoped and exit
                    delayAndExit();
                }
                else {
                    return;
                }
            } else {
                System.exit(0);   
            }
        }        
	}    

    private void delayAndExit() {
        pb.setString("Terminating Program...");
        pb.setStringPainted(true);
        pb.setValue(0); 
        pb.setVisible(true);
        // Wait until the execution stops
        //Create a timer.
        timer = new Timer(20, new ActionListener() {
            int i = 0;
            public void actionPerformed(ActionEvent evt) {
                if (i == 100) {
                    timer.stop();
                    System.exit(0);
                }
                i++;                  
                if(!pfc.isEndOfFile() && i == 100)
                    i = 0;                                                       
                pb.setValue(i);                    
            }
        });
        timer.start();
    }

    private void selectAllScreen() {
        ta.grabFocus();
        ta.selectAll();
    }
    
    private void cutText() {
        ta.cut();
    }
    
    private void copyText() {
        ta.copy();
    }
    
    private void pasteText() {
        ta.paste();
    }
    
    private void showFile(String fileType) {
        String filePath = "";
        
        if(loadedFile != null) {
            if(loadedFile.exists()) {
                if(fileType == "programFile") {
                    filePath = loadedFile.getAbsolutePath();
                }            
                else if(fileType == "listFile") {
                    filePath = pfc.getListFilePath(loadedFile);                
                }
                else if(fileType == "pmdFile") {
                    filePath = pfc.getPmdFilePath(loadedFile);                
                }
                
                if(filePath == "") {
                    errorMessage("File does not exist", "Missing File");
                    return;
                }
                
                if(new File(filePath).length() == 0) {
                    errorMessage("File is empty", "Empty File");
                    return;                    
                }
                
                // Append the file on the Output Screen
                
                // Disable the Run Buttons
                disableRunButtons();
                
                // Add visual separator
                ta.append("\n\n********************************************\n\n");   
                        
                try {
                    BufferedReader br = new BufferedReader(
                        new FileReader(new File(filePath)));               
                    
                    String line = "";
                    
                    while((line = br.readLine()) != null) {
                        ta.append(line + "\n");
                        scrollDown();
                    }
                    
                    br.close();
                                  
                } catch (Exception e) {
                    errorMessage("Unable to read file!", "I/O error");
                }   

                // Enable the Run Buttons
                enableRunButtons();                             
            } else {
                errorMessage("The loaded file's path is no longer valid!",
                    "Missing Loaded File");                
            }
        } else {
            errorMessage("A file has not been loaded!",
                "Load File");
        }
    }
    
    private void showAbout() {
        String text;
        
        text = "The present software is a GUI targeting   "
            +  "\nWin2000 platforms for the Pascal FC     "
            +  "\n programming language.                  "
            +  "\n\nReport bugs to:  burns@cs.york.ac.uk  "
            +  "\n\nAuthor:    Periklis Sochos            "
            +  "\nemail:    periklis78@hotmail.com        "
            +  "\n                                        "
            +  "\n Department of Computer Science         "
            +  "\n           University of York, UK       "; 
            
            JOptionPane.showMessageDialog(f,
                text,
                "About",
                JOptionPane.INFORMATION_MESSAGE,
                new ImageIcon(jpack + "images/About.gif"));           
    }
    
    private void scrollDown() {
        // Scrolls bown the text area
        Rectangle visibleRect = new Rectangle(0, ta.getHeight(), 1, 1);                
        sp.getViewport().scrollRectToVisible(visibleRect);
    }    
    
    private void enableRunButtons() {
        // Enadle run buttons    
        mRunFile.setEnabled(true);
        tRun.setEnabled(true);
        mFair.setEnabled(true);
        mUnFair.setEnabled(true);    
    }
    
    private void disableRunButtons() {
        // Disable run buttons    
        mRunFile.setEnabled(false);
        tRun.setEnabled(false);
        mFair.setEnabled(false);
        mUnFair.setEnabled(false);        
    }
    
    private void enableShowMenuButtons() {
        // Enable the Show Menu buttons
        mProgramFile.setEnabled(true);
        mListFile.setEnabled(true);
        mPmdFile.setEnabled(true);
    }
    
    
    private void disableShowMenuButtons() {
        // Disable the Show Menu buttons
        mProgramFile.setEnabled(false);
        mListFile.setEnabled(false);
        mPmdFile.setEnabled(false);        
    }
    
    private void enableRunDisableStop() {
        // Enadle run disable stop
        // (Note: The arm method call
        // to the stop button is used to detect
        // if a program is running in the exitProgram())        
        tRun.setEnabled(true);
        mRunFile.setEnabled(true);
        mFair.setEnabled(true);
        mUnFair.setEnabled(true);        
        tStop.setEnabled(false);
        mStop.setEnabled(false);
        mStop.setArmed(false);
        
        // Set the label message
        l.setText("Loaded: " + loadedFile.getName()
            + ", Click Run to execute ");
        String mode = "";
        if(mFair.isSelected())mode = "fair";
        else mode = "unfair";        
        showMode(mode);
        // Scroll down and repaint text area
        scrollDown();        
        ta.paintImmediately(ta.getVisibleRect());           
    }
    
    private void disableRunEnableStop() {
        // Disable run enable stop
        // (Note: The arm method call
        // to the stop button is used to detect
        // if a program is running in the exitProgram())
        tRun.setEnabled(false);
        mRunFile.setEnabled(false);
        mFair.setEnabled(false);
        mUnFair.setEnabled(false);
        tStop.setEnabled(true);
        tStop.grabFocus();
        mStop.setEnabled(true);
        mStop.setArmed(true);
        // Set the label message
        l.setText("Loaded: " + loadedFile.getName()
            + ", Click Stop to terminate ");
        String mode = "";
        if(mFair.isSelected())mode = "fair";
        else mode = "unfair";        
        showMode(mode);         
    }
    
    private void showMode(String mode) {
        // Find the "(" and replace it with the given mode
        int i;
        for(i = 0; i < l.getText().length(); i++)
            if(l.getText().charAt(i) == '(')
                break;                
        l.setText(l.getText().substring(0, i) + "(" + mode + ")");   
    }
    
    private void errorMessage(String text, String title) {
        //custom title, warning icon
        JOptionPane.showMessageDialog(f,
            text,
            title,
            JOptionPane.WARNING_MESSAGE);           
    }
    
	private void populateMenuBar(JMenuBar mb) {
        // Menus
        // File
        mFile = new JMenu("File");
        mFile.setMnemonic(KeyEvent.VK_I);
            // Open
            mOpen = new JMenuItem("Open...", new ImageIcon(jpack + "images/Open.gif"));
            mOpen.setMnemonic(KeyEvent.VK_O);
            mOpen.setAccelerator(KeyStroke.getKeyStroke(
                KeyEvent.VK_O, ActionEvent.ALT_MASK));
            mOpen.addActionListener(this);
            // Exit
            mExit = new JMenuItem("Exit", new ImageIcon(jpack + "images/Exit.gif"));
            mExit.setMnemonic(KeyEvent.VK_X);
            mExit.setAccelerator(KeyStroke.getKeyStroke(
                KeyEvent.VK_X, ActionEvent.ALT_MASK));
            mExit.addActionListener(this);
        mFile.add(mOpen);
        mFile.addSeparator();
        mFile.add(mExit);
        // Edit
        mEdit = new JMenu("Edit");
        mEdit.setMnemonic(KeyEvent.VK_E);
            // Select all
            mSelectAll = new JMenuItem("Select all", new ImageIcon(jpack + "images/SelectAll.gif"));
            mSelectAll.setMnemonic(KeyEvent.VK_A);
            mSelectAll.setAccelerator(KeyStroke.getKeyStroke(
                KeyEvent.VK_A, ActionEvent.ALT_MASK));
            mSelectAll.addActionListener(this);             
            // Cut
            mCut = new JMenuItem("Cut", new ImageIcon(jpack + "images/Cut.gif"));
            mCut.setMnemonic(KeyEvent.VK_C);
            mCut.setAccelerator(KeyStroke.getKeyStroke(
                KeyEvent.VK_C, ActionEvent.ALT_MASK));
            mCut.addActionListener(this);             
            // Copy
            mCopy = new JMenuItem("Copy", new ImageIcon(jpack + "images/Copy.gif"));
            mCopy.setMnemonic(KeyEvent.VK_V);
            mCopy.setAccelerator(KeyStroke.getKeyStroke(
                KeyEvent.VK_V, ActionEvent.ALT_MASK));
            mCopy.addActionListener(this);             
            // Paste
            mPaste = new JMenuItem("Paste", new ImageIcon(jpack + "images/Paste.gif"));
            mPaste.setMnemonic(KeyEvent.VK_P);
            mPaste.setAccelerator(KeyStroke.getKeyStroke(
                KeyEvent.VK_P, ActionEvent.ALT_MASK));
            mPaste.addActionListener(this);             
        mEdit.add(mSelectAll);
        mEdit.addSeparator();
        mEdit.add(mCut);
        mEdit.add(mCopy);
        mEdit.add(mPaste);
        // Show
        mShow = new JMenu("Show");
        mShow.setMnemonic(KeyEvent.VK_S);
            // ProgramFile
            mProgramFile = new JMenuItem("Program File");
            mProgramFile.setMnemonic(KeyEvent.VK_P);
            mProgramFile.addActionListener(this);             
            // ListFile
            mListFile = new JMenuItem("List File");
            mListFile.setMnemonic(KeyEvent.VK_L);
            mListFile.addActionListener(this);             
            // PmdFile
            mPmdFile = new JMenuItem("Pmd File");
            mPmdFile.setMnemonic(KeyEvent.VK_M);
            mPmdFile.addActionListener(this);             
        mShow.add(mProgramFile);
        mShow.add(mListFile);
        mShow.add(mPmdFile);
        // Run
        mRun = new JMenu("Run");
        mRun.setMnemonic(KeyEvent.VK_N );
            // Run pfc file
            mRunFile = new JMenuItem("Run pfc file", new ImageIcon(jpack + "images/Run.gif"));
            mRunFile.setEnabled(false);
            mRunFile.setMnemonic(KeyEvent.VK_R);
            mRunFile.setAccelerator(KeyStroke.getKeyStroke(
                KeyEvent.VK_R, ActionEvent.ALT_MASK));
            mRunFile.addActionListener(this);
            // Stop execution
            mStop = new JMenuItem("Stop execution", new ImageIcon(jpack + "images/Stop.gif"));
            mStop.setEnabled(false);
            mStop.setMnemonic(KeyEvent.VK_S);            
            mStop.setAccelerator(KeyStroke.getKeyStroke(
                KeyEvent.VK_S, ActionEvent.ALT_MASK));
            mStop.addActionListener(this);            
            //fair and unfair interpeter
            mGroup = new ButtonGroup();
                // Fair
                mFair = new JRadioButtonMenuItem("fair interpeter (-f)");
                mFair.setSelected(true);
                mFair.setEnabled(false);
                mFair.setMnemonic(KeyEvent.VK_F);
                mFair.setAccelerator(KeyStroke.getKeyStroke(
                    KeyEvent.VK_F, ActionEvent.ALT_MASK));                
                mFair.addActionListener(this);            
                // Unfair
                mUnFair = new JRadioButtonMenuItem("unfair interpeter (-uf)");                
                mUnFair.setMnemonic(KeyEvent.VK_U);
                mUnFair.setEnabled(false);
                mUnFair.setAccelerator(KeyStroke.getKeyStroke(
                    KeyEvent.VK_U, ActionEvent.ALT_MASK)); 
                mUnFair.addActionListener(this);                    
            mGroup.add(mFair);            
            mGroup.add(mUnFair);            
        // Add menu items to the menu
        mRun.add(mRunFile);
        mRun.add(mStop);        
        mRun.addSeparator();
        mRun.add(mFair); 
        mRun.add(mUnFair);
        // About Menu
        mAbout = new JMenu("About");     
        mAbout.setMnemonic(KeyEvent.VK_B);
            // About->About PascalFC
            mAboutPascalFC = new JMenuItem("About PascalFC...", new ImageIcon(jpack + "images/About.gif"));
            mAboutPascalFC.setMnemonic(KeyEvent.VK_B);
            mAboutPascalFC.addActionListener(this);        
        // Add menu items to the menu
        mAbout.add(mAboutPascalFC);  
        
        // Add all menus to the menu bar
        mb.add(mFile);
        mb.add(mEdit);
        mb.add(mShow);
        mb.add(mRun);
        mb.add(mAbout);
	}    
    
	private void populateToolBar(JToolBar tb) {			
		tOpen = new JButton(new ImageIcon(jpack + "images/Open.gif"));
		tOpen.setToolTipText("Open a pfc file from disk");
		tOpen.addActionListener(this);
		tRun = new JButton(new ImageIcon(jpack + "images/Run.gif"));
        tRun.setEnabled(false);
		tRun.setToolTipText("Run the loaded pfc file");
		tRun.addActionListener(this);
		tStop = new JButton(new ImageIcon(jpack + "images/Stop.gif"));
        tStop.setEnabled(false);
		tStop.setToolTipText("Stop program execution");
		tStop.addActionListener(this);
        
		tb.add(tOpen);
        tb.add(new JToolBar.Separator());        
        tb.add(tRun);
        tb.add(tStop);
	}           
}

class PFCFileFilter extends FileFilter {
    
    // Accept all directories only pfc files.
    public boolean accept(File f) {
        if (f.isDirectory()) {
            return true;
        }

        String extension = Utils.getExtension(f);
	   if (extension != null) {
            if (extension.equals(Utils.pfc)) {
                    return true;
            } else {
                return false;
            }
    	}
        
        return false;
    }
    
    // The description of this filter
    public String getDescription() {
        return "pfc files";
    }
}

class Utils {

    public final static String pfc = "pfc";

    /*
     * Get the extension of a file.
     */
    public static String getExtension(File f) {
        String ext = null;
        String s = f.getName();
        int i = s.lastIndexOf('.');

        if (i > 0 &&  i < s.length() - 1) {
            ext = s.substring(i+1).toLowerCase();
        }
        return ext;
    }
}

/**
 * This is the 3rd version of SwingWorker (also known as
 * SwingWorker 3), an abstract class that you subclass to
 * perform GUI-related work in a dedicated thread.  For
 * instructions on using this class, see:
 * 
 * http://java.sun.com/docs/books/tutorial/uiswing/misc/threads.html
 *
 * Note that the API changed slightly in the 3rd version:
 * You must now invoke start() on the SwingWorker after
 * creating it.
 */
abstract class SwingWorker {
    private Object value;  // see getValue(), setValue()
    private Thread thread;

    /** 
     * Class to maintain reference to current worker thread
     * under separate synchronization control.
     */
    private static class ThreadVar {
        private Thread thread;
        ThreadVar(Thread t) { thread = t; }
        synchronized Thread get() { return thread; }
        synchronized void SelectAll() { thread = null; }
    }

    private ThreadVar threadVar;

    /** 
     * Get the value produced by the worker thread, or null if it 
     * hasn't been constructed yet.
     */
    protected synchronized Object getValue() { 
        return value; 
    }

    /** 
     * Set the value produced by worker thread 
     */
    private synchronized void setValue(Object x) { 
        value = x; 
    }

    /** 
     * Compute the value to be returned by the <code>get</code> method. 
     */
    public abstract Object construct();

    /**
     * Called on the event dispatching thread (not on the worker thread)
     * after the <code>construct</code> method has returned.
     */
    public void finished() {
    }

    /**
     * A new method that interrupts the worker thread.  Call this method
     * to force the worker to stop what it's doing.
     */
    public void interrupt() {
        Thread t = threadVar.get();
        if (t != null) {
            t.interrupt();
        }
        threadVar.SelectAll();
    }

    /**
     * Return the value created by the <code>construct</code> method.  
     * Returns null if either the constructing thread or the current
     * thread was interrupted before a value was produced.
     * 
     * @return the value created by the <code>construct</code> method
     */
    public Object get() {
        while (true) {  
            Thread t = threadVar.get();
            if (t == null) {
                return getValue();
            }
            try {
                t.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt(); // propagate
                return null;
            }
        }
    }


    /**
     * Start a thread that will call the <code>construct</code> method
     * and then exit.
     */
    public SwingWorker() {
        final Runnable doFinished = new Runnable() {
           public void run() { finished(); }
        };

        Runnable doConstruct = new Runnable() { 
            public void run() {
                try {
                    setValue(construct());
                }
                finally {
                    threadVar.SelectAll();
                }

                SwingUtilities.invokeLater(doFinished);
            }
        };

        Thread t = new Thread(doConstruct);
        threadVar = new ThreadVar(t);
    }

    /**
     * Start the worker thread.
     */
    public void start() {
        Thread t = threadVar.get();
        if (t != null) {
            t.start();
        }
    }
}
