/*
Copyright 2001 Periklis Sochos

This file is part of PascalFCv1.0.

PascalFCv1.0 is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

PascalFCv1.0 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with PascalFCv1.0; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

package pfc.bin.console;

import java.io.*;

import javax.swing.SwingUtilities;

/**
    The "heart" of the PFCI system
    providing all interaction with
    the operating system
 */
public class Console {
    // Variables to hold the directory where the class resides.
    // In this directory all the binary resourses are located
    // the pfccomp, pint and ufpint, kill and ps. We use the
    // variables to locate the binary resourses and run them 
    // through commands to the operating systems' console    
    final private String topDir;        
    final private String pack;        
    final private String classPath;    
    
    // Conditions to exit the program's execution
    final private String normalTerm;
    final private String abnormalTerm;    
    
    // The streams used to read the program's output
    private Process pr;
    private InputStream is;
    private InputStreamReader sr;
    private BufferedReader br;

    // The following variables are used to 
    // allow the detection of a programs normal 
    // termination(see getChar()), they are declared
    // as instance variables to retain their values
    // betweens calls for the next stream char    
    private boolean streamsOpen;      
    private boolean isEOF;    
    private String lineBuffer;   
    private char streamChar = ' ';
    
    public Console() {
        topDir = System.getProperty("user.dir");
        pack = "/pfc/bin/console/";
        classPath =  topDir + pack;

        normalTerm = "Program terminated normally";
        abnormalTerm = "See pmdfile for post-mortem report";
        streamsOpen = false;        
        isEOF = false;        
        lineBuffer = "";
        streamChar = ' ';
    }            

    /**
        Checks if another instance of the PFCI 
        system is running, it is also platform depended
        @return true if another instance is running
     */    
    public boolean anotherInstance() {
        try {
            // Run the ps command
            Process pr = runCommand("ps");
            BufferedReader br = new BufferedReader(
                new InputStreamReader(pr.getInputStream()));
			
			// Find another application using java.exe
            String line = "";
            int instances = 0;
            boolean anotherInstanceExists = false;
            
            while((line = br.readLine()) != null) {                     
                if(getRunningProcess(line).compareTo("java.exe") == 0) {
                    instances++;
                    if(instances > 1) {
                        anotherInstanceExists = true;
                        break;
                    }
                }  
            }
            return anotherInstanceExists;                              
        } catch (Exception e) {
            System.out.println("Unable to detect instance! Exiting...");
            System.out.println(e);
            return true;
        }                                
    }
    
    /**
        Returns the path of the list file path created
        for the currently loaded file. If the file does
        not exist then it returns an empty string.
        @return The list file's absolute path or
        the empty string ("") if the file does not
        exist.
    */
    public String getListFilePath(File f) {
        String listFilePath = "";
       
        listFilePath = getPath(f.getAbsolutePath())
            + getName(f.getAbsolutePath())
            + ".LST";

        if(!(new File(listFilePath).exists())) {
            listFilePath = "";   
        }
        
        return listFilePath;
    }
    
    /**
        Returns the path of the pmd file path created
        for the currently loaded file. If the file does
        not exist then it returns an empty string.
        @return The pmd file's absolute path or
        the empty string ("") if the file does not
        exist.
    */
    public String getPmdFilePath(File f) {
        String pmdFilePath = "";
       
        pmdFilePath = getPath(f.getAbsolutePath())
            + getName(f.getAbsolutePath())
            + ".PMD";

        if(!(new File(pmdFilePath).exists())) {
            pmdFilePath = "";   
        }
        
        return pmdFilePath;
    }
    
    /**
        Returns the next character of running files
        output
        @return a char
     */    
    public char getChar() {  
        // Initialize variable                  
        int streamInt = 0;
         
	try {    
            // Set up the streams only the first time you
            // are called to return a char from the stream
            if(!streamsOpen) {			
                openStreams();
            }            
            // Read and return the next char from the stream
            if( !isEOF && ((streamInt = br.read()) != -1)) {
                streamChar = (char)streamInt;            
                //  Test if you have reached the end of 
                // a normal-abnormal program execution or 
                // maximum allowed output for infinate programs
                if(lineBuffer.compareTo(normalTerm) == 0 
                   || lineBuffer.compareTo(abnormalTerm) == 0) {
                    // Close streams, kill processes and reset variables
                    cleanUp();                    
                    return ' ';                    
                }
                // Buffer the output lines
                if(streamChar != '\n') lineBuffer += streamChar;
                else lineBuffer = "";
                
                return streamChar;
            } 
            // At this point of the method something
            // has really gone wrong! Under normal
            // conditions we should never reach this point
            // it is here for the java compiler not to complane
            cleanUp();
            
            return ' ';
	}
	catch( Exception ex ) {
		return '!';
	}           
    }
    
    /**
        It submits the specified file for execution
        @param f the pfc file currently loaded
        @param mode the mode of execution either "-f"
            for a "fair" execution or "-uf" for "unfair"
            execution
     */    
    public void executeFile(File f, String mode) {
		// In unix only -f is accepted
		mode = "";
        // The commands to be send
        String command = "";
        
        // Store the file's path\name
        String runFilePath = f.getAbsolutePath();
        
        // Run the pfc command, syntax:
        // pfc mode(-f, -uf) progfile listfile objfile pmdfile
        // (the pfc batch file used to compile and interprete
        //  a pfc program creates only 8 character long file
        //  names, in order to distinguish between files
        //  belonging to a specific program I use the 8 first
        //  letters of the name of the pfc file and add a suffix)
        command = 		classPath       
                        + "shell "
                        + classPath
                        + " " + runFilePath
                        + " " + getPath(runFilePath)
                        + getName(runFilePath) + ".LST"
                        + " " + getPath(runFilePath)
                        + getName(runFilePath) + ".OBJ"
                        + " " + getPath(runFilePath)
                        + getName(runFilePath) + ".PMD";             
        pr = runCommand(command);		        
        isEOF = false;                   
    }

    /**
        Stops the currenly running file
        @return a char
     */    
    public void stopExecution() {
        // Close streams, kill processes and reset variables
        cleanUp();   
    }
    
    /**
        A method to check if the running pfc file
        has completed execution
        @rerutn true if the file has completed execution
            false otherwise
     */    
    public boolean isEndOfFile() {
        // Returns true if the file's execution is
        // for some reason terminated
        return isEOF;   
    }
    
    private void cleanUp() {
        // ATTENTION: The order of killing the processes
        //            is significant!!!!!
        isEOF = true;
        lineBuffer = "";        
        closeStreams();            
        pr.destroy();
        //Windows
	//killProcess();            
    }
    
    private String getName(String absPath) {
        // Gets the absolute path of the file (path\name)
        // currently running and returns the name part
        // without the suffix (.pfc). If the name is 
        // longer than 8 characters it returns the first
        // 8 characters.
        int start = absPath.length()-1;
        int end = 0;
        int lenCounter = 0;
        
        while(absPath.charAt(start) != '/')
            start--;
        start++;
        
        end = start;
        while(absPath.charAt(end) != '.' && lenCounter < 8) {
            lenCounter++;
            end++;
        }
 
        return absPath.substring(start, end);           
    }
    
    private String getPath(String absPath) {
        // Gets the absolute path of the file (path\name)
        // currently running and returns the path part
        int i = absPath.length()-1;
        
        while(absPath.charAt(i) != '/')
            i--;
        
        return absPath.substring(0, i+1);
    }
    
    private void openStreams() {
        // Assign the streams to the currently
        // running process
        is = pr.getInputStream();
        sr = new InputStreamReader(is);
        br = new BufferedReader(sr);
        streamsOpen = true;
    }
    
    private void closeStreams() {
        // Closes all the streams
        try {
            is.close();
            sr.close();        
            br.close();
            streamsOpen = false;
        } catch (IOException e) {
            System.out.println("Unable to close sreams!");   
        }
    }
    
    private Process getProcess(String s) {
        return runCommand(s);       
    }

    private void killProcess() {        
        try {
            Process pr = runCommand("ps -e");
            BufferedReader br = new BufferedReader(
                new InputStreamReader(pr.getInputStream()));               
            
            String line = "";
            String process = "";
            String pid = "";
            
            while((line = br.readLine()) != null) {
                if(getRunningProcess(line).compareTo("java") == 0) {
                    runCommand("kill " + getPid(line));
                }                 
            }                               
        } catch (Exception e) {
            System.out.println(e);
        }                                
    }     
    
    private String getPid(String line) {
        String pid = "";
        int i = 0;
        
        // Ignore empty chars        
        while(line.charAt(i) == ' ')
            i++;
        
        // Read the pid until you hit an empty char
        while(line.charAt(i) != ' ') {
            pid += line.charAt(i);
            i++;
        }
        return pid;
    }
    
    private String getRunningProcess(String line) {
        String process = "";
        int i = line.length()-1;
        
        // Go the start of the process name by
        // reading backwards the line until you hit
        // an empty char
        while(line.charAt(i) != ' ')
            i--;
                    
        // Now go to the start of the processe's name
        // and start reading the name until the
        // end of the line
        i++;
        while(i < line.length()) {
            process += line.charAt(i);
            i++;
        }
        return process;
    }   
    
    private Process runCommand(String s) {
        try {		
            return Runtime.getRuntime().exec(s);        
        } catch (IOException e) {
            System.out.println("Unable to execute command!");		
			System.out.println(e);
            return null;
        }
    }
}
