! ==============================================================================
! Subroutine: RBS_FIRE (RNA,AMAX) --- CELL MRNA
! 
! Purpose: Fires one of the Ribosomal Binding Site (RBS) reactions
!          for the hairpins/multi-loops in an RNA Loop.
!
! Method:  The info array holds the following information:
!
!          istate = info(1,i) --> state of the RNA or enzyme object
!            ipro = info(2,i) --> protein number bound to the helix
!           ileaf = info(3,i) --> link to the first helix in the multi-loop
!           iroot = info(4,i) --> link to the root helix (or 0 if ext loop)
!            iloc = info(5,i) --> 5' nt of closing base pair
!            jloc = info(6,i) --> 3' nt of closing base pair
!
! Arguments:
!
!             RNA - Class structure containing information on the
!                   RNA and its possible reactions.
!            IMIN - 5p nucleotide of the region in the RNA in which
!                   the fold has changed.
!            IMAX - 3p nucleotide of the region in the RNA in which
!                   the fold has changed.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - RIBOSOME, PROTEIN, CONSTANTS
! Functions -
! Subroutines - RESUM
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE RBS_FIRE (RNA,AMAX)

        USE Ribosome, ONLY : r_b1f,r_b2f
        USE Constants,ONLY : i5p_ribo,i3p_ribo,time
        USE Protein,  ONLY : npro,id_30pi,n_ribo,NSTATS

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        TYPE(MRNA), INTENT(INOUT) :: rna

        DOUBLE PRECISION, INTENT(IN) :: amax

        !=== VARIABLES ===!

        INTEGER :: indx,jndx,kndx,imin,imax,iaug

        DOUBLE PRECISION :: atot,xp


        atot = 0.0d0
        indx  = rna% ihead

        !=== One RBS Site to calc - cell mrna ===!

        iaug = 1

        imin = 1
        imax = 1 + i3p_ribo

        !=== Add Ribosome 30S:PIC ===!

        npro(id_30pi) = npro(id_30pi) - 1
        n_ribo(5) = n_ribo(5) - 1

        jndx = rna% inext(indx)

        kndx = rna% iadd
        kndx = rna% ilist(kndx)

        rna% iadd = rna% iadd + 1

        rna% inext(indx) = kndx
        IF ( jndx /= 0 ) THEN
        rna% iprev(jndx) = kndx
        ENDIF

        rna% inext(kndx) = jndx
        rna% iprev(kndx) = indx

        rna% info(1,kndx) = 1
        rna% info(2,kndx) = 1
        rna% info(3,kndx) = 0
        rna% info(4,kndx) = 0
        rna% info(5,kndx) = iaug
        rna% info(6,kndx) = iaug

        CALL RIBO_REAC (rna,kndx)

        !=== UPDATE REACTIONS ===!

        !=== Update RBS Reactions ===!

        CALL RBS_REAC (rna,imin,imax)
            nstats(1) = nstats(1) + 1 !gain ribo initiation
            rna% t(kndx) = time !start time for initiation time
        RETURN

      END SUBROUTINE RBS_FIRE
