! ==============================================================================
! Subroutine: RBS_REAC (RNA,IMIN,IMAX) --- CELL MRNA
! 
! Purpose: Computes the possible Ribosomal Binding Site (RBS) reactions
!          for the hairpins/multi-loops in an RNA Loop.
!
! Method:  The info array holds the following information:
!
!          istate = info(1,i) --> state of the RNA or enzyme object
!            ipro = info(2,i) --> protein number bound to the helix
!           ileaf = info(3,i) --> link to the first helix in the multi-loop
!           iroot = info(4,i) --> link to the root helix (or 0 if ext loop)
!            iloc = info(5,i) --> 5' nt of closing base pair
!            jloc = info(6,i) --> 3' nt of closing base pair
!
! Arguments:
!
!             RNA - Class structure containing information on the
!                   RNA and its possible reactions.
!            IMIN - 5p nucleotide of the region in the RNA in which
!                   the fold has changed.
!            IMAX - 3p nucleotide of the region in the RNA in which
!                   the fold has changed.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - RIBOSOME, PROTEIN, CONSTANTS
! Functions -
! Subroutines - RESUM
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE RBS_REAC (RNA,IMIN,IMAX)

        USE Ribosome, ONLY : r_b1f,r_b2f
        USE Constants,ONLY : i5p_ribo,i3p_ribo
        USE Protein,  ONLY : npro,id_30pi

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        TYPE(MRNA), INTENT(INOUT) :: rna
        INTEGER, INTENT(IN) :: imin,imax

        !=== VARIABLES ===!

        INTEGER :: indx,jndx,iw

        LOGICAL :: ifree

        DOUBLE PRECISION :: atot,a(0:mxpro)


        a(:) = 0.0d0
        atot = 0.0d0
        indx  = rna% ihead

        !=== One RBS Site to calc - cell mrna ===!

        !check if RBS is free!

        ifree = .true.

        jndx = rna% inext(indx)

        IF ( jndx /= 0 ) THEN

          iw = rna% info(6,jndx)

          iw = iw - i5p_ribo
          iw = iw - 1 - i3p_ribo

          IF ( iw < 3 ) ifree = .false.

        ENDIF

        IF ( ifree ) atot = r_b2f

        !=== Save Total Flux ===!

        indx = rna% ihead

        a(id_30pi) = atot

        !=== RESUM ===!

        CALL RESUM (rna,indx,a)

        RETURN

      END SUBROUTINE RBS_REAC
