! ==============================================================================
! Subroutine: RIBO_REAC (RNA,INDX)
! 
! Purpose: Computes the possible reactions for a Ribosome bound to an mRNA
!          in the protein synthesis model.
!
! Method:  The info array holds the following information:
!
!          istate = info(1,i) --> ribosome state
!          iasite = info(2,i) --> ribosome a-site state
!          irna_p = info(3,i) --> tRNA in p-site
!          irna_a = info(4,i) --> tRNA in a-site
!          istart = info(5,i) --> start codon of protein being synthesized
!            iloc = info(6,i) --> location of p-site on mRNA
!
!          The possible states are (iasite in brackets []):
!          * Denotes GDP bound
!          $ Denotes changed conformation
!
!          *** RIBOSOMAL INITIATION ***
!
!          (istate = 1) 30S:IC
!
!          [1] 30S:IC
!
!              30S:IC --> 30S:PIC + mRNA           30S:PIC Unbinding
!              30S:IC + 50S --> 70SIC              70S Complex formation
!
!          [2] 30S:IC*
!
!              30S:IC* --> 30S:PIC + mRNA          30S:PIC Unbinding
!              30S:IC* + 50S --> 70S:PIC           70S:PIC Formation
!
!          (istate = 2) 70S:PIC Bound
!
!          [1] 70S:PIC
!
!              70S:PIC --> 30S:IC + 50S            50S Unbinding
!              70S:PIC --> 70S:PIC*                GTP Hydrolysis
!
!          [2] 70S:PIC*
!
!              70S:PIC* --> 30S:IC* + 50S          50S Unbinding
!              70S:PIC* --> 70S:PIC$               Conf. Change
!
!          [3] 70S:PIC$
!
!              70S:PIC$ --> 70S:IC + Pi            Pi Release
!              70S:PIC$ --> 70S:PIC*               Reverse Conf. Change
!
!          [4] 70S:IC
!
!              70S:IC --> 70S:PE + IF1 + IF2:GDP   IF1 and IF2:GDP release
!
!          [5] 70S:PE
!
!              70S:PE --> 70S:EC + IF3             IF3 Release
!
!          *** RIBOSOMAL ELONGATION ***
!
!          (istate = 3) 70S:EC
! 
!          [1] A-Site Vacant
!
!              R + EfTu  --> (R+Ef)                EfTu Binding
!              R + RFx   --> (R+RFx)               RFx Binding
!
!          [2] Ef-Tu Bound
!
!              (R+Ef) --> (R+EfA)                  EfTu Move to A-site
!              (R+Ef) --> R + EfTu                 EfTu Unbinding
!
!          [3] EF-Tu @ A-Site
!
!              (R+EfA) --> (R+Ef*)                 GTP Hydrolysis
!              (R+EfA) --> (R+Ef)                  Ef-Tu Unbind A-Site
!
!          [4] Ef-Tu* @ A-site
!
!              (R+Ef*) --> (R+Ef$)                 EfTu Conf. Change
!
!          [5] Ef-Tu$ @ A-site
!
!              (R+Ef$) --> (R+T) + EfTu*           ACCOMODATION
!              (R+Ef$) --> R + EfTu* + Trna        REJECTION
!
!          [6] TRNA+Peptide @ A-site
!
!              (R+T) + EFG --> (R+T+EFG)           EfG Binding
!
!          [7] EfG @ A-site
!
!              (R+T+EFG) --> (R+T) + EfG           EfG Unbind
!              (R+T+EFG) --> (R+T+E*)              GTP Hydrolysis
!
!          [8] EfG* @ A-site
!
!              (R+T+E*) --> (R+T+E$)               TRANSLOCATION
!
!          [9] TRNA @ E-site
!
!              (R+T+E$) --> R + TRNA + EfG*        TRNA + EFG:GDP Release
!
!          *** RIBOSOMAL TERMINATION ***
!
!          (istate = 4) 70S:Pre/PostHC
!
!          [1] RFx Bound (x=1/2)
!
!              (R+RFx) --> (R+RFx) + P             Protein release
!              (R+RFx) --> R + RFx                 RFx Unbinding
!
!          [2] RFx Bound (x=1/2)
!
!              (R+RFx) + RF3 --> (R+RFx+RF3)       RF3 Binding
!              (R+RFx) --> (R) + RFx               N->R + RFx Release
!
!          [3] RFx+RF3 Bound (N State)
!
!              (R+RFx+RF3) --> (R+RFx+RF3$)        N -> R state
!              (R+RFx+RF3) --> (R+RFx) + RF3       RF3 Unbinding
!
!          [4] RFx+RF3 Bound (R State)
!
!              (R+RFx+RF3$) --> (R+RFx+RF3)        R -> N state
!              (R+RFx+RF3$) --> (R+RF3) + RFx      RFx Release
!              (R+RFx+RF3$) --> (R+RFx) + RF3      RF3 Release
!
!          [5] RF3 Bound (R State)
!
!              (R+RF3$) --> (R+RF3*)               GTP Hydrolysis
!
!          [6] RF3:GDP Bound (R State)
!
!              (R+RF3*) --> R + RF3                RF3 release
!
!          [7] RFx Bound (R State)
!
!              (R+RFx$) --> R + RFx                RFx release
!
!          *** RIBOSOMAL RECYCLING ***
!
!          (istate = 5) 70S:RC
!
!          [1] 70S:RC
!
!              R + RRF --> (R+RRF)                 RRF Binding
!
!          [2] RRF Bound
!
!              (R+RRF) + EfG --> (R+RRF+EfG)       Ef-G Binding
!              (R+RRF)       -->  R + RRF          RRF Unbinding
!
!          [3] RRF+EfG Bound
!
!              (R+RRF+EfG) --> (R+RRF) + EfG       Ef-G Unbinding
!              (R+RRF+EfG) --> (R+RRF+EfG*)        GTP Hydrolysis
!
!          [4] 70S:PreSplit
!
!              (R+RRF+EfG*) --> R + EfG* + RRF     Ribosomal Splitting
!
!          *** RIBOSOMAL SCANNING ***
!
!          (istate = 6) 30S:Scanning
!
! Arguments:
!
!             RNA - Class structure containing information on the
!                   RNA and its possible reactions.
!             INDX- Ribosome number to calculate reactions for.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - RIBOSOME, PROTEIN
! Functions -
! Subroutines - RESUM
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE RIBO_REAC (RNA,INDX)

        USE Ribosome
        USE Protein

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        TYPE(MRNA), INTENT(INOUT) :: rna
        INTEGER, INTENT(IN) :: indx

        !=== VARIABLES ===!

        INTEGER :: istate,iasite,iloc
        INTEGER :: icod_p,icod_a
        INTEGER :: irna_p,irna_a,i

        DOUBLE PRECISION :: a(0:mxpro),r


        istate = rna% info(1,indx)
        iasite = rna% info(2,indx)

        irna_p = rna% info(3,indx)
        irna_a = rna% info(4,indx)

        iloc   = rna% info(6,indx)

        icod_p = rna% icdn(iloc)
        icod_a = rna% icdn(iloc+3)

        a(:) = 0.00d0


        !=== CALCULATE REACTIONS ===!

        !=== STATE 1: 30S:IC ===!

        IF ( istate == 1 ) THEN

          IF ( iasite == 1 ) THEN

            !=== 50S Binding ===!

            a(id_50s) = a(id_50s) + r_i1f

            !=== Unbind from Start Codon ===!

            a(0) = a(0) + 0.0060d0

          ELSEIF ( iasite == 2 ) THEN

            !=== 50S Binding ===!

            a(id_50s) = a(id_50s) + r_i2f

            !=== Unbind from Start Codon ===!

            a(0) = a(0) + 0.0060d0

          ENDIF

        ENDIF

        !=== STATE 2: 70S:IC ===!

        IF ( istate == 2 ) THEN

          IF ( iasite == 1 ) THEN

            !=== 50S Unbinding ===!

            a(0) = a(0) + r_i1b

            !=== GTP Hydrolosis ===!

            a(0) = a(0) + r_i3

          ELSEIF ( iasite == 2 ) THEN

            !=== 50S Unbinding ===!

            a(0) = a(0) + r_i2b

            !=== 50S Closure ===!

            a(0) = a(0) + r_i4f

          ELSEIF ( iasite == 3 ) THEN

            !=== 50S Opening ===!

            a(0) = a(0) + r_i4b

            !=== Pi Release ===!

            a(0) = a(0) + r_i5

          ELSEIF ( iasite == 4 ) THEN

            !=== IF1 and IF2 Release ===!

            a(0) = a(0) + r_i6

          ELSEIF ( iasite == 5 ) THEN

            !=== IF3 Release ===!

            a(0) = a(0) + r_i7

          ELSEIF ( iasite == 6 ) THEN

            !=== Maturation ===!

            a(0) = a(0) + r_i8

          ENDIF

        ENDIF

        !=== STATE 3: 70S:Elongation ===!

        IF ( istate == 3 ) THEN

          IF ( iasite == 1 ) THEN

            !=== EfTu-TRNA Binding ===!

            a(id_tcx) = a(id_tcx) + r_e1f

            !=== RFx Binding ===!

            a(id_rf1) = a(id_rf1) + r_t1f(icod_a,1)
            a(id_rf2) = a(id_rf2) + r_t1f(icod_a,2)

          ELSEIF ( iasite == 2 ) THEN

            !=== EfTu-TRNA Unbinding ===!

            a(0) = a(0) + r_e1b

            !=== EfTu-TRNA Enter Asite ===!

            a(0) = a(0) + r_e2f(icod_a,irna_a)

          ELSEIF ( iasite == 3 ) THEN

            !=== EfTu-TRNA Exit Asite ===!

            a(0) = a(0) + r_e2b(icod_a,irna_a)

            !=== EfTu-TRNA GTP Hydrolosis ===!

            a(0) = a(0) + r_e3 (icod_a,irna_a)

          ELSEIF ( iasite == 4 ) THEN

            !=== EfTu-TRNA Change ===!

            a(0) = a(0) + r_e4(icod_a,irna_a)

          ELSEIF ( iasite == 5 ) THEN

            !=== Amino-Acid Transfer ===!

            a(0) = a(0) + r_e5(icod_a,irna_a)

            !=== TRNA Rejection ===!

            a(0) = a(0) + r_e6(icod_a,irna_a)

          ELSEIF ( iasite == 6 ) THEN

            !=== EfG Binding ===!

            a(id_efg) = a(id_efg) + r_e7f

          ELSEIF ( iasite == 7 ) THEN

            !=== EfG Unbinding ===!

            a(0) = a(0) + r_e7b

            !=== GTP Hydrolosis ===!

            a(0) = a(0) + r_e8

          ELSEIF ( iasite == 8 ) THEN

            !=== Translocation ===!

            a(0) = a(0) + r_e9

          ELSEIF ( iasite == 9 ) THEN

            !=== EfG + TRNA Release ===!

            a(0) = a(0) + r_e10

          ENDIF

        ENDIF

        !=== STATE 4: 70S:Pre/PostHC ===!

        IF ( istate == 4 ) THEN

          IF ( iasite == 1 ) THEN

            !=== RFx Unbinding ===!

            a(0) = a(0) + r_t1b(icod_a,irna_a)

            !=== Protein Release ===!

            a(0) = a(0) + r_t2(icod_a,irna_a)

          ELSEIF ( iasite == 2 ) THEN

            !=== RF3 Binding ===!

            a(id_rf3) = a(id_rf3) + r_t3f

            !=== RFx Release + Ratcheting (RF3-FREE) ===!

            a(0) = a(0) + r_t9(irna_a)

          ELSEIF ( iasite == 3 ) THEN

            !=== RF3 Unbinding ===!

            a(0) = a(0) + r_t3b

            !=== Ratcheting to R State ===!

            a(0) = a(0) + r_t4f

          ELSEIF ( iasite == 4 ) THEN

            !=== Ratcheting to N State ===!

            a(0) = a(0) + r_t4b

            !=== RF3:GTP Release ===!

            a(0) = a(0) + r_t8

            !=== RFx Release ===!

            a(0) = a(0) + r_t5(irna_a)

          ELSEIF ( iasite == 5 ) THEN

            !=== RF3 GTP Hydrolosis ===!

            a(0) = a(0) + r_t6

          ELSEIF ( iasite == 6 ) THEN

            !=== RF3:GDP Release ===!

            a(0) = a(0) + r_t7

          ELSEIF ( iasite == 7 ) THEN

            !=== RFx Release ===!

            a(0) = a(0) + r_t5(irna_a)

          ELSEIF ( iasite == 8 ) THEN

            !=== RF3:GDP Unbinding ===!

            a(0) = a(0) + r_t7

            !=== GDP Release ===!

            a(0) = a(0) + 35.0d0

          ENDIF

        ENDIF

        !=== STATE 5: 70S:Recycling ===!

        IF ( istate == 5 ) THEN

          IF ( iasite == 1 ) THEN

            !=== RRF Binding ===!

            a(id_rrf) = a(id_rrf) + r_r1f

          ELSEIF ( iasite == 2 ) THEN

            !=== Ef-G Binding ===!

            a(id_efg) = a(id_efg) + r_r2f

            !=== RRF Unbinding ===!

            a(0) = a(0) + r_r1b

          ELSEIF ( iasite == 3 ) THEN

            !=== Ef-G Unbinding ===!

            a(0) = a(0) + r_r2b

            !=== Ef-G GTP Hydrolosis ===!

            a(0) = a(0) + r_r3

          ELSEIF ( iasite == 4 ) THEN

            !=== Ribosomal Splitting ===!

            a(0) = a(0) + r_r4

          ENDIF

        ENDIF

        !=== STATE 6: 30S:Scanning ===!

        IF ( istate == 6 ) THEN

        ENDIF


        !=== RESUM Partial Sum Table ===!

        CALL RESUM (rna,indx,a)

        RETURN

      END SUBROUTINE RIBO_REAC
