! ==============================================================================
! Subroutine: TINT (FLAG)
! 
! Purpose: Sets up the parameter tables for internal loop (11,12,22)
!          energy terms (Default = TURNER 99 MODEL).
!
! Method: Uses the Turner 99 energy parameter table for RNA @ T=37.
!
!          stk_11(i,j,k)     stk_12(i,j,k,m)     stk_22(i,j,k,m)
!
!          1/1 Int Loop       1/2 Int Loop        2/2 Int Loop
!              i k j             i k  m j            i k  m j
!         5'-- A . G --3'   5'-- A .    G --3'  5'-- A .  . G --3'
!         3'-- U . C --5'   3'-- U .  . C --5'  3'-- U .  . C --5'
!
! Arguments:
!
!            FLAG - (LOGICAL) If  TRUE: Read in new parameters.
!                             if FALSE: Use Default parameters.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependencies:
!
! Modules - FOLDING, CONSTANTS
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE TINT (FLAG)

        USE Folding,   ONLY : stk_11,stk_12,stk_22
        USE Constants, ONLY : temp

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        LOGICAL, INTENT(IN) :: flag

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m,is

        DOUBLE PRECISION :: tf,s,h11(6,6,16),g11(6,6,16)
        DOUBLE PRECISION :: h12(6,6,16,4),g12(6,6,16,4)
        DOUBLE PRECISION :: h22(6,6,16,16),g22(6,6,16,16)


        tf = temp / 310.150d0

        !=== DEFAULT PARAMETERS ===!

        !=== DG TERMS ===!

        !=== 1/1 Int Loop ===!

        DATA (((g11(i,j,k),k=1,16),j=1,6),i=1,6) / &
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.50d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.00d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.20d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   0.40d0, -0.40d0,  0.40d0,  0.40d0,  0.30d0,  0.50d0,  0.40d0,  0.50d0,&
        &  -0.10d0,  0.40d0, -1.70d0,  0.40d0,  0.40d0,  0.00d0,  0.40d0, -0.30d0,&
        &   1.10d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,&
        &   0.40d0,  0.40d0, -1.40d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   0.80d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0,&
        &   0.40d0,  0.40d0, -2.10d0,  0.40d0,  0.40d0,  0.40d0,  0.40d0, -0.70d0,&
        &   0.40d0,  0.30d0, -0.10d0,  0.40d0, -0.40d0,  0.50d0,  0.40d0,  0.00d0,&
        &   0.40d0,  0.40d0, -1.70d0,  0.40d0,  0.40d0,  0.50d0,  0.40d0, -0.30d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.00d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.80d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.50d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.10d0,  1.10d0, -1.00d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,  1.10d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,&
        &   1.70d0,  1.70d0, -0.40d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0,  1.70d0 /

        !=== 1/2 Int Loop ===!

        DATA ((((g12(i,j,k,m),k=1,16),m=1,4),j=1,6),i=1,6) / &
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   2.30d0,  2.20d0,  1.10d0,  4.00d0,  2.30d0,  2.20d0,  4.00d0,  2.20d0,&
        &   1.70d0,  4.00d0,  0.80d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   2.10d0,  1.70d0,  1.60d0,  4.00d0,  2.20d0,  2.50d0,  4.00d0,  2.20d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  2.20d0,  4.00d0,  1.50d0,&
        &   0.80d0,  0.60d0,  0.40d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   0.80d0,  4.00d0,  2.20d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  2.50d0,  1.90d0,  4.00d0,  2.20d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  1.70d0,  4.00d0,  1.20d0,&
        &   2.40d0,  2.20d0,  1.60d0,  4.00d0,  2.30d0,  2.20d0,  4.00d0,  2.20d0,&
        &   1.70d0,  4.00d0,  0.80d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   2.10d0,  1.70d0,  1.60d0,  4.00d0,  2.20d0,  2.50d0,  4.00d0,  2.20d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  2.20d0,  4.00d0,  1.30d0,&
        &   1.00d0,  0.60d0,  0.40d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   0.80d0,  4.00d0,  2.20d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  2.50d0,  1.90d0,  4.00d0,  2.20d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  1.70d0,  4.00d0,  1.20d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   2.40d0,  2.20d0,  1.60d0,  4.00d0,  2.30d0,  2.20d0,  4.00d0,  2.20d0,&
        &   1.70d0,  4.00d0,  0.80d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   2.10d0,  1.70d0,  1.60d0,  4.00d0,  2.20d0,  2.50d0,  4.00d0,  2.20d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  2.20d0,  4.00d0,  1.30d0,&
        &   1.00d0,  0.60d0,  0.40d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   0.80d0,  4.00d0,  2.20d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  2.50d0,  1.90d0,  4.00d0,  2.20d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  1.70d0,  4.00d0,  1.20d0,&
        &   2.50d0,  2.20d0,  2.10d0,  4.00d0,  2.30d0,  2.20d0,  4.00d0,  2.20d0,&
        &   1.70d0,  4.00d0,  0.80d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   2.10d0,  1.70d0,  1.60d0,  4.00d0,  2.20d0,  2.50d0,  4.00d0,  2.20d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  2.20d0,  4.00d0,  1.20d0,&
        &   1.20d0,  0.60d0,  0.40d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   0.80d0,  4.00d0,  2.20d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  2.50d0,  1.90d0,  4.00d0,  2.20d0,&
        &   4.00d0,  4.00d0,  4.00d0,  4.00d0,  4.00d0,  1.70d0,  4.00d0,  1.20d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.20d0,  3.00d0,  2.40d0,  4.80d0,  3.10d0,  3.00d0,  4.80d0,  3.00d0,&
        &   2.50d0,  4.80d0,  1.60d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   2.90d0,  2.50d0,  2.40d0,  4.80d0,  3.00d0,  3.30d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.00d0,  4.80d0,  2.10d0,&
        &   1.80d0,  1.40d0,  1.20d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   1.60d0,  4.80d0,  3.00d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  3.30d0,  2.70d0,  4.80d0,  3.00d0,&
        &   4.80d0,  4.80d0,  4.80d0,  4.80d0,  4.80d0,  2.50d0,  4.80d0,  2.00d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0,&
        &   3.90d0,  3.70d0,  3.10d0,  5.50d0,  3.80d0,  3.70d0,  5.50d0,  3.70d0,&
        &   3.20d0,  5.50d0,  2.30d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   3.60d0,  3.20d0,  3.10d0,  5.50d0,  3.70d0,  4.00d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.70d0,  5.50d0,  2.80d0,&
        &   2.50d0,  2.10d0,  1.90d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   2.30d0,  5.50d0,  3.70d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  4.00d0,  3.40d0,  5.50d0,  3.70d0,&
        &   5.50d0,  5.50d0,  5.50d0,  5.50d0,  5.50d0,  3.20d0,  5.50d0,  2.70d0 /

        !=== 2/2 Int Loop ===!

        DATA ((((g22(i,j,k,m),m=1,16),k=1,16),j=1,6),i=1,6) / &
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  3.40d0,  2.00d0,  3.40d0,&
        &   1.70d0,  2.00d0,  2.10d0,  1.00d0,  2.00d0,  3.10d0,  2.20d0,  2.90d0,&
        &   2.60d0,  2.20d0,  1.60d0,  2.00d0,  2.60d0,  2.60d0,  2.00d0,  2.60d0,&
        &   1.60d0,  2.00d0,  2.00d0, -0.20d0,  2.00d0,  2.30d0,  1.10d0,  1.80d0,&
        &   1.50d0,  1.10d0,  0.50d0,  2.00d0,  1.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   0.50d0,  2.00d0,  0.90d0,  0.50d0,  2.00d0,  2.20d0,  1.80d0,  2.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  2.10d0,  1.50d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   1.50d0,  2.00d0,  1.90d0, -0.30d0,  2.00d0,  2.20d0,  1.00d0,  1.70d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.50d0,  1.10d0,  0.50d0,  2.00d0,  1.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   0.50d0,  2.00d0,  0.90d0,  0.50d0,  2.00d0,  2.20d0,  1.80d0,  2.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.60d0,  1.00d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.00d0,  2.00d0,  1.40d0,  0.30d0,  2.00d0,  2.40d0,  1.50d0,  2.20d0,&
        &   2.30d0,  0.90d0,  2.10d0,  2.00d0,  1.30d0,  2.30d0,  2.00d0,  2.30d0,&
        &   2.10d0,  2.00d0,  1.70d0, -1.50d0,  2.00d0,  2.00d0, -0.20d0,  2.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.30d0, -0.10d0,  1.10d0,  2.00d0,  0.30d0,  1.30d0,  2.00d0,  1.30d0,&
        &   1.10d0,  2.00d0,  0.70d0, -2.50d0,  2.00d0,  1.00d0, -1.20d0,  1.30d0,&
        &   2.70d0,  1.20d0,  2.40d0,  2.00d0,  1.70d0,  1.70d0,  2.00d0,  1.70d0,&
        &   2.40d0,  2.00d0,  2.00d0,  0.70d0,  2.00d0,  1.40d0,  1.90d0,  0.80d0,&
        &   2.10d0,  1.90d0,  0.10d0,  2.00d0,  1.80d0,  2.50d0,  2.00d0,  1.50d0,&
        &   0.70d0,  2.00d0,  1.80d0,  0.00d0,  2.00d0,  2.50d0,  0.40d0,  2.10d0,&
        &   2.00d0,  1.70d0,  0.00d0,  2.00d0,  1.70d0,  1.70d0,  2.00d0,  0.70d0,&
        &   0.60d0,  2.00d0,  1.60d0, -1.20d0,  2.00d0,  1.80d0, -0.80d0,  1.00d0,&
        &   0.90d0,  0.60d0, -1.10d0,  2.00d0,  0.60d0,  1.60d0,  2.00d0,  0.70d0,&
        &  -0.50d0,  2.00d0,  0.50d0, -0.50d0,  2.00d0,  1.70d0, -0.10d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.60d0, -0.10d0,  2.00d0,  1.60d0,  1.60d0,  2.00d0,  0.60d0,&
        &   0.50d0,  2.00d0,  1.50d0, -1.30d0,  2.00d0,  1.70d0, -0.90d0,  0.90d0,&
        &   2.40d0,  1.60d0,  0.80d0,  2.00d0,  1.50d0,  1.60d0,  2.00d0,  0.60d0,&
        &   1.40d0,  2.00d0,  2.10d0, -0.30d0,  2.00d0,  1.60d0,  0.10d0,  0.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.40d0,  1.60d0,  0.80d0,  2.00d0,  1.50d0,  1.60d0,  2.00d0,  0.60d0,&
        &   1.40d0,  2.00d0,  2.10d0, -0.30d0,  2.00d0,  1.60d0,  0.10d0,  0.80d0,&
        &   0.90d0,  0.60d0, -1.10d0,  2.00d0,  0.60d0,  1.60d0,  2.00d0,  0.70d0,&
        &  -0.50d0,  2.00d0,  0.50d0, -0.50d0,  2.00d0,  1.70d0, -0.10d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.40d0,  1.20d0, -0.60d0,  2.00d0,  1.10d0,  1.80d0,  2.00d0,  0.80d0,&
        &   0.00d0,  2.00d0,  1.10d0, -0.70d0,  2.00d0,  1.80d0, -0.30d0,  1.40d0,&
        &   1.70d0,  0.40d0,  0.50d0,  2.00d0,  0.40d0,  1.40d0,  2.00d0,  0.50d0,&
        &   1.10d0,  2.00d0,  1.30d0, -2.50d0,  2.00d0,  1.50d0, -2.10d0,  1.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.40d0,  1.60d0,  0.80d0,  2.00d0,  1.50d0,  1.60d0,  2.00d0,  0.60d0,&
        &   1.40d0,  2.00d0,  2.10d0, -0.30d0,  2.00d0,  1.60d0,  0.10d0,  0.80d0,&
        &   0.70d0, -0.50d0, -0.50d0,  2.00d0, -0.60d0,  0.50d0,  2.00d0, -0.50d0,&
        &   0.10d0,  2.00d0,  0.40d0, -3.50d0,  2.00d0,  0.50d0, -3.10d0,  0.50d0,&
        &   2.00d0,  0.80d0,  0.80d0,  2.00d0,  0.70d0,  0.80d0,  2.00d0, -0.20d0,&
        &   1.50d0,  2.00d0,  1.70d0, -0.30d0,  2.00d0,  0.80d0,  0.10d0,  0.00d0,&
        &   2.00d0,  1.90d0,  1.00d0,  2.00d0,  2.40d0,  2.80d0,  2.00d0,  2.70d0,&
        &   1.00d0,  2.00d0,  1.80d0,  0.30d0,  2.00d0,  2.70d0,  1.80d0,  2.20d0,&
        &   1.90d0,  1.80d0,  0.90d0,  2.00d0,  2.20d0,  2.10d0,  2.00d0,  1.90d0,&
        &   0.90d0,  2.00d0,  1.60d0, -0.80d0,  2.00d0,  1.90d0,  0.70d0,  1.00d0,&
        &   0.80d0,  0.70d0, -0.20d0,  2.00d0,  1.10d0,  2.00d0,  2.00d0,  1.80d0,&
        &  -0.20d0,  2.00d0,  0.50d0, -0.10d0,  2.00d0,  1.80d0,  1.40d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.70d0,  0.80d0,  2.00d0,  2.10d0,  2.00d0,  2.00d0,  1.80d0,&
        &   0.80d0,  2.00d0,  1.50d0, -0.90d0,  2.00d0,  1.80d0,  0.60d0,  0.90d0,&
        &   2.30d0,  1.60d0,  1.70d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  1.80d0,  1.50d0,  0.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.60d0,  1.70d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  1.80d0,  1.50d0,  0.90d0,&
        &   0.80d0,  0.70d0, -0.20d0,  2.00d0,  1.10d0,  2.00d0,  2.00d0,  1.80d0,&
        &  -0.20d0,  2.00d0,  0.50d0, -0.10d0,  2.00d0,  1.80d0,  1.40d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.30d0,  1.20d0,  0.30d0,  2.00d0,  1.70d0,  2.10d0,  2.00d0,  2.00d0,&
        &   0.30d0,  2.00d0,  1.10d0, -0.40d0,  2.00d0,  2.00d0,  1.10d0,  1.50d0,&
        &   1.60d0,  0.50d0,  1.40d0,  2.00d0,  0.90d0,  1.80d0,  2.00d0,  1.60d0,&
        &   1.40d0,  2.00d0,  1.30d0, -2.10d0,  2.00d0,  1.60d0, -0.60d0,  1.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.60d0,  1.70d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  1.80d0,  1.50d0,  0.90d0,&
        &   0.60d0, -0.50d0,  0.40d0,  2.00d0,  0.00d0,  0.80d0,  2.00d0,  0.70d0,&
        &   0.40d0,  2.00d0,  0.40d0, -3.10d0,  2.00d0,  0.70d0, -1.60d0,  0.60d0,&
        &   1.90d0,  0.80d0,  1.80d0,  2.00d0,  1.30d0,  1.10d0,  2.00d0,  1.00d0,&
        &   1.80d0,  2.00d0,  1.70d0,  0.00d0,  2.00d0,  1.00d0,  1.60d0,  0.10d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  3.40d0,  2.00d0,  3.40d0,&
        &   1.70d0,  2.00d0,  2.10d0,  1.00d0,  2.00d0,  3.10d0,  2.20d0,  2.90d0,&
        &   2.60d0,  2.20d0,  1.60d0,  2.00d0,  2.60d0,  2.60d0,  2.00d0,  2.60d0,&
        &   1.60d0,  2.00d0,  2.00d0, -0.20d0,  2.00d0,  2.30d0,  1.10d0,  1.80d0,&
        &   1.50d0,  1.10d0,  0.50d0,  2.00d0,  1.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   0.50d0,  2.00d0,  0.90d0,  0.50d0,  2.00d0,  2.20d0,  1.80d0,  2.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  2.10d0,  1.50d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   1.50d0,  2.00d0,  1.90d0, -0.30d0,  2.00d0,  2.20d0,  1.00d0,  1.70d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.50d0,  1.10d0,  0.50d0,  2.00d0,  1.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   0.50d0,  2.00d0,  0.90d0,  0.50d0,  2.00d0,  2.20d0,  1.80d0,  2.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.60d0,  1.00d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.00d0,  2.00d0,  1.40d0,  0.30d0,  2.00d0,  2.40d0,  1.50d0,  2.20d0,&
        &   2.30d0,  0.90d0,  2.10d0,  2.00d0,  1.30d0,  2.30d0,  2.00d0,  2.30d0,&
        &   2.10d0,  2.00d0,  1.70d0, -1.50d0,  2.00d0,  2.00d0, -0.20d0,  2.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.30d0, -0.10d0,  1.10d0,  2.00d0,  0.30d0,  1.30d0,  2.00d0,  1.30d0,&
        &   1.10d0,  2.00d0,  0.70d0, -2.50d0,  2.00d0,  1.00d0, -1.20d0,  1.30d0,&
        &   2.70d0,  1.20d0,  2.40d0,  2.00d0,  1.70d0,  1.70d0,  2.00d0,  1.70d0,&
        &   2.40d0,  2.00d0,  2.00d0,  0.70d0,  2.00d0,  1.40d0,  1.90d0,  0.80d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  3.10d0,  2.00d0,  3.10d0,&
        &   1.50d0,  2.00d0,  2.10d0,  1.30d0,  2.00d0,  3.10d0,  2.30d0,  2.70d0,&
        &   2.60d0,  2.40d0,  1.40d0,  2.00d0,  2.50d0,  2.30d0,  2.00d0,  2.30d0,&
        &   1.40d0,  2.00d0,  1.90d0,  0.20d0,  2.00d0,  2.30d0,  1.20d0,  1.50d0,&
        &   1.50d0,  1.30d0,  0.30d0,  2.00d0,  1.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   0.30d0,  2.00d0,  0.80d0,  0.90d0,  2.00d0,  2.20d0,  1.90d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  2.30d0,  1.30d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   1.30d0,  2.00d0,  1.80d0,  0.10d0,  2.00d0,  2.20d0,  1.10d0,  1.40d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.50d0,  1.30d0,  0.30d0,  2.00d0,  1.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   0.30d0,  2.00d0,  0.80d0,  0.90d0,  2.00d0,  2.20d0,  1.90d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.80d0,  0.80d0,  2.00d0,  1.90d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.80d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.60d0,  2.00d0,&
        &   2.30d0,  1.10d0,  1.90d0,  2.00d0,  1.20d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  2.00d0,  1.60d0, -1.10d0,  2.00d0,  2.00d0, -0.10d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.30d0,  0.10d0,  0.90d0,  2.00d0,  0.20d0,  1.00d0,  2.00d0,  1.00d0,&
        &   0.90d0,  2.00d0,  0.70d0, -2.10d0,  2.00d0,  1.00d0, -1.10d0,  1.10d0,&
        &   2.70d0,  1.40d0,  2.20d0,  2.00d0,  1.50d0,  1.40d0,  2.00d0,  1.40d0,&
        &   2.20d0,  2.00d0,  2.00d0,  1.10d0,  2.00d0,  1.40d0,  2.00d0,  0.60d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  3.10d0,  2.00d0,  3.10d0,&
        &   1.50d0,  2.00d0,  2.10d0,  1.30d0,  2.00d0,  3.10d0,  2.30d0,  2.70d0,&
        &   2.60d0,  2.40d0,  1.40d0,  2.00d0,  2.50d0,  2.30d0,  2.00d0,  2.30d0,&
        &   1.40d0,  2.00d0,  1.90d0,  0.20d0,  2.00d0,  2.30d0,  1.20d0,  1.50d0,&
        &   1.50d0,  1.30d0,  0.30d0,  2.00d0,  1.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   0.30d0,  2.00d0,  0.80d0,  0.90d0,  2.00d0,  2.20d0,  1.90d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  2.30d0,  1.30d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   1.30d0,  2.00d0,  1.80d0,  0.10d0,  2.00d0,  2.20d0,  1.10d0,  1.40d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.50d0,  1.30d0,  0.30d0,  2.00d0,  1.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   0.30d0,  2.00d0,  0.80d0,  0.90d0,  2.00d0,  2.20d0,  1.90d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.80d0,  0.80d0,  2.00d0,  1.90d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.80d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.60d0,  2.00d0,&
        &   2.30d0,  1.10d0,  1.90d0,  2.00d0,  1.20d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  2.00d0,  1.60d0, -1.10d0,  2.00d0,  2.00d0, -0.10d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.30d0,  0.10d0,  0.90d0,  2.00d0,  0.20d0,  1.00d0,  2.00d0,  1.00d0,&
        &   0.90d0,  2.00d0,  0.70d0, -2.10d0,  2.00d0,  1.00d0, -1.10d0,  1.10d0,&
        &   2.70d0,  1.40d0,  2.20d0,  2.00d0,  1.50d0,  1.40d0,  2.00d0,  1.40d0,&
        &   2.20d0,  2.00d0,  2.00d0,  1.10d0,  2.00d0,  1.40d0,  2.00d0,  0.60d0,&
        &   2.00d0,  1.60d0,  1.00d0,  2.00d0,  2.00d0,  2.60d0,  2.00d0,  2.60d0,&
        &   1.00d0,  2.00d0,  1.40d0,  0.20d0,  2.00d0,  2.30d0,  1.50d0,  2.20d0,&
        &   2.40d0,  1.90d0,  1.30d0,  2.00d0,  2.40d0,  2.40d0,  2.00d0,  2.40d0,&
        &   1.30d0,  2.00d0,  1.70d0, -0.40d0,  2.00d0,  2.10d0,  0.80d0,  1.50d0,&
        &   1.00d0,  0.60d0,  0.00d0,  2.00d0,  1.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  0.40d0,  0.00d0,  2.00d0,  1.70d0,  1.30d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.50d0,  0.90d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   0.90d0,  2.00d0,  1.30d0, -0.90d0,  2.00d0,  1.60d0,  0.40d0,  1.10d0,&
        &   2.80d0,  1.80d0,  2.20d0,  2.00d0,  2.20d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.20d0,  0.40d0,  2.00d0,  1.90d0,  1.70d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.60d0,  2.00d0,  2.00d0,  2.10d0,  2.10d0,  2.00d0,  2.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  0.30d0,  2.00d0,  1.80d0,  1.50d0,  1.20d0,&
        &   1.00d0,  0.60d0,  0.00d0,  2.00d0,  1.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  0.40d0,  0.00d0,  2.00d0,  1.70d0,  1.30d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.30d0,  0.70d0,  2.00d0,  1.80d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.00d0,  2.00d0,  2.10d0,  1.20d0,  1.90d0,&
        &   1.80d0,  0.40d0,  1.60d0,  2.00d0,  0.80d0,  1.80d0,  2.00d0,  1.80d0,&
        &   1.60d0,  2.00d0,  1.20d0, -2.00d0,  2.00d0,  1.50d0, -0.70d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.60d0,  2.00d0,  2.00d0,  2.10d0,  2.10d0,  2.00d0,  2.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  0.30d0,  2.00d0,  1.80d0,  1.50d0,  1.20d0,&
        &   0.30d0, -1.10d0,  0.10d0,  2.00d0, -0.70d0,  0.30d0,  2.00d0,  0.30d0,&
        &   0.10d0,  2.00d0, -0.30d0, -3.50d0,  2.00d0,  0.00d0, -2.20d0,  0.30d0,&
        &   2.20d0,  0.70d0,  1.90d0,  2.00d0,  1.20d0,  1.20d0,  2.00d0,  1.20d0,&
        &   1.90d0,  2.00d0,  1.50d0,  0.20d0,  2.00d0,  0.90d0,  1.50d0,  0.30d0,&
        &   0.50d0,  1.10d0, -0.30d0,  2.00d0,  1.10d0,  1.70d0,  2.00d0,  0.70d0,&
        &   0.40d0,  2.00d0,  1.00d0,  0.10d0,  2.00d0,  1.80d0, -0.50d0,  1.50d0,&
        &   0.60d0,  1.50d0,  0.10d0,  2.00d0,  1.10d0,  1.50d0,  2.00d0,  0.50d0,&
        &   0.50d0,  2.00d0,  1.40d0, -0.70d0,  2.00d0,  1.50d0, -0.60d0,  0.00d0,&
        &   0.00d0, -0.70d0, -1.60d0,  2.00d0, -1.00d0, -0.60d0,  2.00d0,  0.20d0,&
        &  -0.70d0,  2.00d0,  0.00d0, -0.80d0,  2.00d0,  1.20d0, -0.60d0,  0.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.30d0,  1.00d0, -0.70d0,  2.00d0,  1.00d0,  1.00d0,  2.00d0,  0.00d0,&
        &   0.70d0,  2.00d0,  0.90d0, -1.90d0,  2.00d0,  1.10d0, -1.50d0, -0.20d0,&
        &   2.20d0,  1.30d0,  0.70d0,  2.00d0,  1.90d0,  1.30d0,  2.00d0,  0.30d0,&
        &   0.70d0,  2.00d0,  1.80d0, -0.30d0,  2.00d0,  1.40d0, -0.20d0, -0.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  1.20d0,  0.40d0,  2.00d0,  1.10d0,  1.20d0,  2.00d0,  1.70d0,&
        &   1.00d0,  2.00d0,  1.70d0, -0.70d0,  2.00d0,  1.20d0, -0.30d0,  0.20d0,&
        &  -0.20d0, -0.40d0, -1.70d0,  2.00d0,  0.70d0,  1.10d0,  2.00d0,  0.20d0,&
        &  -0.50d0,  2.00d0,  0.00d0, -0.90d0,  2.00d0,  1.20d0, -1.30d0,  0.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.10d0,  0.90d0, -0.90d0,  2.00d0,  0.80d0,  1.50d0,  2.00d0,  0.50d0,&
        &  -0.20d0,  2.00d0,  0.80d0, -1.00d0,  2.00d0,  1.50d0, -0.60d0,  1.10d0,&
        &   0.90d0,  0.00d0,  0.30d0,  2.00d0, -0.10d0,  1.00d0,  2.00d0,  0.00d0,&
        &   0.60d0,  2.00d0,  0.90d0, -3.00d0,  2.00d0,  1.00d0, -2.40d0,  0.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  1.20d0,  0.40d0,  2.00d0,  1.10d0,  1.20d0,  2.00d0,  0.20d0,&
        &   0.50d0,  2.00d0,  1.70d0, -0.70d0,  2.00d0,  1.20d0, -0.10d0,  0.40d0,&
        &  -0.10d0, -1.60d0, -1.60d0,  2.00d0, -1.60d0, -0.60d0,  2.00d0, -1.60d0,&
        &  -0.60d0,  2.00d0, -0.70d0, -4.40d0,  2.00d0, -0.50d0, -4.10d0, -1.00d0,&
        &   1.40d0,  0.30d0,  0.50d0,  2.00d0,  0.30d0,  0.30d0,  2.00d0,  0.10d0,&
        &   1.40d0,  2.00d0,  1.20d0, -1.00d0,  2.00d0,  0.30d0,  0.10d0,  0.60d0,&
        &   1.30d0,  1.20d0,  0.30d0,  2.00d0,  1.60d0,  2.10d0,  2.00d0,  1.90d0,&
        &   0.30d0,  2.00d0,  1.00d0, -0.40d0,  2.00d0,  1.90d0,  1.10d0,  1.40d0,&
        &   1.60d0,  1.50d0,  0.60d0,  2.00d0,  2.00d0,  1.80d0,  2.00d0,  1.70d0,&
        &   0.60d0,  2.00d0,  1.40d0, -1.10d0,  2.00d0,  1.70d0,  0.40d0,  0.80d0,&
        &   0.30d0,  0.20d0, -0.70d0,  2.00d0,  0.60d0,  1.50d0,  2.00d0,  1.30d0,&
        &  -0.70d0,  2.00d0,  0.00d0, -0.60d0,  2.00d0,  1.30d0,  0.90d0,  1.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.20d0,  1.10d0,  0.20d0,  2.00d0,  1.50d0,  1.40d0,  2.00d0,  1.20d0,&
        &   0.20d0,  2.00d0,  0.90d0, -1.50d0,  2.00d0,  1.20d0,  0.00d0,  0.30d0,&
        &   2.10d0,  1.40d0,  1.50d0,  2.00d0,  1.80d0,  1.70d0,  2.00d0,  1.50d0,&
        &   1.50d0,  2.00d0,  1.80d0, -0.20d0,  2.00d0,  1.50d0,  1.30d0,  0.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.20d0,  1.30d0,  2.00d0,  1.70d0,  1.50d0,  2.00d0,  1.40d0,&
        &   1.30d0,  2.00d0,  1.70d0, -0.40d0,  2.00d0,  1.40d0,  1.10d0,  0.50d0,&
        &   0.30d0,  0.20d0, -0.70d0,  2.00d0,  0.60d0,  1.50d0,  2.00d0,  1.30d0,&
        &  -0.70d0,  2.00d0,  0.00d0, -0.60d0,  2.00d0,  1.30d0,  0.90d0,  1.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.00d0,  0.90d0,  0.00d0,  2.00d0,  1.40d0,  1.80d0,  2.00d0,  1.70d0,&
        &   0.00d0,  2.00d0,  0.80d0, -0.70d0,  2.00d0,  1.70d0,  0.90d0,  1.20d0,&
        &   1.10d0,  0.00d0,  0.90d0,  2.00d0,  0.40d0,  1.30d0,  2.00d0,  1.10d0,&
        &   0.90d0,  2.00d0,  0.90d0, -2.60d0,  2.00d0,  1.10d0, -1.10d0,  1.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.20d0,  1.30d0,  2.00d0,  1.70d0,  1.50d0,  2.00d0,  1.40d0,&
        &   1.30d0,  2.00d0,  1.70d0, -0.40d0,  2.00d0,  1.40d0,  1.10d0,  0.50d0,&
        &  -0.40d0, -1.50d0, -0.60d0,  2.00d0, -1.10d0, -0.20d0,  2.00d0, -0.40d0,&
        &  -0.60d0,  2.00d0, -0.70d0, -4.20d0,  2.00d0, -0.40d0, -2.60d0, -0.50d0,&
        &   1.40d0,  0.30d0,  1.30d0,  2.00d0,  0.80d0,  0.60d0,  2.00d0,  0.50d0,&
        &   1.30d0,  2.00d0,  1.20d0, -0.50d0,  2.00d0,  0.50d0,  1.10d0, -0.40d0,&
        &   2.00d0,  1.60d0,  1.00d0,  2.00d0,  2.00d0,  2.60d0,  2.00d0,  2.60d0,&
        &   1.00d0,  2.00d0,  1.40d0,  0.20d0,  2.00d0,  2.30d0,  1.50d0,  2.20d0,&
        &   2.40d0,  1.90d0,  1.30d0,  2.00d0,  2.40d0,  2.40d0,  2.00d0,  2.40d0,&
        &   1.30d0,  2.00d0,  1.70d0, -0.40d0,  2.00d0,  2.10d0,  0.80d0,  1.50d0,&
        &   1.00d0,  0.60d0,  0.00d0,  2.00d0,  1.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  0.40d0,  0.00d0,  2.00d0,  1.70d0,  1.30d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.50d0,  0.90d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   0.90d0,  2.00d0,  1.30d0, -0.90d0,  2.00d0,  1.60d0,  0.40d0,  1.10d0,&
        &   2.80d0,  1.80d0,  2.20d0,  2.00d0,  2.20d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.20d0,  0.40d0,  2.00d0,  1.90d0,  1.70d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.60d0,  2.00d0,  2.00d0,  2.10d0,  2.10d0,  2.00d0,  2.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  0.30d0,  2.00d0,  1.80d0,  1.50d0,  1.20d0,&
        &   1.00d0,  0.60d0,  0.00d0,  2.00d0,  1.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  0.40d0,  0.00d0,  2.00d0,  1.70d0,  1.30d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.30d0,  0.70d0,  2.00d0,  1.80d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.00d0,  2.00d0,  2.10d0,  1.20d0,  1.90d0,&
        &   1.80d0,  0.40d0,  1.60d0,  2.00d0,  0.80d0,  1.80d0,  2.00d0,  1.80d0,&
        &   1.60d0,  2.00d0,  1.20d0, -2.00d0,  2.00d0,  1.50d0, -0.70d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.60d0,  2.00d0,  2.00d0,  2.10d0,  2.10d0,  2.00d0,  2.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  0.30d0,  2.00d0,  1.80d0,  1.50d0,  1.20d0,&
        &   0.30d0, -1.10d0,  0.10d0,  2.00d0, -0.70d0,  0.30d0,  2.00d0,  0.30d0,&
        &   0.10d0,  2.00d0, -0.30d0, -3.50d0,  2.00d0,  0.00d0, -2.20d0,  0.30d0,&
        &   2.20d0,  0.70d0,  1.90d0,  2.00d0,  1.20d0,  1.20d0,  2.00d0,  1.20d0,&
        &   1.90d0,  2.00d0,  1.50d0,  0.20d0,  2.00d0,  0.90d0,  1.50d0,  0.30d0,&
        &   2.00d0,  1.80d0,  0.80d0,  2.00d0,  1.90d0,  2.30d0,  2.00d0,  2.30d0,&
        &   0.80d0,  2.00d0,  1.30d0,  0.60d0,  2.00d0,  2.30d0,  1.60d0,  1.90d0,&
        &   2.40d0,  2.10d0,  1.10d0,  2.00d0,  2.20d0,  2.10d0,  2.00d0,  2.10d0,&
        &   1.10d0,  2.00d0,  1.70d0,  0.00d0,  2.00d0,  2.10d0,  0.90d0,  1.30d0,&
        &   1.00d0,  0.80d0, -0.20d0,  2.00d0,  0.90d0,  1.70d0,  2.00d0,  1.70d0,&
        &  -0.20d0,  2.00d0,  0.30d0,  0.40d0,  2.00d0,  1.70d0,  1.40d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.70d0,  0.70d0,  2.00d0,  1.80d0,  1.60d0,  2.00d0,  1.60d0,&
        &   0.70d0,  2.00d0,  1.20d0, -0.50d0,  2.00d0,  1.60d0,  0.50d0,  0.80d0,&
        &   2.80d0,  2.00d0,  2.00d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.10d0,  0.80d0,  2.00d0,  1.90d0,  1.80d0,  1.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.80d0,  1.80d0,  2.00d0,  1.90d0,  1.80d0,  2.00d0,  1.80d0,&
        &   1.80d0,  2.00d0,  2.00d0,  0.70d0,  2.00d0,  1.80d0,  1.60d0,  1.00d0,&
        &   1.00d0,  0.80d0, -0.20d0,  2.00d0,  0.90d0,  1.70d0,  2.00d0,  1.70d0,&
        &  -0.20d0,  2.00d0,  0.30d0,  0.40d0,  2.00d0,  1.70d0,  1.40d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.10d0,  2.00d0,  2.10d0,&
        &   0.50d0,  2.00d0,  1.10d0,  0.40d0,  2.00d0,  2.10d0,  1.30d0,  1.70d0,&
        &   1.80d0,  0.60d0,  1.40d0,  2.00d0,  0.70d0,  1.50d0,  2.00d0,  1.50d0,&
        &   1.40d0,  2.00d0,  1.10d0, -1.60d0,  2.00d0,  1.50d0, -0.60d0,  1.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.80d0,  1.80d0,  2.00d0,  1.90d0,  1.80d0,  2.00d0,  1.80d0,&
        &   1.80d0,  2.00d0,  2.00d0,  0.70d0,  2.00d0,  1.80d0,  1.60d0,  1.00d0,&
        &   0.30d0, -0.90d0, -0.10d0,  2.00d0, -0.80d0,  0.00d0,  2.00d0,  0.00d0,&
        &  -0.10d0,  2.00d0, -0.40d0, -3.10d0,  2.00d0,  0.00d0, -2.10d0,  0.00d0,&
        &   2.20d0,  0.90d0,  1.80d0,  2.00d0,  1.00d0,  0.90d0,  2.00d0,  0.90d0,&
        &   1.80d0,  2.00d0,  1.50d0,  0.60d0,  2.00d0,  0.90d0,  1.60d0,  0.10d0,&
        &   2.00d0,  1.80d0,  0.80d0,  2.00d0,  1.90d0,  2.30d0,  2.00d0,  2.30d0,&
        &   0.80d0,  2.00d0,  1.30d0,  0.60d0,  2.00d0,  2.30d0,  1.60d0,  1.90d0,&
        &   2.40d0,  2.10d0,  1.10d0,  2.00d0,  2.20d0,  2.10d0,  2.00d0,  2.10d0,&
        &   1.10d0,  2.00d0,  1.70d0,  0.00d0,  2.00d0,  2.10d0,  0.90d0,  1.30d0,&
        &   1.00d0,  0.80d0, -0.20d0,  2.00d0,  0.90d0,  1.70d0,  2.00d0,  1.70d0,&
        &  -0.20d0,  2.00d0,  0.30d0,  0.40d0,  2.00d0,  1.70d0,  1.40d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.70d0,  0.70d0,  2.00d0,  1.80d0,  1.60d0,  2.00d0,  1.60d0,&
        &   0.70d0,  2.00d0,  1.20d0, -0.50d0,  2.00d0,  1.60d0,  0.50d0,  0.80d0,&
        &   2.80d0,  2.00d0,  2.00d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.10d0,  0.80d0,  2.00d0,  1.90d0,  1.80d0,  1.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.80d0,  1.80d0,  2.00d0,  1.90d0,  1.80d0,  2.00d0,  1.80d0,&
        &   1.80d0,  2.00d0,  2.00d0,  0.70d0,  2.00d0,  1.80d0,  1.60d0,  1.00d0,&
        &   1.00d0,  0.80d0, -0.20d0,  2.00d0,  0.90d0,  1.70d0,  2.00d0,  1.70d0,&
        &  -0.20d0,  2.00d0,  0.30d0,  0.40d0,  2.00d0,  1.70d0,  1.40d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.10d0,  2.00d0,  2.10d0,&
        &   0.50d0,  2.00d0,  1.10d0,  0.40d0,  2.00d0,  2.10d0,  1.30d0,  1.70d0,&
        &   1.80d0,  0.60d0,  1.40d0,  2.00d0,  0.70d0,  1.50d0,  2.00d0,  1.50d0,&
        &   1.40d0,  2.00d0,  1.10d0, -1.60d0,  2.00d0,  1.50d0, -0.60d0,  1.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.80d0,  1.80d0,  2.00d0,  1.90d0,  1.80d0,  2.00d0,  1.80d0,&
        &   1.80d0,  2.00d0,  2.00d0,  0.70d0,  2.00d0,  1.80d0,  1.60d0,  1.00d0,&
        &   0.30d0, -0.90d0, -0.10d0,  2.00d0, -0.80d0,  0.00d0,  2.00d0,  0.00d0,&
        &  -0.10d0,  2.00d0, -0.40d0, -3.10d0,  2.00d0,  0.00d0, -2.10d0,  0.00d0,&
        &   2.20d0,  0.90d0,  1.80d0,  2.00d0,  1.00d0,  0.90d0,  2.00d0,  0.90d0,&
        &   1.80d0,  2.00d0,  1.50d0,  0.60d0,  2.00d0,  0.90d0,  1.60d0,  0.10d0,&
        &   2.10d0,  1.70d0,  1.10d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.10d0,  2.00d0,  1.50d0,  0.30d0,  2.00d0,  2.40d0,  1.60d0,  2.30d0,&
        &   1.80d0,  1.40d0,  0.80d0,  2.00d0,  1.80d0,  1.80d0,  2.00d0,  1.80d0,&
        &   0.80d0,  2.00d0,  1.20d0, -1.00d0,  2.00d0,  1.50d0,  0.30d0,  1.00d0,&
        &   0.70d0,  0.30d0, -0.30d0,  2.00d0,  0.70d0,  1.70d0,  2.00d0,  1.70d0,&
        &  -0.30d0,  2.00d0,  0.10d0, -0.30d0,  2.00d0,  1.40d0,  1.00d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.40d0,  0.80d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   0.80d0,  2.00d0,  1.20d0, -0.90d0,  2.00d0,  1.60d0,  0.30d0,  1.00d0,&
        &   2.50d0,  1.40d0,  1.80d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   1.80d0,  2.00d0,  1.80d0,  0.10d0,  2.00d0,  1.60d0,  1.30d0,  1.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  1.50d0,  1.90d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   1.90d0,  2.00d0,  1.90d0,  0.10d0,  2.00d0,  1.60d0,  1.40d0,  1.10d0,&
        &   0.10d0, -0.30d0, -0.90d0,  2.00d0,  0.10d0,  1.10d0,  2.00d0,  1.10d0,&
        &  -0.90d0,  2.00d0, -0.50d0, -0.90d0,  2.00d0,  0.80d0,  0.40d0,  1.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.30d0,  0.70d0,  2.00d0,  1.80d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.00d0,  2.00d0,  2.10d0,  1.20d0,  1.90d0,&
        &   0.40d0, -1.10d0,  0.10d0,  2.00d0, -0.60d0,  0.40d0,  2.00d0,  0.40d0,&
        &   0.10d0,  2.00d0, -0.30d0, -3.50d0,  2.00d0,  0.10d0, -2.20d0,  0.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.50d0,  0.40d0,  0.90d0,  2.00d0,  0.90d0,  0.90d0,  2.00d0,  0.90d0,&
        &   0.90d0,  2.00d0,  0.80d0, -0.90d0,  2.00d0,  0.60d0,  0.40d0,  0.00d0,&
        &   0.00d0, -1.50d0, -0.30d0,  2.00d0, -1.00d0,  0.00d0,  2.00d0,  0.00d0,&
        &  -0.30d0,  2.00d0, -0.70d0, -3.90d0,  2.00d0, -0.30d0, -2.60d0, -0.10d0,&
        &   2.10d0,  0.70d0,  1.90d0,  2.00d0,  1.10d0,  1.10d0,  2.00d0,  1.10d0,&
        &   1.90d0,  2.00d0,  1.50d0,  0.10d0,  2.00d0,  0.80d0,  1.40d0,  0.30d0,&
        &   1.50d0,  1.20d0, -0.50d0,  2.00d0,  1.20d0,  1.80d0,  2.00d0,  0.80d0,&
        &   0.10d0,  2.00d0,  1.10d0, -0.70d0,  2.00d0,  1.90d0, -0.30d0,  1.50d0,&
        &   1.20d0,  0.90d0, -0.80d0,  2.00d0,  0.90d0,  0.90d0,  2.00d0,  0.00d0,&
        &  -0.20d0,  2.00d0,  0.80d0, -2.00d0,  2.00d0,  1.00d0, -1.60d0,  0.20d0,&
        &   0.10d0, -0.10d0, -1.90d0,  2.00d0, -0.20d0,  0.90d0,  2.00d0, -0.10d0,&
        &  -1.30d0,  2.00d0, -0.20d0, -1.30d0,  2.00d0,  0.90d0, -0.90d0,  0.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.20d0,  1.00d0, -0.80d0,  2.00d0,  0.90d0,  1.00d0,  2.00d0,  0.00d0,&
        &  -0.10d0,  2.00d0,  0.90d0, -1.90d0,  2.00d0,  1.00d0, -1.50d0,  0.20d0,&
        &   1.80d0,  1.00d0,  0.20d0,  2.00d0,  0.90d0,  1.00d0,  2.00d0,  0.00d0,&
        &   0.90d0,  2.00d0,  1.50d0, -0.90d0,  2.00d0,  1.00d0, -0.50d0,  0.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.00d0,  0.30d0,  2.00d0,  1.00d0,  1.00d0,  2.00d0,  0.00d0,&
        &   0.90d0,  2.00d0,  1.50d0, -0.90d0,  2.00d0,  1.10d0, -0.50d0,  0.30d0,&
        &  -0.50d0, -0.80d0, -2.60d0,  2.00d0, -0.80d0,  0.20d0,  2.00d0, -0.80d0,&
        &  -1.90d0,  2.00d0, -0.90d0, -1.90d0,  2.00d0,  0.30d0, -1.50d0,  0.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.10d0,  0.90d0, -0.90d0,  2.00d0,  0.80d0,  1.50d0,  2.00d0,  0.50d0,&
        &  -0.20d0,  2.00d0,  0.80d0, -1.00d0,  2.00d0,  1.50d0, -0.60d0,  1.10d0,&
        &  -0.30d0, -1.50d0, -1.50d0,  2.00d0, -1.60d0, -0.50d0,  2.00d0, -1.50d0,&
        &  -0.90d0,  2.00d0, -0.60d0, -4.50d0,  2.00d0, -0.50d0, -4.10d0, -0.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   0.80d0,  0.00d0, -0.80d0,  2.00d0,  0.00d0,  0.00d0,  2.00d0, -1.00d0,&
        &  -0.10d0,  2.00d0,  0.50d0, -1.90d0,  2.00d0,  0.00d0, -1.50d0, -0.70d0,&
        &  -0.70d0, -1.90d0, -1.90d0,  2.00d0, -2.00d0, -0.90d0,  2.00d0, -1.90d0,&
        &  -1.30d0,  2.00d0, -1.00d0, -4.90d0,  2.00d0, -0.90d0, -4.50d0, -0.90d0,&
        &   1.50d0,  0.20d0,  0.30d0,  2.00d0,  0.20d0,  0.20d0,  2.00d0, -0.70d0,&
        &   0.90d0,  2.00d0,  1.10d0, -0.90d0,  2.00d0,  0.30d0, -0.50d0, -0.50d0,&
        &   0.50d0,  1.30d0, -0.20d0,  2.00d0,  0.60d0,  2.20d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  1.10d0, -0.10d0,  2.00d0,  2.00d0,  0.90d0,  1.40d0,&
        &   1.10d0,  1.00d0,  0.70d0,  2.00d0,  1.10d0,  1.90d0,  2.00d0,  1.10d0,&
        &  -1.00d0,  2.00d0,  0.80d0, -1.60d0,  2.00d0,  1.10d0, -0.10d0,  0.30d0,&
        &   0.40d0,  0.70d0, -0.50d0,  2.00d0,  0.50d0,  0.70d0,  2.00d0,  0.50d0,&
        &  -0.70d0,  2.00d0, -0.20d0, -0.60d0,  2.00d0,  1.00d0,  0.60d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.10d0,  1.00d0, -0.40d0,  2.00d0,  1.50d0,  1.30d0,  2.00d0,  1.20d0,&
        &  -0.70d0,  2.00d0,  0.90d0, -1.60d0,  2.00d0,  1.20d0,  0.00d0,  0.30d0,&
        &   1.70d0,  1.00d0,  1.10d0,  2.00d0,  1.50d0,  1.30d0,  2.00d0,  1.20d0,&
        &  -0.60d0,  2.00d0,  1.50d0, -0.60d0,  2.00d0,  1.20d0,  1.00d0,  0.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.10d0,  1.20d0,  2.00d0,  1.50d0,  1.40d0,  2.00d0,  1.20d0,&
        &   1.20d0,  2.00d0,  1.50d0, -0.50d0,  2.00d0,  1.20d0,  1.00d0,  0.30d0,&
        &  -0.30d0, -0.70d0, -1.70d0,  2.00d0,  0.10d0,  0.70d0,  2.00d0,  0.40d0,&
        &  -1.60d0,  2.00d0, -0.90d0, -1.60d0,  2.00d0,  0.40d0,  0.30d0,  0.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.00d0,  0.90d0,  0.00d0,  2.00d0,  1.40d0,  1.80d0,  2.00d0,  1.70d0,&
        &   0.00d0,  2.00d0,  0.80d0, -0.70d0,  2.00d0,  1.70d0,  0.90d0,  1.20d0,&
        &  -0.50d0, -1.50d0, -1.30d0,  2.00d0, -0.60d0, -0.20d0,  2.00d0, -0.10d0,&
        &  -0.60d0,  2.00d0, -0.60d0, -4.10d0,  2.00d0, -0.30d0, -2.40d0,  0.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   0.70d0,  0.00d0,  0.20d0,  2.00d0,  0.50d0,  0.30d0,  2.00d0,  0.20d0,&
        &   0.20d0,  2.00d0,  0.50d0, -1.60d0,  2.00d0,  1.70d0,  0.00d0,  0.10d0,&
        &   0.10d0, -1.90d0, -0.90d0,  2.00d0, -0.70d0, -0.30d0,  2.00d0, -0.70d0,&
        &  -0.80d0,  2.00d0, -1.00d0, -4.40d0,  2.00d0, -0.70d0, -3.00d0, -1.00d0,&
        &   1.50d0, -0.20d0,  0.90d0,  2.00d0,  0.00d0, -0.10d0,  2.00d0,  0.40d0,&
        &   0.90d0,  2.00d0,  1.10d0, -1.00d0,  2.00d0,  0.20d0,  0.60d0,  0.60d0,&
        &   2.10d0,  1.70d0,  1.10d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.10d0,  2.00d0,  1.50d0,  0.30d0,  2.00d0,  2.40d0,  1.60d0,  2.30d0,&
        &   1.80d0,  1.40d0,  0.80d0,  2.00d0,  1.80d0,  1.80d0,  2.00d0,  1.80d0,&
        &   0.80d0,  2.00d0,  1.20d0, -1.00d0,  2.00d0,  1.50d0,  0.30d0,  1.00d0,&
        &   0.70d0,  0.30d0, -0.30d0,  2.00d0,  0.70d0,  1.70d0,  2.00d0,  1.70d0,&
        &  -0.30d0,  2.00d0,  0.10d0, -0.30d0,  2.00d0,  1.40d0,  1.00d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.40d0,  0.80d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   0.80d0,  2.00d0,  1.20d0, -0.90d0,  2.00d0,  1.60d0,  0.30d0,  1.00d0,&
        &   2.50d0,  1.40d0,  1.80d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   1.80d0,  2.00d0,  1.80d0,  0.10d0,  2.00d0,  1.60d0,  1.30d0,  1.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  1.50d0,  1.90d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   1.90d0,  2.00d0,  1.90d0,  0.10d0,  2.00d0,  1.60d0,  1.40d0,  1.10d0,&
        &   0.10d0, -0.30d0, -0.90d0,  2.00d0,  0.10d0,  1.10d0,  2.00d0,  1.10d0,&
        &  -0.90d0,  2.00d0, -0.50d0, -0.90d0,  2.00d0,  0.80d0,  0.40d0,  1.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.30d0,  0.70d0,  2.00d0,  1.80d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.00d0,  2.00d0,  2.10d0,  1.20d0,  1.90d0,&
        &   0.40d0, -1.10d0,  0.10d0,  2.00d0, -0.60d0,  0.40d0,  2.00d0,  0.40d0,&
        &   0.10d0,  2.00d0, -0.30d0, -3.50d0,  2.00d0,  0.10d0, -2.20d0,  0.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.50d0,  0.40d0,  0.90d0,  2.00d0,  0.90d0,  0.90d0,  2.00d0,  0.90d0,&
        &   0.90d0,  2.00d0,  0.80d0, -0.90d0,  2.00d0,  0.60d0,  0.40d0,  0.00d0,&
        &   0.00d0, -1.50d0, -0.30d0,  2.00d0, -1.00d0,  0.00d0,  2.00d0,  0.00d0,&
        &  -0.30d0,  2.00d0, -0.70d0, -3.90d0,  2.00d0, -0.30d0, -2.60d0, -0.10d0,&
        &   2.10d0,  0.70d0,  1.90d0,  2.00d0,  1.10d0,  1.10d0,  2.00d0,  1.10d0,&
        &   1.90d0,  2.00d0,  1.50d0,  0.10d0,  2.00d0,  0.80d0,  1.40d0,  0.30d0,&
        &   2.10d0,  1.90d0,  0.90d0,  2.00d0,  2.00d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.90d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.70d0,  2.00d0,&
        &   1.80d0,  1.60d0,  0.60d0,  2.00d0,  1.70d0,  1.50d0,  2.00d0,  1.50d0,&
        &   0.60d0,  2.00d0,  1.10d0, -0.60d0,  2.00d0,  1.50d0,  0.40d0,  0.70d0,&
        &   0.70d0,  0.50d0, -0.50d0,  2.00d0,  0.60d0,  1.40d0,  2.00d0,  1.40d0,&
        &  -0.50d0,  2.00d0,  0.00d0,  0.10d0,  2.00d0,  1.40d0,  1.10d0,  1.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.60d0,  0.60d0,  2.00d0,  1.70d0,  1.60d0,  2.00d0,  1.60d0,&
        &   0.60d0,  2.00d0,  1.20d0, -0.50d0,  2.00d0,  1.60d0,  0.40d0,  0.80d0,&
        &   2.50d0,  1.60d0,  1.60d0,  2.00d0,  1.70d0,  1.60d0,  2.00d0,  1.60d0,&
        &   1.60d0,  2.00d0,  1.80d0,  0.50d0,  2.00d0,  1.60d0,  1.40d0,  0.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  1.70d0,  1.70d0,  2.00d0,  1.80d0,  1.60d0,  2.00d0,  1.60d0,&
        &   1.70d0,  2.00d0,  1.80d0,  0.50d0,  2.00d0,  1.60d0,  1.50d0,  0.80d0,&
        &   0.10d0, -0.10d0, -1.10d0,  2.00d0,  0.00d0,  0.80d0,  2.00d0,  0.80d0,&
        &  -1.10d0,  2.00d0, -0.60d0, -0.50d0,  2.00d0,  0.80d0,  0.50d0,  0.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.10d0,  2.00d0,  2.10d0,&
        &   0.50d0,  2.00d0,  1.10d0,  0.40d0,  2.00d0,  2.10d0,  1.30d0,  1.70d0,&
        &   0.40d0, -0.90d0, -0.10d0,  2.00d0, -0.80d0,  0.10d0,  2.00d0,  0.10d0,&
        &  -0.10d0,  2.00d0, -0.30d0, -3.10d0,  2.00d0,  0.10d0, -2.10d0,  0.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.50d0,  0.60d0,  0.70d0,  2.00d0,  0.70d0,  0.60d0,  2.00d0,  0.60d0,&
        &   0.70d0,  2.00d0,  0.80d0, -0.50d0,  2.00d0,  0.60d0,  0.50d0, -0.20d0,&
        &   0.00d0, -1.30d0, -0.50d0,  2.00d0, -1.20d0, -0.30d0,  2.00d0, -0.30d0,&
        &  -0.50d0,  2.00d0, -0.70d0, -3.50d0,  2.00d0, -0.30d0, -2.50d0, -0.30d0,&
        &   2.10d0,  0.90d0,  1.70d0,  2.00d0,  1.00d0,  0.80d0,  2.00d0,  0.80d0,&
        &   1.70d0,  2.00d0,  1.40d0,  0.50d0,  2.00d0,  0.80d0,  1.50d0,  0.00d0,&
        &   2.10d0,  1.90d0,  0.90d0,  2.00d0,  2.00d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.90d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.70d0,  2.00d0,&
        &   1.80d0,  1.60d0,  0.60d0,  2.00d0,  1.70d0,  1.50d0,  2.00d0,  1.50d0,&
        &   0.60d0,  2.00d0,  1.10d0, -0.60d0,  2.00d0,  1.50d0,  0.40d0,  0.70d0,&
        &   0.70d0,  0.50d0, -0.50d0,  2.00d0,  0.60d0,  1.40d0,  2.00d0,  1.40d0,&
        &  -0.50d0,  2.00d0,  0.00d0,  0.10d0,  2.00d0,  1.40d0,  1.10d0,  1.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.60d0,  0.60d0,  2.00d0,  1.70d0,  1.60d0,  2.00d0,  1.60d0,&
        &   0.60d0,  2.00d0,  1.20d0, -0.50d0,  2.00d0,  1.60d0,  0.40d0,  0.80d0,&
        &   2.50d0,  1.60d0,  1.60d0,  2.00d0,  1.70d0,  1.60d0,  2.00d0,  1.60d0,&
        &   1.60d0,  2.00d0,  1.80d0,  0.50d0,  2.00d0,  1.60d0,  1.40d0,  0.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  1.70d0,  1.70d0,  2.00d0,  1.80d0,  1.60d0,  2.00d0,  1.60d0,&
        &   1.70d0,  2.00d0,  1.80d0,  0.50d0,  2.00d0,  1.60d0,  1.50d0,  0.80d0,&
        &   0.10d0, -0.10d0, -1.10d0,  2.00d0,  0.00d0,  0.80d0,  2.00d0,  0.80d0,&
        &  -1.10d0,  2.00d0, -0.60d0, -0.50d0,  2.00d0,  0.80d0,  0.50d0,  0.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.10d0,  2.00d0,  2.10d0,&
        &   0.50d0,  2.00d0,  1.10d0,  0.40d0,  2.00d0,  2.10d0,  1.30d0,  1.70d0,&
        &   0.40d0, -0.90d0, -0.10d0,  2.00d0, -0.80d0,  0.10d0,  2.00d0,  0.10d0,&
        &  -0.10d0,  2.00d0, -0.30d0, -3.10d0,  2.00d0,  0.10d0, -2.10d0,  0.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.50d0,  0.60d0,  0.70d0,  2.00d0,  0.70d0,  0.60d0,  2.00d0,  0.60d0,&
        &   0.70d0,  2.00d0,  0.80d0, -0.50d0,  2.00d0,  0.60d0,  0.50d0, -0.20d0,&
        &   0.00d0, -1.30d0, -0.50d0,  2.00d0, -1.20d0, -0.30d0,  2.00d0, -0.30d0,&
        &  -0.50d0,  2.00d0, -0.70d0, -3.50d0,  2.00d0, -0.30d0, -2.50d0, -0.30d0,&
        &   2.10d0,  0.90d0,  1.70d0,  2.00d0,  1.00d0,  0.80d0,  2.00d0,  0.80d0,&
        &   1.70d0,  2.00d0,  1.40d0,  0.50d0,  2.00d0,  0.80d0,  1.50d0,  0.00d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  3.40d0,  2.00d0,  3.40d0,&
        &   1.70d0,  2.00d0,  2.10d0,  1.00d0,  2.00d0,  3.10d0,  2.20d0,  2.90d0,&
        &   2.60d0,  2.20d0,  1.60d0,  2.00d0,  2.60d0,  2.60d0,  2.00d0,  2.60d0,&
        &   1.60d0,  2.00d0,  2.00d0, -0.20d0,  2.00d0,  2.30d0,  1.10d0,  1.80d0,&
        &   1.50d0,  1.10d0,  0.50d0,  2.00d0,  1.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   0.50d0,  2.00d0,  0.90d0,  0.50d0,  2.00d0,  2.20d0,  1.80d0,  2.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  2.10d0,  1.50d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   1.50d0,  2.00d0,  1.90d0, -0.30d0,  2.00d0,  2.20d0,  1.00d0,  1.70d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.50d0,  1.10d0,  0.50d0,  2.00d0,  1.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   0.50d0,  2.00d0,  0.90d0,  0.50d0,  2.00d0,  2.20d0,  1.80d0,  2.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.60d0,  1.00d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.00d0,  2.00d0,  1.40d0,  0.30d0,  2.00d0,  2.40d0,  1.50d0,  2.20d0,&
        &   2.30d0,  0.90d0,  2.10d0,  2.00d0,  1.30d0,  2.30d0,  2.00d0,  2.30d0,&
        &   2.10d0,  2.00d0,  1.70d0, -1.50d0,  2.00d0,  2.00d0, -0.20d0,  2.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.30d0, -0.10d0,  1.10d0,  2.00d0,  0.30d0,  1.30d0,  2.00d0,  1.30d0,&
        &   1.10d0,  2.00d0,  0.70d0, -2.50d0,  2.00d0,  1.00d0, -1.20d0,  1.30d0,&
        &   2.70d0,  1.20d0,  2.40d0,  2.00d0,  1.70d0,  1.70d0,  2.00d0,  1.70d0,&
        &   2.40d0,  2.00d0,  2.00d0,  0.70d0,  2.00d0,  1.40d0,  1.90d0,  0.80d0,&
        &   2.10d0,  1.90d0,  0.10d0,  2.00d0,  1.80d0,  2.50d0,  2.00d0,  1.50d0,&
        &   0.70d0,  2.00d0,  1.80d0,  0.00d0,  2.00d0,  2.50d0,  0.40d0,  2.10d0,&
        &   2.00d0,  1.70d0,  0.00d0,  2.00d0,  1.70d0,  1.70d0,  2.00d0,  0.70d0,&
        &   0.60d0,  2.00d0,  1.60d0, -1.20d0,  2.00d0,  1.80d0, -0.80d0,  1.00d0,&
        &   0.90d0,  0.60d0, -1.10d0,  2.00d0,  0.60d0,  1.60d0,  2.00d0,  0.70d0,&
        &  -0.50d0,  2.00d0,  0.50d0, -0.50d0,  2.00d0,  1.70d0, -0.10d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  1.60d0, -0.10d0,  2.00d0,  1.60d0,  1.60d0,  2.00d0,  0.60d0,&
        &   0.50d0,  2.00d0,  1.50d0, -1.30d0,  2.00d0,  1.70d0, -0.90d0,  0.90d0,&
        &   2.40d0,  1.60d0,  0.80d0,  2.00d0,  1.50d0,  1.60d0,  2.00d0,  0.60d0,&
        &   1.40d0,  2.00d0,  2.10d0, -0.30d0,  2.00d0,  1.60d0,  0.10d0,  0.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.40d0,  1.60d0,  0.80d0,  2.00d0,  1.50d0,  1.60d0,  2.00d0,  0.60d0,&
        &   1.40d0,  2.00d0,  2.10d0, -0.30d0,  2.00d0,  1.60d0,  0.10d0,  0.80d0,&
        &   0.90d0,  0.60d0, -1.10d0,  2.00d0,  0.60d0,  1.60d0,  2.00d0,  0.70d0,&
        &  -0.50d0,  2.00d0,  0.50d0, -0.50d0,  2.00d0,  1.70d0, -0.10d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.40d0,  1.20d0, -0.60d0,  2.00d0,  1.10d0,  1.80d0,  2.00d0,  0.80d0,&
        &   0.00d0,  2.00d0,  1.10d0, -0.70d0,  2.00d0,  1.80d0, -0.30d0,  1.40d0,&
        &   1.70d0,  0.40d0,  0.50d0,  2.00d0,  0.40d0,  1.40d0,  2.00d0,  0.50d0,&
        &   1.10d0,  2.00d0,  1.30d0, -2.50d0,  2.00d0,  1.50d0, -2.10d0,  1.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.40d0,  1.60d0,  0.80d0,  2.00d0,  1.50d0,  1.60d0,  2.00d0,  0.60d0,&
        &   1.40d0,  2.00d0,  2.10d0, -0.30d0,  2.00d0,  1.60d0,  0.10d0,  0.80d0,&
        &   0.70d0, -0.50d0, -0.50d0,  2.00d0, -0.60d0,  0.50d0,  2.00d0, -0.50d0,&
        &   0.10d0,  2.00d0,  0.40d0, -3.50d0,  2.00d0,  0.50d0, -3.10d0,  0.50d0,&
        &   2.00d0,  0.80d0,  0.80d0,  2.00d0,  0.70d0,  0.80d0,  2.00d0, -0.20d0,&
        &   1.50d0,  2.00d0,  1.70d0, -0.30d0,  2.00d0,  0.80d0,  0.10d0,  0.00d0,&
        &   2.00d0,  1.90d0,  1.00d0,  2.00d0,  2.40d0,  2.80d0,  2.00d0,  2.70d0,&
        &   1.00d0,  2.00d0,  1.80d0,  0.30d0,  2.00d0,  2.70d0,  1.80d0,  2.20d0,&
        &   1.90d0,  1.80d0,  0.90d0,  2.00d0,  2.20d0,  2.10d0,  2.00d0,  1.90d0,&
        &   0.90d0,  2.00d0,  1.60d0, -0.80d0,  2.00d0,  1.90d0,  0.70d0,  1.00d0,&
        &   0.80d0,  0.70d0, -0.20d0,  2.00d0,  1.10d0,  2.00d0,  2.00d0,  1.80d0,&
        &  -0.20d0,  2.00d0,  0.50d0, -0.10d0,  2.00d0,  1.80d0,  1.40d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.80d0,  1.70d0,  0.80d0,  2.00d0,  2.10d0,  2.00d0,  2.00d0,  1.80d0,&
        &   0.80d0,  2.00d0,  1.50d0, -0.90d0,  2.00d0,  1.80d0,  0.60d0,  0.90d0,&
        &   2.30d0,  1.60d0,  1.70d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  1.80d0,  1.50d0,  0.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.60d0,  1.70d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  1.80d0,  1.50d0,  0.90d0,&
        &   0.80d0,  0.70d0, -0.20d0,  2.00d0,  1.10d0,  2.00d0,  2.00d0,  1.80d0,&
        &  -0.20d0,  2.00d0,  0.50d0, -0.10d0,  2.00d0,  1.80d0,  1.40d0,  1.80d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.30d0,  1.20d0,  0.30d0,  2.00d0,  1.70d0,  2.10d0,  2.00d0,  2.00d0,&
        &   0.30d0,  2.00d0,  1.10d0, -0.40d0,  2.00d0,  2.00d0,  1.10d0,  1.50d0,&
        &   1.60d0,  0.50d0,  1.40d0,  2.00d0,  0.90d0,  1.80d0,  2.00d0,  1.60d0,&
        &   1.40d0,  2.00d0,  1.30d0, -2.10d0,  2.00d0,  1.60d0, -0.60d0,  1.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.60d0,  1.70d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  1.80d0,  1.50d0,  0.90d0,&
        &   0.60d0, -0.50d0,  0.40d0,  2.00d0,  0.00d0,  0.80d0,  2.00d0,  0.70d0,&
        &   0.40d0,  2.00d0,  0.40d0, -3.10d0,  2.00d0,  0.70d0, -1.60d0,  0.60d0,&
        &   1.90d0,  0.80d0,  1.80d0,  2.00d0,  1.30d0,  1.10d0,  2.00d0,  1.00d0,&
        &   1.80d0,  2.00d0,  1.70d0,  0.00d0,  2.00d0,  1.00d0,  1.60d0,  0.10d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  3.40d0,  2.00d0,  3.40d0,&
        &   1.70d0,  2.00d0,  2.10d0,  1.00d0,  2.00d0,  3.10d0,  2.20d0,  2.90d0,&
        &   2.60d0,  2.20d0,  1.60d0,  2.00d0,  2.60d0,  2.60d0,  2.00d0,  2.60d0,&
        &   1.60d0,  2.00d0,  2.00d0, -0.20d0,  2.00d0,  2.30d0,  1.10d0,  1.80d0,&
        &   1.50d0,  1.10d0,  0.50d0,  2.00d0,  1.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   0.50d0,  2.00d0,  0.90d0,  0.50d0,  2.00d0,  2.20d0,  1.80d0,  2.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  2.10d0,  1.50d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   1.50d0,  2.00d0,  1.90d0, -0.30d0,  2.00d0,  2.20d0,  1.00d0,  1.70d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.50d0,  1.10d0,  0.50d0,  2.00d0,  1.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   0.50d0,  2.00d0,  0.90d0,  0.50d0,  2.00d0,  2.20d0,  1.80d0,  2.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.60d0,  1.00d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.00d0,  2.00d0,  1.40d0,  0.30d0,  2.00d0,  2.40d0,  1.50d0,  2.20d0,&
        &   2.30d0,  0.90d0,  2.10d0,  2.00d0,  1.30d0,  2.30d0,  2.00d0,  2.30d0,&
        &   2.10d0,  2.00d0,  1.70d0, -1.50d0,  2.00d0,  2.00d0, -0.20d0,  2.30d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.30d0, -0.10d0,  1.10d0,  2.00d0,  0.30d0,  1.30d0,  2.00d0,  1.30d0,&
        &   1.10d0,  2.00d0,  0.70d0, -2.50d0,  2.00d0,  1.00d0, -1.20d0,  1.30d0,&
        &   2.70d0,  1.20d0,  2.40d0,  2.00d0,  1.70d0,  1.70d0,  2.00d0,  1.70d0,&
        &   2.40d0,  2.00d0,  2.00d0,  0.70d0,  2.00d0,  1.40d0,  1.90d0,  0.80d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  3.10d0,  2.00d0,  3.10d0,&
        &   1.50d0,  2.00d0,  2.10d0,  1.30d0,  2.00d0,  3.10d0,  2.30d0,  2.70d0,&
        &   2.60d0,  2.40d0,  1.40d0,  2.00d0,  2.50d0,  2.30d0,  2.00d0,  2.30d0,&
        &   1.40d0,  2.00d0,  1.90d0,  0.20d0,  2.00d0,  2.30d0,  1.20d0,  1.50d0,&
        &   1.50d0,  1.30d0,  0.30d0,  2.00d0,  1.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   0.30d0,  2.00d0,  0.80d0,  0.90d0,  2.00d0,  2.20d0,  1.90d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  2.30d0,  1.30d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   1.30d0,  2.00d0,  1.80d0,  0.10d0,  2.00d0,  2.20d0,  1.10d0,  1.40d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.50d0,  1.30d0,  0.30d0,  2.00d0,  1.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   0.30d0,  2.00d0,  0.80d0,  0.90d0,  2.00d0,  2.20d0,  1.90d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.80d0,  0.80d0,  2.00d0,  1.90d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.80d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.60d0,  2.00d0,&
        &   2.30d0,  1.10d0,  1.90d0,  2.00d0,  1.20d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  2.00d0,  1.60d0, -1.10d0,  2.00d0,  2.00d0, -0.10d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.30d0,  0.10d0,  0.90d0,  2.00d0,  0.20d0,  1.00d0,  2.00d0,  1.00d0,&
        &   0.90d0,  2.00d0,  0.70d0, -2.10d0,  2.00d0,  1.00d0, -1.10d0,  1.10d0,&
        &   2.70d0,  1.40d0,  2.20d0,  2.00d0,  1.50d0,  1.40d0,  2.00d0,  1.40d0,&
        &   2.20d0,  2.00d0,  2.00d0,  1.10d0,  2.00d0,  1.40d0,  2.00d0,  0.60d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  3.10d0,  2.00d0,  3.10d0,&
        &   1.50d0,  2.00d0,  2.10d0,  1.30d0,  2.00d0,  3.10d0,  2.30d0,  2.70d0,&
        &   2.60d0,  2.40d0,  1.40d0,  2.00d0,  2.50d0,  2.30d0,  2.00d0,  2.30d0,&
        &   1.40d0,  2.00d0,  1.90d0,  0.20d0,  2.00d0,  2.30d0,  1.20d0,  1.50d0,&
        &   1.50d0,  1.30d0,  0.30d0,  2.00d0,  1.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   0.30d0,  2.00d0,  0.80d0,  0.90d0,  2.00d0,  2.20d0,  1.90d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.50d0,  2.30d0,  1.30d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   1.30d0,  2.00d0,  1.80d0,  0.10d0,  2.00d0,  2.20d0,  1.10d0,  1.40d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.50d0,  1.30d0,  0.30d0,  2.00d0,  1.40d0,  2.20d0,  2.00d0,  2.20d0,&
        &   0.30d0,  2.00d0,  0.80d0,  0.90d0,  2.00d0,  2.20d0,  1.90d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.80d0,  0.80d0,  2.00d0,  1.90d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.80d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.60d0,  2.00d0,&
        &   2.30d0,  1.10d0,  1.90d0,  2.00d0,  1.20d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.90d0,  2.00d0,  1.60d0, -1.10d0,  2.00d0,  2.00d0, -0.10d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.30d0,  0.10d0,  0.90d0,  2.00d0,  0.20d0,  1.00d0,  2.00d0,  1.00d0,&
        &   0.90d0,  2.00d0,  0.70d0, -2.10d0,  2.00d0,  1.00d0, -1.10d0,  1.10d0,&
        &   2.70d0,  1.40d0,  2.20d0,  2.00d0,  1.50d0,  1.40d0,  2.00d0,  1.40d0,&
        &   2.20d0,  2.00d0,  2.00d0,  1.10d0,  2.00d0,  1.40d0,  2.00d0,  0.60d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  3.40d0,  2.00d0,  3.40d0,&
        &   1.70d0,  2.00d0,  2.10d0,  1.00d0,  2.00d0,  3.10d0,  2.20d0,  2.90d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  2.50d0,  1.20d0,  1.90d0,&
        &   1.70d0,  1.30d0,  0.70d0,  2.00d0,  1.70d0,  2.70d0,  2.00d0,  2.70d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.70d0,  2.00d0,  2.40d0,  2.00d0,  2.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.90d0,  1.30d0,  2.00d0,  2.30d0,  2.30d0,  2.00d0,  2.30d0,&
        &   1.30d0,  2.00d0,  1.70d0, -0.50d0,  2.00d0,  2.00d0,  0.80d0,  1.50d0,&
        &   3.40d0,  2.30d0,  2.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   2.70d0,  2.00d0,  2.70d0,  1.00d0,  2.00d0,  2.50d0,  2.20d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.70d0,  1.30d0,  0.70d0,  2.00d0,  1.70d0,  2.70d0,  2.00d0,  2.70d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.70d0,  2.00d0,  2.40d0,  2.00d0,  2.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.70d0,  1.10d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.10d0,  2.00d0,  1.50d0,  0.30d0,  2.00d0,  2.40d0,  1.60d0,  2.30d0,&
        &   2.20d0,  0.80d0,  2.00d0,  2.00d0,  1.20d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.00d0,  2.00d0,  1.60d0, -1.60d0,  2.00d0,  1.90d0, -0.30d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.40d0,  2.30d0,  2.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   2.70d0,  2.00d0,  2.70d0,  1.00d0,  2.00d0,  2.50d0,  2.20d0,  1.90d0,&
        &   1.00d0, -0.50d0,  0.70d0,  2.00d0,  0.00d0,  1.00d0,  2.00d0,  1.00d0,&
        &   0.70d0,  2.00d0,  0.30d0, -2.90d0,  2.00d0,  0.70d0, -1.60d0,  0.90d0,&
        &   2.90d0,  1.50d0,  2.70d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   2.70d0,  2.00d0,  2.30d0,  0.90d0,  2.00d0,  1.60d0,  2.20d0,  1.10d0,&
        &   2.10d0,  1.90d0,  0.10d0,  2.00d0,  1.80d0,  2.50d0,  2.00d0,  1.50d0,&
        &   0.70d0,  2.00d0,  1.80d0,  0.00d0,  2.00d0,  2.50d0,  0.40d0,  2.10d0,&
        &   2.10d0,  1.90d0,  0.10d0,  2.00d0,  1.80d0,  1.90d0,  2.00d0,  0.90d0,&
        &   0.70d0,  2.00d0,  1.80d0, -1.00d0,  2.00d0,  1.90d0, -0.60d0,  1.10d0,&
        &   1.10d0,  0.80d0, -0.90d0,  2.00d0,  0.80d0,  1.80d0,  2.00d0,  0.90d0,&
        &  -0.30d0,  2.00d0,  0.70d0, -0.30d0,  2.00d0,  1.90d0,  0.10d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.70d0,  1.40d0, -0.30d0,  2.00d0,  1.40d0,  1.40d0,  2.00d0,  0.40d0,&
        &   0.30d0,  2.00d0,  1.30d0, -1.50d0,  2.00d0,  1.50d0, -1.10d0,  0.70d0,&
        &   2.70d0,  1.90d0,  1.10d0,  2.00d0,  1.80d0,  1.90d0,  2.00d0,  0.90d0,&
        &   1.70d0,  2.00d0,  2.40d0,  0.00d0,  2.00d0,  1.90d0,  0.40d0,  1.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.40d0,  1.60d0,  0.80d0,  2.00d0,  1.50d0,  1.60d0,  2.00d0,  0.60d0,&
        &   1.40d0,  2.00d0,  2.10d0, -0.30d0,  2.00d0,  1.60d0,  0.10d0,  0.80d0,&
        &   1.10d0,  0.80d0, -0.90d0,  2.00d0,  0.80d0,  1.80d0,  2.00d0,  0.90d0,&
        &  -0.30d0,  2.00d0,  0.70d0, -0.30d0,  2.00d0,  1.90d0,  0.10d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.50d0,  1.20d0, -0.50d0,  2.00d0,  1.20d0,  1.80d0,  2.00d0,  0.80d0,&
        &   0.10d0,  2.00d0,  1.10d0, -0.70d0,  2.00d0,  1.90d0, -0.30d0,  1.50d0,&
        &   1.60d0,  0.30d0,  0.40d0,  2.00d0,  0.30d0,  1.30d0,  2.00d0,  0.40d0,&
        &   1.00d0,  2.00d0,  1.20d0, -2.60d0,  2.00d0,  1.40d0, -2.20d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.90d0,  1.10d0,  2.00d0,  1.80d0,  1.90d0,  2.00d0,  0.90d0,&
        &   1.70d0,  2.00d0,  2.40d0,  0.00d0,  2.00d0,  1.90d0,  0.40d0,  1.10d0,&
        &   0.30d0, -0.90d0, -0.90d0,  2.00d0, -1.00d0,  0.10d0,  2.00d0, -0.90d0,&
        &  -0.30d0,  2.00d0,  0.00d0, -3.90d0,  2.00d0,  0.10d0, -3.50d0,  0.10d0,&
        &   2.30d0,  1.00d0,  1.10d0,  2.00d0,  1.00d0,  1.00d0,  2.00d0,  0.00d0,&
        &   1.70d0,  2.00d0,  1.90d0, -0.10d0,  2.00d0,  1.10d0,  0.30d0,  0.30d0,&
        &   2.00d0,  1.90d0,  1.00d0,  2.00d0,  2.40d0,  2.80d0,  2.00d0,  2.70d0,&
        &   1.00d0,  2.00d0,  1.80d0,  0.30d0,  2.00d0,  2.70d0,  1.80d0,  2.20d0,&
        &   2.00d0,  1.90d0,  1.00d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.10d0,&
        &   1.00d0,  2.00d0,  1.80d0, -0.70d0,  2.00d0,  2.10d0,  0.80d0,  1.20d0,&
        &   1.00d0,  0.90d0,  0.00d0,  2.00d0,  1.30d0,  2.20d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  0.70d0,  0.10d0,  2.00d0,  2.00d0,  1.60d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.60d0,  1.50d0,  0.60d0,  2.00d0,  1.90d0,  1.80d0,  2.00d0,  1.60d0,&
        &   0.60d0,  2.00d0,  1.30d0, -1.10d0,  2.00d0,  1.60d0,  0.40d0,  0.70d0,&
        &   2.60d0,  1.90d0,  2.00d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.10d0,&
        &   2.00d0,  2.00d0,  2.40d0,  0.30d0,  2.00d0,  2.10d0,  1.80d0,  1.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.60d0,  1.70d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  1.80d0,  1.50d0,  0.90d0,&
        &   1.00d0,  0.90d0,  0.00d0,  2.00d0,  1.30d0,  2.20d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  0.70d0,  0.10d0,  2.00d0,  2.00d0,  1.60d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.40d0,  1.30d0,  0.40d0,  2.00d0,  1.70d0,  2.20d0,  2.00d0,  2.00d0,&
        &   0.40d0,  2.00d0,  1.10d0, -0.30d0,  2.00d0,  2.00d0,  1.20d0,  1.50d0,&
        &   1.50d0,  0.40d0,  1.30d0,  2.00d0,  0.80d0,  1.70d0,  2.00d0,  1.50d0,&
        &   1.30d0,  2.00d0,  1.20d0, -2.20d0,  2.00d0,  1.50d0, -0.70d0,  1.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.60d0,  1.90d0,  2.00d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.10d0,&
        &   2.00d0,  2.00d0,  2.40d0,  0.30d0,  2.00d0,  2.10d0,  1.80d0,  1.20d0,&
        &   0.20d0, -0.90d0,  0.00d0,  2.00d0, -0.40d0,  0.40d0,  2.00d0,  0.30d0,&
        &   0.00d0,  2.00d0,  0.00d0, -3.50d0,  2.00d0,  0.30d0, -2.00d0,  0.20d0,&
        &   2.20d0,  1.10d0,  2.00d0,  2.00d0,  1.50d0,  1.40d0,  2.00d0,  1.20d0,&
        &   2.00d0,  2.00d0,  1.90d0,  0.30d0,  2.00d0,  1.20d0,  1.80d0,  0.30d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  3.40d0,  2.00d0,  3.40d0,&
        &   1.70d0,  2.00d0,  2.10d0,  1.00d0,  2.00d0,  3.10d0,  2.20d0,  2.90d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  2.50d0,  1.20d0,  1.90d0,&
        &   1.70d0,  1.30d0,  0.70d0,  2.00d0,  1.70d0,  2.70d0,  2.00d0,  2.70d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.70d0,  2.00d0,  2.40d0,  2.00d0,  2.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.90d0,  1.30d0,  2.00d0,  2.30d0,  2.30d0,  2.00d0,  2.30d0,&
        &   1.30d0,  2.00d0,  1.70d0, -0.50d0,  2.00d0,  2.00d0,  0.80d0,  1.50d0,&
        &   3.40d0,  2.30d0,  2.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   2.70d0,  2.00d0,  2.70d0,  1.00d0,  2.00d0,  2.50d0,  2.20d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.70d0,  1.30d0,  0.70d0,  2.00d0,  1.70d0,  2.70d0,  2.00d0,  2.70d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.70d0,  2.00d0,  2.40d0,  2.00d0,  2.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.70d0,  1.10d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.10d0,  2.00d0,  1.50d0,  0.30d0,  2.00d0,  2.40d0,  1.60d0,  2.30d0,&
        &   2.20d0,  0.80d0,  2.00d0,  2.00d0,  1.20d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.00d0,  2.00d0,  1.60d0, -1.60d0,  2.00d0,  1.90d0, -0.30d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.40d0,  2.30d0,  2.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   2.70d0,  2.00d0,  2.70d0,  1.00d0,  2.00d0,  2.50d0,  2.20d0,  1.90d0,&
        &   1.00d0, -0.50d0,  0.70d0,  2.00d0,  0.00d0,  1.00d0,  2.00d0,  1.00d0,&
        &   0.70d0,  2.00d0,  0.30d0, -2.90d0,  2.00d0,  0.70d0, -1.60d0,  0.90d0,&
        &   2.90d0,  1.50d0,  2.70d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   2.70d0,  2.00d0,  2.30d0,  0.90d0,  2.00d0,  1.60d0,  2.20d0,  1.10d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  3.10d0,  2.00d0,  3.10d0,&
        &   1.50d0,  2.00d0,  2.10d0,  1.30d0,  2.00d0,  3.10d0,  2.30d0,  2.70d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   1.50d0,  2.00d0,  2.10d0,  0.30d0,  2.00d0,  2.50d0,  1.30d0,  1.70d0,&
        &   1.70d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.50d0,  2.00d0,  1.00d0,  1.10d0,  2.00d0,  2.40d0,  2.10d0,  2.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  2.10d0,  1.10d0,  2.00d0,  2.20d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.10d0,  2.00d0,  1.60d0, -0.10d0,  2.00d0,  2.00d0,  0.90d0,  1.20d0,&
        &   3.40d0,  2.50d0,  2.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.50d0,  2.00d0,  2.70d0,  1.30d0,  2.00d0,  2.50d0,  2.30d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.70d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.50d0,  2.00d0,  1.00d0,  1.10d0,  2.00d0,  2.40d0,  2.10d0,  2.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.90d0,  0.90d0,  2.00d0,  2.00d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.90d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.70d0,  2.00d0,&
        &   2.20d0,  1.00d0,  1.80d0,  2.00d0,  1.10d0,  1.90d0,  2.00d0,  1.90d0,&
        &   1.80d0,  2.00d0,  1.50d0, -1.20d0,  2.00d0,  1.90d0, -0.20d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.40d0,  2.50d0,  2.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.50d0,  2.00d0,  2.70d0,  1.30d0,  2.00d0,  2.50d0,  2.30d0,  1.70d0,&
        &   1.00d0, -0.30d0,  0.50d0,  2.00d0, -0.20d0,  0.70d0,  2.00d0,  0.70d0,&
        &   0.50d0,  2.00d0,  0.30d0, -2.50d0,  2.00d0,  0.70d0, -1.50d0,  0.70d0,&
        &   2.90d0,  1.70d0,  2.50d0,  2.00d0,  1.80d0,  1.60d0,  2.00d0,  1.60d0,&
        &   2.50d0,  2.00d0,  2.20d0,  1.30d0,  2.00d0,  1.60d0,  2.30d0,  0.80d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  3.10d0,  2.00d0,  3.10d0,&
        &   1.50d0,  2.00d0,  2.10d0,  1.30d0,  2.00d0,  3.10d0,  2.30d0,  2.70d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   1.50d0,  2.00d0,  2.10d0,  0.30d0,  2.00d0,  2.50d0,  1.30d0,  1.70d0,&
        &   1.70d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.50d0,  2.00d0,  1.00d0,  1.10d0,  2.00d0,  2.40d0,  2.10d0,  2.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  2.10d0,  1.10d0,  2.00d0,  2.20d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.10d0,  2.00d0,  1.60d0, -0.10d0,  2.00d0,  2.00d0,  0.90d0,  1.20d0,&
        &   3.40d0,  2.50d0,  2.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.50d0,  2.00d0,  2.70d0,  1.30d0,  2.00d0,  2.50d0,  2.30d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.70d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.50d0,  2.00d0,  1.00d0,  1.10d0,  2.00d0,  2.40d0,  2.10d0,  2.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.90d0,  0.90d0,  2.00d0,  2.00d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.90d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.70d0,  2.00d0,&
        &   2.20d0,  1.00d0,  1.80d0,  2.00d0,  1.10d0,  1.90d0,  2.00d0,  1.90d0,&
        &   1.80d0,  2.00d0,  1.50d0, -1.20d0,  2.00d0,  1.90d0, -0.20d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.40d0,  2.50d0,  2.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.50d0,  2.00d0,  2.70d0,  1.30d0,  2.00d0,  2.50d0,  2.30d0,  1.70d0,&
        &   1.00d0, -0.30d0,  0.50d0,  2.00d0, -0.20d0,  0.70d0,  2.00d0,  0.70d0,&
        &   0.50d0,  2.00d0,  0.30d0, -2.50d0,  2.00d0,  0.70d0, -1.50d0,  0.70d0,&
        &   2.90d0,  1.70d0,  2.50d0,  2.00d0,  1.80d0,  1.60d0,  2.00d0,  1.60d0,&
        &   2.50d0,  2.00d0,  2.20d0,  1.30d0,  2.00d0,  1.60d0,  2.30d0,  0.80d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  3.40d0,  2.00d0,  3.40d0,&
        &   1.70d0,  2.00d0,  2.10d0,  1.00d0,  2.00d0,  3.10d0,  2.20d0,  2.90d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  2.50d0,  1.20d0,  1.90d0,&
        &   1.70d0,  1.30d0,  0.70d0,  2.00d0,  1.70d0,  2.70d0,  2.00d0,  2.70d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.70d0,  2.00d0,  2.40d0,  2.00d0,  2.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.90d0,  1.30d0,  2.00d0,  2.30d0,  2.30d0,  2.00d0,  2.30d0,&
        &   1.30d0,  2.00d0,  1.70d0, -0.50d0,  2.00d0,  2.00d0,  0.80d0,  1.50d0,&
        &   3.40d0,  2.30d0,  2.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   2.70d0,  2.00d0,  2.70d0,  1.00d0,  2.00d0,  2.50d0,  2.20d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.70d0,  1.30d0,  0.70d0,  2.00d0,  1.70d0,  2.70d0,  2.00d0,  2.70d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.70d0,  2.00d0,  2.40d0,  2.00d0,  2.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.70d0,  1.10d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.10d0,  2.00d0,  1.50d0,  0.30d0,  2.00d0,  2.40d0,  1.60d0,  2.30d0,&
        &   2.20d0,  0.80d0,  2.00d0,  2.00d0,  1.20d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.00d0,  2.00d0,  1.60d0, -1.60d0,  2.00d0,  1.90d0, -0.30d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.40d0,  2.30d0,  2.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   2.70d0,  2.00d0,  2.70d0,  1.00d0,  2.00d0,  2.50d0,  2.20d0,  1.90d0,&
        &   1.00d0, -0.50d0,  0.70d0,  2.00d0,  0.00d0,  1.00d0,  2.00d0,  1.00d0,&
        &   0.70d0,  2.00d0,  0.30d0, -2.90d0,  2.00d0,  0.70d0, -1.60d0,  0.90d0,&
        &   2.90d0,  1.50d0,  2.70d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   2.70d0,  2.00d0,  2.30d0,  0.90d0,  2.00d0,  1.60d0,  2.20d0,  1.10d0,&
        &   2.10d0,  1.90d0,  0.10d0,  2.00d0,  1.80d0,  2.50d0,  2.00d0,  1.50d0,&
        &   0.70d0,  2.00d0,  1.80d0,  0.00d0,  2.00d0,  2.50d0,  0.40d0,  2.10d0,&
        &   2.10d0,  1.90d0,  0.10d0,  2.00d0,  1.80d0,  1.90d0,  2.00d0,  0.90d0,&
        &   0.70d0,  2.00d0,  1.80d0, -1.00d0,  2.00d0,  1.90d0, -0.60d0,  1.10d0,&
        &   1.10d0,  0.80d0, -0.90d0,  2.00d0,  0.80d0,  1.80d0,  2.00d0,  0.90d0,&
        &  -0.30d0,  2.00d0,  0.70d0, -0.30d0,  2.00d0,  1.90d0,  0.10d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.70d0,  1.40d0, -0.30d0,  2.00d0,  1.40d0,  1.40d0,  2.00d0,  0.40d0,&
        &   0.30d0,  2.00d0,  1.30d0, -1.50d0,  2.00d0,  1.50d0, -1.10d0,  0.70d0,&
        &   2.70d0,  1.90d0,  1.10d0,  2.00d0,  1.80d0,  1.90d0,  2.00d0,  0.90d0,&
        &   1.70d0,  2.00d0,  2.40d0,  0.00d0,  2.00d0,  1.90d0,  0.40d0,  1.10d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.40d0,  1.60d0,  0.80d0,  2.00d0,  1.50d0,  1.60d0,  2.00d0,  0.60d0,&
        &   1.40d0,  2.00d0,  2.10d0, -0.30d0,  2.00d0,  1.60d0,  0.10d0,  0.80d0,&
        &   1.10d0,  0.80d0, -0.90d0,  2.00d0,  0.80d0,  1.80d0,  2.00d0,  0.90d0,&
        &  -0.30d0,  2.00d0,  0.70d0, -0.30d0,  2.00d0,  1.90d0,  0.10d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.50d0,  1.20d0, -0.50d0,  2.00d0,  1.20d0,  1.80d0,  2.00d0,  0.80d0,&
        &   0.10d0,  2.00d0,  1.10d0, -0.70d0,  2.00d0,  1.90d0, -0.30d0,  1.50d0,&
        &   1.60d0,  0.30d0,  0.40d0,  2.00d0,  0.30d0,  1.30d0,  2.00d0,  0.40d0,&
        &   1.00d0,  2.00d0,  1.20d0, -2.60d0,  2.00d0,  1.40d0, -2.20d0,  1.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.70d0,  1.90d0,  1.10d0,  2.00d0,  1.80d0,  1.90d0,  2.00d0,  0.90d0,&
        &   1.70d0,  2.00d0,  2.40d0,  0.00d0,  2.00d0,  1.90d0,  0.40d0,  1.10d0,&
        &   0.30d0, -0.90d0, -0.90d0,  2.00d0, -1.00d0,  0.10d0,  2.00d0, -0.90d0,&
        &  -0.30d0,  2.00d0,  0.00d0, -3.90d0,  2.00d0,  0.10d0, -3.50d0,  0.10d0,&
        &   2.30d0,  1.00d0,  1.10d0,  2.00d0,  1.00d0,  1.00d0,  2.00d0,  0.00d0,&
        &   1.70d0,  2.00d0,  1.90d0, -0.10d0,  2.00d0,  1.10d0,  0.30d0,  0.30d0,&
        &   2.00d0,  1.90d0,  1.00d0,  2.00d0,  2.40d0,  2.80d0,  2.00d0,  2.70d0,&
        &   1.00d0,  2.00d0,  1.80d0,  0.30d0,  2.00d0,  2.70d0,  1.80d0,  2.20d0,&
        &   2.00d0,  1.90d0,  1.00d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.10d0,&
        &   1.00d0,  2.00d0,  1.80d0, -0.70d0,  2.00d0,  2.10d0,  0.80d0,  1.20d0,&
        &   1.00d0,  0.90d0,  0.00d0,  2.00d0,  1.30d0,  2.20d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  0.70d0,  0.10d0,  2.00d0,  2.00d0,  1.60d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.60d0,  1.50d0,  0.60d0,  2.00d0,  1.90d0,  1.80d0,  2.00d0,  1.60d0,&
        &   0.60d0,  2.00d0,  1.30d0, -1.10d0,  2.00d0,  1.60d0,  0.40d0,  0.70d0,&
        &   2.60d0,  1.90d0,  2.00d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.10d0,&
        &   2.00d0,  2.00d0,  2.40d0,  0.30d0,  2.00d0,  2.10d0,  1.80d0,  1.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.60d0,  1.70d0,  2.00d0,  2.10d0,  1.90d0,  2.00d0,  1.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  1.80d0,  1.50d0,  0.90d0,&
        &   1.00d0,  0.90d0,  0.00d0,  2.00d0,  1.30d0,  2.20d0,  2.00d0,  2.00d0,&
        &   0.00d0,  2.00d0,  0.70d0,  0.10d0,  2.00d0,  2.00d0,  1.60d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.40d0,  1.30d0,  0.40d0,  2.00d0,  1.70d0,  2.20d0,  2.00d0,  2.00d0,&
        &   0.40d0,  2.00d0,  1.10d0, -0.30d0,  2.00d0,  2.00d0,  1.20d0,  1.50d0,&
        &   1.50d0,  0.40d0,  1.30d0,  2.00d0,  0.80d0,  1.70d0,  2.00d0,  1.50d0,&
        &   1.30d0,  2.00d0,  1.20d0, -2.20d0,  2.00d0,  1.50d0, -0.70d0,  1.50d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.60d0,  1.90d0,  2.00d0,  2.00d0,  2.40d0,  2.20d0,  2.00d0,  2.10d0,&
        &   2.00d0,  2.00d0,  2.40d0,  0.30d0,  2.00d0,  2.10d0,  1.80d0,  1.20d0,&
        &   0.20d0, -0.90d0,  0.00d0,  2.00d0, -0.40d0,  0.40d0,  2.00d0,  0.30d0,&
        &   0.00d0,  2.00d0,  0.00d0, -3.50d0,  2.00d0,  0.30d0, -2.00d0,  0.20d0,&
        &   2.20d0,  1.10d0,  2.00d0,  2.00d0,  1.50d0,  1.40d0,  2.00d0,  1.20d0,&
        &   2.00d0,  2.00d0,  1.90d0,  0.30d0,  2.00d0,  1.20d0,  1.80d0,  0.30d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  3.40d0,  2.00d0,  3.40d0,&
        &   1.70d0,  2.00d0,  2.10d0,  1.00d0,  2.00d0,  3.10d0,  2.20d0,  2.90d0,&
        &   2.80d0,  2.30d0,  1.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   1.70d0,  2.00d0,  2.10d0,  0.00d0,  2.00d0,  2.50d0,  1.20d0,  1.90d0,&
        &   1.70d0,  1.30d0,  0.70d0,  2.00d0,  1.70d0,  2.70d0,  2.00d0,  2.70d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.70d0,  2.00d0,  2.40d0,  2.00d0,  2.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  1.90d0,  1.30d0,  2.00d0,  2.30d0,  2.30d0,  2.00d0,  2.30d0,&
        &   1.30d0,  2.00d0,  1.70d0, -0.50d0,  2.00d0,  2.00d0,  0.80d0,  1.50d0,&
        &   3.40d0,  2.30d0,  2.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   2.70d0,  2.00d0,  2.70d0,  1.00d0,  2.00d0,  2.50d0,  2.20d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.00d0,  2.40d0,  2.00d0,  2.50d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.40d0,  2.00d0,  2.40d0,  0.70d0,  2.00d0,  2.20d0,  1.90d0,  1.60d0,&
        &   1.70d0,  1.30d0,  0.70d0,  2.00d0,  1.70d0,  2.70d0,  2.00d0,  2.70d0,&
        &   0.70d0,  2.00d0,  1.10d0,  0.70d0,  2.00d0,  2.40d0,  2.00d0,  2.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.70d0,  1.10d0,  2.00d0,  2.10d0,  2.70d0,  2.00d0,  2.70d0,&
        &   1.10d0,  2.00d0,  1.50d0,  0.30d0,  2.00d0,  2.40d0,  1.60d0,  2.30d0,&
        &   2.20d0,  0.80d0,  2.00d0,  2.00d0,  1.20d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.00d0,  2.00d0,  1.60d0, -1.60d0,  2.00d0,  1.90d0, -0.30d0,  2.20d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.40d0,  2.30d0,  2.70d0,  2.00d0,  2.80d0,  2.80d0,  2.00d0,  2.80d0,&
        &   2.70d0,  2.00d0,  2.70d0,  1.00d0,  2.00d0,  2.50d0,  2.20d0,  1.90d0,&
        &   1.00d0, -0.50d0,  0.70d0,  2.00d0,  0.00d0,  1.00d0,  2.00d0,  1.00d0,&
        &   0.70d0,  2.00d0,  0.30d0, -2.90d0,  2.00d0,  0.70d0, -1.60d0,  0.90d0,&
        &   2.90d0,  1.50d0,  2.70d0,  2.00d0,  1.90d0,  1.90d0,  2.00d0,  1.90d0,&
        &   2.70d0,  2.00d0,  2.30d0,  0.90d0,  2.00d0,  1.60d0,  2.20d0,  1.10d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  3.10d0,  2.00d0,  3.10d0,&
        &   1.50d0,  2.00d0,  2.10d0,  1.30d0,  2.00d0,  3.10d0,  2.30d0,  2.70d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   1.50d0,  2.00d0,  2.10d0,  0.30d0,  2.00d0,  2.50d0,  1.30d0,  1.70d0,&
        &   1.70d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.50d0,  2.00d0,  1.00d0,  1.10d0,  2.00d0,  2.40d0,  2.10d0,  2.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  2.10d0,  1.10d0,  2.00d0,  2.20d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.10d0,  2.00d0,  1.60d0, -0.10d0,  2.00d0,  2.00d0,  0.90d0,  1.20d0,&
        &   3.40d0,  2.50d0,  2.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.50d0,  2.00d0,  2.70d0,  1.30d0,  2.00d0,  2.50d0,  2.30d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.70d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.50d0,  2.00d0,  1.00d0,  1.10d0,  2.00d0,  2.40d0,  2.10d0,  2.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.90d0,  0.90d0,  2.00d0,  2.00d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.90d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.70d0,  2.00d0,&
        &   2.20d0,  1.00d0,  1.80d0,  2.00d0,  1.10d0,  1.90d0,  2.00d0,  1.90d0,&
        &   1.80d0,  2.00d0,  1.50d0, -1.20d0,  2.00d0,  1.90d0, -0.20d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.40d0,  2.50d0,  2.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.50d0,  2.00d0,  2.70d0,  1.30d0,  2.00d0,  2.50d0,  2.30d0,  1.70d0,&
        &   1.00d0, -0.30d0,  0.50d0,  2.00d0, -0.20d0,  0.70d0,  2.00d0,  0.70d0,&
        &   0.50d0,  2.00d0,  0.30d0, -2.50d0,  2.00d0,  0.70d0, -1.50d0,  0.70d0,&
        &   2.90d0,  1.70d0,  2.50d0,  2.00d0,  1.80d0,  1.60d0,  2.00d0,  1.60d0,&
        &   2.50d0,  2.00d0,  2.20d0,  1.30d0,  2.00d0,  1.60d0,  2.30d0,  0.80d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  3.10d0,  2.00d0,  3.10d0,&
        &   1.50d0,  2.00d0,  2.10d0,  1.30d0,  2.00d0,  3.10d0,  2.30d0,  2.70d0,&
        &   2.80d0,  2.50d0,  1.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   1.50d0,  2.00d0,  2.10d0,  0.30d0,  2.00d0,  2.50d0,  1.30d0,  1.70d0,&
        &   1.70d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.50d0,  2.00d0,  1.00d0,  1.10d0,  2.00d0,  2.40d0,  2.10d0,  2.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.30d0,  2.10d0,  1.10d0,  2.00d0,  2.20d0,  2.00d0,  2.00d0,  2.00d0,&
        &   1.10d0,  2.00d0,  1.60d0, -0.10d0,  2.00d0,  2.00d0,  0.90d0,  1.20d0,&
        &   3.40d0,  2.50d0,  2.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.50d0,  2.00d0,  2.70d0,  1.30d0,  2.00d0,  2.50d0,  2.30d0,  1.70d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.10d0,  2.20d0,  2.20d0,  2.00d0,  2.30d0,  2.20d0,  2.00d0,  2.20d0,&
        &   2.20d0,  2.00d0,  2.40d0,  1.00d0,  2.00d0,  2.20d0,  2.00d0,  1.40d0,&
        &   1.70d0,  1.50d0,  0.50d0,  2.00d0,  1.60d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.50d0,  2.00d0,  1.00d0,  1.10d0,  2.00d0,  2.40d0,  2.10d0,  2.40d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.10d0,  1.90d0,  0.90d0,  2.00d0,  2.00d0,  2.40d0,  2.00d0,  2.40d0,&
        &   0.90d0,  2.00d0,  1.40d0,  0.70d0,  2.00d0,  2.40d0,  1.70d0,  2.00d0,&
        &   2.20d0,  1.00d0,  1.80d0,  2.00d0,  1.10d0,  1.90d0,  2.00d0,  1.90d0,&
        &   1.80d0,  2.00d0,  1.50d0, -1.20d0,  2.00d0,  1.90d0, -0.20d0,  1.90d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,  2.00d0,&
        &   3.40d0,  2.50d0,  2.50d0,  2.00d0,  2.60d0,  2.50d0,  2.00d0,  2.50d0,&
        &   2.50d0,  2.00d0,  2.70d0,  1.30d0,  2.00d0,  2.50d0,  2.30d0,  1.70d0,&
        &   1.00d0, -0.30d0,  0.50d0,  2.00d0, -0.20d0,  0.70d0,  2.00d0,  0.70d0,&
        &   0.50d0,  2.00d0,  0.30d0, -2.50d0,  2.00d0,  0.70d0, -1.50d0,  0.70d0,&
        &   2.90d0,  1.70d0,  2.50d0,  2.00d0,  1.80d0,  1.60d0,  2.00d0,  1.60d0,&
        &   2.50d0,  2.00d0,  2.20d0,  1.30d0,  2.00d0,  1.60d0,  2.30d0,  0.80d0 /


        !=== DH TERMS ===!

        !=== 1/1 Int Loop ===!

        DATA (((h11(i,j,k),k=1,16),j=1,6),i=1,6) / &
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0 /

        !=== 1/2 Int Loop ===!

        DATA ((((h12(i,j,k,m),k=1,16),m=1,4),j=1,6),i=1,6) / &
        &  -8.28d0, -9.98d0, -9.98d0, -9.98d0, -8.88d0, -8.68d0, -8.68d0, -8.68d0,&
        & -12.28d0,-15.78d0,-15.78d0,-15.78d0, -9.18d0, -9.18d0, -9.18d0, -9.18d0,&
        &  -8.58d0,-10.28d0,-10.28d0,-10.28d0, -7.68d0, -7.48d0, -7.48d0, -7.48d0,&
        &  -5.38d0, -8.88d0, -8.88d0, -8.88d0, -5.08d0, -5.08d0, -5.08d0, -5.08d0,&
        &  -8.08d0, -9.78d0, -9.78d0, -9.78d0, -9.38d0, -9.18d0, -9.18d0, -9.18d0,&
        & -12.28d0,-15.78d0,-15.78d0,-15.78d0,-10.08d0,-10.08d0,-10.08d0,-10.08d0,&
        &  -7.08d0, -8.78d0, -8.78d0, -8.78d0, -3.98d0, -3.78d0, -3.78d0, -3.78d0,&
        &  -6.18d0, -9.68d0, -9.68d0, -9.68d0, -4.69d0, -4.69d0, -4.69d0, -4.69d0,&
        &  -9.48d0,-11.18d0,-11.18d0,-11.18d0,-11.38d0,-11.18d0,-11.18d0,-11.18d0,&
        &  -8.98d0,-12.48d0,-12.48d0,-12.48d0,-12.08d0,-12.08d0,-12.08d0,-12.08d0,&
        & -11.48d0,-13.18d0,-13.18d0,-13.18d0, -5.68d0, -5.48d0, -5.48d0, -5.48d0,&
        &  -6.48d0, -9.98d0, -9.98d0, -9.98d0, -7.18d0, -7.18d0, -7.18d0, -7.18d0,&
        & -11.38d0,-13.08d0,-13.08d0,-13.08d0, -9.98d0, -9.78d0, -9.78d0, -9.78d0,&
        &  -9.58d0,-13.08d0,-13.08d0,-13.08d0,-10.68d0,-10.68d0,-10.68d0,-10.68d0,&
        &  -9.28d0,-10.98d0,-10.98d0,-10.98d0, -7.58d0, -7.38d0, -7.38d0, -7.38d0,&
        &  -8.38d0,-11.88d0,-11.88d0,-11.88d0, -8.98d0, -8.98d0, -8.98d0, -8.98d0,&
        & -14.58d0,-16.28d0,-16.28d0,-16.28d0,-12.08d0,-11.88d0,-11.88d0,-11.88d0,&
        & -13.68d0,-17.18d0,-17.18d0,-17.18d0,-13.58d0,-13.58d0,-13.58d0,-13.58d0,&
        &  -9.48d0,-11.18d0,-11.18d0,-11.18d0, -7.08d0, -6.88d0, -6.88d0, -6.88d0,&
        &  -8.58d0,-12.08d0,-12.08d0,-12.08d0, -9.98d0, -9.98d0, -9.98d0, -9.98d0,&
        & -13.68d0,-15.38d0,-15.38d0,-15.38d0,-11.98d0,-11.78d0,-11.78d0,-11.78d0,&
        & -12.78d0,-16.28d0,-16.28d0,-16.28d0,-12.68d0,-12.68d0,-12.68d0,-12.68d0,&
        & -12.38d0,-14.08d0,-14.08d0,-14.08d0, -9.98d0, -9.78d0, -9.78d0, -9.78d0,&
        & -11.48d0,-14.98d0,-14.98d0,-14.98d0,-11.88d0,-11.88d0,-11.88d0,-11.88d0,&
        & -11.48d0,-13.18d0,-13.18d0,-13.18d0,-10.48d0,-10.28d0,-10.28d0,-10.28d0,&
        & -12.98d0,-16.48d0,-16.48d0,-16.48d0,-11.38d0,-11.38d0,-11.38d0,-11.38d0,&
        &  -9.08d0,-10.78d0,-10.78d0,-10.78d0, -7.38d0, -7.18d0, -7.18d0, -7.18d0,&
        &  -6.98d0,-10.48d0,-10.48d0,-10.48d0, -8.08d0, -8.08d0, -8.08d0, -8.08d0,&
        & -10.88d0,-12.58d0,-12.58d0,-12.58d0,-10.68d0,-10.48d0,-10.48d0,-10.48d0,&
        & -12.58d0,-16.08d0,-16.08d0,-16.08d0,-11.38d0,-11.38d0,-11.38d0,-11.38d0,&
        &  -9.78d0,-11.48d0,-11.48d0,-11.48d0, -6.98d0, -6.78d0, -6.78d0, -6.78d0,&
        &  -8.88d0,-12.38d0,-12.38d0,-12.38d0, -6.88d0, -6.88d0, -6.88d0, -6.88d0,&
        &  -8.58d0,-10.28d0,-10.28d0,-10.28d0, -8.58d0, -8.38d0, -8.38d0, -8.38d0,&
        &  -9.38d0,-12.88d0,-12.88d0,-12.88d0, -9.28d0, -9.28d0, -9.28d0, -9.28d0,&
        &  -6.88d0, -8.58d0, -8.58d0, -8.58d0, -4.98d0, -4.78d0, -4.78d0, -4.78d0,&
        &  -5.78d0, -9.28d0, -9.28d0, -9.28d0, -5.68d0, -5.68d0, -5.68d0, -5.68d0,&
        &  -7.68d0, -9.38d0, -9.38d0, -9.38d0, -9.48d0, -9.28d0, -9.28d0, -9.28d0,&
        & -10.28d0,-13.78d0,-13.78d0,-13.78d0,-10.18d0,-10.18d0,-10.18d0,-10.18d0,&
        &  -7.58d0, -9.28d0, -9.28d0, -9.28d0, -5.88d0, -5.68d0, -5.68d0, -5.68d0,&
        &  -6.68d0,-10.18d0,-10.18d0,-10.18d0, -6.58d0, -6.58d0, -6.58d0, -6.58d0,&
        &  -8.58d0,-10.28d0,-10.28d0,-10.28d0, -8.58d0, -8.38d0, -8.38d0, -8.38d0,&
        &  -9.38d0,-12.88d0,-12.88d0,-12.88d0, -9.28d0, -9.28d0, -9.28d0, -9.28d0,&
        &  -6.88d0, -8.58d0, -8.58d0, -8.58d0, -4.98d0, -4.78d0, -4.78d0, -4.78d0,&
        &  -5.78d0, -9.28d0, -9.28d0, -9.28d0, -5.68d0, -5.68d0, -5.68d0, -5.68d0,&
        &  -7.68d0, -9.38d0, -9.38d0, -9.38d0, -9.48d0, -9.28d0, -9.28d0, -9.28d0,&
        & -10.28d0,-13.78d0,-13.78d0,-13.78d0,-10.18d0,-10.18d0,-10.18d0,-10.18d0,&
        &  -7.58d0, -9.28d0, -9.28d0, -9.28d0, -5.88d0, -5.68d0, -5.68d0, -5.68d0,&
        &  -6.68d0,-10.18d0,-10.18d0,-10.18d0, -6.58d0, -6.58d0, -6.58d0, -6.58d0,&
        & -14.28d0,-13.48d0,-14.28d0,-14.28d0,-11.48d0,-10.78d0,-11.48d0,-12.98d0,&
        & -18.28d0,-18.28d0,-18.28d0,-18.28d0,-13.98d0,-13.28d0,-13.98d0,-14.48d0,&
        & -14.58d0,-13.78d0,-14.58d0,-14.58d0,-10.28d0, -9.58d0,-10.28d0,-11.78d0,&
        & -11.38d0,-11.38d0,-11.38d0,-11.38d0, -9.88d0, -9.18d0, -9.88d0,-10.38d0,&
        & -14.08d0,-13.28d0,-14.08d0,-14.08d0,-11.98d0,-11.28d0,-11.98d0,-13.48d0,&
        & -18.28d0,-18.28d0,-18.28d0,-18.28d0,-14.88d0,-14.18d0,-14.88d0,-15.38d0,&
        & -13.08d0,-12.28d0,-13.08d0,-13.08d0, -6.58d0, -5.88d0, -6.58d0, -8.08d0,&
        & -12.18d0,-12.18d0,-12.18d0,-12.18d0, -9.49d0, -8.79d0, -9.49d0, -9.99d0,&
        & -15.48d0,-14.68d0,-15.48d0,-15.48d0,-13.98d0,-13.28d0,-13.98d0,-15.48d0,&
        & -14.98d0,-14.98d0,-14.98d0,-14.98d0,-16.88d0,-16.18d0,-16.88d0,-17.38d0,&
        & -17.48d0,-16.68d0,-17.48d0,-17.48d0, -8.28d0, -7.58d0, -8.28d0, -9.78d0,&
        & -12.48d0,-12.48d0,-12.48d0,-12.48d0,-11.98d0,-11.28d0,-11.98d0,-12.48d0,&
        & -17.38d0,-16.58d0,-17.38d0,-17.38d0,-12.58d0,-11.88d0,-12.58d0,-14.08d0,&
        & -15.58d0,-15.58d0,-15.58d0,-15.58d0,-15.48d0,-14.78d0,-15.48d0,-15.98d0,&
        & -15.28d0,-14.48d0,-15.28d0,-15.28d0,-10.18d0, -9.48d0,-10.18d0,-11.68d0,&
        & -14.38d0,-14.38d0,-14.38d0,-14.38d0,-13.78d0,-13.08d0,-13.78d0,-14.28d0,&
        & -20.58d0,-19.78d0,-20.58d0,-20.58d0,-14.68d0,-13.98d0,-14.68d0,-16.18d0,&
        & -19.68d0,-19.68d0,-19.68d0,-19.68d0,-18.38d0,-17.68d0,-18.38d0,-18.88d0,&
        & -15.48d0,-14.68d0,-15.48d0,-15.48d0, -9.68d0, -8.98d0, -9.68d0,-11.18d0,&
        & -14.58d0,-14.58d0,-14.58d0,-14.58d0,-14.78d0,-14.08d0,-14.78d0,-15.28d0,&
        & -19.68d0,-18.88d0,-19.68d0,-19.68d0,-14.58d0,-13.88d0,-14.58d0,-16.08d0,&
        & -18.78d0,-18.78d0,-18.78d0,-18.78d0,-17.48d0,-16.78d0,-17.48d0,-17.98d0,&
        & -18.38d0,-17.58d0,-18.38d0,-18.38d0,-12.58d0,-11.88d0,-12.58d0,-14.08d0,&
        & -17.48d0,-17.48d0,-17.48d0,-17.48d0,-16.68d0,-15.98d0,-16.68d0,-17.18d0,&
        & -17.48d0,-16.68d0,-17.48d0,-17.48d0,-13.08d0,-12.38d0,-13.08d0,-14.58d0,&
        & -18.98d0,-18.98d0,-18.98d0,-18.98d0,-16.18d0,-15.48d0,-16.18d0,-16.68d0,&
        & -15.08d0,-14.28d0,-15.08d0,-15.08d0, -9.98d0, -9.28d0, -9.98d0,-11.48d0,&
        & -12.98d0,-12.98d0,-12.98d0,-12.98d0,-12.88d0,-12.18d0,-12.88d0,-13.38d0,&
        & -16.88d0,-16.08d0,-16.88d0,-16.88d0,-13.28d0,-12.58d0,-13.28d0,-14.78d0,&
        & -18.58d0,-18.58d0,-18.58d0,-18.58d0,-16.18d0,-15.48d0,-16.18d0,-16.68d0,&
        & -15.78d0,-14.98d0,-15.78d0,-15.78d0, -9.58d0, -8.88d0, -9.58d0,-11.08d0,&
        & -14.88d0,-14.88d0,-14.88d0,-14.88d0,-11.68d0,-10.98d0,-11.68d0,-12.18d0,&
        & -14.58d0,-13.78d0,-14.58d0,-14.58d0,-11.18d0,-10.48d0,-11.18d0,-12.68d0,&
        & -15.38d0,-15.38d0,-15.38d0,-15.38d0,-14.08d0,-13.38d0,-14.08d0,-14.58d0,&
        & -12.88d0,-12.08d0,-12.88d0,-12.88d0, -7.58d0, -6.88d0, -7.58d0, -9.08d0,&
        & -11.78d0,-11.78d0,-11.78d0,-11.78d0,-10.48d0, -9.78d0,-10.48d0,-10.98d0,&
        & -13.68d0,-12.88d0,-13.68d0,-13.68d0,-12.08d0,-11.38d0,-12.08d0,-13.58d0,&
        & -16.28d0,-16.28d0,-16.28d0,-16.28d0,-14.98d0,-14.28d0,-14.98d0,-15.48d0,&
        & -13.58d0,-12.78d0,-13.58d0,-13.58d0, -8.48d0, -7.78d0, -8.48d0, -9.98d0,&
        & -12.68d0,-12.68d0,-12.68d0,-12.68d0,-11.38d0,-10.68d0,-11.38d0,-11.88d0,&
        & -14.58d0,-13.78d0,-14.58d0,-14.58d0,-11.18d0,-10.48d0,-11.18d0,-12.68d0,&
        & -15.38d0,-15.38d0,-15.38d0,-15.38d0,-14.08d0,-13.38d0,-14.08d0,-14.58d0,&
        & -12.88d0,-12.08d0,-12.88d0,-12.88d0, -7.58d0, -6.88d0, -7.58d0, -9.08d0,&
        & -11.78d0,-11.78d0,-11.78d0,-11.78d0,-10.48d0, -9.78d0,-10.48d0,-10.98d0,&
        & -13.68d0,-12.88d0,-13.68d0,-13.68d0,-12.08d0,-11.38d0,-12.08d0,-13.58d0,&
        & -16.28d0,-16.28d0,-16.28d0,-16.28d0,-14.98d0,-14.28d0,-14.98d0,-15.48d0,&
        & -13.58d0,-12.78d0,-13.58d0,-13.58d0, -8.48d0, -7.78d0, -8.48d0, -9.98d0,&
        & -12.68d0,-12.68d0,-12.68d0,-12.68d0,-11.38d0,-10.68d0,-11.38d0,-11.88d0,&
        &  -9.18d0,-12.78d0, -9.58d0,-12.78d0,-13.48d0, -9.38d0, -9.38d0,-10.18d0,&
        & -15.98d0,-16.28d0,-15.08d0,-16.28d0,-10.88d0,-10.88d0,-10.88d0,-11.58d0,&
        &  -9.48d0,-13.08d0, -9.88d0,-13.08d0,-12.28d0, -8.18d0, -8.18d0, -8.98d0,&
        &  -9.08d0, -9.38d0, -8.18d0, -9.38d0, -6.78d0, -6.78d0, -6.78d0, -7.48d0,&
        &  -8.98d0,-12.58d0, -9.38d0,-12.58d0,-13.98d0, -9.88d0, -9.88d0,-10.68d0,&
        & -15.98d0,-16.28d0,-15.08d0,-16.28d0,-11.78d0,-11.78d0,-11.78d0,-12.48d0,&
        &  -7.98d0,-11.58d0, -8.38d0,-11.58d0, -8.58d0, -4.48d0, -4.48d0, -5.28d0,&
        &  -9.88d0,-10.18d0, -8.98d0,-10.18d0, -6.39d0, -6.39d0, -6.39d0, -7.09d0,&
        & -10.38d0,-13.98d0,-10.78d0,-13.98d0,-15.98d0,-11.88d0,-11.88d0,-12.68d0,&
        & -12.68d0,-12.98d0,-11.78d0,-12.98d0,-13.78d0,-13.78d0,-13.78d0,-14.48d0,&
        & -12.38d0,-15.98d0,-12.78d0,-15.98d0,-10.28d0, -6.18d0, -6.18d0, -6.98d0,&
        & -10.18d0,-10.48d0, -9.28d0,-10.48d0, -8.88d0, -8.88d0, -8.88d0, -9.58d0,&
        & -12.28d0,-15.88d0,-12.68d0,-15.88d0,-14.58d0,-10.48d0,-10.48d0,-11.28d0,&
        & -13.28d0,-13.58d0,-12.38d0,-13.58d0,-12.38d0,-12.38d0,-12.38d0,-13.08d0,&
        & -10.18d0,-13.78d0,-10.58d0,-13.78d0,-12.18d0, -8.08d0, -8.08d0, -8.88d0,&
        & -12.08d0,-12.38d0,-11.18d0,-12.38d0,-10.68d0,-10.68d0,-10.68d0,-11.38d0,&
        & -15.48d0,-19.08d0,-15.88d0,-19.08d0,-16.68d0,-12.58d0,-12.58d0,-13.38d0,&
        & -17.38d0,-17.68d0,-16.48d0,-17.68d0,-15.28d0,-15.28d0,-15.28d0,-15.98d0,&
        & -10.38d0,-13.98d0,-10.78d0,-13.98d0,-11.68d0, -7.58d0, -7.58d0, -8.38d0,&
        & -12.28d0,-12.58d0,-11.38d0,-12.58d0,-11.68d0,-11.68d0,-11.68d0,-12.38d0,&
        & -14.58d0,-18.18d0,-14.98d0,-18.18d0,-16.58d0,-12.48d0,-12.48d0,-13.28d0,&
        & -16.48d0,-16.78d0,-15.58d0,-16.78d0,-14.38d0,-14.38d0,-14.38d0,-15.08d0,&
        & -13.28d0,-16.88d0,-13.68d0,-16.88d0,-14.58d0,-10.48d0,-10.48d0,-11.28d0,&
        & -15.18d0,-15.48d0,-14.28d0,-15.48d0,-13.58d0,-13.58d0,-13.58d0,-14.28d0,&
        & -12.38d0,-15.98d0,-12.78d0,-15.98d0,-15.08d0,-10.98d0,-10.98d0,-11.78d0,&
        & -16.68d0,-16.98d0,-15.78d0,-16.98d0,-13.08d0,-13.08d0,-13.08d0,-13.78d0,&
        &  -9.98d0,-13.58d0,-10.38d0,-13.58d0,-11.98d0, -7.88d0, -7.88d0, -8.68d0,&
        & -10.68d0,-10.98d0, -9.78d0,-10.98d0, -9.78d0, -9.78d0, -9.78d0,-10.48d0,&
        & -11.78d0,-15.38d0,-12.18d0,-15.38d0,-15.28d0,-11.18d0,-11.18d0,-11.98d0,&
        & -16.28d0,-16.58d0,-15.38d0,-16.58d0,-13.08d0,-13.08d0,-13.08d0,-13.78d0,&
        & -10.68d0,-14.28d0,-11.08d0,-14.28d0,-11.58d0, -7.48d0, -7.48d0, -8.28d0,&
        & -12.58d0,-12.88d0,-11.68d0,-12.88d0, -8.58d0, -8.58d0, -8.58d0, -9.28d0,&
        &  -9.48d0,-13.08d0, -9.88d0,-13.08d0,-13.18d0, -9.08d0, -9.08d0, -9.88d0,&
        & -13.08d0,-13.38d0,-12.18d0,-13.38d0,-10.98d0,-10.98d0,-10.98d0,-11.68d0,&
        &  -7.78d0,-11.38d0, -8.18d0,-11.38d0, -9.58d0, -5.48d0, -5.48d0, -6.28d0,&
        &  -9.48d0, -9.78d0, -8.58d0, -9.78d0, -7.38d0, -7.38d0, -7.38d0, -8.08d0,&
        &  -8.58d0,-12.18d0, -8.98d0,-12.18d0,-14.08d0, -9.98d0, -9.98d0,-10.78d0,&
        & -13.98d0,-14.28d0,-13.08d0,-14.28d0,-11.88d0,-11.88d0,-11.88d0,-12.58d0,&
        &  -8.48d0,-12.08d0, -8.88d0,-12.08d0,-10.48d0, -6.38d0, -6.38d0, -7.18d0,&
        & -10.38d0,-10.68d0, -9.48d0,-10.68d0, -8.28d0, -8.28d0, -8.28d0, -8.98d0,&
        &  -9.48d0,-13.08d0, -9.88d0,-13.08d0,-13.18d0, -9.08d0, -9.08d0, -9.88d0,&
        & -13.08d0,-13.38d0,-12.18d0,-13.38d0,-10.98d0,-10.98d0,-10.98d0,-11.68d0,&
        &  -7.78d0,-11.38d0, -8.18d0,-11.38d0, -9.58d0, -5.48d0, -5.48d0, -6.28d0,&
        &  -9.48d0, -9.78d0, -8.58d0, -9.78d0, -7.38d0, -7.38d0, -7.38d0, -8.08d0,&
        &  -8.58d0,-12.18d0, -8.98d0,-12.18d0,-14.08d0, -9.98d0, -9.98d0,-10.78d0,&
        & -13.98d0,-14.28d0,-13.08d0,-14.28d0,-11.88d0,-11.88d0,-11.88d0,-12.58d0,&
        &  -8.48d0,-12.08d0, -8.88d0,-12.08d0,-10.48d0, -6.38d0, -6.38d0, -7.18d0,&
        & -10.38d0,-10.68d0, -9.48d0,-10.68d0, -8.28d0, -8.28d0, -8.28d0, -8.98d0,&
        &  -8.28d0, -9.98d0, -9.98d0, -9.98d0, -8.88d0, -8.68d0, -8.68d0, -8.68d0,&
        & -12.28d0,-15.78d0,-15.78d0,-15.78d0, -9.18d0, -9.18d0, -9.18d0, -9.18d0,&
        &  -8.58d0,-10.28d0,-10.28d0,-10.28d0, -7.68d0, -7.48d0, -7.48d0, -7.48d0,&
        &  -5.38d0, -8.88d0, -8.88d0, -8.88d0, -5.08d0, -5.08d0, -5.08d0, -5.08d0,&
        &  -8.08d0, -9.78d0, -9.78d0, -9.78d0, -9.38d0, -9.18d0, -9.18d0, -9.18d0,&
        & -12.28d0,-15.78d0,-15.78d0,-15.78d0,-10.08d0,-10.08d0,-10.08d0,-10.08d0,&
        &  -7.08d0, -8.78d0, -8.78d0, -8.78d0, -3.98d0, -3.78d0, -3.78d0, -3.78d0,&
        &  -6.18d0, -9.68d0, -9.68d0, -9.68d0, -4.69d0, -4.69d0, -4.69d0, -4.69d0,&
        &  -9.48d0,-11.18d0,-11.18d0,-11.18d0,-11.38d0,-11.18d0,-11.18d0,-11.18d0,&
        &  -8.98d0,-12.48d0,-12.48d0,-12.48d0,-12.08d0,-12.08d0,-12.08d0,-12.08d0,&
        & -11.48d0,-13.18d0,-13.18d0,-13.18d0, -5.68d0, -5.48d0, -5.48d0, -5.48d0,&
        &  -6.48d0, -9.98d0, -9.98d0, -9.98d0, -7.18d0, -7.18d0, -7.18d0, -7.18d0,&
        & -11.38d0,-13.08d0,-13.08d0,-13.08d0, -9.98d0, -9.78d0, -9.78d0, -9.78d0,&
        &  -9.58d0,-13.08d0,-13.08d0,-13.08d0,-10.68d0,-10.68d0,-10.68d0,-10.68d0,&
        &  -9.28d0,-10.98d0,-10.98d0,-10.98d0, -7.58d0, -7.38d0, -7.38d0, -7.38d0,&
        &  -8.38d0,-11.88d0,-11.88d0,-11.88d0, -8.98d0, -8.98d0, -8.98d0, -8.98d0,&
        & -14.58d0,-16.28d0,-16.28d0,-16.28d0,-12.08d0,-11.88d0,-11.88d0,-11.88d0,&
        & -13.68d0,-17.18d0,-17.18d0,-17.18d0,-13.58d0,-13.58d0,-13.58d0,-13.58d0,&
        &  -9.48d0,-11.18d0,-11.18d0,-11.18d0, -7.08d0, -6.88d0, -6.88d0, -6.88d0,&
        &  -8.58d0,-12.08d0,-12.08d0,-12.08d0, -9.98d0, -9.98d0, -9.98d0, -9.98d0,&
        & -13.68d0,-15.38d0,-15.38d0,-15.38d0,-11.98d0,-11.78d0,-11.78d0,-11.78d0,&
        & -12.78d0,-16.28d0,-16.28d0,-16.28d0,-12.68d0,-12.68d0,-12.68d0,-12.68d0,&
        & -12.38d0,-14.08d0,-14.08d0,-14.08d0, -9.98d0, -9.78d0, -9.78d0, -9.78d0,&
        & -11.48d0,-14.98d0,-14.98d0,-14.98d0,-11.88d0,-11.88d0,-11.88d0,-11.88d0,&
        & -11.48d0,-13.18d0,-13.18d0,-13.18d0,-10.48d0,-10.28d0,-10.28d0,-10.28d0,&
        & -12.98d0,-16.48d0,-16.48d0,-16.48d0,-11.38d0,-11.38d0,-11.38d0,-11.38d0,&
        &  -9.08d0,-10.78d0,-10.78d0,-10.78d0, -7.38d0, -7.18d0, -7.18d0, -7.18d0,&
        &  -6.98d0,-10.48d0,-10.48d0,-10.48d0, -8.08d0, -8.08d0, -8.08d0, -8.08d0,&
        & -10.88d0,-12.58d0,-12.58d0,-12.58d0,-10.68d0,-10.48d0,-10.48d0,-10.48d0,&
        & -12.58d0,-16.08d0,-16.08d0,-16.08d0,-11.38d0,-11.38d0,-11.38d0,-11.38d0,&
        &  -9.78d0,-11.48d0,-11.48d0,-11.48d0, -6.98d0, -6.78d0, -6.78d0, -6.78d0,&
        &  -8.88d0,-12.38d0,-12.38d0,-12.38d0, -6.88d0, -6.88d0, -6.88d0, -6.88d0,&
        &  -8.58d0,-10.28d0,-10.28d0,-10.28d0, -8.58d0, -8.38d0, -8.38d0, -8.38d0,&
        &  -9.38d0,-12.88d0,-12.88d0,-12.88d0, -9.28d0, -9.28d0, -9.28d0, -9.28d0,&
        &  -6.88d0, -8.58d0, -8.58d0, -8.58d0, -4.98d0, -4.78d0, -4.78d0, -4.78d0,&
        &  -5.78d0, -9.28d0, -9.28d0, -9.28d0, -5.68d0, -5.68d0, -5.68d0, -5.68d0,&
        &  -7.68d0, -9.38d0, -9.38d0, -9.38d0, -9.48d0, -9.28d0, -9.28d0, -9.28d0,&
        & -10.28d0,-13.78d0,-13.78d0,-13.78d0,-10.18d0,-10.18d0,-10.18d0,-10.18d0,&
        &  -7.58d0, -9.28d0, -9.28d0, -9.28d0, -5.88d0, -5.68d0, -5.68d0, -5.68d0,&
        &  -6.68d0,-10.18d0,-10.18d0,-10.18d0, -6.58d0, -6.58d0, -6.58d0, -6.58d0,&
        &  -8.58d0,-10.28d0,-10.28d0,-10.28d0, -8.58d0, -8.38d0, -8.38d0, -8.38d0,&
        &  -9.38d0,-12.88d0,-12.88d0,-12.88d0, -9.28d0, -9.28d0, -9.28d0, -9.28d0,&
        &  -6.88d0, -8.58d0, -8.58d0, -8.58d0, -4.98d0, -4.78d0, -4.78d0, -4.78d0,&
        &  -5.78d0, -9.28d0, -9.28d0, -9.28d0, -5.68d0, -5.68d0, -5.68d0, -5.68d0,&
        &  -7.68d0, -9.38d0, -9.38d0, -9.38d0, -9.48d0, -9.28d0, -9.28d0, -9.28d0,&
        & -10.28d0,-13.78d0,-13.78d0,-13.78d0,-10.18d0,-10.18d0,-10.18d0,-10.18d0,&
        &  -7.58d0, -9.28d0, -9.28d0, -9.28d0, -5.88d0, -5.68d0, -5.68d0, -5.68d0,&
        &  -6.68d0,-10.18d0,-10.18d0,-10.18d0, -6.58d0, -6.58d0, -6.58d0, -6.58d0,&
        &  -7.98d0,-10.28d0,-12.88d0, -9.88d0,-10.58d0,-11.38d0, -8.28d0, -8.08d0,&
        & -12.68d0,-15.68d0,-17.78d0,-15.68d0, -8.68d0, -7.28d0, -8.68d0, -7.29d0,&
        &  -8.28d0,-10.58d0,-13.18d0,-10.18d0, -9.38d0,-10.18d0, -7.08d0, -6.88d0,&
        &  -5.78d0, -8.78d0,-10.88d0, -8.78d0, -4.58d0, -3.18d0, -4.58d0, -3.19d0,&
        &  -7.78d0,-10.08d0,-12.68d0, -9.68d0,-11.08d0,-11.88d0, -8.78d0, -8.58d0,&
        & -12.68d0,-15.68d0,-17.78d0,-15.68d0, -9.58d0, -8.18d0, -9.58d0, -8.19d0,&
        &  -6.78d0, -9.08d0,-11.68d0, -8.68d0, -5.68d0, -6.48d0, -3.38d0, -3.18d0,&
        &  -6.58d0, -9.58d0,-11.68d0, -9.58d0, -4.19d0, -2.79d0, -4.19d0, -2.80d0,&
        &  -9.18d0,-11.48d0,-14.08d0,-11.08d0,-13.08d0,-13.88d0,-10.78d0,-10.58d0,&
        &  -9.38d0,-12.38d0,-14.48d0,-12.38d0,-11.58d0,-10.18d0,-11.58d0,-10.19d0,&
        & -11.18d0,-13.48d0,-16.08d0,-13.08d0, -7.38d0, -8.18d0, -5.08d0, -4.88d0,&
        &  -6.88d0, -9.88d0,-11.98d0, -9.88d0, -6.68d0, -5.28d0, -6.68d0, -5.29d0,&
        & -11.08d0,-13.38d0,-15.98d0,-12.98d0,-11.68d0,-12.48d0, -9.38d0, -9.18d0,&
        &  -9.98d0,-12.98d0,-15.08d0,-12.98d0,-10.18d0, -8.78d0,-10.18d0, -8.79d0,&
        &  -8.98d0,-11.28d0,-13.88d0,-10.88d0, -9.28d0,-10.08d0, -6.98d0, -6.78d0,&
        &  -8.78d0,-11.78d0,-13.88d0,-11.78d0, -8.48d0, -7.08d0, -8.48d0, -7.09d0,&
        & -14.28d0,-16.58d0,-19.18d0,-16.18d0,-13.78d0,-14.58d0,-11.48d0,-11.28d0,&
        & -14.08d0,-17.08d0,-19.18d0,-17.08d0,-13.08d0,-11.68d0,-13.08d0,-11.69d0,&
        &  -9.18d0,-11.48d0,-14.08d0,-11.08d0, -8.78d0, -9.58d0, -6.48d0, -6.28d0,&
        &  -8.98d0,-11.98d0,-14.08d0,-11.98d0, -9.48d0, -8.08d0, -9.48d0, -8.09d0,&
        & -13.38d0,-15.68d0,-18.28d0,-15.28d0,-13.68d0,-14.48d0,-11.38d0,-11.18d0,&
        & -13.18d0,-16.18d0,-18.28d0,-16.18d0,-12.18d0,-10.78d0,-12.18d0,-10.79d0,&
        & -12.08d0,-14.38d0,-16.98d0,-13.98d0,-11.68d0,-12.48d0, -9.38d0, -9.18d0,&
        & -11.88d0,-14.88d0,-16.98d0,-14.88d0,-11.38d0, -9.98d0,-11.38d0, -9.99d0,&
        & -11.18d0,-13.48d0,-16.08d0,-13.08d0,-12.18d0,-12.98d0, -9.88d0, -9.68d0,&
        & -13.38d0,-16.38d0,-18.48d0,-16.38d0,-10.88d0, -9.48d0,-10.88d0, -9.49d0,&
        &  -8.78d0,-11.08d0,-13.68d0,-10.68d0, -9.08d0, -9.88d0, -6.78d0, -6.58d0,&
        &  -7.38d0,-10.38d0,-12.48d0,-10.38d0, -7.58d0, -6.18d0, -7.58d0, -6.19d0,&
        & -10.58d0,-12.88d0,-15.48d0,-12.48d0,-12.38d0,-13.18d0,-10.08d0, -9.88d0,&
        & -12.98d0,-15.98d0,-18.08d0,-15.98d0,-10.88d0, -9.48d0,-10.88d0, -9.49d0,&
        &  -9.48d0,-11.78d0,-14.38d0,-11.38d0, -8.68d0, -9.48d0, -6.38d0, -6.18d0,&
        &  -9.28d0,-12.28d0,-14.38d0,-12.28d0, -6.38d0, -4.98d0, -6.38d0, -4.99d0,&
        &  -8.28d0,-10.58d0,-13.18d0,-10.18d0,-10.28d0,-11.08d0, -7.98d0, -7.78d0,&
        &  -9.78d0,-12.78d0,-14.88d0,-12.78d0, -8.78d0, -7.38d0, -8.78d0, -7.39d0,&
        &  -6.58d0, -8.88d0,-11.48d0, -8.48d0, -6.68d0, -7.48d0, -4.38d0, -4.18d0,&
        &  -6.18d0, -9.18d0,-11.28d0, -9.18d0, -5.18d0, -3.78d0, -5.18d0, -3.79d0,&
        &  -7.38d0, -9.68d0,-12.28d0, -9.28d0,-11.18d0,-11.98d0, -8.88d0, -8.68d0,&
        & -10.68d0,-13.68d0,-15.78d0,-13.68d0, -9.68d0, -8.28d0, -9.68d0, -8.29d0,&
        &  -7.28d0, -9.58d0,-12.18d0, -9.18d0, -7.58d0, -8.38d0, -5.28d0, -5.08d0,&
        &  -7.08d0,-10.08d0,-12.18d0,-10.08d0, -6.08d0, -4.68d0, -6.08d0, -4.69d0,&
        &  -8.28d0,-10.58d0,-13.18d0,-10.18d0,-10.28d0,-11.08d0, -7.98d0, -7.78d0,&
        &  -9.78d0,-12.78d0,-14.88d0,-12.78d0, -8.78d0, -7.38d0, -8.78d0, -7.39d0,&
        &  -6.58d0, -8.88d0,-11.48d0, -8.48d0, -6.68d0, -7.48d0, -4.38d0, -4.18d0,&
        &  -6.18d0, -9.18d0,-11.28d0, -9.18d0, -5.18d0, -3.78d0, -5.18d0, -3.79d0,&
        &  -7.38d0, -9.68d0,-12.28d0, -9.28d0,-11.18d0,-11.98d0, -8.88d0, -8.68d0,&
        & -10.68d0,-13.68d0,-15.78d0,-13.68d0, -9.68d0, -8.28d0, -9.68d0, -8.29d0,&
        &  -7.28d0, -9.58d0,-12.18d0, -9.18d0, -7.58d0, -8.38d0, -5.28d0, -5.08d0,&
        &  -7.08d0,-10.08d0,-12.18d0,-10.08d0, -6.08d0, -4.68d0, -6.08d0, -4.69d0,&
        & -11.18d0,-11.88d0,-13.58d0,-12.08d0,-11.08d0,-11.08d0, -9.88d0,-11.08d0,&
        & -15.48d0,-16.98d0,-18.08d0,-16.98d0,-11.38d0,-10.28d0,-11.38d0, -9.48d0,&
        & -11.48d0,-12.18d0,-13.88d0,-12.38d0, -9.88d0, -9.88d0, -8.68d0, -9.88d0,&
        &  -8.58d0,-10.08d0,-11.18d0,-10.08d0, -7.28d0, -6.18d0, -7.28d0, -5.38d0,&
        & -10.98d0,-11.68d0,-13.38d0,-11.88d0,-11.58d0,-11.58d0,-10.38d0,-11.58d0,&
        & -15.48d0,-16.98d0,-18.08d0,-16.98d0,-12.28d0,-11.18d0,-12.28d0,-10.38d0,&
        &  -9.98d0,-10.68d0,-12.38d0,-10.88d0, -6.18d0, -6.18d0, -4.98d0, -6.18d0,&
        &  -9.38d0,-10.88d0,-11.98d0,-10.88d0, -6.89d0, -5.79d0, -6.89d0, -4.99d0,&
        & -12.38d0,-13.08d0,-14.78d0,-13.28d0,-13.58d0,-13.58d0,-12.38d0,-13.58d0,&
        & -12.18d0,-13.68d0,-14.78d0,-13.68d0,-14.28d0,-13.18d0,-14.28d0,-12.38d0,&
        & -14.38d0,-15.08d0,-16.78d0,-15.28d0, -7.88d0, -7.88d0, -6.68d0, -7.88d0,&
        &  -9.68d0,-11.18d0,-12.28d0,-11.18d0, -9.38d0, -8.28d0, -9.38d0, -7.48d0,&
        & -14.28d0,-14.98d0,-16.68d0,-15.18d0,-12.18d0,-12.18d0,-10.98d0,-12.18d0,&
        & -12.78d0,-14.28d0,-15.38d0,-14.28d0,-12.88d0,-11.78d0,-12.88d0,-10.98d0,&
        & -12.18d0,-12.88d0,-14.58d0,-13.08d0, -9.78d0, -9.78d0, -8.58d0, -9.78d0,&
        & -11.58d0,-13.08d0,-14.18d0,-13.08d0,-11.18d0,-10.08d0,-11.18d0, -9.28d0,&
        & -17.48d0,-18.18d0,-19.88d0,-18.38d0,-14.28d0,-14.28d0,-13.08d0,-14.28d0,&
        & -16.88d0,-18.38d0,-19.48d0,-18.38d0,-15.78d0,-14.68d0,-15.78d0,-13.88d0,&
        & -12.38d0,-13.08d0,-14.78d0,-13.28d0, -9.28d0, -9.28d0, -8.08d0, -9.28d0,&
        & -11.78d0,-13.28d0,-14.38d0,-13.28d0,-12.18d0,-11.08d0,-12.18d0,-10.28d0,&
        & -16.58d0,-17.28d0,-18.98d0,-17.48d0,-14.18d0,-14.18d0,-12.98d0,-14.18d0,&
        & -15.98d0,-17.48d0,-18.58d0,-17.48d0,-14.88d0,-13.78d0,-14.88d0,-12.98d0,&
        & -15.28d0,-15.98d0,-17.68d0,-16.18d0,-12.18d0,-12.18d0,-10.98d0,-12.18d0,&
        & -14.68d0,-16.18d0,-17.28d0,-16.18d0,-14.08d0,-12.98d0,-14.08d0,-12.18d0,&
        & -14.38d0,-15.08d0,-16.78d0,-15.28d0,-12.68d0,-12.68d0,-11.48d0,-12.68d0,&
        & -16.18d0,-17.68d0,-18.78d0,-17.68d0,-13.58d0,-12.48d0,-13.58d0,-11.68d0,&
        & -11.98d0,-12.68d0,-14.38d0,-12.88d0, -9.58d0, -9.58d0, -8.38d0, -9.58d0,&
        & -10.18d0,-11.68d0,-12.78d0,-11.68d0,-10.28d0, -9.18d0,-10.28d0, -8.38d0,&
        & -13.78d0,-14.48d0,-16.18d0,-14.68d0,-12.88d0,-12.88d0,-11.68d0,-12.88d0,&
        & -15.78d0,-17.28d0,-18.38d0,-17.28d0,-13.58d0,-12.48d0,-13.58d0,-11.68d0,&
        & -12.68d0,-13.38d0,-15.08d0,-13.58d0, -9.18d0, -9.18d0, -7.98d0, -9.18d0,&
        & -12.08d0,-13.58d0,-14.68d0,-13.58d0, -9.08d0, -7.98d0, -9.08d0, -7.18d0,&
        & -11.48d0,-12.18d0,-13.88d0,-12.38d0,-10.78d0,-10.78d0, -9.58d0,-10.78d0,&
        & -12.58d0,-14.08d0,-15.18d0,-14.08d0,-11.48d0,-10.38d0,-11.48d0, -9.58d0,&
        &  -9.78d0,-10.48d0,-12.18d0,-10.68d0, -7.18d0, -7.18d0, -5.98d0, -7.18d0,&
        &  -8.98d0,-10.48d0,-11.58d0,-10.48d0, -7.88d0, -6.78d0, -7.88d0, -5.98d0,&
        & -10.58d0,-11.28d0,-12.98d0,-11.48d0,-11.68d0,-11.68d0,-10.48d0,-11.68d0,&
        & -13.48d0,-14.98d0,-16.08d0,-14.98d0,-12.38d0,-11.28d0,-12.38d0,-10.48d0,&
        & -10.48d0,-11.18d0,-12.88d0,-11.38d0, -8.08d0, -8.08d0, -6.88d0, -8.08d0,&
        &  -9.88d0,-11.38d0,-12.48d0,-11.38d0, -8.78d0, -7.68d0, -8.78d0, -6.88d0,&
        & -11.48d0,-12.18d0,-13.88d0,-12.38d0,-10.78d0,-10.78d0, -9.58d0,-10.78d0,&
        & -12.58d0,-14.08d0,-15.18d0,-14.08d0,-11.48d0,-10.38d0,-11.48d0, -9.58d0,&
        &  -9.78d0,-10.48d0,-12.18d0,-10.68d0, -7.18d0, -7.18d0, -5.98d0, -7.18d0,&
        &  -8.98d0,-10.48d0,-11.58d0,-10.48d0, -7.88d0, -6.78d0, -7.88d0, -5.98d0,&
        & -10.58d0,-11.28d0,-12.98d0,-11.48d0,-11.68d0,-11.68d0,-10.48d0,-11.68d0,&
        & -13.48d0,-14.98d0,-16.08d0,-14.98d0,-12.38d0,-11.28d0,-12.38d0,-10.48d0,&
        & -10.48d0,-11.18d0,-12.88d0,-11.38d0, -8.08d0, -8.08d0, -6.88d0, -8.08d0,&
        &  -9.88d0,-11.38d0,-12.48d0,-11.38d0, -8.78d0, -7.68d0, -8.78d0, -6.88d0 /

        !=== 2/2 Int Loop ===!

        DATA ((((h22(i,j,k,m),m=1,16),k=1,16),j=1,6),i=1,6) / &
        &  -8.28d0, -8.58d0, -8.08d0, -7.08d0,-10.58d0, -9.38d0,-11.08d0, -5.68d0,&
        & -13.18d0, -6.28d0,-13.18d0, -7.08d0,-10.18d0, -6.08d0,-11.08d0, -5.69d0,&
        &  -9.98d0,-10.28d0, -9.78d0, -8.78d0,-12.28d0,-11.08d0,-12.78d0, -7.38d0,&
        & -14.88d0, -7.98d0,-14.88d0, -8.78d0,-11.88d0, -7.78d0,-12.78d0, -7.39d0,&
        &  -9.98d0,-10.28d0, -9.78d0, -8.78d0,-12.28d0,-11.08d0,-12.78d0, -7.38d0,&
        & -14.88d0, -7.98d0,-14.88d0, -8.78d0,-11.88d0, -7.78d0,-12.78d0, -7.39d0,&
        &  -9.98d0,-10.28d0, -9.78d0, -8.78d0,-12.28d0,-11.08d0,-12.78d0, -7.38d0,&
        & -14.88d0, -7.98d0,-14.88d0, -8.78d0,-11.88d0, -7.78d0,-12.78d0, -7.39d0,&
        &  -6.58d0, -6.88d0, -6.38d0, -5.38d0, -8.88d0, -7.68d0, -9.38d0, -3.98d0,&
        & -11.48d0, -4.58d0,-11.48d0, -5.38d0, -8.48d0, -4.38d0, -9.38d0, -3.99d0,&
        &  -6.38d0, -6.68d0, -6.18d0, -5.18d0, -8.68d0, -7.48d0, -9.18d0, -3.78d0,&
        & -11.28d0, -4.38d0,-11.28d0, -5.18d0, -8.28d0, -4.18d0, -9.18d0, -3.79d0,&
        &  -6.38d0, -6.68d0, -6.18d0, -5.18d0, -8.68d0, -7.48d0, -9.18d0, -3.78d0,&
        & -11.28d0, -4.38d0,-11.28d0, -5.18d0, -8.28d0, -4.18d0, -9.18d0, -3.79d0,&
        &  -6.38d0, -6.68d0, -6.18d0, -5.18d0, -8.68d0, -7.48d0, -9.18d0, -3.78d0,&
        & -11.28d0, -4.38d0,-11.28d0, -5.18d0, -8.28d0, -4.18d0, -9.18d0, -3.79d0,&
        &  -7.38d0, -7.68d0, -7.18d0, -6.18d0, -9.68d0, -8.48d0,-10.18d0, -4.78d0,&
        & -12.28d0, -5.38d0,-12.28d0, -6.18d0, -9.28d0, -5.18d0,-10.18d0, -4.79d0,&
        & -10.88d0,-11.18d0,-10.68d0, -9.68d0,-13.18d0,-11.98d0,-13.68d0, -8.28d0,&
        & -15.78d0, -8.88d0,-15.78d0, -9.68d0,-12.78d0, -8.68d0,-13.68d0, -8.29d0,&
        & -10.88d0,-11.18d0,-10.68d0, -9.68d0,-13.18d0,-11.98d0,-13.68d0, -8.28d0,&
        & -15.78d0, -8.88d0,-15.78d0, -9.68d0,-12.78d0, -8.68d0,-13.68d0, -8.29d0,&
        & -10.88d0,-11.18d0,-10.68d0, -9.68d0,-13.18d0,-11.98d0,-13.68d0, -8.28d0,&
        & -15.78d0, -8.88d0,-15.78d0, -9.68d0,-12.78d0, -8.68d0,-13.68d0, -8.29d0,&
        &  -7.28d0, -7.58d0, -7.08d0, -6.08d0, -9.58d0, -8.38d0,-10.08d0, -4.68d0,&
        & -12.18d0, -5.28d0,-12.18d0, -6.08d0, -9.18d0, -5.08d0,-10.08d0, -4.69d0,&
        &  -7.28d0, -7.58d0, -7.08d0, -6.08d0, -9.58d0, -8.38d0,-10.08d0, -4.68d0,&
        & -12.18d0, -5.28d0,-12.18d0, -6.08d0, -9.18d0, -5.08d0,-10.08d0, -4.69d0,&
        &  -7.28d0, -7.58d0, -7.08d0, -6.08d0, -9.58d0, -8.38d0,-10.08d0, -4.68d0,&
        & -12.18d0, -5.28d0,-12.18d0, -6.08d0, -9.18d0, -5.08d0,-10.08d0, -4.69d0,&
        &  -7.28d0, -7.58d0, -7.08d0, -6.08d0, -9.58d0, -8.38d0,-10.08d0, -4.68d0,&
        & -12.18d0, -5.28d0,-12.18d0, -6.08d0, -9.18d0, -5.08d0,-10.08d0, -4.69d0,&
        &  -9.48d0,-11.48d0,-11.38d0, -9.28d0,-13.08d0, -7.38d0,-11.68d0, -9.28d0,&
        &  -9.88d0, -7.38d0,-10.48d0, -9.28d0,-13.08d0, -8.18d0,-11.68d0, -9.98d0,&
        & -11.18d0,-13.18d0,-13.08d0,-10.98d0,-14.78d0, -9.08d0,-13.38d0,-10.98d0,&
        & -11.58d0, -9.08d0,-12.18d0,-10.98d0,-14.78d0, -9.88d0,-13.38d0,-11.68d0,&
        & -11.18d0,-13.18d0,-13.08d0,-10.98d0,-14.78d0, -9.08d0,-13.38d0,-10.98d0,&
        & -11.58d0, -9.08d0,-12.18d0,-10.98d0,-14.78d0, -9.88d0,-13.38d0,-11.68d0,&
        & -11.18d0,-13.18d0,-13.08d0,-10.98d0,-14.78d0, -9.08d0,-13.38d0,-10.98d0,&
        & -11.58d0, -9.08d0,-12.18d0,-10.98d0,-14.78d0, -9.88d0,-13.38d0,-11.68d0,&
        &  -7.78d0, -9.78d0, -9.68d0, -7.58d0,-11.38d0, -5.68d0, -9.98d0, -7.58d0,&
        &  -8.18d0, -5.68d0, -8.78d0, -7.58d0,-11.38d0, -6.48d0, -9.98d0, -8.28d0,&
        &  -7.58d0, -9.58d0, -9.48d0, -7.38d0,-11.18d0, -5.48d0, -9.78d0, -7.38d0,&
        &  -7.98d0, -5.48d0, -8.58d0, -7.38d0,-11.18d0, -6.28d0, -9.78d0, -8.08d0,&
        &  -7.58d0, -9.58d0, -9.48d0, -7.38d0,-11.18d0, -5.48d0, -9.78d0, -7.38d0,&
        &  -7.98d0, -5.48d0, -8.58d0, -7.38d0,-11.18d0, -6.28d0, -9.78d0, -8.08d0,&
        &  -7.58d0, -9.58d0, -9.48d0, -7.38d0,-11.18d0, -5.48d0, -9.78d0, -7.38d0,&
        &  -7.98d0, -5.48d0, -8.58d0, -7.38d0,-11.18d0, -6.28d0, -9.78d0, -8.08d0,&
        &  -8.58d0,-10.58d0,-10.48d0, -8.38d0,-12.18d0, -6.48d0,-10.78d0, -8.38d0,&
        &  -8.98d0, -6.48d0, -9.58d0, -8.38d0,-12.18d0, -7.28d0,-10.78d0, -9.08d0,&
        & -12.08d0,-14.08d0,-13.98d0,-11.88d0,-15.68d0, -9.98d0,-14.28d0,-11.88d0,&
        & -12.48d0, -9.98d0,-13.08d0,-11.88d0,-15.68d0,-10.78d0,-14.28d0,-12.58d0,&
        & -12.08d0,-14.08d0,-13.98d0,-11.88d0,-15.68d0, -9.98d0,-14.28d0,-11.88d0,&
        & -12.48d0, -9.98d0,-13.08d0,-11.88d0,-15.68d0,-10.78d0,-14.28d0,-12.58d0,&
        & -12.08d0,-14.08d0,-13.98d0,-11.88d0,-15.68d0, -9.98d0,-14.28d0,-11.88d0,&
        & -12.48d0, -9.98d0,-13.08d0,-11.88d0,-15.68d0,-10.78d0,-14.28d0,-12.58d0,&
        &  -8.48d0,-10.48d0,-10.38d0, -8.28d0,-12.08d0, -6.38d0,-10.68d0, -8.28d0,&
        &  -8.88d0, -6.38d0, -9.48d0, -8.28d0,-12.08d0, -7.18d0,-10.68d0, -8.98d0,&
        &  -8.48d0,-10.48d0,-10.38d0, -8.28d0,-12.08d0, -6.38d0,-10.68d0, -8.28d0,&
        &  -8.88d0, -6.38d0, -9.48d0, -8.28d0,-12.08d0, -7.18d0,-10.68d0, -8.98d0,&
        &  -8.48d0,-10.48d0,-10.38d0, -8.28d0,-12.08d0, -6.38d0,-10.68d0, -8.28d0,&
        &  -8.88d0, -6.38d0, -9.48d0, -8.28d0,-12.08d0, -7.18d0,-10.68d0, -8.98d0,&
        &  -8.48d0,-10.48d0,-10.38d0, -8.28d0,-12.08d0, -6.38d0,-10.68d0, -8.28d0,&
        &  -8.88d0, -6.38d0, -9.48d0, -8.28d0,-12.08d0, -7.18d0,-10.68d0, -8.98d0,&
        & -14.58d0, -9.48d0,-13.68d0,-12.38d0,-13.78d0, -8.78d0,-13.68d0,-11.68d0,&
        & -14.58d0, -9.48d0,-13.68d0,-12.38d0,-14.58d0,-10.98d0,-13.68d0,-12.88d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-15.48d0,-10.48d0,-15.38d0,-13.38d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-16.28d0,-12.68d0,-15.38d0,-14.58d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-15.48d0,-10.48d0,-15.38d0,-13.38d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-16.28d0,-12.68d0,-15.38d0,-14.58d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-15.48d0,-10.48d0,-15.38d0,-13.38d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-16.28d0,-12.68d0,-15.38d0,-14.58d0,&
        & -12.88d0, -7.78d0,-11.98d0,-10.68d0,-12.08d0, -7.08d0,-11.98d0, -9.98d0,&
        & -12.88d0, -7.78d0,-11.98d0,-10.68d0,-12.88d0, -9.28d0,-11.98d0,-11.18d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-11.88d0, -6.88d0,-11.78d0, -9.78d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-12.68d0, -9.08d0,-11.78d0,-10.98d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-11.88d0, -6.88d0,-11.78d0, -9.78d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-12.68d0, -9.08d0,-11.78d0,-10.98d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-11.88d0, -6.88d0,-11.78d0, -9.78d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-12.68d0, -9.08d0,-11.78d0,-10.98d0,&
        & -13.68d0, -8.58d0,-12.78d0,-11.48d0,-12.88d0, -7.88d0,-12.78d0,-10.78d0,&
        & -13.68d0, -8.58d0,-12.78d0,-11.48d0,-13.68d0,-10.08d0,-12.78d0,-11.98d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-16.38d0,-11.38d0,-16.28d0,-14.28d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-17.18d0,-13.58d0,-16.28d0,-15.48d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-16.38d0,-11.38d0,-16.28d0,-14.28d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-17.18d0,-13.58d0,-16.28d0,-15.48d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-16.38d0,-11.38d0,-16.28d0,-14.28d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-17.18d0,-13.58d0,-16.28d0,-15.48d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-12.78d0, -7.78d0,-12.68d0,-10.68d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-13.58d0, -9.98d0,-12.68d0,-11.88d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-12.78d0, -7.78d0,-12.68d0,-10.68d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-13.58d0, -9.98d0,-12.68d0,-11.88d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-12.78d0, -7.78d0,-12.68d0,-10.68d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-13.58d0, -9.98d0,-12.68d0,-11.88d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-12.78d0, -7.78d0,-12.68d0,-10.68d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-13.58d0, -9.98d0,-12.68d0,-11.88d0,&
        & -11.48d0, -9.08d0,-10.88d0, -9.78d0,-12.18d0, -9.08d0,-12.38d0, -8.68d0,&
        & -13.88d0, -7.88d0,-13.48d0, -9.78d0,-12.38d0, -9.08d0,-12.38d0, -7.88d0,&
        & -13.18d0,-10.78d0,-12.58d0,-11.48d0,-13.88d0,-10.78d0,-14.08d0,-10.38d0,&
        & -15.58d0, -9.58d0,-15.18d0,-11.48d0,-14.08d0,-10.78d0,-14.08d0, -9.58d0,&
        & -13.18d0,-10.78d0,-12.58d0,-11.48d0,-13.88d0,-10.78d0,-14.08d0,-10.38d0,&
        & -15.58d0, -9.58d0,-15.18d0,-11.48d0,-14.08d0,-10.78d0,-14.08d0, -9.58d0,&
        & -13.18d0,-10.78d0,-12.58d0,-11.48d0,-13.88d0,-10.78d0,-14.08d0,-10.38d0,&
        & -15.58d0, -9.58d0,-15.18d0,-11.48d0,-14.08d0,-10.78d0,-14.08d0, -9.58d0,&
        &  -9.78d0, -7.38d0, -9.18d0, -8.08d0,-10.48d0, -7.38d0,-10.68d0, -6.98d0,&
        & -12.18d0, -6.18d0,-11.78d0, -8.08d0,-10.68d0, -7.38d0,-10.68d0, -6.18d0,&
        &  -9.58d0, -7.18d0, -8.98d0, -7.88d0,-10.28d0, -7.18d0,-10.48d0, -6.78d0,&
        & -11.98d0, -5.98d0,-11.58d0, -7.88d0,-10.48d0, -7.18d0,-10.48d0, -5.98d0,&
        &  -9.58d0, -7.18d0, -8.98d0, -7.88d0,-10.28d0, -7.18d0,-10.48d0, -6.78d0,&
        & -11.98d0, -5.98d0,-11.58d0, -7.88d0,-10.48d0, -7.18d0,-10.48d0, -5.98d0,&
        &  -9.58d0, -7.18d0, -8.98d0, -7.88d0,-10.28d0, -7.18d0,-10.48d0, -6.78d0,&
        & -11.98d0, -5.98d0,-11.58d0, -7.88d0,-10.48d0, -7.18d0,-10.48d0, -5.98d0,&
        & -10.58d0, -8.18d0, -9.98d0, -8.88d0,-11.28d0, -8.18d0,-11.48d0, -7.78d0,&
        & -12.98d0, -6.98d0,-12.58d0, -8.88d0,-11.48d0, -8.18d0,-11.48d0, -6.98d0,&
        & -14.08d0,-11.68d0,-13.48d0,-12.38d0,-14.78d0,-11.68d0,-14.98d0,-11.28d0,&
        & -16.48d0,-10.48d0,-16.08d0,-12.38d0,-14.98d0,-11.68d0,-14.98d0,-10.48d0,&
        & -14.08d0,-11.68d0,-13.48d0,-12.38d0,-14.78d0,-11.68d0,-14.98d0,-11.28d0,&
        & -16.48d0,-10.48d0,-16.08d0,-12.38d0,-14.98d0,-11.68d0,-14.98d0,-10.48d0,&
        & -14.08d0,-11.68d0,-13.48d0,-12.38d0,-14.78d0,-11.68d0,-14.98d0,-11.28d0,&
        & -16.48d0,-10.48d0,-16.08d0,-12.38d0,-14.98d0,-11.68d0,-14.98d0,-10.48d0,&
        & -10.48d0, -8.08d0, -9.88d0, -8.78d0,-11.18d0, -8.08d0,-11.38d0, -7.68d0,&
        & -12.88d0, -6.88d0,-12.48d0, -8.78d0,-11.38d0, -8.08d0,-11.38d0, -6.88d0,&
        & -10.48d0, -8.08d0, -9.88d0, -8.78d0,-11.18d0, -8.08d0,-11.38d0, -7.68d0,&
        & -12.88d0, -6.88d0,-12.48d0, -8.78d0,-11.38d0, -8.08d0,-11.38d0, -6.88d0,&
        & -10.48d0, -8.08d0, -9.88d0, -8.78d0,-11.18d0, -8.08d0,-11.38d0, -7.68d0,&
        & -12.88d0, -6.88d0,-12.48d0, -8.78d0,-11.38d0, -8.08d0,-11.38d0, -6.88d0,&
        & -10.48d0, -8.08d0, -9.88d0, -8.78d0,-11.18d0, -8.08d0,-11.38d0, -7.68d0,&
        & -12.88d0, -6.88d0,-12.48d0, -8.78d0,-11.38d0, -8.08d0,-11.38d0, -6.88d0,&
        &  -8.58d0, -6.88d0, -7.68d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -6.68d0,-11.18d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        &  -6.88d0, -5.18d0, -5.98d0, -5.88d0, -8.58d0, -4.98d0, -9.48d0, -5.88d0,&
        &  -8.58d0, -4.98d0, -9.48d0, -5.88d0, -8.58d0, -4.98d0, -9.48d0, -5.88d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -7.68d0, -5.98d0, -6.78d0, -6.68d0, -9.38d0, -5.78d0,-10.28d0, -6.68d0,&
        &  -9.38d0, -5.78d0,-10.28d0, -6.68d0, -9.38d0, -5.78d0,-10.28d0, -6.68d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -8.58d0, -6.88d0, -7.68d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -6.68d0,-11.18d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        &  -6.88d0, -5.18d0, -5.98d0, -5.88d0, -8.58d0, -4.98d0, -9.48d0, -5.88d0,&
        &  -8.58d0, -4.98d0, -9.48d0, -5.88d0, -8.58d0, -4.98d0, -9.48d0, -5.88d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -7.68d0, -5.98d0, -6.78d0, -6.68d0, -9.38d0, -5.78d0,-10.28d0, -6.68d0,&
        &  -9.38d0, -5.78d0,-10.28d0, -6.68d0, -9.38d0, -5.78d0,-10.28d0, -6.68d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        & -14.28d0,-14.58d0,-14.08d0,-13.08d0,-16.58d0,-15.38d0,-17.08d0,-11.68d0,&
        & -19.18d0,-12.28d0,-19.18d0,-13.08d0,-16.18d0,-12.08d0,-17.08d0,-11.69d0,&
        & -13.48d0,-13.78d0,-13.28d0,-12.28d0,-15.78d0,-14.58d0,-16.28d0,-10.88d0,&
        & -18.38d0,-11.48d0,-18.38d0,-12.28d0,-15.38d0,-11.28d0,-16.28d0,-10.89d0,&
        & -14.28d0,-14.58d0,-14.08d0,-13.08d0,-16.58d0,-15.38d0,-17.08d0,-11.68d0,&
        & -19.18d0,-12.28d0,-19.18d0,-13.08d0,-16.18d0,-12.08d0,-17.08d0,-11.69d0,&
        & -14.28d0,-14.58d0,-14.08d0,-13.08d0,-16.58d0,-15.38d0,-17.08d0,-11.68d0,&
        & -19.18d0,-12.28d0,-19.18d0,-13.08d0,-16.18d0,-12.08d0,-17.08d0,-11.69d0,&
        &  -9.18d0, -9.48d0, -8.98d0, -7.98d0,-11.48d0,-10.28d0,-11.98d0, -6.58d0,&
        & -14.08d0, -7.18d0,-14.08d0, -7.98d0,-11.08d0, -6.98d0,-11.98d0, -6.59d0,&
        &  -8.48d0, -8.78d0, -8.28d0, -7.28d0,-10.78d0, -9.58d0,-11.28d0, -5.88d0,&
        & -13.38d0, -6.48d0,-13.38d0, -7.28d0,-10.38d0, -6.28d0,-11.28d0, -5.89d0,&
        &  -9.18d0, -9.48d0, -8.98d0, -7.98d0,-11.48d0,-10.28d0,-11.98d0, -6.58d0,&
        & -14.08d0, -7.18d0,-14.08d0, -7.98d0,-11.08d0, -6.98d0,-11.98d0, -6.59d0,&
        & -10.68d0,-10.98d0,-10.48d0, -9.48d0,-12.98d0,-11.78d0,-13.48d0, -8.08d0,&
        & -15.58d0, -8.68d0,-15.58d0, -9.48d0,-12.58d0, -8.48d0,-13.48d0, -8.09d0,&
        & -13.38d0,-13.68d0,-13.18d0,-12.18d0,-15.68d0,-14.48d0,-16.18d0,-10.78d0,&
        & -18.28d0,-11.38d0,-18.28d0,-12.18d0,-15.28d0,-11.18d0,-16.18d0,-10.79d0,&
        & -13.38d0,-13.68d0,-13.18d0,-12.18d0,-15.68d0,-14.48d0,-16.18d0,-10.78d0,&
        & -18.28d0,-11.38d0,-18.28d0,-12.18d0,-15.28d0,-11.18d0,-16.18d0,-10.79d0,&
        & -13.38d0,-13.68d0,-13.18d0,-12.18d0,-15.68d0,-14.48d0,-16.18d0,-10.78d0,&
        & -18.28d0,-11.38d0,-18.28d0,-12.18d0,-15.28d0,-11.18d0,-16.18d0,-10.79d0,&
        & -13.38d0,-13.68d0,-13.18d0,-12.18d0,-15.68d0,-14.48d0,-16.18d0,-10.78d0,&
        & -18.28d0,-11.38d0,-18.28d0,-12.18d0,-15.28d0,-11.18d0,-16.18d0,-10.79d0,&
        & -12.08d0,-12.38d0,-11.88d0,-10.88d0,-14.38d0,-13.18d0,-14.88d0, -9.48d0,&
        & -16.98d0,-10.08d0,-16.98d0,-10.88d0,-13.98d0, -9.88d0,-14.88d0, -9.49d0,&
        & -11.38d0,-11.68d0,-11.18d0,-10.18d0,-13.68d0,-12.48d0,-14.18d0, -8.78d0,&
        & -16.28d0, -9.38d0,-16.28d0,-10.18d0,-13.28d0, -9.18d0,-14.18d0, -8.79d0,&
        & -12.08d0,-12.38d0,-11.88d0,-10.88d0,-14.38d0,-13.18d0,-14.88d0, -9.48d0,&
        & -16.98d0,-10.08d0,-16.98d0,-10.88d0,-13.98d0, -9.88d0,-14.88d0, -9.49d0,&
        & -12.58d0,-12.88d0,-12.38d0,-11.38d0,-14.88d0,-13.68d0,-15.38d0, -9.98d0,&
        & -17.48d0,-10.58d0,-17.48d0,-11.38d0,-14.48d0,-10.38d0,-15.38d0, -9.99d0,&
        & -15.48d0,-17.48d0,-17.38d0,-15.28d0,-19.08d0,-13.38d0,-17.68d0,-15.28d0,&
        & -15.88d0,-13.38d0,-16.48d0,-15.28d0,-19.08d0,-14.18d0,-17.68d0,-15.98d0,&
        & -14.68d0,-16.68d0,-16.58d0,-14.48d0,-18.28d0,-12.58d0,-16.88d0,-14.48d0,&
        & -15.08d0,-12.58d0,-15.68d0,-14.48d0,-18.28d0,-13.38d0,-16.88d0,-15.18d0,&
        & -15.48d0,-17.48d0,-17.38d0,-15.28d0,-19.08d0,-13.38d0,-17.68d0,-15.28d0,&
        & -15.88d0,-13.38d0,-16.48d0,-15.28d0,-19.08d0,-14.18d0,-17.68d0,-15.98d0,&
        & -15.48d0,-17.48d0,-17.38d0,-15.28d0,-19.08d0,-13.38d0,-17.68d0,-15.28d0,&
        & -15.88d0,-13.38d0,-16.48d0,-15.28d0,-19.08d0,-14.18d0,-17.68d0,-15.98d0,&
        & -10.38d0,-12.38d0,-12.28d0,-10.18d0,-13.98d0, -8.28d0,-12.58d0,-10.18d0,&
        & -10.78d0, -8.28d0,-11.38d0,-10.18d0,-13.98d0, -9.08d0,-12.58d0,-10.88d0,&
        &  -9.68d0,-11.68d0,-11.58d0, -9.48d0,-13.28d0, -7.58d0,-11.88d0, -9.48d0,&
        & -10.08d0, -7.58d0,-10.68d0, -9.48d0,-13.28d0, -8.38d0,-11.88d0,-10.18d0,&
        & -10.38d0,-12.38d0,-12.28d0,-10.18d0,-13.98d0, -8.28d0,-12.58d0,-10.18d0,&
        & -10.78d0, -8.28d0,-11.38d0,-10.18d0,-13.98d0, -9.08d0,-12.58d0,-10.88d0,&
        & -11.88d0,-13.88d0,-13.78d0,-11.68d0,-15.48d0, -9.78d0,-14.08d0,-11.68d0,&
        & -12.28d0, -9.78d0,-12.88d0,-11.68d0,-15.48d0,-10.58d0,-14.08d0,-12.38d0,&
        & -14.58d0,-16.58d0,-16.48d0,-14.38d0,-18.18d0,-12.48d0,-16.78d0,-14.38d0,&
        & -14.98d0,-12.48d0,-15.58d0,-14.38d0,-18.18d0,-13.28d0,-16.78d0,-15.08d0,&
        & -14.58d0,-16.58d0,-16.48d0,-14.38d0,-18.18d0,-12.48d0,-16.78d0,-14.38d0,&
        & -14.98d0,-12.48d0,-15.58d0,-14.38d0,-18.18d0,-13.28d0,-16.78d0,-15.08d0,&
        & -14.58d0,-16.58d0,-16.48d0,-14.38d0,-18.18d0,-12.48d0,-16.78d0,-14.38d0,&
        & -14.98d0,-12.48d0,-15.58d0,-14.38d0,-18.18d0,-13.28d0,-16.78d0,-15.08d0,&
        & -14.58d0,-16.58d0,-16.48d0,-14.38d0,-18.18d0,-12.48d0,-16.78d0,-14.38d0,&
        & -14.98d0,-12.48d0,-15.58d0,-14.38d0,-18.18d0,-13.28d0,-30.80d0,-15.08d0,&
        & -13.28d0,-15.28d0,-15.18d0,-13.08d0,-16.88d0,-11.18d0,-15.48d0,-13.08d0,&
        & -13.68d0,-11.18d0,-14.28d0,-13.08d0,-16.88d0,-11.98d0,-15.48d0,-13.78d0,&
        & -12.58d0,-14.58d0,-14.48d0,-12.38d0,-16.18d0,-10.48d0,-14.78d0,-12.38d0,&
        & -12.98d0,-10.48d0,-13.58d0,-12.38d0,-16.18d0,-11.28d0,-14.78d0,-13.08d0,&
        & -13.28d0,-15.28d0,-15.18d0,-13.08d0,-16.88d0,-11.18d0,-15.48d0,-13.08d0,&
        & -13.68d0,-11.18d0,-14.28d0,-13.08d0,-16.88d0,-11.98d0,-15.48d0,-13.78d0,&
        & -13.78d0,-15.78d0,-15.68d0,-13.58d0,-17.38d0,-11.68d0,-15.98d0,-13.58d0,&
        & -14.18d0,-11.68d0,-14.78d0,-13.58d0,-17.38d0,-12.48d0,-15.98d0,-14.28d0,&
        & -20.58d0,-15.48d0,-19.68d0,-18.38d0,-19.78d0,-14.78d0,-19.68d0,-17.68d0,&
        & -20.58d0,-15.48d0,-19.68d0,-18.38d0,-20.58d0,-16.98d0,-19.68d0,-18.88d0,&
        & -19.78d0,-14.68d0,-18.88d0,-17.58d0,-18.98d0,-13.98d0,-18.88d0,-16.88d0,&
        & -19.78d0,-14.68d0,-18.88d0,-17.58d0,-19.78d0,-16.18d0,-18.88d0,-18.08d0,&
        & -20.58d0,-15.48d0,-19.68d0,-18.38d0,-19.78d0,-14.78d0,-19.68d0,-17.68d0,&
        & -20.58d0,-15.48d0,-19.68d0,-18.38d0,-20.58d0,-16.98d0,-19.68d0,-18.88d0,&
        & -20.58d0,-15.48d0,-19.68d0,-18.38d0,-19.78d0,-14.78d0,-19.68d0,-17.68d0,&
        & -20.58d0,-15.48d0,-19.68d0,-18.38d0,-20.58d0,-16.98d0,-19.68d0,-18.88d0,&
        & -15.48d0,-10.38d0,-14.58d0,-13.28d0,-14.68d0, -9.68d0,-14.58d0,-12.58d0,&
        & -15.48d0,-10.38d0,-14.58d0,-13.28d0,-15.48d0,-11.88d0,-14.58d0,-13.78d0,&
        & -14.78d0, -9.68d0,-13.88d0,-12.58d0,-13.98d0, -8.98d0,-13.88d0,-11.88d0,&
        & -14.78d0, -9.68d0,-13.88d0,-12.58d0,-14.78d0,-11.18d0,-13.88d0,-13.08d0,&
        & -15.48d0,-10.38d0,-14.58d0,-13.28d0,-14.68d0, -9.68d0,-14.58d0,-12.58d0,&
        & -15.48d0,-10.38d0,-14.58d0,-13.28d0,-15.48d0,-11.88d0,-14.58d0,-13.78d0,&
        & -16.98d0,-11.88d0,-16.08d0,-14.78d0,-16.18d0,-11.18d0,-16.08d0,-14.08d0,&
        & -16.98d0,-11.88d0,-16.08d0,-14.78d0,-16.98d0,-13.38d0,-16.08d0,-15.28d0,&
        & -19.68d0,-14.58d0,-18.78d0,-17.48d0,-18.88d0,-13.88d0,-18.78d0,-16.78d0,&
        & -19.68d0,-14.58d0,-18.78d0,-17.48d0,-19.68d0,-16.08d0,-18.78d0,-17.98d0,&
        & -19.68d0,-14.58d0,-18.78d0,-17.48d0,-18.88d0,-13.88d0,-18.78d0,-16.78d0,&
        & -19.68d0,-14.58d0,-18.78d0,-17.48d0,-19.68d0,-16.08d0,-18.78d0,-17.98d0,&
        & -19.68d0,-14.58d0,-18.78d0,-17.48d0,-18.88d0,-13.88d0,-18.78d0,-16.78d0,&
        & -19.68d0,-14.58d0,-18.78d0,-17.48d0,-19.68d0,-16.08d0,-18.78d0,-17.98d0,&
        & -19.68d0,-14.58d0,-18.78d0,-17.48d0,-18.88d0,-13.88d0,-18.78d0,-16.78d0,&
        & -19.68d0,-14.58d0,-18.78d0,-17.48d0,-19.68d0,-16.08d0,-18.78d0,-17.98d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-17.58d0,-12.58d0,-17.48d0,-15.48d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-18.38d0,-14.78d0,-17.48d0,-16.68d0,&
        & -17.68d0,-12.58d0,-16.78d0,-15.48d0,-16.88d0,-11.88d0,-16.78d0,-14.78d0,&
        & -17.68d0,-12.58d0,-16.78d0,-15.48d0,-17.68d0,-14.08d0,-16.78d0,-15.98d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-17.58d0,-12.58d0,-17.48d0,-15.48d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-18.38d0,-14.78d0,-17.48d0,-16.68d0,&
        & -18.88d0,-13.78d0,-17.98d0,-16.68d0,-18.08d0,-13.08d0,-17.98d0,-15.98d0,&
        & -18.88d0,-13.78d0,-17.98d0,-16.68d0,-18.88d0,-15.28d0,-17.98d0,-17.18d0,&
        & -17.48d0,-15.08d0,-16.88d0,-15.78d0,-18.18d0,-15.08d0,-18.38d0,-14.68d0,&
        & -19.88d0,-13.88d0,-19.48d0,-15.78d0,-18.38d0,-15.08d0,-18.38d0,-13.88d0,&
        & -16.68d0,-14.28d0,-16.08d0,-14.98d0,-17.38d0,-14.28d0,-17.58d0,-13.88d0,&
        & -19.08d0,-13.08d0,-18.68d0,-14.98d0,-17.58d0,-14.28d0,-17.58d0,-13.08d0,&
        & -17.48d0,-15.08d0,-16.88d0,-15.78d0,-18.18d0,-15.08d0,-18.38d0,-14.68d0,&
        & -19.88d0,-13.88d0,-19.48d0,-15.78d0,-18.38d0,-15.08d0,-18.38d0,-13.88d0,&
        & -17.48d0,-15.08d0,-16.88d0,-15.78d0,-18.18d0,-15.08d0,-18.38d0,-14.68d0,&
        & -19.88d0,-13.88d0,-19.48d0,-15.78d0,-18.38d0,-15.08d0,-18.38d0,-13.88d0,&
        & -12.38d0, -9.98d0,-11.78d0,-10.68d0,-13.08d0, -9.98d0,-13.28d0, -9.58d0,&
        & -14.78d0, -8.78d0,-14.38d0,-10.68d0,-13.28d0, -9.98d0,-13.28d0, -8.78d0,&
        & -11.68d0, -9.28d0,-11.08d0, -9.98d0,-12.38d0, -9.28d0,-12.58d0, -8.88d0,&
        & -14.08d0, -8.08d0,-13.68d0, -9.98d0,-12.58d0, -9.28d0,-12.58d0, -8.08d0,&
        & -12.38d0, -9.98d0,-11.78d0,-10.68d0,-13.08d0, -9.98d0,-13.28d0, -9.58d0,&
        & -14.78d0, -8.78d0,-14.38d0,-10.68d0,-13.28d0, -9.98d0,-13.28d0, -8.78d0,&
        & -13.88d0,-11.48d0,-13.28d0,-12.18d0,-14.58d0,-11.48d0,-14.78d0,-11.08d0,&
        & -16.28d0,-10.28d0,-15.88d0,-12.18d0,-14.78d0,-11.48d0,-14.78d0,-10.28d0,&
        & -16.58d0,-14.18d0,-15.98d0,-14.88d0,-17.28d0,-14.18d0,-17.48d0,-13.78d0,&
        & -18.98d0,-12.98d0,-18.58d0,-14.88d0,-17.48d0,-14.18d0,-17.48d0,-12.98d0,&
        & -16.58d0,-14.18d0,-15.98d0,-14.88d0,-17.28d0,-14.18d0,-17.48d0,-13.78d0,&
        & -18.98d0,-12.98d0,-18.58d0,-14.88d0,-17.48d0,-14.18d0,-17.48d0,-12.98d0,&
        & -16.58d0,-14.18d0,-15.98d0,-14.88d0,-17.28d0,-14.18d0,-17.48d0,-13.78d0,&
        & -18.98d0,-12.98d0,-18.58d0,-14.88d0,-17.48d0,-14.18d0,-17.48d0,-12.98d0,&
        & -16.58d0,-14.18d0,-15.98d0,-14.88d0,-17.28d0,-14.18d0,-17.48d0,-13.78d0,&
        & -18.98d0,-12.98d0,-18.58d0,-14.88d0,-17.48d0,-14.18d0,-17.48d0,-12.98d0,&
        & -15.28d0,-12.88d0,-14.68d0,-13.58d0,-15.98d0,-12.88d0,-16.18d0,-12.48d0,&
        & -17.68d0,-11.68d0,-17.28d0,-13.58d0,-16.18d0,-12.88d0,-16.18d0,-11.68d0,&
        & -14.58d0,-12.18d0,-13.98d0,-12.88d0,-15.28d0,-12.18d0,-15.48d0,-11.78d0,&
        & -16.98d0,-10.98d0,-16.58d0,-12.88d0,-15.48d0,-12.18d0,-15.48d0,-10.98d0,&
        & -15.28d0,-12.88d0,-14.68d0,-13.58d0,-15.98d0,-12.88d0,-16.18d0,-12.48d0,&
        & -17.68d0,-11.68d0,-17.28d0,-13.58d0,-16.18d0,-12.88d0,-16.18d0,-11.68d0,&
        & -15.78d0,-13.38d0,-15.18d0,-14.08d0,-16.48d0,-13.38d0,-16.68d0,-12.98d0,&
        & -18.18d0,-12.18d0,-17.78d0,-14.08d0,-16.68d0,-13.38d0,-16.68d0,-12.18d0,&
        & -14.58d0,-12.88d0,-13.68d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -16.28d0,-12.68d0,-17.18d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -13.78d0,-12.08d0,-12.88d0,-12.78d0,-15.48d0,-11.88d0,-16.38d0,-12.78d0,&
        & -15.48d0,-11.88d0,-16.38d0,-12.78d0,-15.48d0,-11.88d0,-16.38d0,-12.78d0,&
        & -14.58d0,-12.88d0,-13.68d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -16.28d0,-12.68d0,-17.18d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -14.58d0,-12.88d0,-13.68d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -16.28d0,-12.68d0,-17.18d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        &  -9.48d0, -7.78d0, -8.58d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -11.18d0, -7.58d0,-12.08d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        &  -8.78d0, -7.08d0, -7.88d0, -7.78d0,-10.48d0, -6.88d0,-11.38d0, -7.78d0,&
        & -10.48d0, -6.88d0,-11.38d0, -7.78d0,-10.48d0, -6.88d0,-11.38d0, -7.78d0,&
        &  -9.48d0, -7.78d0, -8.58d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -11.18d0, -7.58d0,-12.08d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -10.98d0, -9.28d0,-10.08d0, -9.98d0,-12.68d0, -9.08d0,-13.58d0, -9.98d0,&
        & -12.68d0, -9.08d0,-13.58d0, -9.98d0,-12.68d0, -9.08d0,-13.58d0, -9.98d0,&
        & -13.68d0,-11.98d0,-12.78d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -15.38d0,-11.78d0,-16.28d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -13.68d0,-11.98d0,-12.78d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -15.38d0,-11.78d0,-16.28d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -13.68d0,-11.98d0,-12.78d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -15.38d0,-11.78d0,-16.28d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -13.68d0,-11.98d0,-12.78d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -15.38d0,-11.78d0,-16.28d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -11.68d0, -9.98d0,-10.78d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -13.38d0, -9.78d0,-14.28d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -12.88d0,-11.18d0,-11.98d0,-11.88d0,-14.58d0,-10.98d0,-15.48d0,-11.88d0,&
        & -14.58d0,-10.98d0,-15.48d0,-11.88d0,-14.58d0,-10.98d0,-15.48d0,-11.88d0,&
        & -14.58d0,-12.88d0,-13.68d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -16.28d0,-12.68d0,-17.18d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -13.78d0,-12.08d0,-12.88d0,-12.78d0,-15.48d0,-11.88d0,-16.38d0,-12.78d0,&
        & -15.48d0,-11.88d0,-16.38d0,-12.78d0,-15.48d0,-11.88d0,-16.38d0,-12.78d0,&
        & -14.58d0,-12.88d0,-13.68d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -16.28d0,-12.68d0,-17.18d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -14.58d0,-12.88d0,-13.68d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        & -16.28d0,-12.68d0,-17.18d0,-13.58d0,-16.28d0,-12.68d0,-17.18d0,-13.58d0,&
        &  -9.48d0, -7.78d0, -8.58d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -11.18d0, -7.58d0,-12.08d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        &  -8.78d0, -7.08d0, -7.88d0, -7.78d0,-10.48d0, -6.88d0,-11.38d0, -7.78d0,&
        & -10.48d0, -6.88d0,-11.38d0, -7.78d0,-10.48d0, -6.88d0,-11.38d0, -7.78d0,&
        &  -9.48d0, -7.78d0, -8.58d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -11.18d0, -7.58d0,-12.08d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -10.98d0, -9.28d0,-10.08d0, -9.98d0,-12.68d0, -9.08d0,-13.58d0, -9.98d0,&
        & -12.68d0, -9.08d0,-13.58d0, -9.98d0,-12.68d0, -9.08d0,-13.58d0, -9.98d0,&
        & -13.68d0,-11.98d0,-12.78d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -15.38d0,-11.78d0,-16.28d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -13.68d0,-11.98d0,-12.78d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -15.38d0,-11.78d0,-16.28d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -13.68d0,-11.98d0,-12.78d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -15.38d0,-11.78d0,-16.28d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -13.68d0,-11.98d0,-12.78d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -15.38d0,-11.78d0,-16.28d0,-12.68d0,-15.38d0,-11.78d0,-16.28d0,-12.68d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -11.68d0, -9.98d0,-10.78d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -13.38d0, -9.78d0,-14.28d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -12.88d0,-11.18d0,-11.98d0,-11.88d0,-14.58d0,-10.98d0,-15.48d0,-11.88d0,&
        & -14.58d0,-10.98d0,-15.48d0,-11.88d0,-14.58d0,-10.98d0,-15.48d0,-11.88d0,&
        &  -9.18d0, -9.48d0, -8.98d0, -7.98d0,-11.48d0,-10.28d0,-11.98d0, -6.58d0,&
        & -14.08d0, -7.18d0,-14.08d0, -7.98d0,-11.08d0, -6.98d0,-11.98d0, -6.59d0,&
        & -12.78d0,-13.08d0,-12.58d0,-11.58d0,-15.08d0,-13.88d0,-15.58d0,-10.18d0,&
        & -17.68d0,-10.78d0,-17.68d0,-11.58d0,-14.68d0,-10.58d0,-15.58d0,-10.19d0,&
        &  -9.58d0, -9.88d0, -9.38d0, -8.38d0,-11.88d0,-10.68d0,-12.38d0, -6.98d0,&
        & -14.48d0, -7.58d0,-14.48d0, -8.38d0,-11.48d0, -7.38d0,-12.38d0, -6.99d0,&
        & -12.78d0,-13.08d0,-12.58d0,-11.58d0,-15.08d0,-13.88d0,-15.58d0,-10.18d0,&
        & -17.68d0,-10.78d0,-17.68d0,-11.58d0,-14.68d0,-10.58d0,-15.58d0,-10.19d0,&
        & -11.18d0,-11.48d0,-10.98d0, -9.98d0,-13.48d0,-12.28d0,-13.98d0, -8.58d0,&
        & -16.08d0, -9.18d0,-16.08d0, -9.98d0,-13.08d0, -8.98d0,-13.98d0, -8.59d0,&
        &  -7.08d0, -7.38d0, -6.88d0, -5.88d0, -9.38d0, -8.18d0, -9.88d0, -4.48d0,&
        & -11.98d0, -5.08d0,-11.98d0, -5.88d0, -8.98d0, -4.88d0, -9.88d0, -4.49d0,&
        &  -7.08d0, -7.38d0, -6.88d0, -5.88d0, -9.38d0, -8.18d0, -9.88d0, -4.48d0,&
        & -11.98d0, -5.08d0,-11.98d0, -5.88d0, -8.98d0, -4.88d0, -9.88d0, -4.49d0,&
        &  -7.88d0, -8.18d0, -7.68d0, -6.68d0,-10.18d0, -8.98d0,-10.68d0, -5.28d0,&
        & -12.78d0, -5.88d0,-12.78d0, -6.68d0, -9.78d0, -5.68d0,-10.68d0, -5.29d0,&
        & -11.08d0,-11.38d0,-10.88d0, -9.88d0,-13.38d0,-12.18d0,-13.88d0, -8.48d0,&
        & -15.98d0, -9.08d0,-15.98d0, -9.88d0,-12.98d0, -8.88d0,-13.88d0, -8.49d0,&
        & -11.38d0,-11.68d0,-11.18d0,-10.18d0,-13.68d0,-12.48d0,-14.18d0, -8.78d0,&
        & -16.28d0, -9.38d0,-16.28d0,-10.18d0,-13.28d0, -9.18d0,-14.18d0, -8.79d0,&
        & -10.18d0,-10.48d0, -9.98d0, -8.98d0,-12.48d0,-11.28d0,-12.98d0, -7.58d0,&
        & -15.08d0, -8.18d0,-15.08d0, -8.98d0,-12.08d0, -7.98d0,-12.98d0, -7.59d0,&
        & -11.38d0,-11.68d0,-11.18d0,-10.18d0,-13.68d0,-12.48d0,-14.18d0, -8.78d0,&
        & -16.28d0, -9.38d0,-16.28d0,-10.18d0,-13.28d0, -9.18d0,-14.18d0, -8.79d0,&
        &  -8.98d0, -9.28d0, -8.78d0, -7.78d0,-11.28d0,-10.08d0,-11.78d0, -6.38d0,&
        & -13.88d0, -6.98d0,-13.88d0, -7.78d0,-10.88d0, -6.78d0,-11.78d0, -6.39d0,&
        &  -8.98d0, -9.28d0, -8.78d0, -7.78d0,-11.28d0,-10.08d0,-11.78d0, -6.38d0,&
        & -13.88d0, -6.98d0,-13.88d0, -7.78d0,-10.88d0, -6.78d0,-11.78d0, -6.39d0,&
        &  -8.98d0, -9.28d0, -8.78d0, -7.78d0,-11.28d0,-10.08d0,-11.78d0, -6.38d0,&
        & -13.88d0, -6.98d0,-13.88d0, -7.78d0,-10.88d0, -6.78d0,-11.78d0, -6.39d0,&
        &  -9.68d0, -9.98d0, -9.48d0, -8.48d0,-11.98d0,-10.78d0,-12.48d0, -7.08d0,&
        & -14.58d0, -7.68d0,-14.58d0, -8.48d0,-11.58d0, -7.48d0,-12.48d0, -7.09d0,&
        & -10.38d0,-12.38d0,-12.28d0,-10.18d0,-13.98d0, -8.28d0,-12.58d0,-10.18d0,&
        & -10.78d0, -8.28d0,-11.38d0,-10.18d0,-13.98d0, -9.08d0,-12.58d0,-10.88d0,&
        & -13.98d0,-15.98d0,-15.88d0,-13.78d0,-17.58d0,-11.88d0,-16.18d0,-13.78d0,&
        & -14.38d0,-11.88d0,-14.98d0,-13.78d0,-17.58d0,-12.68d0,-16.18d0,-14.48d0,&
        & -10.78d0,-12.78d0,-12.68d0,-10.58d0,-14.38d0, -8.68d0,-12.98d0,-10.58d0,&
        & -11.18d0, -8.68d0,-11.78d0,-10.58d0,-14.38d0, -9.48d0,-12.98d0,-11.28d0,&
        & -13.98d0,-15.98d0,-15.88d0,-13.78d0,-17.58d0,-11.88d0,-16.18d0,-13.78d0,&
        & -14.38d0,-11.88d0,-14.98d0,-13.78d0,-17.58d0,-12.68d0,-16.18d0,-14.48d0,&
        & -12.38d0,-14.38d0,-14.28d0,-12.18d0,-15.98d0,-10.28d0,-14.58d0,-12.18d0,&
        & -12.78d0,-10.28d0,-13.38d0,-12.18d0,-15.98d0,-11.08d0,-14.58d0,-12.88d0,&
        &  -8.28d0,-10.28d0,-10.18d0, -8.08d0,-11.88d0, -6.18d0,-10.48d0, -8.08d0,&
        &  -8.68d0, -6.18d0, -9.28d0, -8.08d0,-11.88d0, -6.98d0,-10.48d0, -8.78d0,&
        &  -8.28d0,-10.28d0,-10.18d0, -8.08d0,-11.88d0, -6.18d0,-10.48d0, -8.08d0,&
        &  -8.68d0, -6.18d0, -9.28d0, -8.08d0,-11.88d0, -6.98d0,-10.48d0, -8.78d0,&
        &  -9.08d0,-11.08d0,-10.98d0, -8.88d0,-12.68d0, -6.98d0,-11.28d0, -8.88d0,&
        &  -9.48d0, -6.98d0,-10.08d0, -8.88d0,-12.68d0, -7.78d0,-11.28d0, -9.58d0,&
        & -12.28d0,-14.28d0,-14.18d0,-12.08d0,-15.88d0,-10.18d0,-14.48d0,-12.08d0,&
        & -12.68d0,-10.18d0,-13.28d0,-12.08d0,-15.88d0,-10.98d0,-14.48d0,-12.78d0,&
        & -12.58d0,-14.58d0,-14.48d0,-12.38d0,-16.18d0,-10.48d0,-14.78d0,-12.38d0,&
        & -12.98d0,-10.48d0,-13.58d0,-12.38d0,-16.18d0,-11.28d0,-14.78d0,-13.08d0,&
        & -11.38d0,-13.38d0,-13.28d0,-11.18d0,-14.98d0, -9.28d0,-13.58d0,-11.18d0,&
        & -11.78d0, -9.28d0,-12.38d0,-11.18d0,-14.98d0,-10.08d0,-13.58d0,-11.88d0,&
        & -12.58d0,-14.58d0,-14.48d0,-12.38d0,-16.18d0,-10.48d0,-14.78d0,-12.38d0,&
        & -12.98d0,-10.48d0,-13.58d0,-12.38d0,-16.18d0,-11.28d0,-30.80d0,-13.08d0,&
        & -10.18d0,-12.18d0,-12.08d0, -9.98d0,-13.78d0, -8.08d0,-12.38d0, -9.98d0,&
        & -10.58d0, -8.08d0,-11.18d0, -9.98d0,-13.78d0, -8.88d0,-12.38d0,-10.68d0,&
        & -10.18d0,-12.18d0,-12.08d0, -9.98d0,-13.78d0, -8.08d0,-12.38d0, -9.98d0,&
        & -10.58d0, -8.08d0,-11.18d0, -9.98d0,-13.78d0, -8.88d0,-12.38d0,-10.68d0,&
        & -10.18d0,-12.18d0,-12.08d0, -9.98d0,-13.78d0, -8.08d0,-12.38d0, -9.98d0,&
        & -10.58d0, -8.08d0,-11.18d0, -9.98d0,-13.78d0, -8.88d0,-12.38d0,-10.68d0,&
        & -10.88d0,-12.88d0,-12.78d0,-10.68d0,-14.48d0, -8.78d0,-13.08d0,-10.68d0,&
        & -11.28d0, -8.78d0,-11.88d0,-10.68d0,-14.48d0, -9.58d0,-13.08d0,-11.38d0,&
        & -15.48d0,-10.38d0,-14.58d0,-13.28d0,-14.68d0, -9.68d0,-14.58d0,-12.58d0,&
        & -15.48d0,-10.38d0,-14.58d0,-13.28d0,-15.48d0,-11.88d0,-14.58d0,-13.78d0,&
        & -19.08d0,-13.98d0,-18.18d0,-16.88d0,-18.28d0,-13.28d0,-18.18d0,-16.18d0,&
        & -19.08d0,-13.98d0,-18.18d0,-16.88d0,-19.08d0,-15.48d0,-18.18d0,-17.38d0,&
        & -15.88d0,-10.78d0,-14.98d0,-13.68d0,-15.08d0,-10.08d0,-14.98d0,-12.98d0,&
        & -15.88d0,-10.78d0,-14.98d0,-13.68d0,-15.88d0,-12.28d0,-14.98d0,-14.18d0,&
        & -19.08d0,-13.98d0,-18.18d0,-16.88d0,-18.28d0,-13.28d0,-18.18d0,-16.18d0,&
        & -19.08d0,-13.98d0,-18.18d0,-16.88d0,-19.08d0,-15.48d0,-18.18d0,-17.38d0,&
        & -17.48d0,-12.38d0,-16.58d0,-15.28d0,-16.68d0,-11.68d0,-16.58d0,-14.58d0,&
        & -17.48d0,-12.38d0,-16.58d0,-15.28d0,-17.48d0,-13.88d0,-16.58d0,-15.78d0,&
        & -13.38d0, -8.28d0,-12.48d0,-11.18d0,-12.58d0, -7.58d0,-12.48d0,-10.48d0,&
        & -13.38d0, -8.28d0,-12.48d0,-11.18d0,-13.38d0, -9.78d0,-12.48d0,-11.68d0,&
        & -13.38d0, -8.28d0,-12.48d0,-11.18d0,-12.58d0, -7.58d0,-12.48d0,-10.48d0,&
        & -13.38d0, -8.28d0,-12.48d0,-11.18d0,-13.38d0, -9.78d0,-12.48d0,-11.68d0,&
        & -14.18d0, -9.08d0,-13.28d0,-11.98d0,-13.38d0, -8.38d0,-13.28d0,-11.28d0,&
        & -14.18d0, -9.08d0,-13.28d0,-11.98d0,-14.18d0,-10.58d0,-13.28d0,-12.48d0,&
        & -17.38d0,-12.28d0,-16.48d0,-15.18d0,-16.58d0,-11.58d0,-16.48d0,-14.48d0,&
        & -17.38d0,-12.28d0,-16.48d0,-15.18d0,-17.38d0,-13.78d0,-16.48d0,-15.68d0,&
        & -17.68d0,-12.58d0,-16.78d0,-15.48d0,-16.88d0,-11.88d0,-16.78d0,-14.78d0,&
        & -17.68d0,-12.58d0,-16.78d0,-15.48d0,-17.68d0,-14.08d0,-16.78d0,-15.98d0,&
        & -16.48d0,-11.38d0,-15.58d0,-14.28d0,-15.68d0,-10.68d0,-15.58d0,-13.58d0,&
        & -16.48d0,-11.38d0,-15.58d0,-14.28d0,-16.48d0,-12.88d0,-15.58d0,-14.78d0,&
        & -17.68d0,-12.58d0,-16.78d0,-15.48d0,-16.88d0,-11.88d0,-16.78d0,-14.78d0,&
        & -17.68d0,-12.58d0,-16.78d0,-15.48d0,-17.68d0,-14.08d0,-30.80d0,-15.98d0,&
        & -15.28d0,-10.18d0,-14.38d0,-13.08d0,-14.48d0, -9.48d0,-14.38d0,-12.38d0,&
        & -15.28d0,-10.18d0,-14.38d0,-13.08d0,-15.28d0,-11.68d0,-14.38d0,-13.58d0,&
        & -15.28d0,-10.18d0,-14.38d0,-13.08d0,-14.48d0, -9.48d0,-14.38d0,-12.38d0,&
        & -15.28d0,-10.18d0,-14.38d0,-13.08d0,-15.28d0,-11.68d0,-14.38d0,-13.58d0,&
        & -15.28d0,-10.18d0,-14.38d0,-13.08d0,-14.48d0, -9.48d0,-14.38d0,-12.38d0,&
        & -15.28d0,-10.18d0,-14.38d0,-13.08d0,-15.28d0,-11.68d0,-14.38d0,-13.58d0,&
        & -15.98d0,-10.88d0,-15.08d0,-13.78d0,-15.18d0,-10.18d0,-15.08d0,-13.08d0,&
        & -15.98d0,-10.88d0,-15.08d0,-13.78d0,-15.98d0,-12.38d0,-15.08d0,-14.28d0,&
        & -12.38d0, -9.98d0,-11.78d0,-10.68d0,-13.08d0, -9.98d0,-13.28d0, -9.58d0,&
        & -14.78d0, -8.78d0,-14.38d0,-10.68d0,-13.28d0, -9.98d0,-13.28d0, -8.78d0,&
        & -15.98d0,-13.58d0,-15.38d0,-14.28d0,-16.68d0,-13.58d0,-16.88d0,-13.18d0,&
        & -18.38d0,-12.38d0,-17.98d0,-14.28d0,-16.88d0,-13.58d0,-16.88d0,-12.38d0,&
        & -12.78d0,-10.38d0,-12.18d0,-11.08d0,-13.48d0,-10.38d0,-13.68d0, -9.98d0,&
        & -15.18d0, -9.18d0,-14.78d0,-11.08d0,-13.68d0,-10.38d0,-13.68d0, -9.18d0,&
        & -15.98d0,-13.58d0,-15.38d0,-14.28d0,-16.68d0,-13.58d0,-16.88d0,-13.18d0,&
        & -18.38d0,-12.38d0,-17.98d0,-14.28d0,-16.88d0,-13.58d0,-16.88d0,-12.38d0,&
        & -14.38d0,-11.98d0,-13.78d0,-12.68d0,-15.08d0,-11.98d0,-15.28d0,-11.58d0,&
        & -16.78d0,-10.78d0,-16.38d0,-12.68d0,-15.28d0,-11.98d0,-15.28d0,-10.78d0,&
        & -10.28d0, -7.88d0, -9.68d0, -8.58d0,-10.98d0, -7.88d0,-11.18d0, -7.48d0,&
        & -12.68d0, -6.68d0,-12.28d0, -8.58d0,-11.18d0, -7.88d0,-11.18d0, -6.68d0,&
        & -10.28d0, -7.88d0, -9.68d0, -8.58d0,-10.98d0, -7.88d0,-11.18d0, -7.48d0,&
        & -12.68d0, -6.68d0,-12.28d0, -8.58d0,-11.18d0, -7.88d0,-11.18d0, -6.68d0,&
        & -11.08d0, -8.68d0,-10.48d0, -9.38d0,-11.78d0, -8.68d0,-11.98d0, -8.28d0,&
        & -13.48d0, -7.48d0,-13.08d0, -9.38d0,-11.98d0, -8.68d0,-11.98d0, -7.48d0,&
        & -14.28d0,-11.88d0,-13.68d0,-12.58d0,-14.98d0,-11.88d0,-15.18d0,-11.48d0,&
        & -16.68d0,-10.68d0,-16.28d0,-12.58d0,-15.18d0,-11.88d0,-15.18d0,-10.68d0,&
        & -14.58d0,-12.18d0,-13.98d0,-12.88d0,-15.28d0,-12.18d0,-15.48d0,-11.78d0,&
        & -16.98d0,-10.98d0,-16.58d0,-12.88d0,-15.48d0,-12.18d0,-15.48d0,-10.98d0,&
        & -13.38d0,-10.98d0,-12.78d0,-11.68d0,-14.08d0,-10.98d0,-14.28d0,-10.58d0,&
        & -15.78d0, -9.78d0,-15.38d0,-11.68d0,-14.28d0,-10.98d0,-14.28d0, -9.78d0,&
        & -14.58d0,-12.18d0,-13.98d0,-12.88d0,-15.28d0,-12.18d0,-15.48d0,-11.78d0,&
        & -16.98d0,-10.98d0,-16.58d0,-12.88d0,-15.48d0,-12.18d0,-15.48d0,-10.98d0,&
        & -12.18d0, -9.78d0,-11.58d0,-10.48d0,-12.88d0, -9.78d0,-13.08d0, -9.38d0,&
        & -14.58d0, -8.58d0,-14.18d0,-10.48d0,-13.08d0, -9.78d0,-13.08d0, -8.58d0,&
        & -12.18d0, -9.78d0,-11.58d0,-10.48d0,-12.88d0, -9.78d0,-13.08d0, -9.38d0,&
        & -14.58d0, -8.58d0,-14.18d0,-10.48d0,-13.08d0, -9.78d0,-13.08d0, -8.58d0,&
        & -12.18d0, -9.78d0,-11.58d0,-10.48d0,-12.88d0, -9.78d0,-13.08d0, -9.38d0,&
        & -14.58d0, -8.58d0,-14.18d0,-10.48d0,-13.08d0, -9.78d0,-13.08d0, -8.58d0,&
        & -12.88d0,-10.48d0,-12.28d0,-11.18d0,-13.58d0,-10.48d0,-13.78d0,-10.08d0,&
        & -15.28d0, -9.28d0,-14.88d0,-11.18d0,-13.78d0,-10.48d0,-13.78d0, -9.28d0,&
        &  -9.48d0, -7.78d0, -8.58d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -11.18d0, -7.58d0,-12.08d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -13.08d0,-11.38d0,-12.18d0,-12.08d0,-14.78d0,-11.18d0,-15.68d0,-12.08d0,&
        & -14.78d0,-11.18d0,-15.68d0,-12.08d0,-14.78d0,-11.18d0,-15.68d0,-12.08d0,&
        &  -9.88d0, -8.18d0, -8.98d0, -8.88d0,-11.58d0, -7.98d0,-12.48d0, -8.88d0,&
        & -11.58d0, -7.98d0,-12.48d0, -8.88d0,-11.58d0, -7.98d0,-12.48d0, -8.88d0,&
        & -13.08d0,-11.38d0,-12.18d0,-12.08d0,-14.78d0,-11.18d0,-15.68d0,-12.08d0,&
        & -14.78d0,-11.18d0,-15.68d0,-12.08d0,-14.78d0,-11.18d0,-15.68d0,-12.08d0,&
        & -11.48d0, -9.78d0,-10.58d0,-10.48d0,-13.18d0, -9.58d0,-14.08d0,-10.48d0,&
        & -13.18d0, -9.58d0,-14.08d0,-10.48d0,-13.18d0, -9.58d0,-14.08d0,-10.48d0,&
        &  -7.38d0, -5.68d0, -6.48d0, -6.38d0, -9.08d0, -5.48d0, -9.98d0, -6.38d0,&
        &  -9.08d0, -5.48d0, -9.98d0, -6.38d0, -9.08d0, -5.48d0, -9.98d0, -6.38d0,&
        &  -7.38d0, -5.68d0, -6.48d0, -6.38d0, -9.08d0, -5.48d0, -9.98d0, -6.38d0,&
        &  -9.08d0, -5.48d0, -9.98d0, -6.38d0, -9.08d0, -5.48d0, -9.98d0, -6.38d0,&
        &  -8.18d0, -6.48d0, -7.28d0, -7.18d0, -9.88d0, -6.28d0,-10.78d0, -7.18d0,&
        &  -9.88d0, -6.28d0,-10.78d0, -7.18d0, -9.88d0, -6.28d0,-10.78d0, -7.18d0,&
        & -11.38d0, -9.68d0,-10.48d0,-10.38d0,-13.08d0, -9.48d0,-13.98d0,-10.38d0,&
        & -13.08d0, -9.48d0,-13.98d0,-10.38d0,-13.08d0, -9.48d0,-13.98d0,-10.38d0,&
        & -11.68d0, -9.98d0,-10.78d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -13.38d0, -9.78d0,-14.28d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -10.48d0, -8.78d0, -9.58d0, -9.48d0,-12.18d0, -8.58d0,-13.08d0, -9.48d0,&
        & -12.18d0, -8.58d0,-13.08d0, -9.48d0,-12.18d0, -8.58d0,-13.08d0, -9.48d0,&
        & -11.68d0, -9.98d0,-10.78d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -13.38d0, -9.78d0,-14.28d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        &  -9.28d0, -7.58d0, -8.38d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        & -10.98d0, -7.38d0,-11.88d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        &  -9.28d0, -7.58d0, -8.38d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        & -10.98d0, -7.38d0,-11.88d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        &  -9.28d0, -7.58d0, -8.38d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        & -10.98d0, -7.38d0,-11.88d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        &  -9.98d0, -8.28d0, -9.08d0, -8.98d0,-11.68d0, -8.08d0,-12.58d0, -8.98d0,&
        & -11.68d0, -8.08d0,-12.58d0, -8.98d0,-11.68d0, -8.08d0,-12.58d0, -8.98d0,&
        &  -9.48d0, -7.78d0, -8.58d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -11.18d0, -7.58d0,-12.08d0, -8.48d0,-11.18d0, -7.58d0,-12.08d0, -8.48d0,&
        & -13.08d0,-11.38d0,-12.18d0,-12.08d0,-14.78d0,-11.18d0,-15.68d0,-12.08d0,&
        & -14.78d0,-11.18d0,-15.68d0,-12.08d0,-14.78d0,-11.18d0,-15.68d0,-12.08d0,&
        &  -9.88d0, -8.18d0, -8.98d0, -8.88d0,-11.58d0, -7.98d0,-12.48d0, -8.88d0,&
        & -11.58d0, -7.98d0,-12.48d0, -8.88d0,-11.58d0, -7.98d0,-12.48d0, -8.88d0,&
        & -13.08d0,-11.38d0,-12.18d0,-12.08d0,-14.78d0,-11.18d0,-15.68d0,-12.08d0,&
        & -14.78d0,-11.18d0,-15.68d0,-12.08d0,-14.78d0,-11.18d0,-15.68d0,-12.08d0,&
        & -11.48d0, -9.78d0,-10.58d0,-10.48d0,-13.18d0, -9.58d0,-14.08d0,-10.48d0,&
        & -13.18d0, -9.58d0,-14.08d0,-10.48d0,-13.18d0, -9.58d0,-14.08d0,-10.48d0,&
        &  -7.38d0, -5.68d0, -6.48d0, -6.38d0, -9.08d0, -5.48d0, -9.98d0, -6.38d0,&
        &  -9.08d0, -5.48d0, -9.98d0, -6.38d0, -9.08d0, -5.48d0, -9.98d0, -6.38d0,&
        &  -7.38d0, -5.68d0, -6.48d0, -6.38d0, -9.08d0, -5.48d0, -9.98d0, -6.38d0,&
        &  -9.08d0, -5.48d0, -9.98d0, -6.38d0, -9.08d0, -5.48d0, -9.98d0, -6.38d0,&
        &  -8.18d0, -6.48d0, -7.28d0, -7.18d0, -9.88d0, -6.28d0,-10.78d0, -7.18d0,&
        &  -9.88d0, -6.28d0,-10.78d0, -7.18d0, -9.88d0, -6.28d0,-10.78d0, -7.18d0,&
        & -11.38d0, -9.68d0,-10.48d0,-10.38d0,-13.08d0, -9.48d0,-13.98d0,-10.38d0,&
        & -13.08d0, -9.48d0,-13.98d0,-10.38d0,-13.08d0, -9.48d0,-13.98d0,-10.38d0,&
        & -11.68d0, -9.98d0,-10.78d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -13.38d0, -9.78d0,-14.28d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -10.48d0, -8.78d0, -9.58d0, -9.48d0,-12.18d0, -8.58d0,-13.08d0, -9.48d0,&
        & -12.18d0, -8.58d0,-13.08d0, -9.48d0,-12.18d0, -8.58d0,-13.08d0, -9.48d0,&
        & -11.68d0, -9.98d0,-10.78d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        & -13.38d0, -9.78d0,-14.28d0,-10.68d0,-13.38d0, -9.78d0,-14.28d0,-10.68d0,&
        &  -9.28d0, -7.58d0, -8.38d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        & -10.98d0, -7.38d0,-11.88d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        &  -9.28d0, -7.58d0, -8.38d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        & -10.98d0, -7.38d0,-11.88d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        &  -9.28d0, -7.58d0, -8.38d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        & -10.98d0, -7.38d0,-11.88d0, -8.28d0,-10.98d0, -7.38d0,-11.88d0, -8.28d0,&
        &  -9.98d0, -8.28d0, -9.08d0, -8.98d0,-11.68d0, -8.08d0,-12.58d0, -8.98d0,&
        & -11.68d0, -8.08d0,-12.58d0, -8.98d0,-11.68d0, -8.08d0,-12.58d0, -8.98d0,&
        &  -8.28d0, -8.58d0, -8.08d0, -7.08d0,-10.58d0, -9.38d0,-11.08d0, -5.68d0,&
        & -13.18d0, -6.28d0,-13.18d0, -7.08d0,-10.18d0, -6.08d0,-11.08d0, -5.69d0,&
        &  -9.98d0,-10.28d0, -9.78d0, -8.78d0,-12.28d0,-11.08d0,-12.78d0, -7.38d0,&
        & -14.88d0, -7.98d0,-14.88d0, -8.78d0,-11.88d0, -7.78d0,-12.78d0, -7.39d0,&
        &  -9.98d0,-10.28d0, -9.78d0, -8.78d0,-12.28d0,-11.08d0,-12.78d0, -7.38d0,&
        & -14.88d0, -7.98d0,-14.88d0, -8.78d0,-11.88d0, -7.78d0,-12.78d0, -7.39d0,&
        &  -9.98d0,-10.28d0, -9.78d0, -8.78d0,-12.28d0,-11.08d0,-12.78d0, -7.38d0,&
        & -14.88d0, -7.98d0,-14.88d0, -8.78d0,-11.88d0, -7.78d0,-12.78d0, -7.39d0,&
        &  -6.58d0, -6.88d0, -6.38d0, -5.38d0, -8.88d0, -7.68d0, -9.38d0, -3.98d0,&
        & -11.48d0, -4.58d0,-11.48d0, -5.38d0, -8.48d0, -4.38d0, -9.38d0, -3.99d0,&
        &  -6.38d0, -6.68d0, -6.18d0, -5.18d0, -8.68d0, -7.48d0, -9.18d0, -3.78d0,&
        & -11.28d0, -4.38d0,-11.28d0, -5.18d0, -8.28d0, -4.18d0, -9.18d0, -3.79d0,&
        &  -6.38d0, -6.68d0, -6.18d0, -5.18d0, -8.68d0, -7.48d0, -9.18d0, -3.78d0,&
        & -11.28d0, -4.38d0,-11.28d0, -5.18d0, -8.28d0, -4.18d0, -9.18d0, -3.79d0,&
        &  -6.38d0, -6.68d0, -6.18d0, -5.18d0, -8.68d0, -7.48d0, -9.18d0, -3.78d0,&
        & -11.28d0, -4.38d0,-11.28d0, -5.18d0, -8.28d0, -4.18d0, -9.18d0, -3.79d0,&
        &  -7.38d0, -7.68d0, -7.18d0, -6.18d0, -9.68d0, -8.48d0,-10.18d0, -4.78d0,&
        & -12.28d0, -5.38d0,-12.28d0, -6.18d0, -9.28d0, -5.18d0,-10.18d0, -4.79d0,&
        & -10.88d0,-11.18d0,-10.68d0, -9.68d0,-13.18d0,-11.98d0,-13.68d0, -8.28d0,&
        & -15.78d0, -8.88d0,-15.78d0, -9.68d0,-12.78d0, -8.68d0,-13.68d0, -8.29d0,&
        & -10.88d0,-11.18d0,-10.68d0, -9.68d0,-13.18d0,-11.98d0,-13.68d0, -8.28d0,&
        & -15.78d0, -8.88d0,-15.78d0, -9.68d0,-12.78d0, -8.68d0,-13.68d0, -8.29d0,&
        & -10.88d0,-11.18d0,-10.68d0, -9.68d0,-13.18d0,-11.98d0,-13.68d0, -8.28d0,&
        & -15.78d0, -8.88d0,-15.78d0, -9.68d0,-12.78d0, -8.68d0,-13.68d0, -8.29d0,&
        &  -7.28d0, -7.58d0, -7.08d0, -6.08d0, -9.58d0, -8.38d0,-10.08d0, -4.68d0,&
        & -12.18d0, -5.28d0,-12.18d0, -6.08d0, -9.18d0, -5.08d0,-10.08d0, -4.69d0,&
        &  -7.28d0, -7.58d0, -7.08d0, -6.08d0, -9.58d0, -8.38d0,-10.08d0, -4.68d0,&
        & -12.18d0, -5.28d0,-12.18d0, -6.08d0, -9.18d0, -5.08d0,-10.08d0, -4.69d0,&
        &  -7.28d0, -7.58d0, -7.08d0, -6.08d0, -9.58d0, -8.38d0,-10.08d0, -4.68d0,&
        & -12.18d0, -5.28d0,-12.18d0, -6.08d0, -9.18d0, -5.08d0,-10.08d0, -4.69d0,&
        &  -7.28d0, -7.58d0, -7.08d0, -6.08d0, -9.58d0, -8.38d0,-10.08d0, -4.68d0,&
        & -12.18d0, -5.28d0,-12.18d0, -6.08d0, -9.18d0, -5.08d0,-10.08d0, -4.69d0,&
        &  -9.48d0,-11.48d0,-11.38d0, -9.28d0,-13.08d0, -7.38d0,-11.68d0, -9.28d0,&
        &  -9.88d0, -7.38d0,-10.48d0, -9.28d0,-13.08d0, -8.18d0,-11.68d0, -9.98d0,&
        & -11.18d0,-13.18d0,-13.08d0,-10.98d0,-14.78d0, -9.08d0,-13.38d0,-10.98d0,&
        & -11.58d0, -9.08d0,-12.18d0,-10.98d0,-14.78d0, -9.88d0,-13.38d0,-11.68d0,&
        & -11.18d0,-13.18d0,-13.08d0,-10.98d0,-14.78d0, -9.08d0,-13.38d0,-10.98d0,&
        & -11.58d0, -9.08d0,-12.18d0,-10.98d0,-14.78d0, -9.88d0,-13.38d0,-11.68d0,&
        & -11.18d0,-13.18d0,-13.08d0,-10.98d0,-14.78d0, -9.08d0,-13.38d0,-10.98d0,&
        & -11.58d0, -9.08d0,-12.18d0,-10.98d0,-14.78d0, -9.88d0,-13.38d0,-11.68d0,&
        &  -7.78d0, -9.78d0, -9.68d0, -7.58d0,-11.38d0, -5.68d0, -9.98d0, -7.58d0,&
        &  -8.18d0, -5.68d0, -8.78d0, -7.58d0,-11.38d0, -6.48d0, -9.98d0, -8.28d0,&
        &  -7.58d0, -9.58d0, -9.48d0, -7.38d0,-11.18d0, -5.48d0, -9.78d0, -7.38d0,&
        &  -7.98d0, -5.48d0, -8.58d0, -7.38d0,-11.18d0, -6.28d0, -9.78d0, -8.08d0,&
        &  -7.58d0, -9.58d0, -9.48d0, -7.38d0,-11.18d0, -5.48d0, -9.78d0, -7.38d0,&
        &  -7.98d0, -5.48d0, -8.58d0, -7.38d0,-11.18d0, -6.28d0, -9.78d0, -8.08d0,&
        &  -7.58d0, -9.58d0, -9.48d0, -7.38d0,-11.18d0, -5.48d0, -9.78d0, -7.38d0,&
        &  -7.98d0, -5.48d0, -8.58d0, -7.38d0,-11.18d0, -6.28d0, -9.78d0, -8.08d0,&
        &  -8.58d0,-10.58d0,-10.48d0, -8.38d0,-12.18d0, -6.48d0,-10.78d0, -8.38d0,&
        &  -8.98d0, -6.48d0, -9.58d0, -8.38d0,-12.18d0, -7.28d0,-10.78d0, -9.08d0,&
        & -12.08d0,-14.08d0,-13.98d0,-11.88d0,-15.68d0, -9.98d0,-14.28d0,-11.88d0,&
        & -12.48d0, -9.98d0,-13.08d0,-11.88d0,-15.68d0,-10.78d0,-14.28d0,-12.58d0,&
        & -12.08d0,-14.08d0,-13.98d0,-11.88d0,-15.68d0, -9.98d0,-14.28d0,-11.88d0,&
        & -12.48d0, -9.98d0,-13.08d0,-11.88d0,-15.68d0,-10.78d0,-14.28d0,-12.58d0,&
        & -12.08d0,-14.08d0,-13.98d0,-11.88d0,-15.68d0, -9.98d0,-14.28d0,-11.88d0,&
        & -12.48d0, -9.98d0,-13.08d0,-11.88d0,-15.68d0,-10.78d0,-14.28d0,-12.58d0,&
        &  -8.48d0,-10.48d0,-10.38d0, -8.28d0,-12.08d0, -6.38d0,-10.68d0, -8.28d0,&
        &  -8.88d0, -6.38d0, -9.48d0, -8.28d0,-12.08d0, -7.18d0,-10.68d0, -8.98d0,&
        &  -8.48d0,-10.48d0,-10.38d0, -8.28d0,-12.08d0, -6.38d0,-10.68d0, -8.28d0,&
        &  -8.88d0, -6.38d0, -9.48d0, -8.28d0,-12.08d0, -7.18d0,-10.68d0, -8.98d0,&
        &  -8.48d0,-10.48d0,-10.38d0, -8.28d0,-12.08d0, -6.38d0,-10.68d0, -8.28d0,&
        &  -8.88d0, -6.38d0, -9.48d0, -8.28d0,-12.08d0, -7.18d0,-10.68d0, -8.98d0,&
        &  -8.48d0,-10.48d0,-10.38d0, -8.28d0,-12.08d0, -6.38d0,-10.68d0, -8.28d0,&
        &  -8.88d0, -6.38d0, -9.48d0, -8.28d0,-12.08d0, -7.18d0,-10.68d0, -8.98d0,&
        & -14.58d0, -9.48d0,-13.68d0,-12.38d0,-13.78d0, -8.78d0,-13.68d0,-11.68d0,&
        & -14.58d0, -9.48d0,-13.68d0,-12.38d0,-14.58d0,-10.98d0,-13.68d0,-12.88d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-15.48d0,-10.48d0,-15.38d0,-13.38d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-16.28d0,-12.68d0,-15.38d0,-14.58d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-15.48d0,-10.48d0,-15.38d0,-13.38d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-16.28d0,-12.68d0,-15.38d0,-14.58d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-15.48d0,-10.48d0,-15.38d0,-13.38d0,&
        & -16.28d0,-11.18d0,-15.38d0,-14.08d0,-16.28d0,-12.68d0,-15.38d0,-14.58d0,&
        & -12.88d0, -7.78d0,-11.98d0,-10.68d0,-12.08d0, -7.08d0,-11.98d0, -9.98d0,&
        & -12.88d0, -7.78d0,-11.98d0,-10.68d0,-12.88d0, -9.28d0,-11.98d0,-11.18d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-11.88d0, -6.88d0,-11.78d0, -9.78d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-12.68d0, -9.08d0,-11.78d0,-10.98d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-11.88d0, -6.88d0,-11.78d0, -9.78d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-12.68d0, -9.08d0,-11.78d0,-10.98d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-11.88d0, -6.88d0,-11.78d0, -9.78d0,&
        & -12.68d0, -7.58d0,-11.78d0,-10.48d0,-12.68d0, -9.08d0,-11.78d0,-10.98d0,&
        & -13.68d0, -8.58d0,-12.78d0,-11.48d0,-12.88d0, -7.88d0,-12.78d0,-10.78d0,&
        & -13.68d0, -8.58d0,-12.78d0,-11.48d0,-13.68d0,-10.08d0,-12.78d0,-11.98d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-16.38d0,-11.38d0,-16.28d0,-14.28d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-17.18d0,-13.58d0,-16.28d0,-15.48d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-16.38d0,-11.38d0,-16.28d0,-14.28d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-17.18d0,-13.58d0,-16.28d0,-15.48d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-16.38d0,-11.38d0,-16.28d0,-14.28d0,&
        & -17.18d0,-12.08d0,-16.28d0,-14.98d0,-17.18d0,-13.58d0,-16.28d0,-15.48d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-12.78d0, -7.78d0,-12.68d0,-10.68d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-13.58d0, -9.98d0,-12.68d0,-11.88d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-12.78d0, -7.78d0,-12.68d0,-10.68d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-13.58d0, -9.98d0,-12.68d0,-11.88d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-12.78d0, -7.78d0,-12.68d0,-10.68d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-13.58d0, -9.98d0,-12.68d0,-11.88d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-12.78d0, -7.78d0,-12.68d0,-10.68d0,&
        & -13.58d0, -8.48d0,-12.68d0,-11.38d0,-13.58d0, -9.98d0,-12.68d0,-11.88d0,&
        & -11.48d0, -9.08d0,-10.88d0, -9.78d0,-12.18d0, -9.08d0,-12.38d0, -8.68d0,&
        & -13.88d0, -7.88d0,-13.48d0, -9.78d0,-12.38d0, -9.08d0,-12.38d0, -7.88d0,&
        & -13.18d0,-10.78d0,-12.58d0,-11.48d0,-13.88d0,-10.78d0,-14.08d0,-10.38d0,&
        & -15.58d0, -9.58d0,-15.18d0,-11.48d0,-14.08d0,-10.78d0,-14.08d0, -9.58d0,&
        & -13.18d0,-10.78d0,-12.58d0,-11.48d0,-13.88d0,-10.78d0,-14.08d0,-10.38d0,&
        & -15.58d0, -9.58d0,-15.18d0,-11.48d0,-14.08d0,-10.78d0,-14.08d0, -9.58d0,&
        & -13.18d0,-10.78d0,-12.58d0,-11.48d0,-13.88d0,-10.78d0,-14.08d0,-10.38d0,&
        & -15.58d0, -9.58d0,-15.18d0,-11.48d0,-14.08d0,-10.78d0,-14.08d0, -9.58d0,&
        &  -9.78d0, -7.38d0, -9.18d0, -8.08d0,-10.48d0, -7.38d0,-10.68d0, -6.98d0,&
        & -12.18d0, -6.18d0,-11.78d0, -8.08d0,-10.68d0, -7.38d0,-10.68d0, -6.18d0,&
        &  -9.58d0, -7.18d0, -8.98d0, -7.88d0,-10.28d0, -7.18d0,-10.48d0, -6.78d0,&
        & -11.98d0, -5.98d0,-11.58d0, -7.88d0,-10.48d0, -7.18d0,-10.48d0, -5.98d0,&
        &  -9.58d0, -7.18d0, -8.98d0, -7.88d0,-10.28d0, -7.18d0,-10.48d0, -6.78d0,&
        & -11.98d0, -5.98d0,-11.58d0, -7.88d0,-10.48d0, -7.18d0,-10.48d0, -5.98d0,&
        &  -9.58d0, -7.18d0, -8.98d0, -7.88d0,-10.28d0, -7.18d0,-10.48d0, -6.78d0,&
        & -11.98d0, -5.98d0,-11.58d0, -7.88d0,-10.48d0, -7.18d0,-10.48d0, -5.98d0,&
        & -10.58d0, -8.18d0, -9.98d0, -8.88d0,-11.28d0, -8.18d0,-11.48d0, -7.78d0,&
        & -12.98d0, -6.98d0,-12.58d0, -8.88d0,-11.48d0, -8.18d0,-11.48d0, -6.98d0,&
        & -14.08d0,-11.68d0,-13.48d0,-12.38d0,-14.78d0,-11.68d0,-14.98d0,-11.28d0,&
        & -16.48d0,-10.48d0,-16.08d0,-12.38d0,-14.98d0,-11.68d0,-14.98d0,-10.48d0,&
        & -14.08d0,-11.68d0,-13.48d0,-12.38d0,-14.78d0,-11.68d0,-14.98d0,-11.28d0,&
        & -16.48d0,-10.48d0,-16.08d0,-12.38d0,-14.98d0,-11.68d0,-14.98d0,-10.48d0,&
        & -14.08d0,-11.68d0,-13.48d0,-12.38d0,-14.78d0,-11.68d0,-14.98d0,-11.28d0,&
        & -16.48d0,-10.48d0,-16.08d0,-12.38d0,-14.98d0,-11.68d0,-14.98d0,-10.48d0,&
        & -10.48d0, -8.08d0, -9.88d0, -8.78d0,-11.18d0, -8.08d0,-11.38d0, -7.68d0,&
        & -12.88d0, -6.88d0,-12.48d0, -8.78d0,-11.38d0, -8.08d0,-11.38d0, -6.88d0,&
        & -10.48d0, -8.08d0, -9.88d0, -8.78d0,-11.18d0, -8.08d0,-11.38d0, -7.68d0,&
        & -12.88d0, -6.88d0,-12.48d0, -8.78d0,-11.38d0, -8.08d0,-11.38d0, -6.88d0,&
        & -10.48d0, -8.08d0, -9.88d0, -8.78d0,-11.18d0, -8.08d0,-11.38d0, -7.68d0,&
        & -12.88d0, -6.88d0,-12.48d0, -8.78d0,-11.38d0, -8.08d0,-11.38d0, -6.88d0,&
        & -10.48d0, -8.08d0, -9.88d0, -8.78d0,-11.18d0, -8.08d0,-11.38d0, -7.68d0,&
        & -12.88d0, -6.88d0,-12.48d0, -8.78d0,-11.38d0, -8.08d0,-11.38d0, -6.88d0,&
        &  -8.58d0, -6.88d0, -7.68d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -6.68d0,-11.18d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        &  -6.88d0, -5.18d0, -5.98d0, -5.88d0, -8.58d0, -4.98d0, -9.48d0, -5.88d0,&
        &  -8.58d0, -4.98d0, -9.48d0, -5.88d0, -8.58d0, -4.98d0, -9.48d0, -5.88d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -7.68d0, -5.98d0, -6.78d0, -6.68d0, -9.38d0, -5.78d0,-10.28d0, -6.68d0,&
        &  -9.38d0, -5.78d0,-10.28d0, -6.68d0, -9.38d0, -5.78d0,-10.28d0, -6.68d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -8.58d0, -6.88d0, -7.68d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -6.68d0,-11.18d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -10.28d0, -8.58d0, -9.38d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        & -11.98d0, -8.38d0,-12.88d0, -9.28d0,-11.98d0, -8.38d0,-12.88d0, -9.28d0,&
        &  -6.88d0, -5.18d0, -5.98d0, -5.88d0, -8.58d0, -4.98d0, -9.48d0, -5.88d0,&
        &  -8.58d0, -4.98d0, -9.48d0, -5.88d0, -8.58d0, -4.98d0, -9.48d0, -5.88d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -6.68d0, -4.98d0, -5.78d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -8.38d0, -4.78d0, -9.28d0, -5.68d0, -8.38d0, -4.78d0, -9.28d0, -5.68d0,&
        &  -7.68d0, -5.98d0, -6.78d0, -6.68d0, -9.38d0, -5.78d0,-10.28d0, -6.68d0,&
        &  -9.38d0, -5.78d0,-10.28d0, -6.68d0, -9.38d0, -5.78d0,-10.28d0, -6.68d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -11.18d0, -9.48d0,-10.28d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        & -12.88d0, -9.28d0,-13.78d0,-10.18d0,-12.88d0, -9.28d0,-13.78d0,-10.18d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.58d0, -5.88d0, -6.68d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -9.28d0, -5.68d0,-10.18d0, -6.58d0, -9.28d0, -5.68d0,-10.18d0, -6.58d0,&
        &  -7.98d0, -8.28d0, -7.78d0, -6.78d0,-10.28d0, -9.08d0,-10.78d0, -5.38d0,&
        & -12.88d0, -5.98d0,-12.88d0, -6.78d0, -9.88d0, -5.78d0,-10.78d0, -5.39d0,&
        & -10.28d0,-10.58d0,-10.08d0, -9.08d0,-12.58d0,-11.38d0,-13.08d0, -7.68d0,&
        & -15.18d0, -8.28d0,-15.18d0, -9.08d0,-12.18d0, -8.08d0,-13.08d0, -7.69d0,&
        & -12.88d0,-13.18d0,-12.68d0,-11.68d0,-15.18d0,-13.98d0,-15.68d0,-10.28d0,&
        & -17.78d0,-10.88d0,-17.78d0,-11.68d0,-14.78d0,-10.68d0,-15.68d0,-10.29d0,&
        &  -9.88d0,-10.18d0, -9.68d0, -8.68d0,-12.18d0,-10.98d0,-12.68d0, -7.28d0,&
        & -14.78d0, -7.88d0,-14.78d0, -8.68d0,-11.78d0, -7.68d0,-12.68d0, -7.29d0,&
        &  -8.28d0, -8.58d0, -8.08d0, -7.08d0,-10.58d0, -9.38d0,-11.08d0, -5.68d0,&
        & -13.18d0, -6.28d0,-13.18d0, -7.08d0,-10.18d0, -6.08d0,-11.08d0, -5.69d0,&
        &  -9.08d0, -9.38d0, -8.88d0, -7.88d0,-11.38d0,-10.18d0,-11.88d0, -6.48d0,&
        & -13.98d0, -7.08d0,-13.98d0, -7.88d0,-10.98d0, -6.88d0,-11.88d0, -6.49d0,&
        &  -5.98d0, -6.28d0, -5.78d0, -4.78d0, -8.28d0, -7.08d0, -8.78d0, -3.38d0,&
        & -10.88d0, -3.98d0,-10.88d0, -4.78d0, -7.88d0, -3.78d0, -8.78d0, -3.39d0,&
        &  -5.78d0, -6.08d0, -5.58d0, -4.58d0, -8.08d0, -6.88d0, -8.58d0, -3.18d0,&
        & -10.68d0, -3.78d0,-10.68d0, -4.58d0, -7.68d0, -3.58d0, -8.58d0, -3.19d0,&
        &  -7.78d0, -8.08d0, -7.58d0, -6.58d0,-10.08d0, -8.88d0,-10.58d0, -5.18d0,&
        & -12.68d0, -5.78d0,-12.68d0, -6.58d0, -9.68d0, -5.58d0,-10.58d0, -5.19d0,&
        & -10.78d0,-11.08d0,-10.58d0, -9.58d0,-13.08d0,-11.88d0,-13.58d0, -8.18d0,&
        & -15.68d0, -8.78d0,-15.68d0, -9.58d0,-12.68d0, -8.58d0,-13.58d0, -8.19d0,&
        & -12.88d0,-13.18d0,-12.68d0,-11.68d0,-15.18d0,-13.98d0,-15.68d0,-10.28d0,&
        & -17.78d0,-10.88d0,-17.78d0,-11.68d0,-14.78d0,-10.68d0,-15.68d0,-10.29d0,&
        & -10.78d0,-11.08d0,-10.58d0, -9.58d0,-13.08d0,-11.88d0,-13.58d0, -8.18d0,&
        & -15.68d0, -8.78d0,-15.68d0, -9.58d0,-12.68d0, -8.58d0,-13.58d0, -8.19d0,&
        &  -6.78d0, -7.08d0, -6.58d0, -5.58d0, -9.08d0, -7.88d0, -9.58d0, -4.18d0,&
        & -11.68d0, -4.78d0,-11.68d0, -5.58d0, -8.68d0, -4.58d0, -9.58d0, -4.19d0,&
        &  -5.38d0, -5.68d0, -5.18d0, -4.18d0, -7.68d0, -6.48d0, -8.18d0, -2.78d0,&
        & -10.28d0, -3.38d0,-10.28d0, -4.18d0, -7.28d0, -3.18d0, -8.18d0, -2.79d0,&
        &  -6.78d0, -7.08d0, -6.58d0, -5.58d0, -9.08d0, -7.88d0, -9.58d0, -4.18d0,&
        & -11.68d0, -4.78d0,-11.68d0, -5.58d0, -8.68d0, -4.58d0, -9.58d0, -4.19d0,&
        &  -5.39d0, -5.69d0, -5.19d0, -4.19d0, -7.69d0, -6.49d0, -8.19d0, -2.79d0,&
        & -10.29d0, -3.39d0,-10.29d0, -4.19d0, -7.29d0, -3.19d0, -8.19d0, -2.80d0,&
        &  -9.18d0,-11.18d0,-11.08d0, -8.98d0,-12.78d0, -7.08d0,-11.38d0, -8.98d0,&
        &  -9.58d0, -7.08d0,-10.18d0, -8.98d0,-12.78d0, -7.88d0,-11.38d0, -9.68d0,&
        & -11.48d0,-13.48d0,-13.38d0,-11.28d0,-15.08d0, -9.38d0,-13.68d0,-11.28d0,&
        & -11.88d0, -9.38d0,-12.48d0,-11.28d0,-15.08d0,-10.18d0,-13.68d0,-11.98d0,&
        & -14.08d0,-16.08d0,-15.98d0,-13.88d0,-17.68d0,-11.98d0,-16.28d0,-13.88d0,&
        & -14.48d0,-11.98d0,-15.08d0,-13.88d0,-17.68d0,-12.78d0,-16.28d0,-14.58d0,&
        & -11.08d0,-13.08d0,-12.98d0,-10.88d0,-14.68d0, -8.98d0,-13.28d0,-10.88d0,&
        & -11.48d0, -8.98d0,-12.08d0,-10.88d0,-14.68d0, -9.78d0,-13.28d0,-11.58d0,&
        &  -9.48d0,-11.48d0,-11.38d0, -9.28d0,-13.08d0, -7.38d0,-11.68d0, -9.28d0,&
        &  -9.88d0, -7.38d0,-10.48d0, -9.28d0,-13.08d0, -8.18d0,-11.68d0, -9.98d0,&
        & -10.28d0,-12.28d0,-12.18d0,-10.08d0,-13.88d0, -8.18d0,-12.48d0,-10.08d0,&
        & -10.68d0, -8.18d0,-11.28d0,-10.08d0,-13.88d0, -8.98d0,-12.48d0,-10.78d0,&
        &  -7.18d0, -9.18d0, -9.08d0, -6.98d0,-10.78d0, -5.08d0, -9.38d0, -6.98d0,&
        &  -7.58d0, -5.08d0, -8.18d0, -6.98d0,-10.78d0, -5.88d0, -9.38d0, -7.68d0,&
        &  -6.98d0, -8.98d0, -8.88d0, -6.78d0,-10.58d0, -4.88d0, -9.18d0, -6.78d0,&
        &  -7.38d0, -4.88d0, -7.98d0, -6.78d0,-10.58d0, -5.68d0, -9.18d0, -7.48d0,&
        &  -8.98d0,-10.98d0,-10.88d0, -8.78d0,-12.58d0, -6.88d0,-11.18d0, -8.78d0,&
        &  -9.38d0, -6.88d0, -9.98d0, -8.78d0,-12.58d0, -7.68d0,-11.18d0, -9.48d0,&
        & -11.98d0,-13.98d0,-13.88d0,-11.78d0,-15.58d0, -9.88d0,-14.18d0,-11.78d0,&
        & -12.38d0, -9.88d0,-12.98d0,-11.78d0,-15.58d0,-10.68d0,-14.18d0,-12.48d0,&
        & -14.08d0,-16.08d0,-15.98d0,-13.88d0,-17.68d0,-11.98d0,-16.28d0,-13.88d0,&
        & -14.48d0,-11.98d0,-15.08d0,-13.88d0,-17.68d0,-12.78d0,-16.28d0,-14.58d0,&
        & -11.98d0,-13.98d0,-13.88d0,-11.78d0,-15.58d0, -9.88d0,-14.18d0,-11.78d0,&
        & -12.38d0, -9.88d0,-12.98d0,-11.78d0,-15.58d0,-10.68d0,-14.18d0,-12.48d0,&
        &  -7.98d0, -9.98d0, -9.88d0, -7.78d0,-11.58d0, -5.88d0,-10.18d0, -7.78d0,&
        &  -8.38d0, -5.88d0, -8.98d0, -7.78d0,-11.58d0, -6.68d0,-10.18d0, -8.48d0,&
        &  -6.58d0, -8.58d0, -8.48d0, -6.38d0,-10.18d0, -4.48d0, -8.78d0, -6.38d0,&
        &  -6.98d0, -4.48d0, -7.58d0, -6.38d0,-10.18d0, -5.28d0, -8.78d0, -7.08d0,&
        &  -7.98d0, -9.98d0, -9.88d0, -7.78d0,-11.58d0, -5.88d0,-10.18d0, -7.78d0,&
        &  -8.38d0, -5.88d0, -8.98d0, -7.78d0,-11.58d0, -6.68d0,-10.18d0, -8.48d0,&
        &  -6.59d0, -8.59d0, -8.49d0, -6.39d0,-10.19d0, -4.49d0, -8.79d0, -6.39d0,&
        &  -6.99d0, -4.49d0, -7.59d0, -6.39d0,-10.19d0, -5.29d0, -8.79d0, -7.09d0,&
        & -14.28d0, -9.18d0,-13.38d0,-12.08d0,-13.48d0, -8.48d0,-13.38d0,-11.38d0,&
        & -14.28d0, -9.18d0,-13.38d0,-12.08d0,-14.28d0,-10.68d0,-13.38d0,-12.58d0,&
        & -16.58d0,-11.48d0,-15.68d0,-14.38d0,-15.78d0,-10.78d0,-15.68d0,-13.68d0,&
        & -16.58d0,-11.48d0,-15.68d0,-14.38d0,-16.58d0,-12.98d0,-15.68d0,-14.88d0,&
        & -19.18d0,-14.08d0,-18.28d0,-16.98d0,-18.38d0,-13.38d0,-18.28d0,-16.28d0,&
        & -19.18d0,-14.08d0,-18.28d0,-16.98d0,-19.18d0,-15.58d0,-18.28d0,-17.48d0,&
        & -16.18d0,-11.08d0,-15.28d0,-13.98d0,-15.38d0,-10.38d0,-15.28d0,-13.28d0,&
        & -16.18d0,-11.08d0,-15.28d0,-13.98d0,-16.18d0,-12.58d0,-15.28d0,-14.48d0,&
        & -14.58d0, -9.48d0,-13.68d0,-12.38d0,-13.78d0, -8.78d0,-13.68d0,-11.68d0,&
        & -14.58d0, -9.48d0,-13.68d0,-12.38d0,-14.58d0,-10.98d0,-13.68d0,-12.88d0,&
        & -15.38d0,-10.28d0,-14.48d0,-13.18d0,-14.58d0, -9.58d0,-14.48d0,-12.48d0,&
        & -15.38d0,-10.28d0,-14.48d0,-13.18d0,-15.38d0,-11.78d0,-14.48d0,-13.68d0,&
        & -12.28d0, -7.18d0,-11.38d0,-10.08d0,-11.48d0, -6.48d0,-11.38d0, -9.38d0,&
        & -12.28d0, -7.18d0,-11.38d0,-10.08d0,-12.28d0, -8.68d0,-11.38d0,-10.58d0,&
        & -12.08d0, -6.98d0,-11.18d0, -9.88d0,-11.28d0, -6.28d0,-11.18d0, -9.18d0,&
        & -12.08d0, -6.98d0,-11.18d0, -9.88d0,-12.08d0, -8.48d0,-11.18d0,-10.38d0,&
        & -14.08d0, -8.98d0,-13.18d0,-11.88d0,-13.28d0, -8.28d0,-13.18d0,-11.18d0,&
        & -14.08d0, -8.98d0,-13.18d0,-11.88d0,-14.08d0,-10.48d0,-13.18d0,-12.38d0,&
        & -17.08d0,-11.98d0,-16.18d0,-14.88d0,-16.28d0,-11.28d0,-16.18d0,-14.18d0,&
        & -17.08d0,-11.98d0,-16.18d0,-14.88d0,-17.08d0,-13.48d0,-16.18d0,-15.38d0,&
        & -19.18d0,-14.08d0,-18.28d0,-16.98d0,-18.38d0,-13.38d0,-18.28d0,-16.28d0,&
        & -19.18d0,-14.08d0,-18.28d0,-16.98d0,-19.18d0,-15.58d0,-18.28d0,-17.48d0,&
        & -17.08d0,-11.98d0,-16.18d0,-14.88d0,-16.28d0,-11.28d0,-16.18d0,-14.18d0,&
        & -17.08d0,-11.98d0,-16.18d0,-14.88d0,-17.08d0,-13.48d0,-16.18d0,-15.38d0,&
        & -13.08d0, -7.98d0,-12.18d0,-10.88d0,-12.28d0, -7.28d0,-12.18d0,-10.18d0,&
        & -13.08d0, -7.98d0,-12.18d0,-10.88d0,-13.08d0, -9.48d0,-12.18d0,-11.38d0,&
        & -11.68d0, -6.58d0,-10.78d0, -9.48d0,-10.88d0, -5.88d0,-10.78d0, -8.78d0,&
        & -11.68d0, -6.58d0,-10.78d0, -9.48d0,-11.68d0, -8.08d0,-10.78d0, -9.98d0,&
        & -13.08d0, -7.98d0,-12.18d0,-10.88d0,-12.28d0, -7.28d0,-12.18d0,-10.18d0,&
        & -13.08d0, -7.98d0,-12.18d0,-10.88d0,-13.08d0, -9.48d0,-12.18d0,-11.38d0,&
        & -11.69d0, -6.59d0,-10.79d0, -9.49d0,-10.89d0, -5.89d0,-10.79d0, -8.79d0,&
        & -11.69d0, -6.59d0,-10.79d0, -9.49d0,-11.69d0, -8.09d0,-10.79d0, -9.99d0,&
        & -11.18d0, -8.78d0,-10.58d0, -9.48d0,-11.88d0, -8.78d0,-12.08d0, -8.38d0,&
        & -13.58d0, -7.58d0,-13.18d0, -9.48d0,-12.08d0, -8.78d0,-12.08d0, -7.58d0,&
        & -13.48d0,-11.08d0,-12.88d0,-11.78d0,-14.18d0,-11.08d0,-14.38d0,-10.68d0,&
        & -15.88d0, -9.88d0,-15.48d0,-11.78d0,-14.38d0,-11.08d0,-14.38d0, -9.88d0,&
        & -16.08d0,-13.68d0,-15.48d0,-14.38d0,-16.78d0,-13.68d0,-16.98d0,-13.28d0,&
        & -18.48d0,-12.48d0,-18.08d0,-14.38d0,-16.98d0,-13.68d0,-16.98d0,-12.48d0,&
        & -13.08d0,-10.68d0,-12.48d0,-11.38d0,-13.78d0,-10.68d0,-13.98d0,-10.28d0,&
        & -15.48d0, -9.48d0,-15.08d0,-11.38d0,-13.98d0,-10.68d0,-13.98d0, -9.48d0,&
        & -11.48d0, -9.08d0,-10.88d0, -9.78d0,-12.18d0, -9.08d0,-12.38d0, -8.68d0,&
        & -13.88d0, -7.88d0,-13.48d0, -9.78d0,-12.38d0, -9.08d0,-12.38d0, -7.88d0,&
        & -12.28d0, -9.88d0,-11.68d0,-10.58d0,-12.98d0, -9.88d0,-13.18d0, -9.48d0,&
        & -14.68d0, -8.68d0,-14.28d0,-10.58d0,-13.18d0, -9.88d0,-13.18d0, -8.68d0,&
        &  -9.18d0, -6.78d0, -8.58d0, -7.48d0, -9.88d0, -6.78d0,-10.08d0, -6.38d0,&
        & -11.58d0, -5.58d0,-11.18d0, -7.48d0,-10.08d0, -6.78d0,-10.08d0, -5.58d0,&
        &  -8.98d0, -6.58d0, -8.38d0, -7.28d0, -9.68d0, -6.58d0, -9.88d0, -6.18d0,&
        & -11.38d0, -5.38d0,-10.98d0, -7.28d0, -9.88d0, -6.58d0, -9.88d0, -5.38d0,&
        & -10.98d0, -8.58d0,-10.38d0, -9.28d0,-11.68d0, -8.58d0,-11.88d0, -8.18d0,&
        & -13.38d0, -7.38d0,-12.98d0, -9.28d0,-11.88d0, -8.58d0,-11.88d0, -7.38d0,&
        & -13.98d0,-11.58d0,-13.38d0,-12.28d0,-14.68d0,-11.58d0,-14.88d0,-11.18d0,&
        & -16.38d0,-10.38d0,-15.98d0,-12.28d0,-14.88d0,-11.58d0,-14.88d0,-10.38d0,&
        & -16.08d0,-13.68d0,-15.48d0,-14.38d0,-16.78d0,-13.68d0,-16.98d0,-13.28d0,&
        & -18.48d0,-12.48d0,-18.08d0,-14.38d0,-16.98d0,-13.68d0,-16.98d0,-12.48d0,&
        & -13.98d0,-11.58d0,-13.38d0,-12.28d0,-14.68d0,-11.58d0,-14.88d0,-11.18d0,&
        & -16.38d0,-10.38d0,-15.98d0,-12.28d0,-14.88d0,-11.58d0,-14.88d0,-10.38d0,&
        &  -9.98d0, -7.58d0, -9.38d0, -8.28d0,-10.68d0, -7.58d0,-10.88d0, -7.18d0,&
        & -12.38d0, -6.38d0,-11.98d0, -8.28d0,-10.88d0, -7.58d0,-10.88d0, -6.38d0,&
        &  -8.58d0, -6.18d0, -7.98d0, -6.88d0, -9.28d0, -6.18d0, -9.48d0, -5.78d0,&
        & -10.98d0, -4.98d0,-10.58d0, -6.88d0, -9.48d0, -6.18d0, -9.48d0, -4.98d0,&
        &  -9.98d0, -7.58d0, -9.38d0, -8.28d0,-10.68d0, -7.58d0,-10.88d0, -7.18d0,&
        & -12.38d0, -6.38d0,-11.98d0, -8.28d0,-10.88d0, -7.58d0,-10.88d0, -6.38d0,&
        &  -8.59d0, -6.19d0, -7.99d0, -6.89d0, -9.29d0, -6.19d0, -9.49d0, -5.79d0,&
        & -10.99d0, -4.99d0,-10.59d0, -6.89d0, -9.49d0, -6.19d0, -9.49d0, -4.99d0,&
        &  -8.28d0, -6.58d0, -7.38d0, -7.28d0, -9.98d0, -6.38d0,-10.88d0, -7.28d0,&
        &  -9.98d0, -6.38d0,-10.88d0, -7.28d0, -9.98d0, -6.38d0,-10.88d0, -7.28d0,&
        & -10.58d0, -8.88d0, -9.68d0, -9.58d0,-12.28d0, -8.68d0,-13.18d0, -9.58d0,&
        & -12.28d0, -8.68d0,-13.18d0, -9.58d0,-12.28d0, -8.68d0,-13.18d0, -9.58d0,&
        & -13.18d0,-11.48d0,-12.28d0,-12.18d0,-14.88d0,-11.28d0,-15.78d0,-12.18d0,&
        & -14.88d0,-11.28d0,-15.78d0,-12.18d0,-14.88d0,-11.28d0,-15.78d0,-12.18d0,&
        & -10.18d0, -8.48d0, -9.28d0, -9.18d0,-11.88d0, -8.28d0,-12.78d0, -9.18d0,&
        & -11.88d0, -8.28d0,-12.78d0, -9.18d0,-11.88d0, -8.28d0,-12.78d0, -9.18d0,&
        &  -8.58d0, -6.88d0, -7.68d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -6.68d0,-11.18d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        &  -9.38d0, -7.68d0, -8.48d0, -8.38d0,-11.08d0, -7.48d0,-11.98d0, -8.38d0,&
        & -11.08d0, -7.48d0,-11.98d0, -8.38d0,-11.08d0, -7.48d0,-11.98d0, -8.38d0,&
        &  -6.28d0, -4.58d0, -5.38d0, -5.28d0, -7.98d0, -4.38d0, -8.88d0, -5.28d0,&
        &  -7.98d0, -4.38d0, -8.88d0, -5.28d0, -7.98d0, -4.38d0, -8.88d0, -5.28d0,&
        &  -6.08d0, -4.38d0, -5.18d0, -5.08d0, -7.78d0, -4.18d0, -8.68d0, -5.08d0,&
        &  -7.78d0, -4.18d0, -8.68d0, -5.08d0, -7.78d0, -4.18d0, -8.68d0, -5.08d0,&
        &  -8.08d0, -6.38d0, -7.18d0, -7.08d0, -9.78d0, -6.18d0,-10.68d0, -7.08d0,&
        &  -9.78d0, -6.18d0,-10.68d0, -7.08d0, -9.78d0, -6.18d0,-10.68d0, -7.08d0,&
        & -11.08d0, -9.38d0,-10.18d0,-10.08d0,-12.78d0, -9.18d0,-13.68d0,-10.08d0,&
        & -12.78d0, -9.18d0,-13.68d0,-10.08d0,-12.78d0, -9.18d0,-13.68d0,-10.08d0,&
        & -13.18d0,-11.48d0,-12.28d0,-12.18d0,-14.88d0,-11.28d0,-15.78d0,-12.18d0,&
        & -14.88d0,-11.28d0,-15.78d0,-12.18d0,-14.88d0,-11.28d0,-15.78d0,-12.18d0,&
        & -11.08d0, -9.38d0,-10.18d0,-10.08d0,-12.78d0, -9.18d0,-13.68d0,-10.08d0,&
        & -12.78d0, -9.18d0,-13.68d0,-10.08d0,-12.78d0, -9.18d0,-13.68d0,-10.08d0,&
        &  -7.08d0, -5.38d0, -6.18d0, -6.08d0, -8.78d0, -5.18d0, -9.68d0, -6.08d0,&
        &  -8.78d0, -5.18d0, -9.68d0, -6.08d0, -8.78d0, -5.18d0, -9.68d0, -6.08d0,&
        &  -5.68d0, -3.98d0, -4.78d0, -4.68d0, -7.38d0, -3.78d0, -8.28d0, -4.68d0,&
        &  -7.38d0, -3.78d0, -8.28d0, -4.68d0, -7.38d0, -3.78d0, -8.28d0, -4.68d0,&
        &  -7.08d0, -5.38d0, -6.18d0, -6.08d0, -8.78d0, -5.18d0, -9.68d0, -6.08d0,&
        &  -8.78d0, -5.18d0, -9.68d0, -6.08d0, -8.78d0, -5.18d0, -9.68d0, -6.08d0,&
        &  -5.69d0, -3.99d0, -4.79d0, -4.69d0, -7.39d0, -3.79d0, -8.29d0, -4.69d0,&
        &  -7.39d0, -3.79d0, -8.29d0, -4.69d0, -7.39d0, -3.79d0, -8.29d0, -4.69d0,&
        &  -8.28d0, -6.58d0, -7.38d0, -7.28d0, -9.98d0, -6.38d0,-10.88d0, -7.28d0,&
        &  -9.98d0, -6.38d0,-10.88d0, -7.28d0, -9.98d0, -6.38d0,-10.88d0, -7.28d0,&
        & -10.58d0, -8.88d0, -9.68d0, -9.58d0,-12.28d0, -8.68d0,-13.18d0, -9.58d0,&
        & -12.28d0, -8.68d0,-13.18d0, -9.58d0,-12.28d0, -8.68d0,-13.18d0, -9.58d0,&
        & -13.18d0,-11.48d0,-12.28d0,-12.18d0,-14.88d0,-11.28d0,-15.78d0,-12.18d0,&
        & -14.88d0,-11.28d0,-15.78d0,-12.18d0,-14.88d0,-11.28d0,-15.78d0,-12.18d0,&
        & -10.18d0, -8.48d0, -9.28d0, -9.18d0,-11.88d0, -8.28d0,-12.78d0, -9.18d0,&
        & -11.88d0, -8.28d0,-12.78d0, -9.18d0,-11.88d0, -8.28d0,-12.78d0, -9.18d0,&
        &  -8.58d0, -6.88d0, -7.68d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        & -10.28d0, -6.68d0,-11.18d0, -7.58d0,-10.28d0, -6.68d0,-11.18d0, -7.58d0,&
        &  -9.38d0, -7.68d0, -8.48d0, -8.38d0,-11.08d0, -7.48d0,-11.98d0, -8.38d0,&
        & -11.08d0, -7.48d0,-11.98d0, -8.38d0,-11.08d0, -7.48d0,-11.98d0, -8.38d0,&
        &  -6.28d0, -4.58d0, -5.38d0, -5.28d0, -7.98d0, -4.38d0, -8.88d0, -5.28d0,&
        &  -7.98d0, -4.38d0, -8.88d0, -5.28d0, -7.98d0, -4.38d0, -8.88d0, -5.28d0,&
        &  -6.08d0, -4.38d0, -5.18d0, -5.08d0, -7.78d0, -4.18d0, -8.68d0, -5.08d0,&
        &  -7.78d0, -4.18d0, -8.68d0, -5.08d0, -7.78d0, -4.18d0, -8.68d0, -5.08d0,&
        &  -8.08d0, -6.38d0, -7.18d0, -7.08d0, -9.78d0, -6.18d0,-10.68d0, -7.08d0,&
        &  -9.78d0, -6.18d0,-10.68d0, -7.08d0, -9.78d0, -6.18d0,-10.68d0, -7.08d0,&
        & -11.08d0, -9.38d0,-10.18d0,-10.08d0,-12.78d0, -9.18d0,-13.68d0,-10.08d0,&
        & -12.78d0, -9.18d0,-13.68d0,-10.08d0,-12.78d0, -9.18d0,-13.68d0,-10.08d0,&
        & -13.18d0,-11.48d0,-12.28d0,-12.18d0,-14.88d0,-11.28d0,-15.78d0,-12.18d0,&
        & -14.88d0,-11.28d0,-15.78d0,-12.18d0,-14.88d0,-11.28d0,-15.78d0,-12.18d0,&
        & -11.08d0, -9.38d0,-10.18d0,-10.08d0,-12.78d0, -9.18d0,-13.68d0,-10.08d0,&
        & -12.78d0, -9.18d0,-13.68d0,-10.08d0,-12.78d0, -9.18d0,-13.68d0,-10.08d0,&
        &  -7.08d0, -5.38d0, -6.18d0, -6.08d0, -8.78d0, -5.18d0, -9.68d0, -6.08d0,&
        &  -8.78d0, -5.18d0, -9.68d0, -6.08d0, -8.78d0, -5.18d0, -9.68d0, -6.08d0,&
        &  -5.68d0, -3.98d0, -4.78d0, -4.68d0, -7.38d0, -3.78d0, -8.28d0, -4.68d0,&
        &  -7.38d0, -3.78d0, -8.28d0, -4.68d0, -7.38d0, -3.78d0, -8.28d0, -4.68d0,&
        &  -7.08d0, -5.38d0, -6.18d0, -6.08d0, -8.78d0, -5.18d0, -9.68d0, -6.08d0,&
        &  -8.78d0, -5.18d0, -9.68d0, -6.08d0, -8.78d0, -5.18d0, -9.68d0, -6.08d0,&
        &  -5.69d0, -3.99d0, -4.79d0, -4.69d0, -7.39d0, -3.79d0, -8.29d0, -4.69d0,&
        &  -7.39d0, -3.79d0, -8.29d0, -4.69d0, -7.39d0, -3.79d0, -8.29d0, -4.69d0,&
        & -11.18d0,-11.48d0,-10.98d0, -9.98d0,-13.48d0,-12.28d0,-13.98d0, -8.58d0,&
        & -16.08d0, -9.18d0,-16.08d0, -9.98d0,-13.08d0, -8.98d0,-13.98d0, -8.59d0,&
        & -11.88d0,-12.18d0,-11.68d0,-10.68d0,-14.18d0,-12.98d0,-14.68d0, -9.28d0,&
        & -16.78d0, -9.88d0,-16.78d0,-10.68d0,-13.78d0, -9.68d0,-14.68d0, -9.29d0,&
        & -13.58d0,-13.88d0,-13.38d0,-12.38d0,-15.88d0,-14.68d0,-16.38d0,-10.98d0,&
        & -18.48d0,-11.58d0,-18.48d0,-12.38d0,-15.48d0,-11.38d0,-16.38d0,-10.99d0,&
        & -12.08d0,-12.38d0,-11.88d0,-10.88d0,-14.38d0,-13.18d0,-14.88d0, -9.48d0,&
        & -16.98d0,-10.08d0,-16.98d0,-10.88d0,-13.98d0, -9.88d0,-14.88d0, -9.49d0,&
        &  -8.78d0, -9.08d0, -8.58d0, -7.58d0,-11.08d0, -9.88d0,-11.58d0, -6.18d0,&
        & -13.68d0, -6.78d0,-13.68d0, -7.58d0,-10.68d0, -6.58d0,-11.58d0, -6.19d0,&
        &  -8.78d0, -9.08d0, -8.58d0, -7.58d0,-11.08d0, -9.88d0,-11.58d0, -6.18d0,&
        & -13.68d0, -6.78d0,-13.68d0, -7.58d0,-10.68d0, -6.58d0,-11.58d0, -6.19d0,&
        &  -7.58d0, -7.88d0, -7.38d0, -6.38d0, -9.88d0, -8.68d0,-10.38d0, -4.98d0,&
        & -12.48d0, -5.58d0,-12.48d0, -6.38d0, -9.48d0, -5.38d0,-10.38d0, -4.99d0,&
        &  -8.78d0, -9.08d0, -8.58d0, -7.58d0,-11.08d0, -9.88d0,-11.58d0, -6.18d0,&
        & -13.68d0, -6.78d0,-13.68d0, -7.58d0,-10.68d0, -6.58d0,-11.58d0, -6.19d0,&
        & -10.58d0,-10.88d0,-10.38d0, -9.38d0,-12.88d0,-11.68d0,-13.38d0, -7.98d0,&
        & -15.48d0, -8.58d0,-15.48d0, -9.38d0,-12.48d0, -8.38d0,-13.38d0, -7.99d0,&
        & -12.08d0,-12.38d0,-11.88d0,-10.88d0,-14.38d0,-13.18d0,-14.88d0, -9.48d0,&
        & -16.98d0,-10.08d0,-16.98d0,-10.88d0,-13.98d0, -9.88d0,-14.88d0, -9.49d0,&
        & -13.18d0,-13.48d0,-12.98d0,-11.98d0,-15.48d0,-14.28d0,-15.98d0,-10.58d0,&
        & -18.08d0,-11.18d0,-18.08d0,-11.98d0,-15.08d0,-10.98d0,-15.98d0,-10.59d0,&
        & -12.08d0,-12.38d0,-11.88d0,-10.88d0,-14.38d0,-13.18d0,-14.88d0, -9.48d0,&
        & -16.98d0,-10.08d0,-16.98d0,-10.88d0,-13.98d0, -9.88d0,-14.88d0, -9.49d0,&
        &  -9.48d0, -9.78d0, -9.28d0, -8.28d0,-11.78d0,-10.58d0,-12.28d0, -6.88d0,&
        & -14.38d0, -7.48d0,-14.38d0, -8.28d0,-11.38d0, -7.28d0,-12.28d0, -6.89d0,&
        &  -8.38d0, -8.68d0, -8.18d0, -7.18d0,-10.68d0, -9.48d0,-11.18d0, -5.78d0,&
        & -13.28d0, -6.38d0,-13.28d0, -7.18d0,-10.28d0, -6.18d0,-11.18d0, -5.79d0,&
        &  -9.48d0, -9.78d0, -9.28d0, -8.28d0,-11.78d0,-10.58d0,-12.28d0, -6.88d0,&
        & -14.38d0, -7.48d0,-14.38d0, -8.28d0,-11.38d0, -7.28d0,-12.28d0, -6.89d0,&
        &  -7.58d0, -7.88d0, -7.38d0, -6.38d0, -9.88d0, -8.68d0,-10.38d0, -4.98d0,&
        & -12.48d0, -5.58d0,-12.48d0, -6.38d0, -9.48d0, -5.38d0,-10.38d0, -4.99d0,&
        & -12.38d0,-14.38d0,-14.28d0,-12.18d0,-15.98d0,-10.28d0,-14.58d0,-12.18d0,&
        & -12.78d0,-10.28d0,-13.38d0,-12.18d0,-15.98d0,-11.08d0,-14.58d0,-12.88d0,&
        & -13.08d0,-15.08d0,-14.98d0,-12.88d0,-16.68d0,-10.98d0,-15.28d0,-12.88d0,&
        & -13.48d0,-10.98d0,-14.08d0,-12.88d0,-16.68d0,-11.78d0,-15.28d0,-13.58d0,&
        & -14.78d0,-16.78d0,-16.68d0,-14.58d0,-18.38d0,-12.68d0,-16.98d0,-14.58d0,&
        & -15.18d0,-12.68d0,-15.78d0,-14.58d0,-18.38d0,-13.48d0,-16.98d0,-15.28d0,&
        & -13.28d0,-15.28d0,-15.18d0,-13.08d0,-16.88d0,-11.18d0,-15.48d0,-13.08d0,&
        & -13.68d0,-11.18d0,-14.28d0,-13.08d0,-16.88d0,-11.98d0,-15.48d0,-13.78d0,&
        &  -9.98d0,-11.98d0,-11.88d0, -9.78d0,-13.58d0, -7.88d0,-12.18d0, -9.78d0,&
        & -10.38d0, -7.88d0,-10.98d0, -9.78d0,-13.58d0, -8.68d0,-12.18d0,-10.48d0,&
        &  -9.98d0,-11.98d0,-11.88d0, -9.78d0,-13.58d0, -7.88d0,-12.18d0, -9.78d0,&
        & -10.38d0, -7.88d0,-10.98d0, -9.78d0,-13.58d0, -8.68d0,-12.18d0,-10.48d0,&
        &  -8.78d0,-10.78d0,-10.68d0, -8.58d0,-12.38d0, -6.68d0,-10.98d0, -8.58d0,&
        &  -9.18d0, -6.68d0, -9.78d0, -8.58d0,-12.38d0, -7.48d0,-10.98d0, -9.28d0,&
        &  -9.98d0,-11.98d0,-11.88d0, -9.78d0,-13.58d0, -7.88d0,-12.18d0, -9.78d0,&
        & -10.38d0, -7.88d0,-10.98d0, -9.78d0,-13.58d0, -8.68d0,-12.18d0,-10.48d0,&
        & -11.78d0,-13.78d0,-13.68d0,-11.58d0,-15.38d0, -9.68d0,-13.98d0,-11.58d0,&
        & -12.18d0, -9.68d0,-12.78d0,-11.58d0,-15.38d0,-10.48d0,-13.98d0,-12.28d0,&
        & -13.28d0,-15.28d0,-15.18d0,-13.08d0,-16.88d0,-11.18d0,-15.48d0,-13.08d0,&
        & -13.68d0,-11.18d0,-14.28d0,-13.08d0,-16.88d0,-11.98d0,-15.48d0,-13.78d0,&
        & -14.38d0,-16.38d0,-16.28d0,-14.18d0,-17.98d0,-12.28d0,-16.58d0,-14.18d0,&
        & -14.78d0,-12.28d0,-15.38d0,-14.18d0,-17.98d0,-13.08d0,-16.58d0,-14.88d0,&
        & -13.28d0,-15.28d0,-15.18d0,-13.08d0,-16.88d0,-11.18d0,-15.48d0,-13.08d0,&
        & -13.68d0,-11.18d0,-14.28d0,-13.08d0,-16.88d0,-11.98d0,-15.48d0,-13.78d0,&
        & -10.68d0,-12.68d0,-12.58d0,-10.48d0,-14.28d0, -8.58d0,-12.88d0,-10.48d0,&
        & -11.08d0, -8.58d0,-11.68d0,-10.48d0,-14.28d0, -9.38d0,-12.88d0,-11.18d0,&
        &  -9.58d0,-11.58d0,-11.48d0, -9.38d0,-13.18d0, -7.48d0,-11.78d0, -9.38d0,&
        &  -9.98d0, -7.48d0,-10.58d0, -9.38d0,-13.18d0, -8.28d0,-11.78d0,-10.08d0,&
        & -10.68d0,-12.68d0,-12.58d0,-10.48d0,-14.28d0, -8.58d0,-12.88d0,-10.48d0,&
        & -11.08d0, -8.58d0,-11.68d0,-10.48d0,-14.28d0, -9.38d0,-12.88d0,-11.18d0,&
        &  -8.78d0,-10.78d0,-10.68d0, -8.58d0,-12.38d0, -6.68d0,-10.98d0, -8.58d0,&
        &  -9.18d0, -6.68d0, -9.78d0, -8.58d0,-12.38d0, -7.48d0,-10.98d0, -9.28d0,&
        & -17.48d0,-12.38d0,-16.58d0,-15.28d0,-16.68d0,-11.68d0,-16.58d0,-14.58d0,&
        & -17.48d0,-12.38d0,-16.58d0,-15.28d0,-17.48d0,-13.88d0,-16.58d0,-15.78d0,&
        & -18.18d0,-13.08d0,-17.28d0,-15.98d0,-17.38d0,-12.38d0,-17.28d0,-15.28d0,&
        & -18.18d0,-13.08d0,-17.28d0,-15.98d0,-18.18d0,-14.58d0,-17.28d0,-16.48d0,&
        & -19.88d0,-14.78d0,-18.98d0,-17.68d0,-19.08d0,-14.08d0,-18.98d0,-16.98d0,&
        & -19.88d0,-14.78d0,-18.98d0,-17.68d0,-19.88d0,-16.28d0,-18.98d0,-18.18d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-17.58d0,-12.58d0,-17.48d0,-15.48d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-18.38d0,-14.78d0,-17.48d0,-16.68d0,&
        & -15.08d0, -9.98d0,-14.18d0,-12.88d0,-14.28d0, -9.28d0,-14.18d0,-12.18d0,&
        & -15.08d0, -9.98d0,-14.18d0,-12.88d0,-15.08d0,-11.48d0,-14.18d0,-13.38d0,&
        & -15.08d0, -9.98d0,-14.18d0,-12.88d0,-14.28d0, -9.28d0,-14.18d0,-12.18d0,&
        & -15.08d0, -9.98d0,-14.18d0,-12.88d0,-15.08d0,-11.48d0,-14.18d0,-13.38d0,&
        & -13.88d0, -8.78d0,-12.98d0,-11.68d0,-13.08d0, -8.08d0,-12.98d0,-10.98d0,&
        & -13.88d0, -8.78d0,-12.98d0,-11.68d0,-13.88d0,-10.28d0,-12.98d0,-12.18d0,&
        & -15.08d0, -9.98d0,-14.18d0,-12.88d0,-14.28d0, -9.28d0,-14.18d0,-12.18d0,&
        & -15.08d0, -9.98d0,-14.18d0,-12.88d0,-15.08d0,-11.48d0,-14.18d0,-13.38d0,&
        & -16.88d0,-11.78d0,-15.98d0,-14.68d0,-16.08d0,-11.08d0,-15.98d0,-13.98d0,&
        & -16.88d0,-11.78d0,-15.98d0,-14.68d0,-16.88d0,-13.28d0,-15.98d0,-15.18d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-17.58d0,-12.58d0,-17.48d0,-15.48d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-18.38d0,-14.78d0,-17.48d0,-16.68d0,&
        & -19.48d0,-14.38d0,-18.58d0,-17.28d0,-18.68d0,-13.68d0,-18.58d0,-16.58d0,&
        & -19.48d0,-14.38d0,-18.58d0,-17.28d0,-19.48d0,-15.88d0,-18.58d0,-17.78d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-17.58d0,-12.58d0,-17.48d0,-15.48d0,&
        & -18.38d0,-13.28d0,-17.48d0,-16.18d0,-18.38d0,-14.78d0,-17.48d0,-16.68d0,&
        & -15.78d0,-10.68d0,-14.88d0,-13.58d0,-14.98d0, -9.98d0,-14.88d0,-12.88d0,&
        & -15.78d0,-10.68d0,-14.88d0,-13.58d0,-15.78d0,-12.18d0,-14.88d0,-14.08d0,&
        & -14.68d0, -9.58d0,-13.78d0,-12.48d0,-13.88d0, -8.88d0,-13.78d0,-11.78d0,&
        & -14.68d0, -9.58d0,-13.78d0,-12.48d0,-14.68d0,-11.08d0,-13.78d0,-12.98d0,&
        & -15.78d0,-10.68d0,-14.88d0,-13.58d0,-14.98d0, -9.98d0,-14.88d0,-12.88d0,&
        & -15.78d0,-10.68d0,-14.88d0,-13.58d0,-15.78d0,-12.18d0,-14.88d0,-14.08d0,&
        & -13.88d0, -8.78d0,-12.98d0,-11.68d0,-13.08d0, -8.08d0,-12.98d0,-10.98d0,&
        & -13.88d0, -8.78d0,-12.98d0,-11.68d0,-13.88d0,-10.28d0,-12.98d0,-12.18d0,&
        & -14.38d0,-11.98d0,-13.78d0,-12.68d0,-15.08d0,-11.98d0,-15.28d0,-11.58d0,&
        & -16.78d0,-10.78d0,-16.38d0,-12.68d0,-15.28d0,-11.98d0,-15.28d0,-10.78d0,&
        & -15.08d0,-12.68d0,-14.48d0,-13.38d0,-15.78d0,-12.68d0,-15.98d0,-12.28d0,&
        & -17.48d0,-11.48d0,-17.08d0,-13.38d0,-15.98d0,-12.68d0,-15.98d0,-11.48d0,&
        & -16.78d0,-14.38d0,-16.18d0,-15.08d0,-17.48d0,-14.38d0,-17.68d0,-13.98d0,&
        & -19.18d0,-13.18d0,-18.78d0,-15.08d0,-17.68d0,-14.38d0,-17.68d0,-13.18d0,&
        & -15.28d0,-12.88d0,-14.68d0,-13.58d0,-15.98d0,-12.88d0,-16.18d0,-12.48d0,&
        & -17.68d0,-11.68d0,-17.28d0,-13.58d0,-16.18d0,-12.88d0,-16.18d0,-11.68d0,&
        & -11.98d0, -9.58d0,-11.38d0,-10.28d0,-12.68d0, -9.58d0,-12.88d0, -9.18d0,&
        & -14.38d0, -8.38d0,-13.98d0,-10.28d0,-12.88d0, -9.58d0,-12.88d0, -8.38d0,&
        & -11.98d0, -9.58d0,-11.38d0,-10.28d0,-12.68d0, -9.58d0,-12.88d0, -9.18d0,&
        & -14.38d0, -8.38d0,-13.98d0,-10.28d0,-12.88d0, -9.58d0,-12.88d0, -8.38d0,&
        & -10.78d0, -8.38d0,-10.18d0, -9.08d0,-11.48d0, -8.38d0,-11.68d0, -7.98d0,&
        & -13.18d0, -7.18d0,-12.78d0, -9.08d0,-11.68d0, -8.38d0,-11.68d0, -7.18d0,&
        & -11.98d0, -9.58d0,-11.38d0,-10.28d0,-12.68d0, -9.58d0,-12.88d0, -9.18d0,&
        & -14.38d0, -8.38d0,-13.98d0,-10.28d0,-12.88d0, -9.58d0,-12.88d0, -8.38d0,&
        & -13.78d0,-11.38d0,-13.18d0,-12.08d0,-14.48d0,-11.38d0,-14.68d0,-10.98d0,&
        & -16.18d0,-10.18d0,-15.78d0,-12.08d0,-14.68d0,-11.38d0,-14.68d0,-10.18d0,&
        & -15.28d0,-12.88d0,-14.68d0,-13.58d0,-15.98d0,-12.88d0,-16.18d0,-12.48d0,&
        & -17.68d0,-11.68d0,-17.28d0,-13.58d0,-16.18d0,-12.88d0,-16.18d0,-11.68d0,&
        & -16.38d0,-13.98d0,-15.78d0,-14.68d0,-17.08d0,-13.98d0,-17.28d0,-13.58d0,&
        & -18.78d0,-12.78d0,-18.38d0,-14.68d0,-17.28d0,-13.98d0,-17.28d0,-12.78d0,&
        & -15.28d0,-12.88d0,-14.68d0,-13.58d0,-15.98d0,-12.88d0,-16.18d0,-12.48d0,&
        & -17.68d0,-11.68d0,-17.28d0,-13.58d0,-16.18d0,-12.88d0,-16.18d0,-11.68d0,&
        & -12.68d0,-10.28d0,-12.08d0,-10.98d0,-13.38d0,-10.28d0,-13.58d0, -9.88d0,&
        & -15.08d0, -9.08d0,-14.68d0,-10.98d0,-13.58d0,-10.28d0,-13.58d0, -9.08d0,&
        & -11.58d0, -9.18d0,-10.98d0, -9.88d0,-12.28d0, -9.18d0,-12.48d0, -8.78d0,&
        & -13.98d0, -7.98d0,-13.58d0, -9.88d0,-12.48d0, -9.18d0,-12.48d0, -7.98d0,&
        & -12.68d0,-10.28d0,-12.08d0,-10.98d0,-13.38d0,-10.28d0,-13.58d0, -9.88d0,&
        & -15.08d0, -9.08d0,-14.68d0,-10.98d0,-13.58d0,-10.28d0,-13.58d0, -9.08d0,&
        & -10.78d0, -8.38d0,-10.18d0, -9.08d0,-11.48d0, -8.38d0,-11.68d0, -7.98d0,&
        & -13.18d0, -7.18d0,-12.78d0, -9.08d0,-11.68d0, -8.38d0,-11.68d0, -7.18d0,&
        & -11.48d0, -9.78d0,-10.58d0,-10.48d0,-13.18d0, -9.58d0,-14.08d0,-10.48d0,&
        & -13.18d0, -9.58d0,-14.08d0,-10.48d0,-13.18d0, -9.58d0,-14.08d0,-10.48d0,&
        & -12.18d0,-10.48d0,-11.28d0,-11.18d0,-13.88d0,-10.28d0,-14.78d0,-11.18d0,&
        & -13.88d0,-10.28d0,-14.78d0,-11.18d0,-13.88d0,-10.28d0,-14.78d0,-11.18d0,&
        & -13.88d0,-12.18d0,-12.98d0,-12.88d0,-15.58d0,-11.98d0,-16.48d0,-12.88d0,&
        & -15.58d0,-11.98d0,-16.48d0,-12.88d0,-15.58d0,-11.98d0,-16.48d0,-12.88d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        &  -9.08d0, -7.38d0, -8.18d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        & -10.78d0, -7.18d0,-11.68d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        &  -9.08d0, -7.38d0, -8.18d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        & -10.78d0, -7.18d0,-11.68d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        &  -7.88d0, -6.18d0, -6.98d0, -6.88d0, -9.58d0, -5.98d0,-10.48d0, -6.88d0,&
        &  -9.58d0, -5.98d0,-10.48d0, -6.88d0, -9.58d0, -5.98d0,-10.48d0, -6.88d0,&
        &  -9.08d0, -7.38d0, -8.18d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        & -10.78d0, -7.18d0,-11.68d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        & -10.88d0, -9.18d0, -9.98d0, -9.88d0,-12.58d0, -8.98d0,-13.48d0, -9.88d0,&
        & -12.58d0, -8.98d0,-13.48d0, -9.88d0,-12.58d0, -8.98d0,-13.48d0, -9.88d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -13.48d0,-11.78d0,-12.58d0,-12.48d0,-15.18d0,-11.58d0,-16.08d0,-12.48d0,&
        & -15.18d0,-11.58d0,-16.08d0,-12.48d0,-15.18d0,-11.58d0,-16.08d0,-12.48d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        &  -9.78d0, -8.08d0, -8.88d0, -8.78d0,-11.48d0, -7.88d0,-12.38d0, -8.78d0,&
        & -11.48d0, -7.88d0,-12.38d0, -8.78d0,-11.48d0, -7.88d0,-12.38d0, -8.78d0,&
        &  -8.68d0, -6.98d0, -7.78d0, -7.68d0,-10.38d0, -6.78d0,-11.28d0, -7.68d0,&
        & -10.38d0, -6.78d0,-11.28d0, -7.68d0,-10.38d0, -6.78d0,-11.28d0, -7.68d0,&
        &  -9.78d0, -8.08d0, -8.88d0, -8.78d0,-11.48d0, -7.88d0,-12.38d0, -8.78d0,&
        & -11.48d0, -7.88d0,-12.38d0, -8.78d0,-11.48d0, -7.88d0,-12.38d0, -8.78d0,&
        &  -7.88d0, -6.18d0, -6.98d0, -6.88d0, -9.58d0, -5.98d0,-10.48d0, -6.88d0,&
        &  -9.58d0, -5.98d0,-10.48d0, -6.88d0, -9.58d0, -5.98d0,-10.48d0, -6.88d0,&
        & -11.48d0, -9.78d0,-10.58d0,-10.48d0,-13.18d0, -9.58d0,-14.08d0,-10.48d0,&
        & -13.18d0, -9.58d0,-14.08d0,-10.48d0,-13.18d0, -9.58d0,-14.08d0,-10.48d0,&
        & -12.18d0,-10.48d0,-11.28d0,-11.18d0,-13.88d0,-10.28d0,-14.78d0,-11.18d0,&
        & -13.88d0,-10.28d0,-14.78d0,-11.18d0,-13.88d0,-10.28d0,-14.78d0,-11.18d0,&
        & -13.88d0,-12.18d0,-12.98d0,-12.88d0,-15.58d0,-11.98d0,-16.48d0,-12.88d0,&
        & -15.58d0,-11.98d0,-16.48d0,-12.88d0,-15.58d0,-11.98d0,-16.48d0,-12.88d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        &  -9.08d0, -7.38d0, -8.18d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        & -10.78d0, -7.18d0,-11.68d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        &  -9.08d0, -7.38d0, -8.18d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        & -10.78d0, -7.18d0,-11.68d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        &  -7.88d0, -6.18d0, -6.98d0, -6.88d0, -9.58d0, -5.98d0,-10.48d0, -6.88d0,&
        &  -9.58d0, -5.98d0,-10.48d0, -6.88d0, -9.58d0, -5.98d0,-10.48d0, -6.88d0,&
        &  -9.08d0, -7.38d0, -8.18d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        & -10.78d0, -7.18d0,-11.68d0, -8.08d0,-10.78d0, -7.18d0,-11.68d0, -8.08d0,&
        & -10.88d0, -9.18d0, -9.98d0, -9.88d0,-12.58d0, -8.98d0,-13.48d0, -9.88d0,&
        & -12.58d0, -8.98d0,-13.48d0, -9.88d0,-12.58d0, -8.98d0,-13.48d0, -9.88d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -13.48d0,-11.78d0,-12.58d0,-12.48d0,-15.18d0,-11.58d0,-16.08d0,-12.48d0,&
        & -15.18d0,-11.58d0,-16.08d0,-12.48d0,-15.18d0,-11.58d0,-16.08d0,-12.48d0,&
        & -12.38d0,-10.68d0,-11.48d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        & -14.08d0,-10.48d0,-14.98d0,-11.38d0,-14.08d0,-10.48d0,-14.98d0,-11.38d0,&
        &  -9.78d0, -8.08d0, -8.88d0, -8.78d0,-11.48d0, -7.88d0,-12.38d0, -8.78d0,&
        & -11.48d0, -7.88d0,-12.38d0, -8.78d0,-11.48d0, -7.88d0,-12.38d0, -8.78d0,&
        &  -8.68d0, -6.98d0, -7.78d0, -7.68d0,-10.38d0, -6.78d0,-11.28d0, -7.68d0,&
        & -10.38d0, -6.78d0,-11.28d0, -7.68d0,-10.38d0, -6.78d0,-11.28d0, -7.68d0,&
        &  -9.78d0, -8.08d0, -8.88d0, -8.78d0,-11.48d0, -7.88d0,-12.38d0, -8.78d0,&
        & -11.48d0, -7.88d0,-12.38d0, -8.78d0,-11.48d0, -7.88d0,-12.38d0, -8.78d0,&
        &  -7.88d0, -6.18d0, -6.98d0, -6.88d0, -9.58d0, -5.98d0,-10.48d0, -6.88d0,&
        &  -9.58d0, -5.98d0,-10.48d0, -6.88d0, -9.58d0, -5.98d0,-10.48d0, -6.88d0 /


        !=== ALTERNATIVE PARAMETERS ===!

        IF ( flag ) THEN

          !=== 1/1 Stack DG ===!

          READ(101,*)

          DO i=1,6
          DO j=1,6
          READ(101,'(16F7.2)')(g11(i,j,k),k=1,16)
          ENDDO
          ENDDO

          !=== 1/1 Stack DH ===!

          READ(101,*)

          DO i=1,6
          DO j=1,6
          READ(101,'(16F7.2)')(h11(i,j,k),k=1,16)
          ENDDO
          ENDDO

          !=== 1/2 Stack DG ===!

          READ(101,*)

          DO i=1,6
          DO j=1,6
          DO k=1,4
          READ(101,'(16F7.2)')(g12(i,j,m,k),m=1,16)
          ENDDO
          ENDDO
          ENDDO

          !=== 1/2 Stack DH ===!

          READ(101,*)

          DO i=1,6
          DO j=1,6
          DO k=1,4
          READ(101,'(16F7.2)')(h12(i,j,m,k),m=1,16)
          ENDDO
          ENDDO
          ENDDO

          !=== 2/2 Stack DG ===!

          READ(101,*)

          DO i=1,6
          DO j=1,6
          DO k=1,16
          READ(101,'(16F7.2)')(g22(i,j,k,m),m=1,16)
          ENDDO
          ENDDO
          ENDDO

          !=== 2/2 Stack DH ===!

          READ(101,*)

          DO i=1,6
          DO j=1,6
          DO k=1,16
          READ(101,'(16F7.2)')(h22(i,j,k,m),m=1,16)
          ENDDO
          ENDDO
          ENDDO

        ENDIF

        IF ( temp == 310.15d0 ) THEN

          stk_11(:,:,:) = g11(:,:,:)
          stk_12(:,:,:,:) = g12(:,:,:,:)
          stk_22(:,:,:,:) = g22(:,:,:,:)

          RETURN

        ENDIF

        !=== Compute DG @ TEMP ===!

        DO i=1,6
        DO j=1,6
        DO k=1,16

          s = tf * ( h11(i,j,k) - g11(i,j,k) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is) 

          stk_11(i,j,k) = h11(i,j,k) - s

        ENDDO
        ENDDO
        ENDDO

        DO i=1,6
        DO j=1,6
        DO k=1,16
        DO m=1,4

          s = tf * ( h12(i,j,k,m) - g12(i,j,k,m) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is) 

          stk_12(i,j,k,m) = h12(i,j,k,m) - s

        ENDDO
        ENDDO
        ENDDO
        ENDDO

        DO i=1,6
        DO j=1,6
        DO k=1,16
        DO m=1,16

          s = tf * ( h22(i,j,k,m) - g22(i,j,k,m) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is) 

          stk_22(i,j,k,m) = h22(i,j,k,m) - s

        ENDDO
        ENDDO
        ENDDO
        ENDDO

        RETURN

      END SUBROUTINE TINT
