! ==============================================================================
! Subroutine: TMISC (FLAG)
! 
! Purpose: Sets up the parameter tables for miscelaneous energy
!          terms (Default = TURNER 99 MODEL).
!
! Method: Uses the Turner 99 energy parameter table for RNA @ T=37.
!
! Arguments:
!
!            FLAG - (LOGICAL) If  TRUE: Read in new parameters.
!                             if FALSE: Use Default parameters.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependencies:
!
! Modules - FOLDING, CONSTANTS
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE TMISC (FLAG)

        USE Folding,   ONLY : elp_hp,elp_bu,elp_in,seq_t4,elp_t4,eninio,&
                            & enf1,enf2,epc,epg,eau,nt4
        USE Constants, ONLY : temp

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        LOGICAL, INTENT(IN) :: flag

        !=== VARIABLES ===!

        INTEGER :: i,is
        CHARACTER (LEN=6) :: cwrk,st4(200)

        DOUBLE PRECISION :: tf,s,ghp(30),gbu(30),gin(30)
        DOUBLE PRECISION :: gt4(200),gni(5),gnf1(3),gnf2(3)
        DOUBLE PRECISION :: gpc(3),gau,gpg,x,y

        DOUBLE PRECISION :: hhp(30),hbu(30),hin(30)
        DOUBLE PRECISION :: ht4(200),hni(5),hnf1(3),hnf2(3)
        DOUBLE PRECISION :: hpc(3),hau,hpg


        tf = temp / 310.150d0

        !=== DEFAULT PARAMETERS ===!

        !=== DG TERMS ===!

        !=== HP Loop Entropy DG ===!

        DATA (ghp(i),i=1,30) / &
                          &   0.00d0,  0.00d0,  5.70d0,  5.60d0,  5.60d0,  5.40d0,&
                          &   5.90d0,  5.60d0,  6.40d0,  6.50d0,  6.60d0,  6.70d0,&
                          &   6.80d0,  6.90d0,  6.90d0,  7.00d0,  7.10d0,  7.10d0,&
                          &   7.20d0,  7.20d0,  7.30d0,  7.30d0,  7.40d0,  7.40d0,&
                          &   7.50d0,  7.50d0,  7.50d0,  7.60d0,  7.60d0,  7.70d0 /

        !=== BL Loop Entropy DG ===!

        DATA (gbu(i),i=1,30) / &
                          &   3.80d0,  2.80d0,  3.20d0,  3.60d0,  4.00d0,  4.40d0,&
                          &   4.60d0,  4.70d0,  4.80d0,  4.90d0,  5.00d0,  5.10d0,&
                          &   5.20d0,  5.30d0,  5.40d0,  5.40d0,  5.50d0,  5.50d0,&
                          &   5.60d0,  5.70d0,  5.70d0,  5.80d0,  5.80d0,  5.80d0,&
                          &   5.90d0,  5.90d0,  6.00d0,  6.00d0,  6.00d0,  6.10d0 /

        !=== IN Loop Entropy DG ===!

        DATA (gin(i),i=1,30) / &
                          &   0.00d0,  0.00d0,  0.00d0,  1.70d0,  1.80d0,  2.00d0,&
                          &   2.20d0,  2.30d0,  2.40d0,  2.50d0,  2.60d0,  2.70d0,&
                          &   2.80d0,  2.90d0,  3.00d0,  3.00d0,  3.10d0,  3.10d0,&
                          &   3.20d0,  3.30d0,  3.30d0,  3.40d0,  3.40d0,  3.40d0,&
                          &   3.50d0,  3.50d0,  3.60d0,  3.60d0,  3.60d0,  3.70d0 /

        !=== Tetra-Loops DG ===!

        nt4 = 30

        DATA (st4(i),i=1,30) / &
                          & 'GGGGAC','GGUGAC','CGAAAG','GGAGAC','CGCAAG','GGAAAC',&
                          & 'CGGAAG','CUUCGG','CGUGAG','CGAAGG','CUACGG','GGCAAC',&
                          & 'CGCGAG','UGAGAG','CGAGAG','AGAAAU','CGUAAG','CUAACG',&
                          & 'UGAAAG','GGAAGC','GGGAAC','UGAAAA','AGCAAU','AGUAAU',&
                          & 'CGGGAG','AGUGAU','GGCGAC','GGGAGC','GUGAAC','UGGAAA' /

        DATA (gt4(i),i=1,30) / &
                          &  -3.00d0, -3.00d0, -3.00d0, -3.00d0, -3.00d0, -3.00d0,&
                          &  -3.00d0, -3.00d0, -3.00d0, -2.50d0, -2.50d0, -2.50d0,&
                          &  -2.50d0, -2.50d0, -2.00d0, -2.00d0, -2.00d0, -2.00d0,&
                          &  -2.00d0, -1.50d0, -1.50d0, -1.50d0, -1.50d0, -1.50d0,&
                          &  -1.50d0, -1.50d0, -1.50d0, -1.50d0, -1.50d0, -1.50d0 /

        !=== Ninio Equation DG ===!

        DATA (gni(i),i=1,5) / &
                          &   0.50d0,  0.50d0,  0.50d0,  0.50d0,  3.00d0 /

        !=== ENF1 Multi-Loop DG ===!

        DATA (gnf1(i),i=1,3) / &
                          &   3.40d0,  0.00d0,  0.40d0 /

        !=== ENF2 Multi-Loop DG ===!

        DATA (gnf2(i),i=1,3) / &
                          &  10.10d0, -0.30d0, -0.30d0 /

        !=== Poly-C Loop DG ===!

        DATA (gpc(i),i=1,3 ) / &
                          &   1.40d0,  1.60d0,  0.30d0 /

        !=== AU Penalty / Poly G Bonus DG ===!

        gau = +0.50d0
        gpg = -2.20d0


        !=== DH TERMS ===!

        !=== HP Loop Entropy DH ===!

        DATA (hhp(i),i=1,30) / &
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0 /

        !=== BL Loop Entropy DH ===!

        DATA (hbu(i),i=1,30) / &
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0 /

        !=== IN Loop Entropy DH ===!

        DATA (hin(i),i=1,30) / &
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0 /

        !=== Tetra-Loops DH ===!


        DATA (ht4(i),i=1,30) / &
                          &  -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0,&
                          &  -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0,&
                          &  -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0,&
                          &  -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0,&
                          &  -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0, -4.00d0 /

        !=== Ninio Equation DH ===!

        DATA (hni(i),i=1,5) / &
                          &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  3.00d0 /

        !=== ENF1 Multi-Loop DH ===!

        DATA (hnf1(i),i=1,3) / &
                          &   0.00d0,  0.00d0,  0.00d0 /

        !=== ENF2 Multi-Loop DH ===!

        DATA (hnf2(i),i=1,3) / &
                          &   0.00d0,  0.00d0,  0.00d0 /

        !=== Poly-C Loop DH ===!

        DATA (hpc(i),i=1,3 ) / &
                          &  18.60d0,  7.60d0,  3.40d0 /

        !=== AU Penalty / Poly G Bonus DH ===!

        hau =   0.00d0
        hpg = -14.80d0


        !=== ALTERNATIVE PARAMETERS ===!

        IF ( flag ) THEN

          !=== HP Loop Entropy DG ===!

          READ(101,*)

          READ(101,'(10F7.2)')(ghp(i),i=1,30)

          !=== HP Loop Entropy DH ===!

          READ(101,*)

          READ(101,'(10F7.2)')(hhp(i),i=1,30)

          !=== Bulge Loop Entropy DG ===!

          READ(101,*)

          READ(101,'(10F7.2)')(gbu(i),i=1,30)

          !=== Bulge Loop Entropy DH ===!

          READ(101,*)

          READ(101,'(10F7.2)')(hbu(i),i=1,30)

          !=== Internal Loop Entropy DG ===!

          READ(101,*)

          READ(101,'(10F7.2)')(gin(i),i=1,30)

          !=== Internal Loop Entropy DH ===!

          READ(101,*)

          READ(101,'(10F7.2)')(hin(i),i=1,30)

          !=== Ninio Terms DG ===!

          READ(101,*)

          READ(101,'(5F7.2)')(gni(i),i=1,5)

          !=== Ninio Terms DH ===!

          READ(101,*)

          READ(101,'(5F7.2)')(hni(i),i=1,5)

          !=== ENF1 Multi-Loop Terms DG ===!

          READ(101,*)

          READ(101,'(3F7.2)')(gnf1(i),i=1,3)

          !=== ENF1 Multi-Loop Terms DH ===!

          READ(101,*)

          READ(101,'(3F7.2)')(hnf1(i),i=1,3)


          !=== ENF2 Multi-Loop Terms DG ===!

          READ(101,*)

          READ(101,'(3F7.2)')(gnf2(i),i=1,3)

          !=== ENF2 Multi-Loop Terms DH ===!

          READ(101,*)

          READ(101,'(3F7.2)')(hnf2(i),i=1,3)

          !=== PolyC-Loop Terms DG ===!

          READ(101,*)

          READ(101,'(3F7.2)')(gpc(i),i=1,3)

          !=== PolyC-Loop Terms DH ===!

          READ(101,*)

          READ(101,'(3F7.2)')(hpc(i),i=1,3)

          !=== AU Penalty / Poly G Bonus DG ===!

          READ(101,*)

          READ(101,'(2F7.2)')gau,gpg

          !=== AU Penalty / Poly G Bonus DH ===!

          READ(101,*)

          READ(101,'(2F7.2)')hau,hpg

          !=== Tetra-Loop Bonuses ===!

          nt4 = 0

          READ(101,*)

          READ(101,'(A6,2F7.2)',IOSTAT=is)cwrk,x,y

          DO WHILE ( is == 0 )

            nt4 = nt4 + 1

            st4(nt4) = cwrk
            gt4(nt4) = x
            ht4(nt4) = y

            READ(101,'(A6,F7.2)',IOSTAT=is)cwrk,x,y

          ENDDO

        ENDIF


        IF ( temp == 310.15d0 ) THEN

          elp_hp(:) = ghp(:)
          elp_bu(:) = gbu(:)
          elp_in(:) = gin(:)
          seq_t4(:) = st4(:)
          elp_t4(:) = gt4(:)
          eninio(:) = gni(:)

          enf1(:) = gnf1(:)
          enf2(:) = gnf2(:)
          epc(:)  = gpc(:)

          eau = gau
          epg = gpg

          RETURN

        ENDIF

        !=== Compute DG @ TEMP ===!

        DO i=1,30

          !=== HP ===!

          s = tf * ( hhp(i) - ghp(i) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is)

          elp_hp(i) = hhp(i) - s

          !=== BU ===!

          s = tf * ( hbu(i) - gbu(i) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is)

          elp_bu(i) = hbu(i) - s

          !=== INT ===!

          s = tf * ( hin(i) - gin(i) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is)

          elp_in(i) = hin(i) - s

        ENDDO

        DO i=1,nt4

          !=== TetraLoops ===!

          s = tf * ( ht4(i) - gt4(i) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is)

          elp_t4(i) = ht4(i) - s

        ENDDO

        DO i=1,5

          !=== NINIO ===!

          s = tf * ( hni(i) - gni(i) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is)

          eninio(i) = hni(i) - s

        ENDDO

        DO i=1,3

          !=== ENF1 ===!

          s = tf * ( hnf1(i) - gnf1(i) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is)

          enf1(i) = hnf1(i) - s

          !=== ENF2 ===!

          s = tf * ( hnf2(i) - gnf2(i) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is)

          enf2(i) = hnf2(i) - s

          !=== POLY C ===!

          s = tf * ( hpc(i) - gpc(i) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is)

          epc(i) = hpc(i) - s

        ENDDO

        !=== AU Penalty ===!

        s = tf * ( hau - gau )

        s = 1.0d2 * s
        is = NINT(s)
        s = 1.0d-2 * DBLE(is)

        eau = hau - s

        !=== Poly G Bonus ===!

        s = tf * ( hpg - gpg )

        s = 1.0d2 * s
        is = NINT(s)
        s = 1.0d-2 * DBLE(is)

        epg = hpg - s

        RETURN

      END SUBROUTINE TMISC
