! ==============================================================================
! Subroutine: TRIBO (FLAG)
! 
! Purpose: Sets up the parameter tables for the ribosome kinetics.
!
!          For a list of parameters see:
!
!          Dykeman (2019)
!
! Arguments:
!
!            FLAG - (LOGICAL) If  TRUE: Read in new parameters.
!                             if FALSE: Use Default parameters.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependencies:
!
! Modules - RIBOSOME, CONSTANTS
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE TRIBO (FLAG)

        USE Ribosome
        USE Constants, ONLY : vol,cfac

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        LOGICAL, INTENT(IN) :: flag

        !=== VARIBLES ===!

        INTEGER :: i,j,k,i1,i2,i3
        INTEGER :: icog(64,64)

        DOUBLE PRECISION :: x,y,c
        DOUBLE PRECISION :: rf(6),rb(6)
        DOUBLE PRECISION :: xf(6),xb(6)
        DOUBLE PRECISION :: x1f(64,2),x1b(64,2),x2(64,2)


        c = cfac / vol

        !=== DEFAULT PARAMETERS ===!

        !=== 30 PIC Formation ===!

        r_pic_f(1) = 11.60d2 * c
        r_pic_f(2) = 28.00d1 * c
        r_pic_f(3) = 12.00d0 * c
        r_pic_f(4) =  5.00d0 * c

        r_pic_b(1) = 36.00d0
        r_pic_b(2) = 12.00d0
        r_pic_b(3) =  0.02d0
        r_pic_b(4) =  1.50d0

        !=== Ribosome 30PIC Binding ===!

        r_b1f = 25.00d0 * c
        r_b2f = 25.00d1 * c

        r_b1b = 25.00d0
        r_b2b =  2.50d0

        r_b3f = 1.00d6

        !=== Ribosome Initiation ===!

        r_i1f = 34.00d0 * c
        r_i2f =  4.80d0 * c

        r_i1b = 35.00d0
        r_i2b = 12.00d0

        r_i3  = 44.00d0
        r_i4f = 24.00d0
        r_i4b =  2.10d0
        r_i5  =  3.70d0
        r_i6  =  6.00d0
        r_i7  =  3.20d0
        r_i8  =  6.20d0

        !=== Ribosome Elongation (COGNATE) ===!

        rf(1) = 94.00d0 * c
        rf(2) = 33.00d2
        rf(3) = 17.00d2
        rf(4) = 54.00d1
        rf(5) = 35.00d1
        rf(6) =  1.00d0

        rb(1) = 23.00d2
        rb(2) =  2.00d0

        !=== Ribosome Elongation (NEAR-COGNATE) ===!

        xf(1) = 94.00d0 * c
        xf(2) = 33.00d2
        xf(3) =  4.00d0
        xf(4) = 54.00d1
        xf(5) =  0.27d0
        xf(6) =  7.00d0

        xb(1) = 23.00d2
        xb(2) = 43.00d2

        !=== Ribosome Translocation ===!

        r_e7f = 70.0d0 * c

        r_e7b = 35.0d0
        r_e8  = 25.0d1
        r_e9  = 30.0d1
        r_e10 = 20.0d1

        !=== Ribosome Termination ===!

        DATA (x1f(i,1),i=1,64) / &
        &  18.00d0,  0.00d0, 18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  18.00d0,  0.00d0, 18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  18.00d0,  0.00d0, 18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  60.00d0, 18.00d0, 60.00d0, 18.00d0, 18.00d0,  0.00d0, 18.00d0,  0.00d0,&
        &  18.00d0,  0.00d0, 18.00d0,  0.00d0, 18.00d0,  0.00d0, 18.00d0,  0.00d0 /

        DATA (x1b(i,1),i=1,64) / &
        &  50.00d1,  0.00d0, 50.00d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  29.80d1,  0.00d0, 44.80d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  50.00d1,  0.00d0, 50.00d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.10d0, 17.60d1,  0.10d0, 60.00d0, 14.90d1,  0.00d0, 14.80d1,  0.00d0,&
        &  35.80d1,  0.00d0, 18.60d1,  0.00d0, 17.90d1,  0.00d0, 14.90d1,  0.00d0 /

        DATA (x2(i,1),i=1,64) / &
        &  1.00d-3,  0.00d0, 1.00d-3,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  2.00d-3,  0.00d0, 3.00d-3,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  1.00d-3,  0.00d0, 1.00d-3,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  23.00d0, 0.29d-1, 23.00d0,  0.20d0, 0.34d-1,  0.00d0, 0.26d-1,  0.00d0,&
        &  9.00d-3,  0.00d0, 0.13d-1,  0.00d0, 0.22d-1,  0.00d0, 0.41d-1,  0.00d0 /

        DATA (x1f(i,2),i=1,64) / &
        &  18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  18.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  23.00d0, 18.00d0, 18.00d0, 18.00d0, 18.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  23.00d0, 18.00d0, 18.00d0, 18.00d0, 18.00d0,  0.00d0,  0.00d0,  0.00d0 /


        DATA (x1b(i,2),i=1,64) / &
        &  50.00d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  34.40d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  52.20d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  28.50d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  30.80d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  34.40d1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &   1.30d0, 59.40d1, 29.50d1, 59.40d1, 25.00d2,  0.00d0,  0.00d0,  0.00d0,&
        &   1.30d0, 59.40d1, 36.80d1, 50.70d1, 14.20d1,  0.00d0,  0.00d0,  0.00d0 /

        DATA (x2(i,2),i=1,64) / &
        &  1.00d-3,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  1.00d-3,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  0.16d-1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  0.24d-1,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  1.00d-3,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        &  3.00d-3,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,  0.00d0,&
        & 23.000d0,  0.05d0,  0.21d0, 0.18d-1,  0.48d0,  0.00d0,  0.00d0,  0.00d0,&
        & 23.000d0, 0.44d-1,  0.19d0, 0.29d-1, 8.00d-3,  0.00d0,  0.00d0,  0.00d0 /


        r_t1f(:,:) = x1f(:,:) * c
        r_t1b(:,:) = x1b(:,:)
        r_t2(:,:)  = x2(:,:)


        !=== Ribosome term/part 2 ===!

        r_t3f = 36.0d0 * c
        r_t3b = 1.30d0

        r_t4f = 18.09d0
        r_t4b = 2.270d0

        r_t5(1) = 13.34d0
        r_t5(2) = 13.34d0

        r_t6  =  5.00d0
        r_t7  = 15.48d0
        r_t8  = 13.34d0

        r_t9(1) = 0.20d0
        r_t9(2) = 2.00d0

        !=== Ribosome Recycling ===!

        r_r1f = 15.0d0 * c
        r_r2f = 70.0d0 * c

        r_r1b = 58.0d0
        r_r2b = 35.0d0
        r_r3  = 4.30d1
        r_r4  = 2.50d1


        !=== ALTERNATIVE PARAMETERS ===!

        IF ( flag ) THEN

          !=== 30 PIC Formation ===!

          READ(101,*)

          DO i=1,4

            READ(101,*)x,y

            r_pic_f(i) = x * c
            r_pic_b(i) = y

          ENDDO

          !=== Ribosome Binding ===!

          READ(101,*)

          READ(101,*)x,y

          r_b1f = x * c
          r_b1b = y

          READ(101,*)x,y

          r_b2f = x * c
          r_b2b = y

          READ(101,*)r_b3f

          !=== Ribosome Initiation ===!

          READ(101,*)

          READ(101,*)x,y

          r_i1f = x * c
          r_i1b = y

          READ(101,*)x,y

          r_i2f = x * c
          r_i2b = y

          READ(101,*)r_i3
          READ(101,*)r_i4f,r_i4b
          READ(101,*)r_i5
          READ(101,*)r_i6

          !=== Ribosome Elongation (COGNATE) ===!

          READ(101,*)

          READ(101,*)x,y

          rf(1) = x * c
          rb(1) = y

          READ(101,*)rf(2),rb(2)
          READ(101,*)rf(3)
          READ(101,*)rf(4)
          READ(101,*)rf(5)
          READ(101,*)rf(6)

          !=== Ribosome Elongation (NEAR-COGNATE) ===!

          READ(101,*)x,y

          xf(1) = x * c
          xb(1) = y

          READ(101,*)xf(2),xb(2)
          READ(101,*)xf(3)
          READ(101,*)xf(4)
          READ(101,*)xf(5)
          READ(101,*)xf(6)

          !=== Ribosome Translocation ===!

          READ(101,*)

          READ(101,*)x,y

          r_e7f = x * c
          r_e7b = y

          READ(101,*)r_e8
          READ(101,*)r_e9
          READ(101,*)r_e10

          !=== Ribosome Termination ===!

          READ(101,*)

          READ(101,'(8E12.4)')(r_t1f(i,1),i=1,64)
          READ(101,'(8E12.4)')(r_t1b(i,1),i=1,64)
          READ(101,'(8E12.4)')(r_t2(i,1) ,i=1,64)

          READ(101,'(8E12.4)')(r_t1f(i,2),i=1,64)
          READ(101,'(8E12.4)')(r_t1b(i,2),i=1,64)
          READ(101,'(8E12.4)')(r_t2(i,2) ,i=1,64)

          r_t1f(:,:) = r_t1f(:,:) * c

          READ(101,*)x,y

          r_t3f = x * c
          r_t3b = y

          READ(101,*)r_t4f,r_t4b
          READ(101,*)r_t5(1),r_t5(2)
          READ(101,*)r_t6
          READ(101,*)r_t7
          READ(101,*)r_t8
          READ(101,*)r_t9(1),r_t9(2)

          !=== Ribosome Recycling ===!

          READ(101,*)

          READ(101,*)x,y

          r_r1f = x * c
          r_r1b = y

          READ(101,*)x,y

          r_r2f = x * c
          r_r2b = y

          READ(101,*)r_r3
          READ(101,*)r_r4

        ENDIF


        !=== SET ELONGATION TABLES ===!

        !=== EColi TRNA Recognition ===!

        icog(:,:) = 0

        !=== NEAR COGNATE ===!

        DO i1=1,4
        DO i2=1,4
        DO i3=1,4

          i = (i1-1) * 16 + (i2-1) * 4 + i3

          !=== First Position ===!

          DO k=1,4

            j = (k-1) * 16 + (i2-1) * 4 + i3

            IF ( k /= i2 ) icog(i,j) = 2

          ENDDO

          !=== Second Position ===!

          DO k=1,4

            j = (i1-1) * 16 + (k-1) * 4 + i3

            IF ( k /= i2 ) icog(i,j) = 2

          ENDDO

          !=== Third Position ===!

          DO k=1,4

            j = (i1-1) * 16 + (i2-1) * 4 + k

            IF ( k /= i3 ) icog(i,j) = 2

          ENDDO

        ENDDO
        ENDDO
        ENDDO

        !=== COGNATE ===!

        icog(:,:) = 0 !ECD - NO NEAR COGNATE REACTIONS!

        DO i=1,64
        icog(i,i) = 1
        ENDDO

        !=== LYS ===!

        i = 1  !AAA TRNA
        i1 = 3 !AAG Codon
        icog(i1,i) = 1

        !=== ASN ===!

        i = 2  !AAC TRNA
        i1 = 4 !AAU Codon
        icog(i1,i) = 1

        !=== THR-4 ===!

        i = 5  !ACA TRNA
        i1 = 8 !ACU Codon
        i2 = 7 !ACG Codon
        icog(i1,i) = 1
        icog(i2,i) = 1

        !=== THR-1+3 ===!

        i = 6  !ACC TRNA
        i1 = 8 !ACU Codon
        icog(i1,i) = 1

        !=== SER-3 ===!

        i = 10  !AGC TRNA
        i1 = 12 !AGU Codon
        icog(i1,i) = 1

        !=== ILE-1 ===!

        i = 14  !AUC TRNA
        i1 = 16 !AUU Codon
        icog(i1,i) = 1

        !=== HIS ===!

        i = 18  !CAC TRNA
        i1 = 20 !CAU Codon
        icog(i1,i) = 1

        !=== PRO-3 ===!

        i = 21  !CCA TRNA
        i1 = 24 !CCU Codon
        i2 = 23 !CCG Codon
        icog(i1,i) = 1
        icog(i2,i) = 1

        !=== PRO-2 ===!

        i = 22  !CCC TRNA
        i1 = 24 !CCU Codon
        icog(i1,i) = 1

        !=== ARG-2 ===!

        i = 28  !CGU TRNA
        i1 = 25 !CGA Codon
        i2 = 26 !CGC Codon
        icog(i1,i) = 1
        icog(i2,i) = 1

        !=== LEU-3 ===!

        i = 29  !CUA TRNA
        i1 = 31 !CUG Codon
        icog(i1,i) = 1

        !=== LEU-2 ===!

        i = 30  !CUC TRNA
        i1 = 32 !CUU Codon
        icog(i1,i) = 1

        !=== GLU-2 ===!

        i = 33  !GAA TRNA
        i1 = 35 !GAG Codon
        icog(i1,i) = 1

        !=== ASP-1 ===!

        i = 34  !GAC TRNA
        i1 = 36 !GAU Codon
        icog(i1,i) = 1

        !=== ALA-1B ===!

        i = 37  !GCA TRNA
        i1 = 40 !GCU Codon
        i2 = 39 !GCG Codon
        icog(i1,i) = 1
        icog(i2,i) = 1

        !=== GLY-2 ===!

        i = 41  !GGA TRNA
        i1 = 43 !GGG Codon
        icog(i1,i) = 1

        !=== GLY-3 ===!

        i = 42  !GGC TRNA
        i1 = 44 !GGU Codon
        icog(i1,i) = 1

        !=== VAL-1 ===!

        i = 45  !GUA TRNA
        i1 = 48 !GUU Codon
        i2 = 47 !GUG Codon
        icog(i1,i) = 1
        icog(i2,i) = 1

        !=== VAL-2A+B ===!

        i = 46  !GUC TRNA
        i1 = 48 !GUU Codon
        icog(i1,i) = 1

        !=== TYR-1+2 ===!

        i = 50  !UAC TRNA
        i1 = 52 !UAU Codon
        icog(i1,i) = 1

        !=== SER-1 ===!

        i = 53  !UCA TRNA
        i1 = 56 !UCU Codon
        i2 = 55 !UCG Codon
        icog(i1,i) = 1
        icog(i2,i) = 1

        !=== SER-5 ===!

        i = 54  !UCC TRNA
        i1 = 56 !UCU Codon
        icog(i1,i) = 1

        !=== CYS ===!

        i = 58  !UGC TRNA
        i1 = 60 !UGU Codon
        icog(i1,i) = 1

        !=== LEU-5 ===!

        i = 61  !UUA TRNA
        i1 = 63 !UUG Codon
        icog(i1,i) = 1

        !=== PHE ===!

        i = 62  !UUC TRNA
        i1 = 64 !UUU Codon
        icog(i1,i) = 1


        !=== Set Rate Table ===!

        r_e1f = rf(1)
        r_e1b = rb(1)

        r_e2f(:,:) = 0.0d0
        r_e2b(:,:) = 0.0d0

        r_e3(:,:) = 0.0d0
        r_e4(:,:) = 0.0d0
        r_e5(:,:) = 0.0d0
        r_e6(:,:) = 0.0d0

        DO i=1,64
        DO j=1,64

          IF ( icog(i,j) == 1 ) THEN

            r_e2f(i,j) = rf(2)
            r_e2b(i,j) = rb(2)

            r_e3(i,j) = rf(3)
            r_e4(i,j) = rf(4)
            r_e5(i,j) = rf(5)
            r_e6(i,j) = rf(6)

          ELSEIF ( icog(i,j) == 2 ) THEN

            r_e2f(i,j) = xf(2)
            r_e2b(i,j) = xb(2)

            r_e3(i,j) = xf(3)
            r_e4(i,j) = xf(4)
            r_e5(i,j) = xf(5)
            r_e6(i,j) = xf(6)

          ENDIF

          !=== No ReadThrough ===!

          IF ( i == 49 .or. i == 51 .or. i == 57 ) THEN

            r_e2f(i,j) = 0.0d0
            r_e2b(i,j) = 0.0d0

          ENDIF

        ENDDO
        ENDDO

        RETURN

      END SUBROUTINE TRIBO
