! ==============================================================================
! Subroutine: TSTACK (FLAG)
! 
! Purpose: Sets up the parameter tables for stacking (bp,hp,in,dangles)
!          energy terms (Default = TURNER 99 MODEL).
!
! Method: Uses the Turner 99 energy parameter table for RNA @ T=37.
!
!           3p Dangle       5p Dangle        Stacking
!              i j             i j             i j
!         5'-- A X --3'   5'-- A   --3'   5'-- A X --3'
!         3'-- U   --5'   3'-- U X --5'   3'-- U Y --5'
!
! Arguments:
!
!            FLAG - (LOGICAL) If  TRUE: Read in new parameters.
!                             if FALSE: Use Default parameters.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependencies:
!
! Modules - FOLDING, CONSTANTS
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE TSTACK (FLAG)

        USE Folding,   ONLY : stk_bp,stk_hp,stk_in,dng_5p,dng_3p
        USE Constants, ONLY : temp

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        LOGICAL, INTENT(IN) :: flag

        !=== VARIBLES ===!

        INTEGER :: i,j,is

        DOUBLE PRECISION :: tf,s,g5p(6,4),g3p(6,4)
        DOUBLE PRECISION :: gin(6,16),ghp(6,16)
        DOUBLE PRECISION :: gbp(6,6)

        DOUBLE PRECISION :: h5p(6,4),h3p(6,4)
        DOUBLE PRECISION :: hin(6,16),hhp(6,16)
        DOUBLE PRECISION :: hbp(6,6)


        tf = temp / 310.150d0

        !=== DEFAULT PARAMETERS ===!

        !=== DG TERMS ===!

        !=== 5' Dangle ===!

        DATA ((g5p(i,j),j=1,4),i=1,6) / &
                          &  -0.30d0, -0.10d0, -0.20d0, -0.20d0,&
                          &  -0.20d0, -0.30d0, -0.00d0, -0.00d0,&
                          &  -0.50d0, -0.30d0, -0.20d0, -0.10d0,&
                          &  -0.30d0, -0.10d0, -0.20d0, -0.20d0,&
                          &  -0.30d0, -0.30d0, -0.40d0, -0.20d0,&
                          &  -0.30d0, -0.30d0, -0.40d0, -0.20d0 /

        !=== 3' Dangle ===!

        DATA ((g3p(i,j),j=1,4),i=1,6) / &
                          &  -0.80d0, -0.50d0, -0.80d0, -0.60d0,&
                          &  -1.70d0, -0.80d0, -1.70d0, -1.20d0,&
                          &  -1.10d0, -0.40d0, -1.30d0, -0.60d0,&
                          &  -0.80d0, -0.50d0, -0.80d0, -0.60d0,&
                          &  -0.70d0, -0.10d0, -0.70d0, -0.10d0,&
                          &  -0.70d0, -0.10d0, -0.70d0, -0.10d0 /

        !=== BP Stacking ===!

        DATA ((gbp(i,j),j=1,6),i=1,6) / &
                          &  -0.90d0, -2.20d0, -2.10d0, -0.60d0, -1.10d0, -1.40d0,&
                          &  -2.10d0, -3.30d0, -2.40d0, -1.40d0, -2.10d0, -2.10d0,&
                          &  -2.40d0, -3.40d0, -3.30d0, -1.50d0, -2.20d0, -2.50d0,&
                          &  -1.30d0, -2.50d0, -2.10d0, -0.50d0, -1.40d0, +1.30d0,&
                          &  -1.30d0, -2.40d0, -2.10d0, -1.00d0, -0.90d0, -1.30d0,&
                          &  -1.00d0, -1.50d0, -1.40d0, +0.30d0, -0.60d0, -0.50d0 /

        !=== HP Stacking ===!

        DATA ((ghp(i,j),j=1,16),i=1,6) / &
        &  -0.30d0, -0.50d0, -0.30d0, -0.30d0, -0.10d0, -0.20d0, -1.50d0, -0.20d0,&
        &  -1.10d0, -1.20d0, -0.20d0,  0.20d0, -0.30d0, -0.30d0, -0.60d0, -1.10d0,&
        &  -1.50d0, -1.50d0, -1.40d0, -1.80d0, -1.00d0, -0.90d0, -2.90d0, -0.80d0,&
        &  -2.20d0, -2.00d0, -1.60d0, -1.10d0, -1.70d0, -1.40d0, -1.80d0, -2.00d0,&
        &  -1.10d0, -1.50d0, -1.30d0, -2.10d0, -1.10d0, -0.70d0, -2.40d0, -0.50d0,&
        &  -2.40d0, -2.90d0, -1.40d0, -1.20d0, -1.90d0, -1.00d0, -2.20d0, -1.50d0,&
        &   0.20d0, -0.50d0, -0.30d0, -0.30d0, -0.10d0, -0.20d0, -1.50d0, -0.20d0,&
        &  -0.90d0, -1.10d0, -0.30d0,  0.00d0, -0.30d0, -0.30d0, -0.40d0, -1.10d0,&
        &  -0.50d0, -0.30d0, -0.60d0, -0.50d0, -0.20d0, -0.10d0, -1.20d0, -0.00d0,&
        &  -1.40d0, -1.20d0, -0.70d0, -0.20d0, -0.30d0, -0.10d0, -0.50d0, -0.80d0,&
        &  -0.50d0, -0.30d0, -0.60d0, -0.50d0, -0.20d0, -0.10d0, -1.70d0,  0.00d0,&
        &  -0.80d0, -1.20d0, -0.30d0, -0.70d0, -0.60d0, -0.10d0, -0.60d0, -0.80d0 /

        !=== IN Stacking ===!

        DATA ((gin(i,j),j=1,16),i=1,6) / &
        &   0.70d0,  0.70d0, -0.40d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,&
        &  -0.40d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.00d0,&
        &  -0.00d0, -0.00d0, -1.10d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0,&
        &  -1.10d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0, -0.70d0,&
        &  -0.00d0, -0.00d0, -1.10d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0,&
        &  -1.10d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0, -0.00d0, -0.70d0,&
        &   0.70d0,  0.70d0, -0.40d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,&
        &  -0.40d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.00d0,&
        &   0.70d0,  0.70d0, -0.40d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,&
        &  -0.40d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.00d0,&
        &   0.70d0,  0.70d0, -0.40d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,&
        &  -0.40d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.70d0,  0.00d0 /

        !=== DH TERMS ===!

        !=== 5' Dangle ===!

        DATA ((h5p(i,j),j=1,4),i=1,6) / &
                          &  -0.50d0,  6.90d0, -0.60d0, -0.60d0,&
                          &  -1.60d0,  0.70d0, -4.60d0, -0.40d0,&
                          &  -2.40d0,  3.30d0,  0.80d0, -1.40d0,&
                          &  -1.50d0,  5.10d0,  0.10d0,  1.00d0,&
                          &   1.60d0,  2.20d0,  0.70d0,  3.10d0,&
                          &   1.60d0,  2.20d0,  0.70d0,  3.10d0 /

        !=== 3' Dangle ===!

        DATA ((h3p(i,j),j=1,4),i=1,6) / &
                          &  -4.90d0, -0.90d0, -5.50d0, -2.30d0,&
                          &  -9.00d0, -4.10d0, -8.60d0, -7.50d0,&
                          &  -7.40d0, -2.80d0, -6.40d0, -3.60d0,&
                          &  -4.90d0, -0.90d0, -5.50d0, -2.30d0,&
                          &  -5.70d0, -0.70d0, -5.80d0, -2.20d0,&
                          &  -7.40d0, -2.40d0, -7.20d0, -4.90d0 /

        !=== BP Stacking ===!

        DATA ((hbp(i,j),j=1,6),i=1,6) / &
                          &  -6.80d0,-11.40d0,-10.50d0, -3.20d0, -9.40d0, -8.80d0,&
                          & -10.40d0,-13.40d0,-10.60d0, -5.60d0,-10.50d0,-12.10d0,&
                          & -12.40d0,-14.90d0,-13.40d0, -8.30d0,-11.40d0,-12.60d0,&
                          & -12.80d0,-12.60d0,-12.10d0,-13.50d0, -8.80d0,-14.60d0,&
                          &  -7.70d0,-12.40d0,-10.40d0, -7.00d0, -6.80d0,-12.80d0,&
                          &  -7.00d0, -8.30d0, -5.60d0, -9.30d0, -3.20d0,-13.50d0 /

        !=== HP Stacking ===!

        DATA ((hhp(i,j),j=1,16),i=1,6) / &
        &  -4.30d0, -6.00d0, -6.00d0, -6.00d0, -2.60d0, -2.40d0, -2.40d0, -2.40d0,&
        &  -3.40d0, -6.90d0, -6.90d0, -6.90d0, -3.30d0, -3.30d0, -3.30d0, -3.30d0,&
        & -10.30d0, -9.50d0,-10.30d0,-10.30d0, -5.20d0, -4.50d0, -5.20d0, -6.70d0,&
        &  -9.40d0, -9.40d0, -9.40d0, -9.40d0, -8.10d0, -7.40d0, -8.10d0, -8.60d0,&
        &  -5.20d0, -8.80d0, -5.60d0, -8.80d0, -7.20d0, -3.10d0, -3.10d0, -3.90d0,&
        &  -7.10d0, -7.40d0, -6.20d0, -7.40d0, -5.00d0, -5.00d0, -5.00d0, -5.70d0,&
        &  -4.30d0, -6.00d0, -6.00d0, -6.00d0, -2.60d0, -2.40d0, -2.40d0, -2.40d0,&
        &  -3.40d0, -6.90d0, -6.90d0, -6.90d0, -3.30d0, -3.30d0, -3.30d0, -3.30d0,&
        &  -4.00d0, -6.30d0, -8.90d0, -5.90d0, -4.30d0, -5.10d0, -2.00d0, -1.80d0,&
        &  -3.80d0, -6.80d0, -8.90d0, -6.80d0, -2.80d0, -1.40d0, -2.80d0, -1.40d0,&
        &  -7.20d0, -7.90d0, -9.60d0, -8.10d0, -4.80d0, -4.80d0, -3.60d0, -4.80d0,&
        &  -6.60d0, -8.10d0, -9.20d0, -8.10d0, -5.50d0, -4.40d0, -5.50d0, -3.60d0 /

        !=== IN Stacking ===!

        DATA ((hin(i,j),j=1,16),i=1,6) / &
        &  -4.30d0, -6.00d0, -6.00d0, -6.00d0, -2.60d0, -2.40d0, -2.40d0, -2.40d0,&
        &  -3.40d0, -6.90d0, -6.90d0, -6.90d0, -3.30d0, -3.30d0, -3.30d0, -3.30d0,&
        & -10.30d0, -9.50d0,-10.30d0,-10.30d0, -5.20d0, -4.50d0, -5.20d0, -6.70d0,&
        &  -9.40d0, -9.40d0, -9.40d0, -9.40d0, -8.10d0, -7.40d0, -8.10d0, -8.60d0,&
        &  -5.20d0, -8.80d0, -5.60d0, -8.80d0, -7.20d0, -3.10d0, -3.10d0, -3.90d0,&
        &  -7.10d0, -7.40d0, -6.20d0, -7.40d0, -5.00d0, -5.00d0, -5.00d0, -5.70d0,&
        &  -4.30d0, -6.00d0, -6.00d0, -6.00d0, -2.60d0, -2.40d0, -2.40d0, -2.40d0,&
        &  -3.40d0, -6.90d0, -6.90d0, -6.90d0, -3.30d0, -3.30d0, -3.30d0, -3.30d0,&
        &  -4.00d0, -6.30d0, -8.90d0, -5.90d0, -4.30d0, -5.10d0, -2.00d0, -1.80d0,&
        &  -3.80d0, -6.80d0, -8.90d0, -6.80d0, -2.80d0, -1.40d0, -2.80d0, -1.40d0,&
        &  -7.20d0, -7.90d0, -9.60d0, -8.10d0, -4.80d0, -4.80d0, -3.60d0, -4.80d0,&
        &  -6.60d0, -8.10d0, -9.20d0, -8.10d0, -5.50d0, -4.40d0, -5.50d0, -3.60d0 /


        !=== ALTERNATIVE PARAMETERS ===!

        IF ( flag ) THEN

          !=== 5' Dangle DG ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(4F7.2)')(g5p(i,j),j=1,4)
          ENDDO

          !=== 5' Dangle DH ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(4F7.2)')(h5p(i,j),j=1,4)
          ENDDO

          !=== 3' Dangle DG ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(4F7.2)')(g3p(i,j),j=1,4)
          ENDDO

          !=== 3' Dangle DH ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(4F7.2)')(h3p(i,j),j=1,4)
          ENDDO

          !=== BP Stack DG ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(6F7.2)')(gbp(i,j),j=1,6)
          ENDDO

          !=== BP Stack DH ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(6F7.2)')(hbp(i,j),j=1,6)
          ENDDO

          !=== HP Stack DG ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(16F7.2)')(ghp(i,j),j=1,16)
          ENDDO

          !=== HP Stack DH ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(16F7.2)')(hhp(i,j),j=1,16)
          ENDDO

          !=== INT Stack DG ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(16F7.2)')(gin(i,j),j=1,16)
          ENDDO

          !=== INT Stack DH ===!

          READ(101,*)

          DO i=1,6
          READ(101,'(16F7.2)')(hin(i,j),j=1,16)
          ENDDO

        ENDIF


        IF ( temp == 310.15d0 ) THEN

          dng_5p(:,:) = g5p(:,:)
          dng_3p(:,:) = g3p(:,:)
          stk_bp(:,:) = gbp(:,:)
          stk_hp(:,:) = ghp(:,:)
          stk_in(:,:) = gin(:,:)

          RETURN

        ENDIF

        !=== Compute DG @ TEMP ===!

        DO i=1,6
        DO j=1,4

          !=== 5p ===!

          s = tf * ( h5p(i,j) - g5p(i,j) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is) 

          dng_5p(i,j) = h5p(i,j) - s

          !=== 3p ===!

          s = tf * ( h3p(i,j) - g3p(i,j) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is) 

          dng_3p(i,j) = h3p(i,j) - s

        ENDDO
        ENDDO

        DO i=1,6
        DO j=1,6

          !=== Stack ===!

          s = tf * ( hbp(i,j) - gbp(i,j) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is) 

          stk_bp(i,j) = hbp(i,j) - s

        ENDDO
        ENDDO

        DO i=1,6
        DO j=1,16

          !=== HP ===!

          s = tf * ( hhp(i,j) - ghp(i,j) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is) 

          stk_hp(i,j) = hhp(i,j) - s

          !=== INT ===!

          s = tf * ( hin(i,j) - gin(i,j) )

          s = 1.0d2 * s
          is = NINT(s)
          s = 1.0d-2 * DBLE(is) 

          stk_in(i,j) = hin(i,j) - s

        ENDDO
        ENDDO

        RETURN

      END SUBROUTINE TSTACK
