! ==============================================================================
! Subroutine: MRNA_FIRE (RNA,AMAX,IP)
! 
! Purpose: Computes the possible reactions for an mRNA HP or ML structure
!          in the protein synthesis model.
!
! Method:  The info array holds the following information:
!
!          istate = info(1,i) --> state of the RNA or enzyme object
!            ipro = info(2,i) --> protein number bound to the helix
!           ileaf = info(3,i) --> link to the first helix in the multi-loop
!           iroot = info(4,i) --> link to the root helix (or 0 if ext loop)
!            iloc = info(5,i) --> 5' nt of closing base pair
!            jloc = info(6,i) --> 3' nt of closing base pair
!
! Arguments:
!
!             RNA - Class structure containing information on the
!                   RNA and its possible reactions.
!            AMAX - Threshold at which to fire a reaction.
!              IP - Protein ID number which is involved in the next
!                   reaction to fire.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - PROTEIN
! Functions -
! Subroutines - FOLD_FIRE, PRO_FIRE, RBS_FIRE, RIBO_FIRE
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE MRNA_FIRE (RNA,AMAX,IP)

        USE Protein

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        TYPE(MRNA), INTENT(INOUT) :: rna
        INTEGER, INTENT(IN) :: ip

        DOUBLE PRECISION, INTENT(IN) :: amax

        !=== VARIABLES ===!

        INTEGER :: i,j,n,istate,ipro
        INTEGER :: indx,icase

        DOUBLE PRECISION :: r,am,xp


        IF ( ip == 0 ) xp = 1.0d0
        IF ( ip  > 0 ) xp = DBLE(npro(ip))


        !=== Folding RNA Reaction ===!

        IF ( ip < 0 ) THEN

           write(*,*)'ERROR - foldfire'
           STOP
!          CALL FOLD_FIRE (rna)

          RETURN

        ENDIF


        !=== Non-Folding RNA Reaction ===!

        am = amax

        n = mxobj / 2
        i = n

        DO WHILE ( MOD(n,2) == 0 )

          n = n / 2
          j = i - n

          r = rna% psum(ip,j) * xp

          IF ( r >= am ) THEN
            i = i - n
          ELSE
            i = i + n
            am = am - r
          ENDIF

        ENDDO

        r = rna% atot(ip,i) * xp

        IF ( r >= am ) THEN
          indx = i
        ELSE
          am = am - r
          indx = i + 1
        ENDIF


        !=== RNA state info ===!

        istate = rna% info(1,indx) 
        ipro   = rna% info(2,indx)


        !=== Get CASE for Reaction Type ===!

        IF ( istate > 0 ) THEN
          icase = 1
        ELSEIF ( ip == id_30pi ) THEN
          icase = 2
        ELSE
          icase = 3
        ENDIF


        !=== CASE 1: Enzyme Present ===!

        IF ( icase == 1 ) THEN

          CALL RIBO_FIRE (rna,am,indx,ip)

        ENDIF


        !=== CASE 2: 30s:PIC Bind ===!

        IF ( icase == 2 ) THEN

          CALL RBS_FIRE (rna,am)

        ENDIF


        !=== CASE 3: Protein Binding/Unbinding ===!

        IF ( icase == 3 ) THEN

           write(*,*)'ERROR - profire'
           STOP
!          CALL PRO_FIRE (rna,am,indx,ip)

        ENDIF

        RETURN

      END SUBROUTINE MRNA_FIRE
