! ==============================================================================
! Subroutine: RIBO_FIRE (RNA,AMAX,INDX,IP)
! 
! Purpose: Chooses a reaction to fire for a Ribosome bound to an mRNA
!          in the protein synthesis model.
!
! Method:  The info array holds the following information:
!
!          istate = info(1,i) --> ribosome state
!          iasite = info(2,i) --> ribosome a-site state
!          irna_p = info(3,i) --> tRNA in p-site
!          irna_a = info(4,i) --> tRNA in a-site
!          istart = info(5,i) --> start codon of protein being synthesized
!            iloc = info(6,i) --> location of p-site on mRNA
!
!          The possible states are (iasite in brackets []):
!          * Denotes GDP bound
!          $ Denotes changed conformation
!
!          *** RIBOSOMAL INITIATION ***
!
!          (istate = 1) 30S:IC
!
!          [1] 30S:IC
!
!              30S:IC --> 30S:PIC + mRNA           30S:PIC Unbinding
!              30S:IC + 50S --> 70SIC              70S Complex formation
!
!          [2] 30S:IC*
!
!              30S:IC* --> 30S:PIC + mRNA          30S:PIC Unbinding
!              30S:IC* + 50S --> 70S:PIC           70S:PIC Formation
!
!          (istate = 2) 70S:PIC Bound
!
!          [1] 70S:PIC
!
!              70S:PIC --> 30S:IC + 50S            50S Unbinding
!              70S:PIC --> 70S:PIC*                GTP Hydrolysis
!
!          [2] 70S:PIC*
!
!              70S:PIC* --> 30S:IC* + 50S          50S Unbinding
!              70S:PIC* --> 70S:PIC$               Conf. Change
!
!          [3] 70S:PIC$
!
!              70S:PIC$ --> 70S:IC + Pi            Pi Release
!              70S:PIC$ --> 70S:PIC*               Reverse Conf. Change
!
!          [4] 70S:IC
!
!              70S:IC --> 70S:PE + IF1 + IF2:GDP   IF1 and IF2:GDP release
!
!          [5] 70S:PE
!
!              70S:PE --> 70S:EC + IF3             IF3 Release
!
!          *** RIBOSOMAL ELONGATION ***
!
!          (istate = 3) 70S:EC
! 
!          [1] A-Site Vacant
!
!              R + EfTu  --> (R+Ef)                EfTu Binding
!              R + RFx   --> (R+RFx)               RFx Binding
!
!          [2] Ef-Tu Bound
!
!              (R+Ef) --> (R+EfA)                  EfTu Move to A-site
!              (R+Ef) --> R + EfTu                 EfTu Unbinding
!
!          [3] EF-Tu @ A-Site
!
!              (R+EfA) --> (R+Ef*)                 GTP Hydrolysis
!              (R+EfA) --> (R+Ef)                  Ef-Tu Unbind A-Site
!
!          [4] Ef-Tu* @ A-site
!
!              (R+Ef*) --> (R+Ef$)                 EfTu Conf. Change
!
!          [5] Ef-Tu$ @ A-site
!
!              (R+Ef$) --> (R+T) + EfTu*           ACCOMODATION
!              (R+Ef$) --> R + EfTu* + Trna        REJECTION
!
!          [6] TRNA+Peptide @ A-site
!
!              (R+T) + EFG --> (R+T+EFG)           EfG Binding
!
!          [7] EfG @ A-site
!
!              (R+T+EFG) --> (R+T) + EfG           EfG Unbind
!              (R+T+EFG) --> (R+T+E*)              GTP Hydrolysis
!
!          [8] EfG* @ A-site
!
!              (R+T+E*) --> (R+T+E$)               TRANSLOCATION
!
!          [9] TRNA @ E-site
!
!              (R+T+E$) --> R + TRNA + EfG*        TRNA + EFG:GDP Release
!
!          *** RIBOSOMAL TERMINATION ***
!
!          (istate = 4) 70S:Pre/PostHC
!
!          [1] RFx Bound (x=1/2)
!
!              (R+RFx) --> (R+RFx) + P             Protein release
!              (R+RFx) --> R + RFx                 RFx Unbinding
!
!          [2] RFx Bound (x=1/2)
!
!              (R+RFx) + RF3 --> (R+RFx+RF3)       RF3 Binding
!              (R+RFx) --> (R) + RFx               N->R + RFx Release
!
!          [3] RFx+RF3 Bound (N State)
!
!              (R+RFx+RF3) --> (R+RFx+RF3$)        N -> R state
!              (R+RFx+RF3) --> (R+RFx) + RF3       RF3 Unbinding
!
!          [4] RFx+RF3 Bound (R State)
!
!              (R+RFx+RF3$) --> (R+RFx+RF3)        R -> N state
!              (R+RFx+RF3$) --> (R+RF3) + RFx      RFx Release
!              (R+RFx+RF3$) --> (R+RFx) + RF3      RF3 Release
!
!          [5] RF3 Bound (R State)
!
!              (R+RF3$) --> (R+RF3*)               GTP Hydrolysis
!
!          [6] RF3:GDP Bound (R State)
!
!              (R+RF3*) --> R + RF3                RF3 release
!
!          [7] RFx Bound (R State)
!
!              (R+RFx$) --> R + RFx                RFx release
!
!          *** RIBOSOMAL RECYCLING ***
!
!          (istate = 5) 70S:RC
!
!          [1] 70S:RC
!
!              R + RRF --> (R+RRF)                 RRF Binding
!
!          [2] RRF Bound
!
!              (R+RRF) + EfG --> (R+RRF+EfG)       Ef-G Binding
!              (R+RRF)       -->  R + RRF          RRF Unbinding
!
!          [3] RRF+EfG Bound
!
!              (R+RRF+EfG) --> (R+RRF) + EfG       Ef-G Unbinding
!              (R+RRF+EfG) --> (R+RRF+EfG*)        GTP Hydrolysis
!
!          [4] 70S:PreSplit
!
!              (R+RRF+EfG*) --> R + EfG* + RRF     Ribosomal Splitting
!
!          *** RIBOSOMAL SCANNING ***
!
!          (istate = 6) 30S:Scanning
!
! Arguments:
!
!             RNA - Class structure containing information on the
!                   RNA and its possible reactions.
!            AMAX - Threshold at which to fire a reaction.
!            INDX - Index number of the RNA object to fire a reaction.
!              IP - Protein ID number which is involved in the next
!                   reaction to fire.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - CONSTANTS, RIBOSOME, PROTEIN
! Functions -
! Subroutines - RESUM, RIBO_REAC, FOLD_REAC, RBS_REAC
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE RIBO_FIRE (RNA,AMAX,INDX,IP)

        USE Constants, ONLY : i5p_ribo,i3p_ribo,time
        USE Ribosome
        USE Protein

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        TYPE(MRNA), INTENT(INOUT) :: rna
        INTEGER, INTENT(IN) :: indx,ip

        DOUBLE PRECISION, INTENT(IN) :: amax

        !=== VARIABLES ===!

        INTEGER :: istate,iasite,iloc,i5p,i3p,i
        INTEGER :: icod_p,icod_a,jndx,kndx,ic,j
        INTEGER :: irna_p,irna_a,imin,imax

        DOUBLE PRECISION :: r,atot,a(0:mxpro)


        istate = rna% info(1,indx)
        iasite = rna% info(2,indx)

        irna_p = rna% info(3,indx)
        irna_a = rna% info(4,indx)

        iloc   = rna% info(6,indx)

        icod_p = rna% icdn(iloc)
        icod_a = rna% icdn(iloc+3)


        !=== FIRE REACTIONS ===!

        !=== STATE 1: 30S:IC ===!

        IF ( istate == 1 ) THEN
        IF ( iasite == 1 .or. iasite == 2 ) THEN

          IF ( ip == id_50s ) THEN

            !=== 50S Binding ===!
            !=== Update Protein Species ===!

            npro(id_50s) = npro(id_50s) - 1
            n_ribo(6) = n_ribo(6) - 1

            !=== Update Reactions ===!

            rna% info(1,indx) = 2

            CALL RIBO_REAC (rna,indx)

          ELSE

            !=== 30S:PIC Unbinding ===!
            !=== Update Protein Species ===!

            IF ( iasite == 1 ) THEN
              npro(id_30pi) = npro(id_30pi) + 1
              n_ribo(5) = n_ribo(5) + 1
            ELSE
              n_ifac(1) = n_ifac(1) + 1
              n_ifac(3) = n_ifac(3) + 1
              n_ifac(6) = n_ifac(6) + 1
              n_ribo(2) = n_ribo(2) + 1
            ENDIF
              nstats(1) = nstats(1) - 1 !lose ribo initiation ECD

            !=== Unbind from Start ===!

            jndx = rna% iprev(indx)
            kndx = rna% inext(indx)

            rna% inext(jndx) = kndx
            IF ( kndx /= 0 ) THEN
            rna% iprev(kndx) = jndx
            ENDIF

            !=== Delete Ribosome ===!

            ic = rna% iadd - 1
            rna% iadd = ic
            rna% ilist(ic) = indx

            rna% info(:,indx) = 0
            rna% inext(indx) = 0
            rna% iprev(indx) = 0

            a(:) = 0.0d0

            CALL RESUM (rna,indx,a)

            !=== Update Reactions ===!

            imin = iloc - i5p_ribo
            imax = iloc + i3p_ribo

            CALL RBS_REAC (rna,imax,imin)

          ENDIF

          RETURN

        ENDIF
        ENDIF


        !=== STATE 2: 70S:IC ===!

        IF ( istate == 2 ) THEN
        IF ( iasite == 1 ) THEN

          IF ( r_i1b >= amax ) THEN

            !=== 50S Unbinding ===!

            rna% info(1,indx) = 1

            !=== Update Protein Species ===!

            npro(id_50s) = npro(id_50s) + 1
            n_ribo(6) = n_ribo(6) + 1

          ELSE

            !=== GTP Hydrolosis ===!

            rna% info(2,indx) = 2

            !=== GTP/GDP Conservation ===!

            n_ndp(3) = n_ndp(3) - 1
            n_ntp(3) = n_ntp(3) + 1
              ncount(5) = ncount(5) + 1 !GTP consumption ECD
              ncount(1) = ncount(1) + 1 !IF2:GTP consumption ECD

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 2 ) THEN

          IF ( r_i2b >= amax ) THEN

            !=== 50S Unbinding ===!

            rna% info(1,indx) = 1

            !=== Update Protein Species ===!

            npro(id_50s) = npro(id_50s) + 1
            n_ribo(6) = n_ribo(6) + 1

          ELSE

            !=== 50s Closure ===!

            rna% info(2,indx) = 3

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 3 ) THEN

          IF ( r_i4b >= amax ) THEN

            !=== 50S Opening ===!

            rna% info(2,indx) = 2

          ELSE

            !=== Pi Release ===!

            rna% info(2,indx) = 4

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 4 ) THEN

          !=== IF1 and IF2-GDP Release ===!

          rna% info(2,indx) = 5

          !=== Update Protein Species ===!

          n_ifac(1) = n_ifac(1) + 1
          n_ifac(3) = n_ifac(3) + 1

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 5 ) THEN

          !=== IF3 Release ===!

          rna% info(2,indx) = 6

          !=== Update Protein Species ===!

          n_ifac(5) = n_ifac(5) + 1
          npro(id_if3) = npro(id_if3) + 1

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 6 ) THEN

          !=== Maturation ===!

          !=== Update Reactions ===!
!ecd-stats
            nstats(1) = nstats(1) - 1 !lose ribo initiation
            nstats(2) = nstats(2) + 1 !gain ribo elongation
            jndx = rna% inext(indx)
            if ( jndx /= 0 ) jndx = rna% info(6,jndx)
            open(unit=212,file='init.time',access='append')
            write(212,*)time,time-rna%t(indx),jndx
            close(unit=212)
            rna% t(indx) = time !start elongation time
!ecd-end
          rna% info(1,indx) = 3
          rna% info(2,indx) = 1
          rna% info(3,indx) = 0
          rna% info(4,indx) = 0
          rna% info(5,indx) = iloc

          CALL RIBO_REAC (rna,indx)

          RETURN

        ENDIF
        ENDIF


        !=== STATE 3: 70S:Elongation ===!

        IF ( istate == 3 ) THEN
        IF ( iasite == 1 ) THEN

          IF ( ip == id_rf1 ) THEN

            !=== RF1 Binding ===!

            rna% info(1,indx) = 4
            rna% info(2,indx) = 1

            rna% info(4,indx) = 1

            !=== Update Protein Species ===!

            npro(id_rf1) = npro(id_rf1) - 1
            n_rfac(1) = n_rfac(1) - 1

          ELSEIF ( ip == id_rf2 ) THEN

            !=== RF2 Binding ===!

            rna% info(1,indx) = 4
            rna% info(2,indx) = 1

            rna% info(4,indx) = 2

            !=== Update Protein Species ===!

            npro(id_rf2) = npro(id_rf2) - 1
            n_rfac(2) = n_rfac(2) - 1

          ELSE

            !=== EfTu-TRNA Binding ===!

            atot = 0.0d0

            i = 0

            DO WHILE ( atot < amax ) 

              i = i + 1

              r = r_e1f * DBLE(ntcx(i))

              atot = atot + r

              IF ( i == 64 ) EXIT

            ENDDO

            rna% info(2,indx) = 2
            rna% info(4,indx) = i

            !=== Update Protein Species ===!

            npro(id_tcx) = npro(id_tcx) - 1
            ntcx(i) = ntcx(i) - 1

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 2 ) THEN

          r = r_e2f(icod_a,irna_a)

          IF ( r >= amax ) THEN

            !=== EfTu-TRNA Enter Asite ===!

            rna% info(2,indx) = 3

          ELSE

            !=== EfTu-TRNA Unbinding ===!

            i = irna_a

            rna% info(2,indx) = 1
            rna% info(4,indx) = 0

            !=== Update Protein Species ===!

            npro(id_tcx) = npro(id_tcx) + 1
            ntcx(i) = ntcx(i) + 1

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 3 ) THEN

          r = r_e3(icod_a,irna_a)

          IF ( r >= amax ) THEN

            !=== EfTu-TRNA GTP Hydrolosis ===!

            rna% info(2,indx) = 4

            !=== GTP/GDP Conservation ===!

            n_ndp(3) = n_ndp(3) - 1
            n_ntp(3) = n_ntp(3) + 1
              ncount(5) = ncount(5) + 1 !GTP consumption ECD
              ncount(2) = ncount(2) + 1 !EF-Tu:GTP consumption ECD
          ELSE

            !=== EfTu-TRNA Exit Asite ===!

            rna% info(2,indx) = 2

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 4 ) THEN

          !=== EfTu Change ===!

          rna% info(2,indx) = 5

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 5 ) THEN

          r = r_e5(icod_a,irna_a)

          IF ( r >= amax ) THEN

            !=== Amino-Acid Transfer ===!
            !=== EfTu-GDP Release ===!

            rna% info(2,indx) = 6

            !=== Update Protein Species ===!

            n_elong(3) = n_elong(3) + 1

          ELSE

            !=== Reject TRNA ===!

            i = irna_a

            rna% info(2,indx) = 1
            rna% info(4,indx) = 0

            !=== Update Protein Species ===!

            ntrna_aa(i) = ntrna_aa(i) + 1
            n_elong(3) = n_elong(3) + 1

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 6 ) THEN

          !=== EfG Binding ===!

          rna% info(2,indx) = 7

          !=== Update Protein Species ===!

          npro(id_efg) = npro(id_efg) - 1
          n_elong(10) = n_elong(10) - 1

          !=== Update Reactions ===!

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 7 ) THEN

          r = r_e7b

          IF ( r >= amax ) THEN

            !=== EfG Unbinding ===!

            rna% info(2,indx) = 6

            !=== Update Protein Species ===!

            npro(id_efg) = npro(id_efg) + 1
            n_elong(10) = n_elong(10) + 1

          ELSE

            !=== GTP Hydrolosis ===!

            !=== GTP/GDP Conservation ===!

            n_ndp(3) = n_ndp(3) - 1
            n_ntp(3) = n_ntp(3) + 1
              ncount(5) = ncount(5) + 1 !GTP consumption ECD
              ncount(3) = ncount(3) + 1 !EFG:GTP consumption ECD
            !=== Is Ribosome Translocatable? ==!

            i = iloc + i3p_ribo
            j = rna% nnuc + 100

            i3p = rna% inext(indx)

            IF ( i3p /= 0 ) THEN
            IF ( rna% info(1,i3p) > 0 ) THEN
              j = rna% info(6,i3p) - i5p_ribo
            ELSEIF ( rna% info(2,i3p) > 0 ) THEN
              j = rna% info(5,i3p)
            ENDIF
            ENDIF

            IF ( j-i >= 4 ) THEN

              !=== Translocatable ===!
              rna% info(2,indx) = +8

            ELSE

              !=== Stall ===!
              rna% info(2,indx) = -8
               nstats(4) = nstats(4) + 1 !gain stalled ribo ECD
            ENDIF

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 8 ) THEN

          !=== Translocation ===!

          rna% info(2,indx) = 9
          rna% info(6,indx) = iloc + 3

          CALL RIBO_REAC (rna,indx)

          !=== Make Upstream Ribosome Moveable ===!

          i5p = rna% iprev(indx)

          IF ( i5p /= 0 ) THEN

            i = rna% info(1,i5p)
            j = rna% info(2,i5p)

            IF ( i == 3 .and. j == -8 ) THEN
              rna% info(2,i5p) = 8
              CALL RIBO_REAC (rna,i5p)
               nstats(4) = nstats(4) - 1 !lose stalled ribo ECD
            ENDIF

          ENDIF

          !=== Folding/RBS Reactions ===!

          imin = iloc - i5p_ribo
          imax = iloc + i3p_ribo + 3

          CALL RBS_REAC (rna,imax,imin)

          RETURN

        ELSEIF ( iasite == 9 ) THEN

          !=== EfG + TRNA Release ===!

          i = irna_p

          rna% info(2,indx) = 1

          rna% info(3,indx) = irna_a
          rna% info(4,indx) = 0

          !=== Update Protein Species ===!

          n_elong(9) = n_elong(9) + 1

          IF ( irna_p /= 0 ) THEN
!            ntrna(i) = ntrna(i) + 1
            ntrna_aa(i) = ntrna_aa(i) + 1
            ntr_tot(i) = ntr_tot(i) + 1 !TRNA AA consumption
          ELSE
!            ntrna(65) = ntrna(65) + 1
            n_ifac(6) = n_ifac(6) + 1
            ntr_tot(65) = ntr_tot(65) + 1 !TRNA AA consumption
          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ENDIF
        ENDIF


        !=== STATE 4: 70S:Pre/PostHC ===!

        IF ( istate == 4 ) THEN
        IF ( iasite == 1 ) THEN

          r = r_t2(icod_a,irna_a)

          IF ( r >= amax ) THEN

            !=== Protein Release ===!

            rna% info(2,indx) = 2
!ecd-stats
            nstats(2) = nstats(2) - 1 !lose ribo elongation
            nstats(3) = nstats(3) + 1 !gain ribo termination
            if ( icod_a == 49 .or. icod_a == 51 .or. icod_a == 57 ) then
            ncount(7) = ncount(7) + 1
            else
            ncount(6) = ncount(6) + 1
            ncount(7) = ncount(7) + 1
            endif
            open(unit=212,file='elon.time',access='append')
            write(212,*)time,time-rna%t(indx),iloc+2,&
                      & (iloc+2)/(3.0d0*(time-rna%t(indx)))
            close(unit=212)
            rna% t(indx) = time !start postHC time
!ecd-end
          ELSE

            !=== RFx Unbinding ===!

            rna% info(1,indx) = 3
            rna% info(2,indx) = 1
            rna% info(4,indx) = 0

            !=== Update Protein Species ===!

            i = irna_a

            IF ( i == 1 ) j = id_rf1
            IF ( i == 2 ) j = id_rf2

            npro(j) = npro(j) + 1
            n_rfac(i) = n_rfac(i) + 1

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 2 ) THEN

          IF ( ip == id_rf3 ) THEN

            !=== RF3 Binding ===!

            atot = r_t3f * DBLE(n_rfac(4))

            IF ( atot > amax ) THEN

              !=== GDP Form ===!

              rna% info(2,indx) = 8

              !=== Update Protein Species ===!

              npro(id_rf3) = npro(id_rf3) - 1
              n_rfac(4) = n_rfac(4) - 1

            ELSE

              !=== GTP Form ===!

              rna% info(2,indx) = 3

              !=== Update Protein Species ===!

              npro(id_rf3) = npro(id_rf3) - 1
              n_rfac(5) = n_rfac(5) - 1

            ENDIF

          ELSE

            !=== RFx Release + Ratcheting ===!

            rna% info(1,indx) = 5
            rna% info(2,indx) = 1
            !rna% info(4,indx) = 0
!ecd-stats
            open(unit=212,file='post.time',access='append')
            write(212,*)time,time-rna%t(indx),iloc+2,1
            close(unit=212)
            rna% t(indx) = time !start term time
!ecd-end
            !=== Update Protein Species ===!

            i = irna_a

            IF ( i == 1 ) j = id_rf1
            IF ( i == 2 ) j = id_rf2

            npro(j) = npro(j) + 1
            n_rfac(i) = n_rfac(i) + 1

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 3 ) THEN

          IF ( r_t3b >= amax ) THEN

            !=== RF3 Unbinding ===!

            rna% info(2,indx) = 2

            !=== Update Protein Species ===!

            npro(id_rf3) = npro(id_rf3) + 1
            n_rfac(5) = n_rfac(5) + 1

          ELSE

            !=== Ratcheting to R State ===!

            rna% info(2,indx) = 4

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 4 ) THEN

          r = r_t4b + r_t5(irna_a)

          IF ( r_t4b >= amax ) THEN

            !=== Ratcheting to N State ===!

            rna% info(2,indx) = 3

          ELSEIF ( r >= amax ) THEN

            !=== RFx Release ===!

            rna% info(2,indx) = 5
            !rna% info(4,indx) = 0

            !=== Update Protein Species ===!

            i = irna_a

            IF ( i == 1 ) j = id_rf1
            IF ( i == 2 ) j = id_rf2

            npro(j) = npro(j) + 1
            n_rfac(i) = n_rfac(i) + 1

          ELSE

            !=== RF3:GTP Release ===!

            rna% info(2,indx) = 7

            !=== Update Protein Species ===!

            npro(id_rf3) = npro(id_rf3) + 1
            n_rfac(5) = n_rfac(5) + 1

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 5 ) THEN

          !=== RF3 GTP Hydrolysis ===!

          !=== GTP/GDP Conservation ===!

          n_ndp(3) = n_ndp(3) - 1
          n_ntp(3) = n_ntp(3) + 1
            ncount(5) = ncount(5) + 1 !GTP consumption ECD
            ncount(4) = ncount(4) + 1 !RF3:GTP consumption ECD

          rna% info(2,indx) = 6

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 6 ) THEN

          !=== RF3:GDP Release ===!

          rna% info(1,indx) = 5
          rna% info(2,indx) = 1
!ecd-stats
            open(unit=212,file='post.time',access='append')
            write(212,*)time,time-rna%t(indx),iloc+2,3
            close(unit=212)
            rna% t(indx) = time !start term time
!ecd-end
          !=== Update Protein Species ===!

          npro(id_rf3) = npro(id_rf3) + 1
          n_rfac(4) = n_rfac(4) + 1

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 7 ) THEN

          !=== RFx Release ===!

          rna% info(1,indx) = 5
          rna% info(2,indx) = 1
          !rna% info(4,indx) = 0
!ecd-stats
            open(unit=212,file='post.time',access='append')
            write(212,*)time,time-rna%t(indx),iloc+2,2
            close(unit=212)
            rna% t(indx) = time !start term time
!ecd-end
          !=== Update Protein Species ===!

          i = irna_a

          IF ( i == 1 ) j = id_rf1
          IF ( i == 2 ) j = id_rf2

          npro(j) = npro(j) + 1
          n_rfac(i) = n_rfac(i) + 1

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 8 ) THEN

          !=== RF3 Unbinding ===!

          rna% info(2,indx) = 2

          !=== Update Protein Species ===!

          IF ( r_t7 >= amax ) THEN
            npro(id_rf3) = npro(id_rf3) + 1
            n_rfac(4) = n_rfac(4) + 1
          ELSE
            n_ndp(3) = n_ndp(3) + 1
            n_rfac(3) = n_rfac(3) + 1
          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ENDIF
        ENDIF


        !=== STATE 5: 70S:Recycling ===!

        IF ( istate == 5 ) THEN
        IF ( iasite == 1 ) THEN

          !=== RRF Binding ===!

          rna% info(2,indx) = 2

          !=== Update Protein Species ===!

          npro(id_rrf) = npro(id_rrf) - 1
          n_rfac(6) = n_rfac(6) - 1

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 2 ) THEN

          IF ( ip == id_efg ) THEN

            !=== EF-G Binding ===!

            rna% info(2,indx) = 3

            !=== Update Protein Species ===!

            npro(id_efg) = npro(id_efg) - 1
            n_elong(10) = n_elong(10) - 1

          ELSE

            !=== RRF Unbinding ===!

            rna% info(2,indx) = 1

            !=== Update Protein Species ===!

            npro(id_rrf) = npro(id_rrf) + 1
            n_rfac(6) = n_rfac(6) + 1

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 3 ) THEN

          IF ( r_r3 >= amax ) THEN

            !=== Ef-G GTP Hydrolosis ===!

            rna% info(2,indx) = 4

            !=== GTP/GDP Conservation ===!

            n_ndp(3) = n_ndp(3) - 1
            n_ntp(3) = n_ntp(3) + 1
             ncount(5) = ncount(5) + 1 !GTP consumption
             ncount(3) = ncount(3) + 1 !EFG:GTP consumption
          ELSE

            !=== Ef-G Unbinding ===!

            rna% info(2,indx) = 2

            !=== Update Protein Species ===!

            npro(id_efg) = npro(id_efg) + 1
            n_elong(10) = n_elong(10) + 1

          ENDIF

          CALL RIBO_REAC (rna,indx)

          RETURN

        ELSEIF ( iasite == 4 ) THEN

          !=== Ribosomal Splitting ===!

          i = irna_p

          !=== Update Protein Species ===!

          npro(id_50s) = npro(id_50s) + 1
          n_ribo(1) = n_ribo(1) + 1
          n_ribo(6) = n_ribo(6) + 1

          npro(id_rrf) = npro(id_rrf) + 1
          n_rfac(6) = n_rfac(6) + 1

          n_elong(9) = n_elong(9) + 1

          IF ( irna_p /= 0 ) THEN
!            ntrna(i) = ntrna(i) + 1
            ntrna_aa(i) = ntrna_aa(i) + 1
            ntr_tot(i) = ntr_tot(i) + 1 !TRNA AA consumption
          ELSE
!            ntrna(65) = ntrna(65) + 1
            n_ifac(6) = n_ifac(6) + 1
            ntr_tot(65) = ntr_tot(65) + 1 !TRNA AA consumption
          ENDIF
!ecd-start
            nstats(3) = nstats(3) - 1 !lose ribo termination
            jndx = rna% iprev(indx)
            jndx = rna% info(6,jndx)
            open(unit=212,file='term.time',access='append')
            write(212,*)time,time-rna%t(indx),iloc-jndx,irna_a
            close(unit=212)
!ecd-end
          !=== Make Upstream Ribosome Moveable ===!

          i5p = rna% iprev(indx)

          IF ( i5p /= 0 ) THEN

            i = rna% info(1,i5p)
            j = rna% info(2,i5p)

            IF ( i == 3 .and. j == -8 ) THEN
              rna% info(2,i5p) = 8
              CALL RIBO_REAC (rna,i5p)
              nstats(4) = nstats(4) - 1 !lose stalled ribo
            ENDIF

          ENDIF

          !=== Unbind Ribosome ===!

          jndx = rna% iprev(indx)
          kndx = rna% inext(indx)

          rna% inext(jndx) = kndx
          IF ( kndx /= 0 ) THEN
          rna% iprev(kndx) = jndx
          ENDIF

          !=== Delete Ribosome ===!

          ic = rna% iadd - 1
          rna% iadd = ic
          rna% ilist(ic) = indx

          rna% info(:,indx) = 0
          rna% inext(indx) = 0
          rna% iprev(indx) = 0

          a(:) = 0.0d0

          CALL RESUM (rna,indx,a)

          !=== Folding/RBS Reactions ===!

          imin = iloc - i5p_ribo
          imax = iloc + i3p_ribo

          CALL RBS_REAC (rna,imax,imin)

          RETURN

        ENDIF
        ENDIF


        !=== STATE 6: 70S:Scanning ===!

        IF ( istate == 6 ) THEN

        ENDIF

        RETURN

      END SUBROUTINE RIBO_FIRE
