! ==============================================================================
! Subroutine: DELTAG (IBSP,LOOP,NH,NS,I,J,DG)
! 
! Purpose: Computes the difference in free energy of an RNA loop due
!          to the additon or deletion of the i-j base pair.
!
! Method:
!
! Arguments:
!
!          IBSP - Array of dimension (NNUC) containing the information
!                 on base pairs in the RNA fold.
!                 IBSP(i) = j [i base pairs with j]
!                 IBSP(i) = 0 [i is single stranded]
!             I - Nucleotide position of the 5' most nucleotide.
!             J - Nucleotide position of the 3' most nucleotide.
!            DG - (OUTPUT) The energy difference from deletion of the
!                 base-pair i-j
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependencies:
!
! Modules - FOLDING, RNADATA
! Functions -
! Subroutines - EHAIR, EBULGE, EDANGLE
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE DELTAG (IBSP,LOOP,NH,NS,I,J,DG)

        USE Folding,   ONLY : enf1,eaup
        USE RNAData,   ONLY : iseq,nnuc
        USE Constants, ONLY : mxnt


        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(IN) :: i,j
        INTEGER, INTENT(IN) :: ibsp(mxnt),loop(mxnt)
        INTEGER, INTENT(IN) :: nh(mxnt),ns(mxnt)

        DOUBLE PRECISION, INTENT(OUT) :: dg

        !=== VARIABLES ===!

        INTEGER :: i1,i2,j1,j2,ip,jp,ilp,jlp
        INTEGER :: ih,is,jh,js,kh,ks,klp,n

        DOUBLE PRECISION :: e,e3,e5,eli,elj
        DOUBLE PRECISION :: ei,ef,em,es,eh,ed


        n = nnuc

        !=== Set Multi-Loop Terms ===!

        em = enf1(1)
        es = enf1(2)
        eh = enf1(3)

        ilp = loop(i)
        jlp = loop(j)


        !=== Check if we add or del BP ===!

        IF ( ibsp(i) == 0 ) THEN

          !=== Add BP ===!

          ih = 1
          is = 0

          ip = i + 1

          DO WHILE ( ip < j )

            IF ( ibsp(ip) == 0 ) is = is + 1

            IF ( ibsp(ip)  > i ) THEN
              ih = ih + 1
              ip = ibsp(ip)
            ENDIF

            ip = ip + 1

          ENDDO

          jh = nh(ilp) - ih + 2
          js = ns(ilp) - is - 2

        ELSE

          !=== Del BP ===!

          IF ( ilp == 0 ) THEN
            ih = 2
            is = 0
          ELSE
            ih = nh(ilp)
            is = ns(ilp)
          ENDIF

          IF ( jlp == 0 ) THEN
            jh = 2
            js = 0
          ELSE
            jh = nh(jlp)
            js = ns(jlp)
          ENDIF

        ENDIF

        !=== Final Loop Size ===!

        klp = jlp

        kh = ih + jh - 2
        ks = is + js + 2



        ei = 0.0d0
        ef = 0.0d0
        ed = 0.0d0

        !=== INITIAL ENERGY ===!

        !==== ILOOP ===!

        IF ( ih == 1 ) THEN

          CALL EHAIR (i,j,eli)

          IF ( kh == 2 .and. klp /= 1 ) THEN

            ip = i - 1
            DO WHILE ( ibsp(ip) == 0 )
            ip = ip - 1
            ENDDO

            jp = j + 1
            DO WHILE ( ibsp(jp) == 0 )
            jp = jp + 1
            ENDDO

            IF ( ibsp(jp) > jp ) THEN

              !=== 5' HP ===!

              i1 = ip
              i2 = jp

              j1 = ibsp(i1)
              j2 = ibsp(i2)

              CALL EDANGLE (i1,j1,i1+1,e3)
              CALL EDANGLE (i2,j2,i2-1,e5)

              IF ( i1+2 < i ) ei = ei + e3
              IF ( i2-2 > j ) ei = ei + e5

              CALL EDANGLE (i1,j1,j1-1,e5)
              CALL EDANGLE (i2,j2,j2+1,e3)

              IF ( j2+2 < j1 ) THEN
                ei = ei + e5 + e3
              ELSEIF ( j2+2 == j1 ) THEN
                ei = ei + MIN(e5,e3)
              ENDIF

            ELSE

              !=== 3' HP ===!

              j1 = jp
              j2 = ip

              i1 = ibsp(j1)
              i2 = ibsp(j2)

              CALL EDANGLE (i1,j1,j1-1,e5)
              CALL EDANGLE (i2,j2,j2+1,e3)

              IF ( j1-2 > j ) ei = ei + e5
              IF ( j2+2 < i ) ei = ei + e3

              CALL EDANGLE (i1,j1,i1+1,e3)
              CALL EDANGLE (i2,j2,i2-1,e5)

              IF ( i2-2 > i1 ) THEN
                ei = ei + e5 + e3
              ELSEIF ( i2-2 == i1 ) THEN
                ei = ei + MIN(e5,e3)
              ENDIF

            ENDIF

            ei = ei + eaup(iseq(i1),iseq(j1))
            ei = ei + eaup(iseq(i2),iseq(j2))

          ENDIF

        ELSEIF ( ih == 2 ) THEN

          i2 = i + 1
          DO WHILE ( ibsp(i2) == 0 )
          i2 = i2 + 1
          ENDDO

          j2 = ibsp(i2)

          CALL EBULGE (i,j,i2,j2,eli)

          IF ( jh > 2 .or. jlp == 1 ) THEN

            CALL EDANGLE (i2,j2,i2-1,e5)
            CALL EDANGLE (i2,j2,j2+1,e3)

            IF ( i2-1 /= i ) ef = ef + e5
            IF ( j2+1 /= j ) ef = ef + e3

            ef = ef + eaup(iseq(i2),iseq(j2))

          ENDIF

        ELSEIF ( ih > 2 ) THEN

          e5 = 0.0d0
          e3 = 0.0d0

          IF ( ibsp(i+1) == 0 ) THEN

            CALL EDANGLE (i,j,i+1,e3)

            ip = i + 2
            jp = ibsp(ip)

            IF ( jp > 0 ) THEN

              CALL EDANGLE (ip,jp,i+1,e)

              e3 = MIN(e3,e)
              ef = ef + e

            ENDIF

          ENDIF

          IF ( ibsp(j-1) == 0 ) THEN

            CALL EDANGLE (i,j,j-1,e5)

            jp = j - 2
            ip = ibsp(jp)

            IF ( ip > 0 ) THEN

              CALL EDANGLE (ip,jp,j-1,e)

              e5 = MIN(e5,e)
              ef = ef + e

            ENDIF

          ENDIF

          eli = e3 + e5 + eaup(iseq(i),iseq(j))
          eli = eli + em + es * DBLE(is) + eh * DBLE(ih)

        ENDIF

        !==== JLOOP ===!

        IF ( jh > 2 .or. jlp == 1 ) THEN

          e5 = 0.0d0
          e3 = 0.0d0

          IF ( i > 1 ) THEN
          IF ( ibsp(i-1) == 0 ) THEN

            CALL EDANGLE (i,j,i-1,e5)

            IF ( i > 2 ) THEN

              jp = i - 2
              ip = ibsp(jp)

              IF ( ip > 0 ) THEN

                CALL EDANGLE (ip,jp,i-1,e)

                e5 = MIN(e5,e)
                ed = ed + e

              ENDIF

            ENDIF

          ENDIF
          ENDIF

          IF ( j < n ) THEN
          IF ( ibsp(j+1) == 0 ) THEN

            CALL EDANGLE (i,j,j+1,e3)

            IF ( j < n-1 ) THEN

              ip = j + 2
              jp = ibsp(ip)

              IF ( jp > 0 ) THEN

                CALL EDANGLE (ip,jp,j+1,e)

                e3 = MIN(e3,e)
                ed = ed + e

              ENDIF

            ENDIF

          ENDIF
          ENDIF

          elj = e3 + e5 + eaup(iseq(i),iseq(j))

          IF ( jlp > 1 ) THEN
          elj = elj + em + es * DBLE(js) + eh * DBLE(jh)
          ENDIF

        ELSEIF ( jh == 2 ) THEN

          i1 = i - 1
          DO WHILE ( ibsp(i1) == 0 )
          i1 = i1 - 1
          ENDDO

          j1 = ibsp(i1)

          CALL EBULGE (i1,j1,i,j,elj)

          IF ( ih > 2 ) THEN

            CALL EDANGLE (i1,j1,i1+1,e3)
            CALL EDANGLE (i1,j1,j1-1,e5)

            IF ( i1+1 /= i ) ef = ef + e3
            IF ( j1-1 /= j ) ef = ef + e5

            ef = ef + eaup(iseq(i1),iseq(j1))

          ENDIF

        ENDIF

        ei = ei + eli + elj


        !=== FINAL ENERGY ===!

        IF ( kh > 2 .or. klp == 1 ) THEN

          e5 = 0.0d0
          e3 = 0.0d0

          !=== Dangle i ===!

          jp = i - 1
          ip = 0

          IF ( jp > 0 ) ip = ibsp(jp)

          IF ( ip > 0 ) CALL EDANGLE (ip,jp,i,e3)

          ip = i + 1
          jp = ibsp(ip)

          IF ( jp > 0 ) THEN

            CALL EDANGLE (ip,jp,i,e)

            e3 = MIN(e3,e)

          ENDIF

          !=== Dangle j ===!

          ip = j + 1
          jp = 0

          IF ( ip <= n ) jp = ibsp(ip)

          IF ( jp  > 0 ) CALL EDANGLE (ip,jp,j,e5) 

          jp = j - 1
          ip = ibsp(jp)

          IF ( ip > 0 ) THEN

            CALL EDANGLE (ip,jp,j,e)

            e5 = MIN(e5,e)

          ENDIF

          !=== Efinal ===!

          ef = ef + e3 + e5 + ed

          IF ( klp /= 1 ) THEN
          ef = ef + em + es * DBLE(ks) + eh * DBLE(kh)
          ENDIF

        ELSEIF ( kh == 1 ) THEN

          CALL EHAIR (i1,j1,ef)

        ELSEIF ( kh == 2 ) THEN

          CALL EBULGE (i1,j1,i2,j2,ef)

        ENDIF


        !=== Calculate DG ===!

        IF ( ibsp(i) == 0 ) THEN
          dg = ei - ef
        ELSE
          dg = ef - ei
        ENDIF

        RETURN

      END SUBROUTINE DELTAG
