! ==============================================================================
! Subroutine: EBULGE (I,J,IP,JP,EB)
! 
! Purpose: Computes the free energy of an RNA buldge with two helices.
!
! Method: EB = E_entropic + E_stack + E_stack + E_asymmetry
!
!               5' (I) X ... W (IP) 3'
!               3' (J) Y ... Z (JP) 5'
!
!               NOTE: I < IP and JP < J
!
! Arguments:
!
!             I - Nucleotide position of the starting basepair 5'.
!             J - Nucleotide position of the starting basepair 3'.
!            IP - Nucleotide position of the ending basepair 3'.
!            JP - Nucleotide position of the ending basepair 5'.
!            EB - (OUTPUT) Free energy of the bulge.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - FOLDING, RNADATA, CONSTANTS
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE EBULGE (I,J,IP,JP,EB)

        USE Folding , ONLY : stk_11,stk_12,stk_22,stk_in,stk_bp,elp_bu,&
                           & elp_in,eninio,einf,eaup,iwc,mxas,mxlp
        USE RNAData , ONLY : iseq,nnuc
        USE Constants,ONLY : beta


        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(IN) :: i,j,ip,jp

        DOUBLE PRECISION, INTENT(OUT) :: eb

        !=== VARIABLES ===!

        INTEGER :: indx,jndx,kndx,mndx
        INTEGER :: k,ibul,imin,imax
        INTEGER :: n1,n2,nt,na,ic

        DOUBLE PRECISION, PARAMETER :: fac = 1.750d0
        DOUBLE PRECISION :: x,c


        eb = 0.0d0

        n1 = ip - i - 1
        n2 = j - jp - 1

        nt = n1 + n2
        na = IABS(n1-n2)

        imin = MIN(n1,n2)
        imax = MAX(n1,n2)

        ic = 1

        IF ( na  > mxas ) ic = 0
        IF ( nt  > mxlp ) ic = 0

        IF ( iwc(iseq(i ),iseq(j )) == 0 ) ic = 0
        IF ( iwc(iseq(ip),iseq(jp)) == 0 ) ic = 0

        IF ( ic == 0 ) THEN
          eb = einf
          RETURN
        ENDIF


        !=== Get Bulge Type ===!

        !n1 = 0 / n2 = 0 --> ibul = 0
        !n1 = 0 / n2 = 1 --> ibul = 1
        !n1 = 1 / n2 = 0 --> ibul = 1
        !n1 = 0 / n2 > 1 --> ibul = 2
        !n1 > 1 / n2 = 0 --> ibul = 2
        !n1 = 1 / n2 = 1 --> ibul = 3
        !n1 = 1 / n2 = 2 --> ibul = 4
        !n1 = 2 / n2 = 1 --> ibul = 5
        !n1 = 2 / n2 = 2 --> ibul = 6

        ibul = 7
        IF ( imin == 0 ) THEN
          ibul = MIN(imax,2)
        ELSEIF ( imax <= 2 ) THEN
          ibul = MIN(imax,2) + MIN(imin,2) + MIN(n1,2)
        ENDIF

        !=== TERM 1 --> Entropic Term ===!

        IF ( ibul == 1 .or. ibul == 2 .or. ibul == 7 ) THEN

          x = 0.0d0
          k = MIN(nt,30)

          IF ( nt > 30 ) THEN

            c = fac / beta

            x = DBLE(nt) / 30.0d0
            x = c * LOG(x)

          ENDIF

          IF ( ibul <= 2 ) eb = elp_bu(k) + x
          IF ( ibul == 7 ) eb = elp_in(k) + x

        ENDIF


        !=== TERM 2 --> Bulge Stacking Energy ===!

        IF ( ibul <= 1 ) THEN

          ! 5' (i) A . X (ip) 3'
          ! 3' (j) U   Y (jp) 5'

          indx = iwc(iseq(i),iseq(j))
          jndx = iwc(iseq(ip),iseq(jp))

          eb = eb + stk_bp(indx,jndx)

        ELSEIF ( ibul == 2 ) THEN

          ! 5' (i) A .. X (ip) 3'
          ! 3' (j) U    Y (jp) 5'

          !=== Closing A-U / G-U Penalty ===!

          eb = eb + eaup(iseq(i),iseq(j))
          eb = eb + eaup(iseq(ip),iseq(jp))

        ELSEIF ( ibul == 3 ) THEN

          ! 5' (i) A . X (ip) 3'
          ! 3' (j) U . Y (jp) 5'

          indx = iwc(iseq(i),iseq(j))
          jndx = iwc(iseq(ip),iseq(jp))
          kndx = 4 * iseq(i+1) + iseq(j-1) - 4

          eb = eb + stk_11(indx,jndx,kndx)

        ELSEIF ( ibul == 4 ) THEN

          ! 5' (i) A .   X (ip) 3'
          ! 3' (j) U . . Y (jp) 5'

          indx = iwc(iseq(i),iseq(j))
          jndx = iwc(iseq(ip),iseq(jp))
          kndx = 4 * iseq(i+1) + iseq(j-1) - 4
          mndx = iseq(j-2)

          eb = eb + stk_12(indx,jndx,kndx,mndx)

        ELSEIF ( ibul == 5 ) THEN

          ! 5' (i) A . . X (ip) 3'
          ! 3' (j) U   . Y (jp) 5'

          indx = iwc(iseq(jp),iseq(ip))
          jndx = iwc(iseq(j),iseq(i))
          kndx = 4 * iseq(j-1) + iseq(i+2) - 4
          mndx = iseq(ip-2)

          eb = eb + stk_12(indx,jndx,kndx,mndx)

        ELSEIF ( ibul == 6 ) THEN

          ! 5' (i) A . . X (ip) 3'
          ! 3' (j) U . . Y (jp) 5'

          indx = iwc(iseq(i),iseq(j))
          jndx = iwc(iseq(ip),iseq(jp))
          kndx = 4 * iseq(i+1) + iseq(j-1) - 4
          mndx = 4 * iseq(i+2) + iseq(j-2) - 4

          eb = eb + stk_22(indx,jndx,kndx,mndx)

        ELSEIF ( ibul == 7 ) THEN

          ! 5' (i) A X .. G (ip) 3'
          ! 3' (j) U Y .. C (jp) 5'

          indx = iwc(iseq(i),iseq(j))
          jndx = 4 * iseq(i+1) + iseq(j-1) - 4

          !=== GAIL Rule ===!

          IF ( imin == 1 .and. imax > 2 ) jndx = 1

          eb = eb + stk_in(indx,jndx)

          ! 5' (i) A .. X G (ip) 3'
          ! 3' (j) U .. Y C (jp) 5'

          indx = iwc(iseq(jp),iseq(ip))
          jndx = 4 * iseq(jp+1) + iseq(ip-1) - 4

          !=== GAIL Rule ===!

          IF ( imin == 1 .and. imax > 2 ) jndx = 1

          eb = eb + stk_in(indx,jndx)

          !=== Asymmetry Penalty ===!

          k = MIN(4,n1,n2)

          x = DBLE(na) * eninio(k)

          x = MIN(x,eninio(5))

          eb = eb + x

        ENDIF

        RETURN

      END SUBROUTINE EBULGE
