! ==============================================================================
! Subroutine: EDANGLE (I,J,K,ED)
! 
! Purpose: Computes the free energy of a dangling nucleotide over a basepair.
!
! Method: ED = E_dangle
!
!               5' (I) X       3'
!               3' (J) Y Z (K) 5'
!
!
! Arguments:
!
!             I - Nucleotide position of the basepair 5'.
!             J - Nucleotide position of the basepair 3'.
!             K - Nucleotide position of the dangling nucleotide.
!            ED - (OUTPUT) Free energy of the dangling nucleotide.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - FOLDING, RNADATA
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE EDANGLE (I,J,K,ED)

        USE Folding, ONLY : iwc,einf,dng_5p,dng_3p
        USE RNAData, ONLY : iseq,nnuc


        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(IN) :: i,j,k

        DOUBLE PRECISION, INTENT(OUT) :: ed

        !=== VARIABLES ===!

        INTEGER :: ic,indx,jndx


        ed = 0.0d0

        ic = iwc(iseq(i),iseq(j))

        IF ( ic == 0 ) THEN
          ed = einf
          RETURN
        ENDIF


        IF ( k == i+1 ) THEN

          !5' (i) A X (k) 3'
          !3' (j) U       5'

          indx = iwc(iseq(i),iseq(j))
          jndx = iseq(k)

          ed = dng_3p(indx,jndx)

        ENDIF

        IF ( k == j-1 ) THEN

          !5' (i) A       3'
          !3' (j) U X (k) 5'

          indx = iwc(iseq(i),iseq(j))
          jndx = iseq(k)

          ed = dng_5p(indx,jndx)

        ENDIF

        IF ( k == j+1 ) THEN

          !5'       A (i) 3'
          !3' (k) X U (j) 5'

          indx = iwc(iseq(j),iseq(i))
          jndx = iseq(k)

          ed = dng_3p(indx,jndx)

        ENDIF

        IF ( k == i-1 ) THEN

          !5' (k) X A (i) 3'
          !3'       U (j) 5'

          indx = iwc(iseq(j),iseq(i))
          jndx = iseq(k)

          ed = dng_5p(indx,jndx)

        ENDIF

        RETURN

      END SUBROUTINE EDANGLE
