! ==============================================================================
! Subroutine: EHAIR (I,J,EH)
! 
! Purpose: Computes the free energy of an RNA hairpin turn.
!
! Method: EH = E_entropic + E_stack + E_bonus + E_penalty
!
!               5' (I) X ... loop 3'
!               3' (J) Y ... loop 5'
!
!               NOTE: I < J
!
! Arguments:
!
!             I - Nucleotide position of the loop basepair 5'.
!             J - Nucleotide position of the loop basepair 3'.
!            EH - (OUTPUT) Free energy of the hairpin turn.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - FOLDING, RNADATA, CONSTANTS
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE EHAIR (I,J,EH)

        USE Folding , ONLY : stk_hp,stk_bp,elp_hp,elp_t4,seq_t4,epc,&
                           & einf,eaup,epg,iwc,nt4
        USE RNAData , ONLY : iseq,nnuc
        USE Constants,ONLY : beta


        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(IN) :: i,j

        DOUBLE PRECISION, INTENT(OUT) :: eh

        !=== VARIABLES ===!

        INTEGER :: k,ic,nl
        INTEGER :: indx,jndx

        CHARACTER :: cl(6)
        CHARACTER (LEN=6) :: cwrk

        DOUBLE PRECISION, PARAMETER :: fac = 1.750d0
        DOUBLE PRECISION :: x,c


        eh = 0.0d0

        nl = j - i - 1

        ic = iwc(iseq(i),iseq(j))

        IF ( j <= i .or. ic == 0 ) THEN
          eh = einf
          RETURN
        ENDIF


        !=== TERM 1 --> Entropic Term ===!

        IF ( nl > 30 ) THEN

          c = fac / beta

          x = DBLE(nl) / 30.0d0
          x = c * LOG(x)

          eh = elp_hp(30) + x

        ELSE

          eh = elp_hp(nl)

        ENDIF


        !=== TERM 2 --> Stacking Energy ===!

        IF ( nl > 3 ) THEN

          ! 5' (i) A X (i+1) LOOP
          ! 3' (j) U Y (j-1) LOOP

          indx = iwc(iseq(i),iseq(j))
          jndx = 4 * iseq(i+1) + iseq(j-1) - 4

          eh = eh + stk_hp(indx,jndx)

        ENDIF


        !=== TERM 3 --> Bonuses ===!

        !=== Tetra-loop Bonus ===!

        IF ( nl == 4 .and. nt4 > 0 ) THEN

          DO k=i,j
          IF ( iseq(k) == 1 ) cl(k-i+1) = 'A'
          IF ( iseq(k) == 2 ) cl(k-i+1) = 'C'
          IF ( iseq(k) == 3 ) cl(k-i+1) = 'G'
          IF ( iseq(k) == 4 ) cl(k-i+1) = 'U'
          ENDDO

          WRITE(cwrk,'(6A1)')(cl(k),k=1,6)
 
          DO k=1,nt4
          IF ( seq_t4(k) == cwrk ) THEN

            eh = eh + elp_t4(k)
            EXIT

          ENDIF
          ENDDO

        ENDIF

        !=== GGG Bonus ===!

        IF ( iseq(i) == 3 .and. iseq(j) == 4 ) THEN

          ic = 0

          DO k=MAX(1,i-2),i
          IF ( iseq(k) == 3 ) ic = ic + 1
          ENDDO

          IF ( ic == 3 ) eh = eh + epg

        ENDIF


        !=== TERM 4 --> Penalties ===!

        !=== Poly C Penalty ===!

        ic = 0

        DO k=i+1,j-1
        IF ( iseq(k) == 2 ) ic = ic + 1
        ENDDO

        IF ( ic == nl ) THEN
        IF ( nl == 3 ) THEN
          eh = eh + epc(1)
        ELSE
          eh = eh + epc(2)
          eh = eh + epc(3) * DBLE(ic)
        ENDIF
        ENDIF

        !=== A-U / G-U closing a Tri-loop ===!

        IF ( nl == 3 ) THEN

          eh = eh + eaup(iseq(i),iseq(j))

        ENDIF

        RETURN

      END SUBROUTINE EHAIR
