! ==============================================================================
! Subroutine: FILL_CT
! 
! Purpose: Computes the arrays V(ij), VB(ij) etc. needed to calculate an
!          RNA secondary structure during cotranslation or cotranscription.
!
! Method: M. Zuker and P. Stiegler "Optimal computer folding of large RNA
!         sequences using thermodynamics and auxiliary information."
!         Nuc. Acids Res. 9, 133 (1981)
!
!         V(ij) = Lowest energy structure on [i,j] i<j
!         VB(ij)= Lowest energy strucutre on [i,j] i<j i and j bp
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependencies:
!
! Modules - FOLDING, RNADATA, CONSTANTS
! Functions -
! Subroutines - EHAIR, EBULGE, EDANGLE
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE FILL_CT

        USE Folding,   ONLY : eaup,einf,iwc,loop,mxlp,mxbp
        USE RNAData,   ONLY : v,vr,vb,iseq,indx_col,indx_row,nnuc
        USE Constants, ONLY : nmax


        IMPLICIT NONE

        !=== ARGUMENTS ===!

        !=== VARIABLES ===!

        INTEGER :: i,j,k,n,ip,jp,js,je,is,ie
        INTEGER :: ij,ij1,ik,kj,ix,jx,imax

        DOUBLE PRECISION :: e,e1,e2,e3,e5
        DOUBLE PRECISION :: vij,vbij,wij,wbij


        n = nnuc

        !=== Initialize V,VB ===!

        ij = indx_col(n) - 1
        imax = MIN(15,ij)

        DO ij=1,imax

          v(ij)  = 0.0d0
          vb(ij) = einf

        ENDDO

        vr(:) = 0.0d0

        IF ( n < 5 ) GOTO 1


        !=== Fill V,VB Matrix [i < j] ===!

        DO j=5,n

          imax = j - 4

          ij = indx_col(j-0) - j
          ij1= indx_col(j-1) - j

          DO k=1,4

            v(ij)  = 0.0d0
            vb(ij) = einf

            ij = ij + 1
            ij1 = ij1 + 1

          ENDDO

          DO i=imax,1,-1

            vij  = 0.0d0
            vbij = einf

            !=== VBij -> i-j BP Term ===!

            !=== CASE k=1: i,j close a hairpin ===!

            CALL EHAIR (i,j,e)

            vbij = MIN(vbij,e)

            !=== CASE k=2: i,j close a bulge/stack with ip,jp ===!

            ie = i + 1
            je = j - 1

             k = MAX(loop+1,j-i-mxlp-2)
            js = MAX(i+loop+2,j-1-mxlp)
            js = MIN(js,je)

            DO jp=js,je

              is = MAX(jp-k,ie)
              ik = indx_col(jp) - is

              DO ip=is,ie,-1

                CALL EBULGE (i,j,ip,jp,e)

                e = e + vb(ik)

                vbij = MIN(vbij,e)

                ik = ik + 1

              ENDDO

            ENDDO

            IF ( j-i > mxbp ) vbij = einf
            IF ( iwc(iseq(i),iseq(j)) == 0 ) vbij = einf


            ip = i + 1
            jp = j - 1

            !=== Branching Terms ===!

            !=== CASE 1: add ss i ===!

            CALL EDANGLE (ip,j,i,e)

            e = e + eaup(iseq(ip),iseq(j))

            e1 = vb(ij-1) + e
            e2 =  v(ij-1)

            vij  = MIN(e1,e2,vij)

            !=== CASE 2: add ss j ===!

            CALL EDANGLE (i,jp,j,e)

            e = e + eaup(iseq(i),iseq(jp))

            e1 = vb(ij1) + e
            e2 =  v(ij1)

            vij  = MIN(e1,e2,vij)

            !=== CASE 3: add ss i and j ===!

            CALL EDANGLE (ip,jp,i,e5)
            CALL EDANGLE (ip,jp,j,e3)

            e = e3 + e5 + eaup(iseq(ip),iseq(jp))

            e1 = vb(ij1-1) + e
            e2 =  v(ij1-1) !!! THINK I NEED TO ADD THIS TERM IN ALL FILLS (TRACE SHOULD BE OK)

            vij  = MIN(e1,e2,vij)

            !=== CASE 4: i-j bp ===!

            e1 = vbij + eaup(iseq(i),iseq(j))

            vij  = MIN(e1,vij)

            !=== CASE 5: Bifurcation ===!

            ik = indx_row(i) + i
            kj = ij - 1

            DO k=i,j-1

              e1 = vr(ik) + v(kj)

              vij = MIN(vij,e1)

              ik = ik + 1
              kj = kj - 1

            ENDDO

            !=== Store Energies in V Array ===!

            v(ij)  = vij
            vb(ij) = vbij

            ij  = ij + 1
            ij1 = ij1 + 1

            ik = indx_row(i) + j

            vr(ik) = vij

          ENDDO

        ENDDO

 1      CONTINUE

        RETURN

      END SUBROUTINE FILL_CT
