! ==============================================================================
! Subroutine: TRACE_CT (IBSP,IH,NH,I5,I3)
!
! Purpose: Computes the minimum energy RNA Hairpin ONLY structure on the
!          [i5-i3] RNA fragment using V(ij) and VB(ij) arrays.
!
! Method: M. Zuker and P. Stiegler "Optimal computer folding of large RNA
!         sequences using thermodynamics and auxiliary information."
!         Nuc. Acids Res. 9, 133 (1981).
!
!         V(ij) = Lowest energy structure on [i,j] i<j
!         VB(ij)= Lowest energy strucutre on [i,j] i<j i and j bp
!
! Arguments:
!
!          IBSP - (OUTPUT) Array of dimension NNUC containing the
!                 information on base pairs in the RNA fold.
!                   IBSP(i) = j [i base pairs with j]
!                   IBSP(i) = 0 [i is single stranded]
!            IH - (OUTPUT) Array of dimension NNUC containing the start/end
!                 start/end position of helices in the loop.
!            NH - (OUTPUT)Total number of helices on [i-j]
!            I5 - (INPUT) 5' end of sequence fragment.
!            I3 - (INPUT) 3' end of sequence fragment.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependencies:
!
! Modules - FOLDING, RNADATA, CONSTANTS
! Functions -
! Subroutines - EHAIR, EBULGE, EDANGLE
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE TRACE_CT (IBSP,IH,NH,I5,I3)

        USE Folding,   ONLY : eaup,einf,prec,iwc,loop,mxlp,mxbp
        USE RNAData,   ONLY : v,vr,vb,iseq,indx_col,indx_row,nnuc
        USE Constants, ONLY : mxnt,nmax


        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(IN) :: i5,i3
        INTEGER, INTENT(OUT) :: ih(mxnt),nh
        INTEGER, INTENT(INOUT) :: ibsp(mxnt)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,n,ip,jp,is,js,ie,je
        INTEGER :: ic,ij,ij1,ik,kj,ix,jx
        INTEGER :: istk(mxnt),jstk(mxnt)

        DOUBLE PRECISION :: e,ep,e1,e2,e3,e4,e5
        DOUBLE PRECISION :: estk(mxnt)

        LOGICAL :: iflg


        n = nnuc

        !=== Helix List ===!

        nh = 0


        !=== Initialize ibsp and stacks ===!

        ij = indx_col(i3) - i5

        ic = 1

        istk(1) = i5
        jstk(1) = i3
        estk(1) = v(ij)

        ibsp(i5:i3) = 0


        !+++ FORWARD TRACE +++!

        !=== Pull fragment i,j from stack ===!

 1      IF ( ic == 0 ) RETURN

        i = istk(ic)
        j = jstk(ic)
        ep= estk(ic)

        ic = ic - 1

        IF ( j <= i ) GOTO 1


        !=== External Loop ===!

        !=== Remove nucleotides 5' ===!

        ij = indx_col(j) - i - 1

        e = v(ij)

        DO WHILE ( DABS(e-ep) < prec )

          ij = ij - 1
           i =  i + 1

          IF ( i == j ) GOTO 1

          ep = e
           e = v(ij)

        ENDDO

        !=== Remove nucleotides 3' ===!

        ij = indx_row(i) + j - 1

        e = vr(ij)

        DO WHILE ( DABS(e-ep) < prec )

          ij = ij - 1
           j =  j - 1

          IF ( i == j ) GOTO 1

          ep = e
           e = vr(ij)

        ENDDO


        !=== Check For Base Pair ===!

        iflg = .false.

        ip = i + 1
        jp = j - 1

         ij = indx_col(j-0) - i
        ij1 = indx_col(j-1) - i

        e1 = vb(ij)
        e2 = vb(ij-1)
        e3 = vb(ij1)
        e4 = vb(ij1-1)

        e1 = e1 + eaup(iseq(i),iseq(j))

        CALL EDANGLE(ip,j,i,e)

        e2 = e2 + e + eaup(iseq(ip),iseq(j))

        CALL EDANGLE(i,jp,j,e)

        e3 = e3 + e + eaup(iseq(i),iseq(jp))

        CALL EDANGLE(ip,jp,i,e5)
        CALL EDANGLE(ip,jp,j,e)

        e4 = e4 + e + e5 + eaup(iseq(ip),iseq(jp))


        IF ( DABS(e1-ep) < prec ) THEN

          ep = vb(ij)

        ELSEIF ( DABS(e2-ep) < prec ) THEN

          i = ip
          ep = vb(ij-1)

        ELSEIF ( DABS(e3-ep) < prec ) THEN

          j = jp
          ep = vb(ij1)

        ELSEIF ( DABS(e4-ep) < prec ) THEN

          i = ip
          j = jp
          ep = vb(ij1-1)

        ELSE

          iflg = .true.

          !=== Bifurcation - External Loop ===!

          ik = indx_row(i) + i
          kj = indx_col(j) - i - 1

          DO k=i,j-1

            e = vr(ik) + v(kj)

            IF ( DABS(e-ep) < prec ) THEN

              ic = ic + 1
              istk(ic) = k + 1
              jstk(ic) = j
              estk(ic) = v(kj)

              ic = ic + 1
              istk(ic) = i
              jstk(ic) = k
              estk(ic) = vr(ik)

              GOTO 1

            ENDIF

            ik = ik + 1
            kj = kj - 1

          ENDDO

        ENDIF

        IF ( iflg ) GOTO 1


        !=== Add Closing BP of Hairpin ===!

        nh = nh + 1
        ih(nh) = i

        !=== Base-Pair Found ===!

 2      ibsp(i) = j
        ibsp(j) = i


        !=== CASE k=1: i,j close a hairpin ===!

        CALL EHAIR (i,j,e)

        IF ( DABS(e-ep) < prec ) THEN

          !=== Add Opening BP of Hairpin ===!

          nh = nh + 1
          ih(nh) = i

          GOTO 1

        ENDIF

        !=== CASE k=2: i,j close a bulge/stack with ip,jp ===!

        ie = i + 1
        je = j - 1

         k = MAX(loop+1,j-i-mxlp-2)
        js = MAX(i+loop+2,j-1-mxlp)
        js = MIN(js,je)

        DO jp=js,je

          is = MAX(jp-k,ie)
          ik = indx_col(jp) - is

          DO ip=is,ie,-1

            CALL EBULGE (i,j,ip,jp,e)

            e = e + vb(ik)

            IF ( DABS(e-ep) < prec ) THEN

              i = ip
              j = jp

              ep = vb(ik)

              GOTO 2

            ENDIF

            ik = ik + 1

          ENDDO

        ENDDO

        IF ( ic  > 0 ) GOTO 1
        IF ( ic == 0 ) RETURN

        RETURN

      END SUBROUTINE TRACE_CT
