! ==============================================================================
! Subroutine: SD_DELTAG
! 
! Purpose: For each nucleotide in the sequence, calculates the minimum
!          possible interaction energy between the Anti-SD and SD and
!          returns a list of ribosome binding sites.
!
! Method:  H.M. Salis, E.A. Mirsky, C.A. Voigt, "Automated design of synthetic
!          ribosome binding sites to control protein expression."
!          Nature BioTech. 27, 946 (2009)
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - FOLDING, RIBOSOME, RNADATA, CONSTANTS
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE SD_DELTAG

        USE RNAData ,  ONLY : iseq,link_rbs,list_rbs,rbs_off,&
                            & icdn,nnuc,nrbs
        USE Constants, ONLY : beta,i5p_ribo,i3p_ribo
        USE Ribosome,  ONLY : r_b1f,r_b1b,r_b2f,r_b2b,r_b3f
        USE Folding

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        !=== VARIABLES ===!

        DOUBLE PRECISION, PARAMETER :: a1 = 0.048d0
        DOUBLE PRECISION, PARAMETER :: a2 = 0.240d0
        DOUBLE PRECISION, PARAMETER :: b1 = 1.220d1
        DOUBLE PRECISION, PARAMETER :: b2 = 2.500d0
        DOUBLE PRECISION, PARAMETER :: ed = 4.100d0

        INTEGER :: i,j,k,n,nr,ip,jp,ix,jx,ic,nt
        INTEGER :: indx,jndx,kndx,mndx,imax,imin
        INTEGER :: ibul,isd_seq(0:9),itmp(nnuc)

        DOUBLE PRECISION :: a,b,c,x,e,e1,e2,emin
        DOUBLE PRECISION :: vsd(0:9),dgs(64),vij
        DOUBLE PRECISION :: esd(nnuc),rtmp(nnuc)


        !=== Initialize ===!

        n = nnuc
        nrbs = 0

        link_rbs(:) = 0
        list_rbs(:) = 0
        rbs_off(:)  = 0.0d0

        !=== Complementary SD seq on Ribosome ===!

        isd_seq(0) = 2
        isd_seq(1) = 1
        isd_seq(2) = 2
        isd_seq(3) = 2
        isd_seq(4) = 4
        isd_seq(5) = 2
        isd_seq(6) = 2
        isd_seq(7) = 4
        isd_seq(8) = 4
        isd_seq(9) = 1

        dgs(:) = einf


        !=== Get DeltaG Start ===!

        !5' (i) U G (ip) 3'
        !3' (j) A C (jp) 5'

        indx = iwc(1,4)
        jndx = iwc(4,1)
        kndx = iwc(3,2)
        mndx = iwc(3,4)

        a = stk_bp(indx,jndx) + dng_3p(iwc(4,1),1) + eaup(4,1)
        b = stk_bp(mndx,jndx) + dng_3p(iwc(4,3),1) + eaup(4,3)
        c = stk_bp(jndx,kndx) + dng_5p(iwc(3,2),4) + eaup(3,2)

        e1 = dng_3p(indx,4) + eaup(1,4)
        e2 = dng_5p(jndx,2) + eaup(4,1)

        !=== AUG ===!

        dgs(15) = a + c + ed

        !=== GUG ===!

        dgs(47) = b + c + ed

        !=== UUG ===!

        dgs(63) = e1 + c + ed + dng_5p(indx,4)

        !=== CUG ===!

        dgs(31) = e1 + c + ed + dng_5p(indx,2)

        !=== AUA ===!

        dgs(13) = e2 + a + ed + dng_3p(jndx,1)

        !=== AUU ===!

        dgs(16) = e2 + a + ed + dng_3p(jndx,4)

        !=== AUC ===!

        dgs(14) = e2 + a + ed + dng_3p(jndx,2)


        !=== Get SD/Anti-SD Interaction ===!

        DO i=1,n

          vsd(:) = einf

          !=== Loop Over SD Sequence ===!

          DO j=1,9

            !5' (ip) ----- GENOME ---- (i) 3'
            !3' (jp) -- SD SEQUENCE -- (1) 5'

            vij = einf

            ip = i - j + 1
            jp = j

            IF ( ip < 1 ) EXIT

            IF ( iwc(iseq(ip),isd_seq(jp)) == 0 ) CYCLE


            !=== CASE 1: ip-jp closing BP ===!

            e = 0.0d0

            !=== 5' Dangle ===!

            IF ( jp > 0 ) THEN

              !5' (ip) A       3'
              !3' (jp) U X (k) 5'

              indx = iwc(iseq(ip),isd_seq(jp))
              jndx = isd_seq(jp-1)

              e = dng_5p(indx,jndx)

            ENDIF

            !=== 3' Dangle ===!

            IF ( ip < n ) THEN

              !5' (ip) A X (k) 3'
              !3' (jp) U       5'

              indx = iwc(iseq(ip),isd_seq(jp))
              jndx = iseq(ip+1)

              e = dng_3p(indx,jndx)

            ENDIF

            e = e + eaup(iseq(ip),isd_seq(jp))

            vij = MIN(vij,e)


            !=== CASE 2: ip,jp close a stack/interior loop ===!

            DO k=j-1,0,-1

              !=== Get Bulge Type ===!

              ibul = jp - k

              ix = ip + ibul
              jx = jp - ibul

              IF ( ix > n ) EXIT

              indx = iwc(iseq(ip),isd_seq(jp))
              jndx = iwc(iseq(ix),isd_seq(jx))

              IF ( jndx == 0 ) CYCLE

              IF ( ibul == 1 ) THEN

                !=== Stacking Energy ===!

                !5' (ip) U X (ix) 3'
                !3' (jp) A Y (jx) 5'

                e = stk_bp(indx,jndx)

              ELSEIF ( ibul == 2 ) THEN

                !5' (ip) A . X (ix) 3'
                !3' (jp) U . Y (jx) 5'

                kndx = 4 * iseq(ip+1) + isd_seq(jp-1) - 4

                e = stk_11(indx,jndx,kndx)

              ELSEIF ( ibul == 3 ) THEN

                !5' (ip) A . . X (ix) 3'
                !3' (jp) U . . Y (jx) 5'

                kndx = 4 * iseq(ip+1) + isd_seq(jp-1) - 4
                mndx = 4 * iseq(ip+2) + isd_seq(jp-2) - 4

                e = stk_22(indx,jndx,kndx,mndx)

              ELSE

                nt = 2 * ( ibul - 1 )

                e = elp_in(nt)

                !5' (ip) A X .. G (ix) 3'
                !3' (jp) U Y .. C (jx) 5'

                kndx = iwc(iseq(ip),isd_seq(jp))
                mndx = 4 * iseq(ip+1) + isd_seq(jp-1) - 4

                e = e + stk_in(kndx,mndx)

                !5' (ip) A .. X G (ix) 3'
                !3' (jp) U .. Y C (jx) 5'

                kndx = iwc(isd_seq(jx),iseq(ix))
                mndx = 4 * isd_seq(jx+1) + iseq(ix-1) - 4

                e = e + stk_in(kndx,mndx)

              ENDIF

              e = e + vsd(jx)

              vij = MIN(vij,e)

            ENDDO

            vsd(j) = vij

          ENDDO

          !=== Find Lowest Energy Interaction ===!

          emin = 0.0d0

          DO j=1,9

            ip = i - j + 1
            jp = j

            IF ( ip < 1 ) EXIT

            IF ( iwc(isd_seq(jp),iseq(ip)) == 0 ) CYCLE

            e = vsd(j) + ed

            !=== Add in 5'3' dangle terms ===!

            !=== 5' Dangle ===!

            IF ( ip > 1 ) THEN

              !5' (k) X A (ip) 3'
              !3'       U (jp) 5'

              indx = iwc(isd_seq(jp),iseq(ip))
              jndx = iseq(ip-1)

              e = e + dng_5p(indx,jndx)

            ENDIF

            !=== 3' Dangle ===!

            IF ( jp < 9 ) THEN

              !5'       A (ip) 3'
              !3' (k) X U (jp) 5'

              indx = iwc(isd_seq(jp),iseq(ip))
              jndx = isd_seq(jp+1)

              e = e + dng_3p(indx,jndx)

            ENDIF

            e = e + eaup(isd_seq(jp),iseq(ip))

            emin = MIN(emin,e)

          ENDDO

          esd(i) = emin

        ENDDO


        nr = 0
        itmp(:) = 0

        !=== Find Minimum E of interaction ===!
        !=== For start Codon @ position i ===!

        DO i=1,n-2

          !=== Codon Number ===!

          ic = icdn(i)

          ix  = 0

          IF ( ic == 15 ) ix = 1
          IF ( ic == 47 ) ix = 2
          IF ( ic == 63 ) ix = 3

          IF ( ic == 31 ) ix = 4
          IF ( ic == 13 ) ix = 5
          IF ( ic == 16 ) ix = 6
          IF ( ic == 14 ) ix = 7

          IF ( ix == 0 ) CYCLE

          !=== DG_SD:mRNA ===!

          k = 0

          emin = 0.0d0

          IF ( i > 1 ) THEN

            DO j=MAX(i-16,1),i-1

              !=== DG_Spacing ===!

              nt = i - j - 1

              IF ( nt >= 5 ) THEN

                x = a1 * DBLE(nt-5) + a2
                x = x * DBLE(nt-5)

              ELSE

                x = b2 * DBLE(nt-3)
                x = 1.0d0 + DEXP(x)
                x = b1 / ( x ** 3 )

              ENDIF

              e = esd(j) + x

              IF ( e < emin ) THEN
                emin = e
                k = j
              ENDIF

            ENDDO

          ENDIF

          !=== DG_start ===!

          e = dgs(ic)

          IF ( iseq(i+0) == 1 .and. i+0 > 1 ) e = e + dng_5p(iwc(4,1),iseq(i-1))
          IF ( iseq(i+2) == 3 .and. i+2 < n ) e = e + dng_3p(iwc(3,2),iseq(i+3))

          !=== Temp. Store RBS Info ===!

          nr = nr + 1

          itmp(nr) = i
          rtmp(nr) = e + emin
!ecd-start ECD TO FIX MEAN FIRST PASSAGE
          x = beta * rtmp(nr)
          x = DEXP(-x)

          a = 1.0d0 / r_b2f
          b = x / r_b3f
          c = x * a

           write(88,*)itmp(nr),rtmp(nr),x,1.0d0/(a+b+c)
!ecd-end
        ENDDO


        !=== Find Unique Non-Overlapping RBS Sites ===!

        ic = nr

        DO WHILE ( ic /= 0 )

          e = einf

          DO i=1,nr
          IF ( itmp(i) > 0 .and. rtmp(i) < e ) THEN

            e = rtmp(i)
            j = i

          ENDIF
          ENDDO

          !=== RBS Off Rate ===!
!ecd-start ECD TO FIX MEAN FIRST PASSAGE
          x = beta * rtmp(j)
          x = DEXP(-x)

          a = 1.0d0 / r_b2f
          b = x / r_b3f
          c = x * a

          !=== MFP RATE ===!

          x = 1.0d0 / (a+b+c)
!ecd-end
          !=== Store RBS Site ===!

          ix = itmp(j)

          nrbs = nrbs + 1

          link_rbs(ix) = nrbs
          rbs_off(ix) = x

          !=== Remove Overlaping Sites ===!

          imin = itmp(j) - i5p_ribo
          imax = itmp(j) + i3p_ribo

          imin = MAX(1,imin)
          imax = MIN(n,imax)

          itmp(j) = 0

          ic = ic - 1

          DO i=1,nr
          IF ( itmp(i) /= 0 ) THEN

            ip = itmp(i) - i5p_ribo
            jp = itmp(i) + i3p_ribo

            IF ( jp >= imin .and. ip <= imax ) THEN

              itmp(i) = 0
              ic = ic - 1

            ENDIF

          ENDIF
          ENDDO

        ENDDO


        !=== Compute Link Array - link_rbs ===!
        !=== Compute Ordered List -list_rbs===!

        ix = 1

        DO i=1,n
        IF ( link_rbs(i) /= 0 ) THEN

          list_rbs(ix) = i
          link_rbs(i) = ix

          ix = ix + 1

        ENDIF
        ENDDO

        ix = 0

        DO i=n,1,-1

          IF ( link_rbs(i) == 0 ) THEN
            link_rbs(i) = ix
          ELSE
            ix = link_rbs(i)
          ENDIF

        ENDDO
!ecd-start
        DO i=1,n
        IF ( link_rbs(i) > 1 ) THEN
          link_rbs(i) = 0
          rbs_off(i) = 0.0d0
        ENDIF
        ENDDO
        list_rbs(2:n) = 0
        nrbs = 1
!ecd-end
        RETURN

      END SUBROUTINE SD_DELTAG
