! ==============================================================================
! Subroutine: SSA (RNA,ISEED,TOUT)
! 
! Purpose: Selects either an mRNA reaction or cellular reaction to fire
!          following the Gillespie stochastic simulation algorithm (SSA).
!
! Method: Uses a binary tree based method to identify from a number
!         of different mRNAs in the simulation the specific mRNA
!         which will react next.
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - CLASS_MRNA, PROTEIN, CONSTANTS
! Functions - RANDOM
! Subroutines - MRNA_FIRE CELL_REAC CELL_FIRE
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      SUBROUTINE SSA (RNA,ISEED,TOUT)

        USE Constants, ONLY : psum_rna,iqueue_rna,mxpro,mxrna,time
        USE Class_MRNA
        USE Protein

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        INTEGER, INTENT(INOUT) :: iseed
        TYPE (MRNA), INTENT(INOUT) :: rna(mxrna)

        DOUBLE PRECISION, INTENT(IN) :: tout

        !=== VARIABLES ===!

        INTEGER :: i,j,k,ip,n,n1,n2
        INTEGER :: i1,i2,irna,ISPACE(4096)

        DOUBLE PRECISION :: r,x,t1,t2,random
        DOUBLE PRECISION :: amax,tau,tfld,treac
        DOUBLE PRECISION :: atot,acell,xp(mxpro)


        n = mxrna / 2

        !=== Non-Folding mRNA Reactions ===!

        xp(:) = DBLE(npro(:))

        atot = psum_rna(0,n)

        DO ip=1,mxpro
        atot = atot + psum_rna(ip,n) * xp(ip)
        ENDDO

        !=== Cell Reactions ===!

        CALL CELL_REAC (acell)

        atot = atot + acell


        !=== Next RNA Folding Reaction ===!

        irna = iqueue_rna(n)
        tfld = rna(irna)% tnext


        !=== Compute Time Increment ===!

        r = RANDOM(iseed)

        tau = DLOG(1.0d0/r)
        tau = tau / atot

        treac = time + tau

        !=== Minimum Time ===!

        IF ( treac <= tfld ) THEN

          time = treac

          r = RANDOM(iseed)
          amax = r * atot

        ELSE

          time = tfld

        ENDIF


        !=== Output Data ===!

        IF ( time > tout ) THEN

          open(unit=113,file='pro-gtp_consumption.dat',access='append')
          open(unit=114,file='ribo_states.dat',access='append')
          open(unit=115,file='ribo_dependent_pro.dat',access='append')

          open(unit=151,file='elong_factors.dat',access='append')
          open(unit=152,file='ribo_subunits.dat',access='append')
          open(unit=153,file='init_factors.dat',access='append')
          open(unit=154,file='recy_factors.dat',access='append')

          open(unit=156,file='tRNA-tcx.dat',access='append')
          open(unit=256,file='tRNA-free.dat',access='append')

          write(113,'(E16.8,7I10)')time,(ncount(i),i=1,7)
          write(114,'(E16.8,4I10)')time,(nstats(i),i=1,4)
          write(115,'(E16.8,9I7)')time,(npro(i),i=1,9)

          write(*,'(E16.8,9I7)')time,(npro(i),i=1,9)

          write(151,'(E16.8,10I7)')time,(n_elong(i),i=1,10)
          write(152,'(E16.8,10I7)')time,(n_ribo(i),i=1,6)
          write(153,'(E16.8,10I7)')time,(n_ifac(i),i=1,6)
          write(154,'(E16.8,10I7)')time,(n_rfac(i),i=1,6)

          write(156,*)time
          write(156,'(16I6)')(ntcx(i),i=1,64)
          write(256,*)time
          write(256,'(16I6)')(ntrna_aa(i),i=1,64)

          close(unit=113)
          close(unit=114)
          close(unit=115)

          close(unit=151)
          close(unit=152)
          close(unit=153)
          close(unit=154)

          close(unit=156)
          close(unit=256)

        ENDIF


        !=== Find which Reaction to Fire ===!

        IF ( treac <= tfld ) THEN

          !=== Fire Cell Reaction ===!

          IF ( acell >= amax ) THEN
            CALL CELL_FIRE (amax)
            RETURN
          ELSE
            amax = amax - acell
          ENDIF


          !=== Fire mRNA Reaction ===!

          !=== Find Protein Index ===!

          ip = 0
          r = psum_rna(ip,n)

          DO WHILE ( r < amax )

            amax = amax - r

            ip = ip + 1

            r = psum_rna(ip,n) * xp(ip)

          ENDDO

          !=== FIND RNA Number ===!

          i = n

          IF ( ip /= 0 ) THEN
            x = xp(ip)
          ELSE
            x = 1.0d0
          ENDIF

          DO WHILE ( MOD(n,2) == 0 )

            n = n / 2
            j = i - n

            r = psum_rna(ip,j) * x

            IF ( r >= amax ) THEN
              i = i - n
            ELSE
              i = i + n
              amax = amax - r
            ENDIF

          ENDDO

          r = rna(i)% a(ip) * x

          IF ( r >= amax ) THEN
            irna = i
          ELSE
            amax = amax - r
            irna = i + 1
          ENDIF

        ELSE

          !=== Set ip=-1 for FOLD reaction ===!

          ip = -1

        ENDIF


        !=== Index into psum/iqueue ===!

        i = irna
        IF ( MOD(irna,2) == 0 ) i = i - 1


        !=== FIRE RNA REACTION ===!

        CALL MRNA_FIRE (rna(irna),amax,ip)


        !=== RESUM & REQUEUE ===!

        n = 1
        n1= 2
        n2= 4

        t1 = rna(i+0)% tnext
        t2 = rna(i+1)% tnext

        IF ( t1 <= t2 ) iqueue_rna(i) = i
        IF ( t2 <  t1 ) iqueue_rna(i) = i+1

        psum_rna(:,i) = rna(i)% a(:) + rna(i+1)% a(:)

        DO WHILE ( n1 < mxrna )

          i = INT(i/n2) * n2 + n1

          j = i - n
          k = i + n

          !=== REQUEUE Folding Times ===!

          i1 = iqueue_rna(j)
          i2 = iqueue_rna(k)

          t1 = rna(i1)% tnext
          t2 = rna(i2)% tnext

          IF ( t1 <= t2 ) iqueue_rna(i) = i1
          IF ( t2 <  t1 ) iqueue_rna(i) = i2

          !=== RESUM Partial Sum Table ===!

          psum_rna(:,i) = psum_rna(:,j) + psum_rna(:,k)

          n  = n1
          n1 = n2
          n2 = 2 * n2

        ENDDO

        !=== Finished ===!

        RETURN

      END SUBROUTINE SSA
