! ==============================================================================
! Subroutine: GETOPTIONS
! 
! Purpose: Reads in the options for the simulation via a namelist.
!          Default values are set for each option.
!
! Method:
!
! Arguments:
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - PROTEIN, CONSTANTS
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE GETOPTIONS

        USE Constants, ONLY : vol,xribo_tot
        USE Protein

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        !=== VARIABLES ===!

        INTEGER :: i,j

        DOUBLE PRECISION :: x,xr


!        NAMELIST /options/ n_ntp,n_ndp,n_nmp,naa,n_ribo,n_ifac,n_elong,&
!                         & n_rfac,nars,ntcx,ntrna,ntrna_aa,nars_aa


        x  = 6.022d2 * vol
        xr = xribo_tot * x

        !=== Set Default Values ===!

        !=== NxPs ===!

        n_ntp(1) = INT(9.60d3 * x)           !ATP - Adenosine Triphosphate
        n_ntp(2) = INT(2.70d3 * x)           !CTP
        n_ntp(3) = INT(4.90d3 * x)           !GTP
        n_ntp(4) = INT(8.30d3 * x)           !UTP

        n_ndp(1) = INT(5.60d2 * x)           !ADP - Adenosine Diphosphate
        n_ndp(2) = INT(0.00d0 * x)           !CDP (??)
        n_ndp(3) = INT(6.80d2 * x)           !GDP
        n_ndp(4) = INT(1.80d3 * x)           !UDP

        n_nmp(1) = INT(2.80d2 * x)           !AMP - Adenosine Monophosphate
        n_nmp(2) = INT(3.60d2 * x)           !CMP
        n_nmp(3) = INT(2.40d1 * x)           !GMP
        n_nmp(4) = INT(0.00d0 * x)           !UMP (??)

        !=== Amino Acids ===!

        naa(1) = INT(1.80d1 * x)             !Amino PHE-F
        naa(2) = INT(3.00d2 * x)             !Amino LEU-L
        naa(3) = INT(3.00d2 * x)             !Amino ILE-I
        naa(4) = INT(1.50d2 * x)             !Amino MET-M
        naa(5) = INT(4.00d3 * x)             !Amino VAL-V
        naa(6) = INT(6.80d1 * x)             !Amino SER-S
        naa(7) = INT(3.90d2 * x)             !Amino PRO-P
        naa(8) = INT(1.80d2 * x)             !Amino THR-T
        naa(9) = INT(2.60d3 * x)             !Amino ALA-A
        naa(10)= INT(2.90d1 * x)             !Amino TYR-Y
        naa(11)= INT(6.80d1 * x)             !Amino HIS-H
        naa(12)= INT(3.80d3 * x)             !Amino GLN-Q
        naa(13)= INT(5.10d2 * x)             !Amino ASN-N
        naa(14)= INT(4.10d2 * x)             !Amino LYS-K
        naa(15)= INT(4.20d3 * x)             !Amino ASP-D
        naa(16)= INT(9.60d4 * x)             !Amino GLU-E
        naa(17)= INT(3.70d2 * x)             !Amino CYS-C
        naa(18)= INT(1.20d1 * x)             !Amino TRP-W
        naa(19)= INT(5.70d2 * x)             !Amino ARG-R
        naa(20)= INT(8.00d2 * x)             !Amino GLY-G (??)

        !=== 30S/50S Subunits ===!

        n_ribo(1) = INT(1.00d0 * xr)         !R30S Subunit
        n_ribo(2) = INT(0.00d0 * x)          !R30S:IF3
        n_ribo(3) = INT(0.00d0 * x)          !R30S:IF2/3
        n_ribo(4) = INT(0.00d0 * x)          !R30S:IF1/2/3
        n_ribo(5) = INT(0.00d0 * x)          !R30S:PIC Subunit
        n_ribo(6) = INT(1.00d0 * xr)         !R50S Subunit

        !=== Read in New Values ===!

!        READ(1,NML=options)

!        IF ( iratio == 0 ) GOTO 10

        !=== Initiation Factors ===!

        n_ifac(1) = INT(0.09d0 * xr)         !Initiation Factor 1
        n_ifac(2) = INT(0.13d0 * xr)         !Initiation Factor 2
        n_ifac(3) = INT(0.00d0 * xr)         !Initiation Factor 2 / GDP
        n_ifac(4) = INT(0.00d0 * xr)         !Initiation Factor 2 / GTP
        n_ifac(5) = INT(0.20d0 * xr)         !Initiation Factor 3
        n_ifac(6) = INT(0.00d0 * xr)         !Initiation Factor TRNA-FMET

        !=== Elongation/Recycling ===!

        n_elong(1) = INT(0.60d0 * xr)        !EFTs
        n_elong(2) = INT(5.40d0 * xr)        !EFTu
        n_elong(3) = INT(0.00d0 * xr)        !EFTu:GDP
        n_elong(4) = INT(0.00d0 * xr)        !EFTu:GTP
        n_elong(5) = INT(0.00d0 * xr)        !EFTu/Ts
        n_elong(6) = INT(0.00d0 * xr)        !EFTu/Ts:GDP
        n_elong(7) = INT(0.00d0 * xr)        !EFTu/Ts:GTP
        n_elong(8) = INT(0.80d0 * xr)        !EFG
        n_elong(9) = INT(0.00d0 * xr)        !EFG:GDP
        n_elong(10)= INT(0.00d0 * xr)        !EFG:GTP

        n_rfac(1) = INT(0.46d-2* xr)         !Release Factor 1
        n_rfac(2) = INT(0.46d-1* xr)         !Release Factor 2
        n_rfac(3) = INT(0.02d0 * xr)         !Release Factor 3
        n_rfac(4) = INT(0.00d0 * xr)         !Release Factor 3 / GDP
        n_rfac(5) = INT(0.00d0 * xr)         !Release Factor 3 / GTP
        n_rfac(6) = INT(0.11d0 * xr)         !Recycling Factor

        !=== Aminos and TRNA-Synthases ===!

        nars_aa(:) = 0                       !Aminoacyl-TRNA-Synthase / AA

        nars(1) = INT(0.1300 * xr)           !Aminoacyl-TRNA-Synthase (PHE)
        nars(2) = INT(0.0540 * xr)           !Aminoacyl-TRNA-Synthase (LEU)
        nars(3) = INT(0.0540 * xr)           !Aminoacyl-TRNA-Synthase (ILE)
        nars(4) = INT(0.0500 * xr)           !Aminoacyl-TRNA-Synthase (MET)
        nars(5) = INT(0.0430 * xr)           !Aminoacyl-TRNA-Synthase (VAL)
        nars(6) = INT(0.0680 * xr)           !Aminoacyl-TRNA-Synthase (SER)
        nars(7) = INT(0.1260 * xr)           !Aminoacyl-TRNA-Synthase (PRO)
        nars(8) = INT(0.0450 * xr)           !Aminoacyl-TRNA-Synthase (THR)
        nars(9) = INT(0.0580 * xr)           !Aminoacyl-TRNA-Synthase (ALA)
        nars(10)= INT(0.0540 * xr)           !Aminoacyl-TRNA-Synthase (TYR)
        nars(11)= INT(0.0400 * xr)           !Aminoacyl-TRNA-Synthase (HIS)
        nars(12)= INT(0.0270 * xr)           !Aminoacyl-TRNA-Synthase (GLN)
        nars(13)= INT(0.1180 * xr)           !Aminoacyl-TRNA-Synthase (ASN)
        nars(14)= INT(0.0800 * xr)           !Aminoacyl-TRNA-Synthase (LYS)
        nars(15)= INT(0.0780 * xr)           !Aminoacyl-TRNA-Synthase (ASP)
        nars(16)= INT(0.0540 * xr)           !Aminoacyl-TRNA-Synthase (GLU)
        nars(17)= INT(0.0110 * xr)           !Aminoacyl-TRNA-Synthase (CYS)
        nars(18)= INT(0.0360 * xr)           !Aminoacyl-TRNA-Synthase (TRP)
        nars(19)= INT(0.0180 * xr)           !Aminoacyl-TRNA-Synthase (ARG)
        nars(20)= INT(0.0900 * xr)           !Aminoacyl-TRNA-Synthase (GLY)

        !=== TRNAs and EFTU-TRNAs ===!

        ntcx(:) = 0                          !Ternary Complex
        ntrna_aa(:) = 0                      !TRNA:AA
        ntrna(:) = 0                         !TRNA

        ntrna(1) = INT(0.2870 * xr)          !TRNA cod=AAA
        ntrna(2) = INT(0.1770 * xr)          !TRNA cod=AAC
        ntrna(3:4) = 0                       !TRNA cod=AAG/U

        ntrna(5) = INT(0.1490 * xr)          !TRNA cod=ACA
        ntrna(6) = INT(0.1720 * xr)          !TRNA cod=ACC
        ntrna(7) = INT(0.0830 * xr)          !TRNA cod=ACG
        ntrna(8) = 0                         !TRNA cod=ACU

        ntrna(9) = INT(0.0990 * xr)          !TRNA cod=AGA
        ntrna(10)= INT(0.1730 * xr)          !TRNA cod=AGC
        ntrna(11)= INT(0.0630 * xr)          !TRNA cod=AGG
        ntrna(12)= 0                         !TRNA cod=AGU

        ntrna(13)= INT(0.2820 * xr)          !TRNA cod=AUA
        ntrna(14)= INT(0.2820 * xr)          !TRNA cod=AUC
        ntrna(15)= INT(0.1140 * xr)          !TRNA cod=AUG
        ntrna(16)= 0                         !TRNA cod=AUU

        ntrna(17)= INT(0.1250 * xr)          !TRNA cod=CAA
        ntrna(18)= INT(0.1040 * xr)          !TRNA cod=CAC
        ntrna(19)= INT(0.1450 * xr)          !TRNA cod=CAG
        ntrna(20)= 0                         !TRNA cod=CAU

        ntrna(21)= INT(0.0780 * xr)          !TRNA cod=CCA
        ntrna(22)= INT(0.0980 * xr)          !TRNA cod=CCC
        ntrna(23)= INT(0.1040 * xr)          !TRNA cod=CCG
        ntrna(24)= 0                         !TRNA cod=CCU

        ntrna(25)= 0                         !TRNA cod=CGA
        ntrna(26)= 0                         !TRNA cod=CGC
        ntrna(27)= INT(0.0760 * xr)          !TRNA cod=CGG
        ntrna(28)= INT(0.6400 * xr)          !TRNA cod=CGU

        ntrna(29)= INT(0.0980 * xr)          !TRNA cod=CUA
        ntrna(30)= INT(0.1550 * xr)          !TRNA cod=CUC
        ntrna(31)= INT(0.6320 * xr)          !TRNA cod=CUG
        ntrna(32)= 0                         !TRNA cod=CUU

        ntrna(33)= INT(0.7040 * xr)          !TRNA cod=GAA
        ntrna(34)= INT(0.3640 * xr)          !TRNA cod=GAC
        ntrna(35:36) = 0                     !TRNA cod=GAG/U

        ntrna(37)= INT(0.5390 * xr)          !TRNA cod=GCA
        ntrna(38)= INT(0.0920 * xr)          !TRNA cod=GCC
        ntrna(39:40) = 0                     !TRNA cod=GCG/U

        ntrna(41)= INT(0.1510 * xr)          !TRNA cod=GGA
        ntrna(42)= INT(0.6600 * xr)          !TRNA cod=GGC
        ntrna(43)= INT(0.1510 * xr)          !TRNA cod=GGG
        ntrna(44)= 0                         !TRNA cod=GGU

        ntrna(45)= INT(0.5000 * xr)          !TRNA cod=GUA
        ntrna(46)= INT(0.1930 * xr)          !TRNA cod=GUC
        ntrna(47:48) = 0                     !TRNA cod=GUG/U

        ntrna(49)= INT(0.00d0 * xr)          !TRNA cod=UAA*
        ntrna(50)= INT(0.2560 * xr)          !TRNA cod=UAC
        ntrna(51)= INT(0.00d0 * xr)          !TRNA cod=UAG*
        ntrna(52)= 0                         !TRNA cod=UAU

        ntrna(53)= INT(0.2160 * xr)          !TRNA cod=UCA
        ntrna(54)= INT(0.1100 * xr)          !TRNA cod=UCC
        ntrna(55)= INT(0.0420 * xr)          !TRNA cod=UCG
        ntrna(56)= 0                         !TRNA cod=UCU

        ntrna(57)= INT(0.00d0 * xr)          !TRNA cod=UGA*
        ntrna(58)= INT(0.2010 * xr)          !TRNA cod=UGC
        ntrna(59)= INT(0.1280 * xr)          !TRNA cod=UGG
        ntrna(60)= 0                         !TRNA cod=UGU

        ntrna(61)= INT(0.1360 * xr)          !TRNA cod=UUA
        ntrna(62)= INT(0.1530 * xr)          !TRNA cod=UUC
        ntrna(63)= INT(0.2650 * xr)          !TRNA cod=UUG
        ntrna(64)= 0                         !TRNA cod=UUU

        ntrna(65)= INT(0.3360 * xr)          !TRNA cod=fMET


 10     CONTINUE


        !=== Set Inital npro Values ===!

        npro(id_30pi) = n_ribo(5)
        npro(id_50s ) = n_ribo(6)

        npro(id_tcx) = 0
        npro(id_efg) = n_elong(10)
        npro(id_rf1) = n_rfac(1)
        npro(id_rf2) = n_rfac(2)
        npro(id_rf3) = n_rfac(5) + n_rfac(4)
        npro(id_rrf) = n_rfac(6)
        npro(id_if3) = n_ifac(5)

        RETURN

      END SUBROUTINE GETOPTIONS
