! ==============================================================================
! Program: PRO-SYN
! 
! Description: A program for computing RNA translation using
!              Drew/Pape RNA translation kinetics model.
!
! Notes:
!        EXECUTION PATH
!
!        Section 1 - Read In Files and Prepare
!        Section 2 - Perform the simulation
!        Section 3 - Clean Up and Finish Program
!        END
!
!        FILE TREE
!
!        Unit = 1  --- Option Settings File
!        Unit = 2  --- 
!        Unit = 5  --- Standard Out (NOT USED)
!        Unit = 6  --- Standard In (NOT USED)
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            01/01/2019   Original Code
!
! Dependancies:
!
! Modules - CLASS_MRNA, CONSTANTS, PROTEIN
! Functions -
! Subroutines - SSA, SETUP_RIBOSOME, SETUP_FOLDING, SETUP_RNA, GETOPTIONS
!
! Author(s): Eric Dykeman
!            Copyright (c) 2019 (Please Refer to LICENCE)
!
! ==============================================================================

      PROGRAM PROSYN

        USE Class_mRNA
        USE Constants
        USE Protein

        IMPLICIT NONE

        !=== VARIABLES ===!

        TYPE(mRNA) :: rna(mxrna)

        DOUBLE PRECISION :: x,t1,t2,dt,tout,tmax

        CHARACTER (LEN=300) :: riboparm,foldparm,seqfile,outfile
        CHARACTER (LEN=300) :: arg,fasta

        INTEGER :: i,j,k,n,io,i1,i2,iseed,narg

        LOGICAL :: iribo,ifold
!ecd-start
        character (len=8192) :: fline
        character :: seq(8192)
        integer :: i3,icodon(8192)
!ecd-end
        !=== Default Settings ===!

        iseed = 1881324

        time = 0.0d0
        tmax = 1.0d5

        seqfile  = 'cell_mrna.dat'
        outfile = 'output.dat'

        io = 1
        dt = 1.0d-6
        tout = dt

        iribo = .false.
        ifold = .false.


        !=== SECTION 0 - Input From Command Line ===!

        narg = IARGC ()

        DO i=1,narg,2

          CALL GETARG (i,arg)

          SELECT CASE (arg)

            CASE ('-s')

              CALL GETARG (i+1,seqfile)

            CASE ('-x')

              CALL GETARG (i+1,fasta)
              READ(fasta,*)xribo_tot

            CASE DEFAULT

              WRITE(*,*)arg,'Invalid Line Argument'
              STOP

          END SELECT

        ENDDO


        !=== SECTION 1a - Get Options ===!

        CALL GETOPTIONS


        !=== SECTION 1b - Read in Parms, Setup Data ===!

        CALL SETUP_FOLDING (foldparm,ifold)

        CALL SETUP_RIBOSOME (riboparm,iribo)

        CALL SETUP_RNA (seqfile)


        !=== SECTION 1c - Initialize RNA ===!

        nrna = 0
        psum_rna(:,:) = 0.0d0
        rna(:)% tnext = 1.0d9

        do i=1,mxrna
        rna(i)% a(:)  = 0.0d0
        enddo

!        CALL MRNA_INIT (rna(1))

!ecd - start (read in cell mrna)
        open(unit=444,file=seqfile,status='unknown')

        read(444,*,IOSTAT=i)fline

        do while ( i == 0 )

          nrna = nrna + 1

          if ( nrna > 1 ) then
          rna(nrna) = rna(1)
          else
          call mrna_init (rna(1))
          endif

          n = LEN_TRIM(fline)

          read(fline,'(10000A1)')(seq(j),j=1,n)

          !codon list!
          icodon(:) = 0

          do j=1,n-2
          if ( seq(j+0) == 'A' ) i1 = 0
          if ( seq(j+0) == 'C' ) i1 = 1
          if ( seq(j+0) == 'G' ) i1 = 2
          if ( seq(j+0) == 'U' ) i1 = 3
          if ( seq(j+1) == 'A' ) i2 = 0
          if ( seq(j+1) == 'C' ) i2 = 1
          if ( seq(j+1) == 'G' ) i2 = 2
          if ( seq(j+1) == 'U' ) i2 = 3
          if ( seq(j+2) == 'A' ) i3 = 0
          if ( seq(j+2) == 'C' ) i3 = 1
          if ( seq(j+2) == 'G' ) i3 = 2
          if ( seq(j+2) == 'U' ) i3 = 3
          k = 16 * i1 + 4 * i2 + i3 + 1
          icodon(j) = k
          enddo

          rna(nrna)% icdn(:) = icodon(:)
          rna(nrna)% nnuc = n

          read(444,*,IOSTAT=i)fline

        enddo

        !reset!

        nstats(:) = 0
        ncount(:) = 0
        ntr_tot(:) = 0

        !=== Default start -> All TRNAs Aminoacylated ===!

        ntrna_aa(:) = ntrna(:)

        n_ifac(6) = ntrna_aa(65)

        !headers in data file for output/testing!

        open(unit=113,file='pro-gtp_consumption.dat',status='Unknown')
        open(unit=114,file='ribo_states.dat',status='Unknown')
        open(unit=115,file='ribo_dependent_pro.dat',status='Unknown')

        open(unit=151,file='elong_factors.dat',status='Unknown')
        open(unit=152,file='ribo_subunits.dat',status='Unknown')
        open(unit=153,file='init_factors.dat',status='Unknown')
        open(unit=154,file='recy_factors.dat',status='Unknown')

        write(113,*)' #TIME               Used IF2    Used TC  &
       & Used Ef-G  Used RF3  Used GTP  Num PRE  Num Pro'
        write(114,*)'#TIME, # 70SIC, # 70S:EC, # 70S:TC, # Stall'
        write(115,*)'#TIME, #30S:PIC, #50S, #TC, #Ef-G:GTP, #RF1, &
                  & #RF2, #RF3, #RRF, #IF3 '

        write(151,*)'# TIME             TS    TU       TU-2   TU-3   &
       &sTU    sTU-2 sTU-3 EFG     EFG-2  EFG-3'

        write(152,*)'# TIME            30S     30S1    30S2   30S3  &
       &30PIC  50S'

        write(153,*)'# TIME             IF1    IF2    IF2-2  IF2-3  &
       &IF3    fMET'

        write(154,*)'# TIME               RF1   RF2    RF3   RF3-2  &
       &RF3-3  RRF'

       close(unit=113)
       close(unit=114)
       close(unit=115)

       close(unit=151)
       close(unit=152)
       close(unit=153)
       close(unit=154)

!ecd - end

        !=== Create Partial Sum Table ===!

        DO i=1,mxrna,2
        psum_rna(:,i) = rna(i)% a(:) + rna(i+1)% a(:)
        ENDDO

        k = 4
        n = 1
        DO WHILE ( k <= mxrna )

          j = k / 2
          DO i=j,mxrna,k
          psum_rna(:,i) = psum_rna(:,i-n) + psum_rna(:,i+n)
          ENDDO

          n = n * 2
          k = k * 2

        ENDDO

        !=== Create Queue Table ===!

        DO i=1,mxrna,2

          t1 = rna(i+0)% tnext
          t2 = rna(i+1)% tnext

          IF ( t1 <= t2 ) THEN
            iqueue_rna(i) = i
          ELSE
            iqueue_rna(i) = i+1
          ENDIF

        ENDDO

        k = 4
        n = 1
        DO WHILE ( k <= mxrna )

          j = k / 2
          DO i=j,mxrna,k

            i1 = iqueue_rna(i-n)
            i2 = iqueue_rna(i+n)

            t1 = rna(i1)% tnext
            t2 = rna(i2)% tnext

            IF ( t1 <= t2 ) THEN
              iqueue_rna(i) = i1
            ELSE
              iqueue_rna(i) = i2
            ENDIF

          ENDDO

          n = n * 2
          k = k * 2

        ENDDO


        !=== SECTION 2 - Perform Stochastic Simulation ===!

        OPEN (UNIT=2, FILE=outfile, STATUS='Unknown')

        DO WHILE ( time < tmax )

          CALL SSA (rna,iseed,tout)

          !=== Increment tout ===!

          IF ( time > tout ) THEN

            tout = tout + dt

            io = io + 1

            IF ( io > 9 ) THEN
              io = 1
              IF ( dt < 1.0d2 ) dt = dt * 10.0d0
              dt = MIN(dt,100.0d0)
            ENDIF

          ENDIF

        ENDDO


        !=== SECTION 3 - Finish ===!

        CLOSE (UNIT = 2)

      END PROGRAM PROSYN
