! ==============================================================================
! Subroutine: RATTLE (FOLD,FNEW)
! 
! Purpose: Calculates the forces of constraint on a set of rigid bonds
!          and updates the force vectors accordingly.
!
! Method: See Ryckaert, et al., J. Comp. Phys. 23, 327 (1977).
!
! Arguments:
!
!           FOLD - Array of dimension (3,NAT) containing the total
!                  force on each atom at time step t. On OUTPUT
!                  contains the total force plus the additional
!                  force required to inforce the velocity constraints.
!           FNEW - Array of dimension (3,NAT) containing the total
!                  force on each atom at time step t+dt.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE RATTLE (FOLD,FNEW)

        USE SystemParam, ONLY : rat,vel,amass,tstep,tols,dcbd,lcbd,&
                              & nat,ncbd,maxsi,mpijob,myproc

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        DOUBLE PRECISION, INTENT(IN) :: fnew(3,nat)
        DOUBLE PRECISION, INTENT(INOUT) :: fold(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,ishk,ichk

        DOUBLE PRECISION :: x1,x2,x3,r1,r2,r3
        DOUBLE PRECISION :: a,b,c,ct,dsq,rm

        DOUBLE PRECISION, PARAMETER :: cfac = 9.6485339840d3


        ct = tstep * cfac
        ct = 0.50d0 * ct


        !=== Rattle Itteration ===!

        DO ishk=1,maxsi

          ichk = 0

          DO i=1,ncbd

            j = lcbd(1,i)
            k = lcbd(2,i)

            dsq = dcbd(2,i)
            dsq = dsq * dsq

            a = ct / amass(j)
            b = ct / amass(k)

            rm = 1.0d0 / ( a + b )

            r1 = rat(1,j) - rat(1,k)
            r2 = rat(2,j) - rat(2,k)
            r3 = rat(3,j) - rat(3,k)

            x1 = vel(1,j) - vel(1,k)
            x2 = vel(2,j) - vel(2,k)
            x3 = vel(3,j) - vel(3,k)

            x1 = x1 + a * fold(1,j) - b * fold(1,k)
            x2 = x2 + a * fold(2,j) - b * fold(2,k)
            x3 = x3 + a * fold(3,j) - b * fold(3,k)

            c = x1 * r1 + x2 * r2 + x3 * r3

            IF ( DABS(c) < tols ) CYCLE

            ichk = 1

            c = c / dsq
            c = c * rm

            x1 = c * r1
            x2 = c * r2
            x3 = c * r3

            fold(1,j) = fold(1,j) - x1
            fold(2,j) = fold(2,j) - x2
            fold(3,j) = fold(3,j) - x3
 
            fold(1,k) = fold(1,k) + x1
            fold(2,k) = fold(2,k) + x2
            fold(3,k) = fold(3,k) + x3

          ENDDO

          IF ( ichk == 0 ) EXIT

        ENDDO

        IF ( ichk == 1 ) THEN

          IF ( myproc == 0 ) THEN

            WRITE(4,*)'ERROR: rattle failed to converge.'
            WRITE(4,*)'Number of iterations -- maxsi = ',maxsi

          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        RETURN

      END SUBROUTINE RATTLE
