! ==============================================================================
! Subroutine: SCALER (DH,DDH,S,XI,DXI)
! 
! Purpose: Scales the position of atoms according to the pressure coupling
!          algorithms of either (1) Nose-Klein or (2) Weak-Coupling
!
! Method: (1) Nose and Klein, Mol. Phys., 50 1055 (1983).
!         (2) Berendsen et al., J. Chem. Phys., 81 3684 (1984).
!
! Arguments:
!
!           DH  - 3 X 3 matrix of the derivatives of the direct
!                 lattice vectors with respect to time.
!           DDH - 3 X 3 matrix of the second derivatives of the
!                 direct lattice vectors with respect to time.
!           S   - Time Scaling element in the Nose-Klein coupling
!                 algorithm. On INPUT, S is the value at time t,
!                 on OUTPUT S is the value at time t + tstep.
!           XI  - Derivative of LN(s) with respect to time.
!           DXI - Second derivative of LN(s) with respect to time.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - CROSS, EWALDSETUP, PMESETUP
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SCALER (DH,DDH,S,XI,DXI)

        USE SystemParam, ONLY : rat,dlv,rlv,vol,tstep,nat,ipc,ircp

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        DOUBLE PRECISION, INTENT(IN) :: xi,dxi,dh(3,3),ddh(3,3)
        DOUBLE PRECISION, INTENT(INOUT) :: s

        !=== VARIABLES ===!

        INTEGER :: i

        DOUBLE PRECISION :: x1,x2,x3,arg,ct


        ct = 0.50d0 * tstep


        !=== New Direct Lattice Vectors ===!

        IF ( ipc == 1 ) THEN

          s = DLOG(s)

          arg = xi + ct * dxi
          s = s + tstep * arg

          s = DEXP(s)

          DO i=1,3

            x1 = dh(1,i) + ct * ddh(1,i)
            x2 = dh(2,i) + ct * ddh(2,i)
            x3 = dh(3,i) + ct * ddh(3,i)

            dlv(1,i) = dlv(1,i) + tstep * x1
            dlv(2,i) = dlv(2,i) + tstep * x2
            dlv(3,i) = dlv(3,i) + tstep * x3

          ENDDO

        ELSEIF ( ipc == 2 ) THEN

          DO i=1,3

            dlv(1,i) = dlv(1,i) + tstep * dh(1,i)
            dlv(2,i) = dlv(2,i) + tstep * dh(2,i)
            dlv(3,i) = dlv(3,i) + tstep * dh(3,i)

          ENDDO

        ENDIF

        !=== Scale Positions ===!

        DO i=1,nat

          x1 = rlv(1,1) * rat(1,i) + rlv(1,2) * rat(2,i) + &
             & rlv(1,3) * rat(3,i)

          x2 = rlv(2,1) * rat(1,i) + rlv(2,2) * rat(2,i) + &
             & rlv(2,3) * rat(3,i)

          x3 = rlv(3,1) * rat(1,i) + rlv(3,2) * rat(2,i) + &
             & rlv(3,3) * rat(3,i)

          rat(1,i) = dlv(1,1) * x1 + dlv(1,2) * x2 + &
                   & dlv(1,3) * x3

          rat(2,i) = dlv(2,1) * x1 + dlv(2,2) * x2 + &
                   & dlv(2,3) * x3

          rat(3,i) = dlv(3,1) * x1 + dlv(3,2) * x2 + &
                   & dlv(3,3) * x3

        ENDDO

        !=== New Reciprical Lattice Vectors ===!

        CALL CROSS (dlv(:,2),dlv(:,3),rlv(1,:))
        CALL CROSS (dlv(:,3),dlv(:,1),rlv(2,:))
        CALL CROSS (dlv(:,1),dlv(:,2),rlv(3,:))

        vol = rlv(1,1) * dlv(1,1) + rlv(1,2) * dlv(2,1) &
          & + rlv(1,3) * dlv(3,1)

        DO i=1,3

          rlv(1,i) = rlv(1,i) / vol
          rlv(2,i) = rlv(2,i) / vol
          rlv(3,i) = rlv(3,i) / vol

        ENDDO


        !=== RE-Setup Reciprocal Ewald Sums ===!

        IF ( ircp == 1 ) THEN

          CALL EWALDSETUP

        ELSEIF ( ircp == 2 ) THEN

          CALL PMESETUP

        ENDIF

        RETURN

      END SUBROUTINE SCALER
