! ==============================================================================
! Subroutine: SCALEV (T,DH,DDH,S,XI,DXI,EK,FOLD,FNEW)
! 
! Purpose: Scales the velocities and finds the force of constraint
!          from pressure / temperature coupling for the either the
!          (1) Nose-Klein or (2) Weak Coupling algorithms.
!
! Method: (1) Nose and Klein, Mol. Phys., 50 1055 (1983).
!         (2) Berendsen et al., J. Chem. Phys., 81 3684 (1984).
!
! Arguments:
!
!           T    - 3 X 3 kinetic energy tensor.
!           DH   - 3 X 3 matrix of the derivatives of the direct
!                  lattice vectors with respect to time.
!           DDH  - 3 X 3 matrix of the second derivatives of the
!                  direct lattive vectors with respect to time.
!           S    - Time scaling element in the Nose-Klein coupling.
!           XI   - Derivative of LN(s) with respect to time.
!           DXI  - Second derivative of LN(s) with respect to time.
!           EK   - Total Kinetic energy of the system.
!           FOLD - Array of dimension (3,NAT) containing the total
!                  force on each atom at time step t. On OUTPUT
!                  contains the total force plus the additional
!                  force from temperature / pressure coupling.
!           FNEW - Array of dimension (3,NAT) containing the total
!                  force on each atom at time step t. On OUTPUT
!                  contains the total force plus the additional
!                  force from temperature / pressure coupling.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines - MATINV
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SCALEV (T,DH,DDH,S,XI,DXI,EK,FOLD,FNEW)

        USE SystemParam, ONLY : rat,vel,amass,dlv,rlv,vir,vol,temp,&
                              & btemp,pres,comp,taut,taup,tstep,nat,&
                              & itc,ipc,ips,ipbc,mpijob,myproc

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        DOUBLE PRECISION, INTENT(INOUT) :: t(3,3),dh(3,3),ddh(3,3)
        DOUBLE PRECISION, INTENT(INOUT) :: fold(3,nat),fnew(3,nat)
        DOUBLE PRECISION, INTENT(INOUT) :: s,xi,dxi,ek

        !=== VARIABLES ===!

        INTEGER :: i,j,ichk,istart,iend

        DOUBLE PRECISION :: x1,x2,x3,c,ct,arg,xin,ekn
        DOUBLE PRECISION :: wt,wp,dsq(3),dhn(3,3),tn(3,3)
        DOUBLE PRECISION :: a(3,3),b(3,3),wrk(3,3)

        INTEGER, PARAMETER :: maxit = 5

        DOUBLE PRECISION, PARAMETER :: cfac1 = 9.6485339840d3
        DOUBLE PRECISION, PARAMETER :: cfac2 = 6.0221417942d-3
        DOUBLE PRECISION, PARAMETER :: tol = 1.00d-8


        ct = 0.50d0 * tstep

        wt = taut * taut * btemp

        wp = taup * taup * pres


        !=== Setup For Pressure Coupling ===!

        IF ( ipbc == 2 ) THEN

          DO i=1,3

            tn(1,i) = t(1,i)
            tn(2,i) = t(2,i)
            tn(3,i) = t(3,i)

            wrk(1,i) = cfac1 * vir(1,i)
            wrk(2,i) = cfac1 * vir(2,i)
            wrk(3,i) = cfac1 * vir(3,i)

            arg = dlv(1,i) * dlv(1,i)
            arg = arg + dlv(2,i) * dlv(2,i)
            dsq(i) = arg + dlv(3,i) * dlv(3,i)

          ENDDO

          arg = cfac2 * pres * vol

          wrk(1,1) = wrk(1,1) - arg
          wrk(2,2) = wrk(2,2) - arg
          wrk(3,3) = wrk(3,3) - arg

        ENDIF


        !=== Nose-Hoover Temperature Coupling ===!

        IF ( ipbc /= 2 .and. itc == 1 ) THEN

          xi = xi + ct * dxi
          dxi = ( temp - btemp ) / wt
          xin = xi + ct * dxi

          ichk = 1

          DO i=1,maxit

            arg = 1.0d0 + ct * xin
            arg = arg * arg

            dxi = temp / arg
            dxi = ( dxi - btemp ) / wt

            arg = xi + ct * dxi

            IF ( DABS(arg-xin) < tol ) THEN

              ichk = 0
              xi = arg

              EXIT

            ENDIF

            xin = arg

          ENDDO

          IF ( ichk == 1 .and. myproc == 0 ) THEN

            WRITE(4,*)'ERROR: Nose-Hoover failed to converge in scalev.'
            WRITE(4,*)'Number of iterations -- maxit = ',maxit

            IF ( mpijob ) CALL CLOSEMPI

            STOP

          ENDIF

          arg = 1.0d0 + ct * xi

          DO i=1,nat

            c = amass(i) * xi / cfac1

            vel(1,i) = vel(1,i) / arg
            vel(2,i) = vel(2,i) / arg
            vel(3,i) = vel(3,i) / arg

            x1 = c * vel(1,i)
            x2 = c * vel(2,i)
            x3 = c * vel(3,i)

            fold(1,i) = fold(1,i) - x1
            fold(2,i) = fold(2,i) - x2
            fold(3,i) = fold(3,i) - x3

            fnew(1,i) = fnew(1,i) - x1
            fnew(2,i) = fnew(2,i) - x2
            fnew(3,i) = fnew(3,i) - x3

          ENDDO

          arg = arg * arg

          ek = ek / arg
          temp = temp / arg

        ENDIF


        !=== Berendsen Temperature Coupling ===!

        IF ( itc == 2 ) THEN

          xi = 1.0d0 - btemp / temp
          xi = 0.5d0 * xi / taut

          ichk = 1

          DO i=1,maxit

            arg = 1.0d0 + ct * xi
            arg = arg * arg
            arg = temp / arg

            xin = 1.0d0 - btemp / arg
            xin = 0.5d0 * xin / taut

            IF ( DABS(xin-xi) < tol ) THEN

              ichk = 0
              xi = xin

              EXIT

            ENDIF

            xi = xin

          ENDDO

          IF ( ichk == 1 .and. myproc == 0 ) THEN

            WRITE(4,*)'ERROR: Berendsen failed to converge in scalev.'
            WRITE(4,*)'Number of iterations -- maxit = ',maxit

            IF ( mpijob ) CALL CLOSEMPI

            STOP

          ENDIF

          arg = 1.0d0 + ct * xi

          DO i=1,nat

            c = amass(i) * xi / cfac1

            vel(1,i) = vel(1,i) / arg
            vel(2,i) = vel(2,i) / arg
            vel(3,i) = vel(3,i) / arg

            x1 = c * vel(1,i)
            x2 = c * vel(2,i)
            x3 = c * vel(3,i)

            fold(1,i) = fold(1,i) - x1
            fold(2,i) = fold(2,i) - x2
            fold(3,i) = fold(3,i) - x3

            fnew(1,i) = fnew(1,i) - x1
            fnew(2,i) = fnew(2,i) - x2
            fnew(3,i) = fnew(3,i) - x3

          ENDDO

          arg = arg * arg

          ek = ek / arg
          temp = temp / arg

        ENDIF


        !=== Nose-Klein Pressure Coupling ===!

        IF ( ipbc == 2 .and. ipc == 1 ) THEN

          xin = 0.0d0
          ekn = ek

          IF ( itc == 1 ) THEN

            xi = xi + ct * dxi

          ENDIF

          DO i=1,3

            dh(1,j) = dh(1,j) + ct * ddh(1,j)
            dh(2,j) = dh(2,j) + ct * ddh(2,j)
            dh(3,j) = dh(3,j) + ct * ddh(3,j)

          ENDDO

          ichk = 1

          DO i=1,maxit

            IF ( itc == 1 ) THEN

              dxi = temp * ekn / ek
              dxi = ( dxi - btemp ) / wt
              xin = xi + ct * dxi

            ENDIF

            DO j=1,3

              ddh(j,1) = ( 2.0d0 * tn(j,1) + wrk(j,1) ) / wp
              ddh(j,2) = ( 2.0d0 * tn(j,2) + wrk(j,2) ) / wp
              ddh(j,3) = ( 2.0d0 * tn(j,3) + wrk(j,3) ) / wp

              IF ( ips /= 1 ) THEN

                x1 = ddh(j,1) * rlv(1,1) + ddh(j,2) * rlv(1,2) &
                 & + ddh(j,3) * rlv(1,3)
                x2 = ddh(j,1) * rlv(2,1) + ddh(j,2) * rlv(2,2) &
                 & + ddh(j,3) * rlv(2,3)
                x3 = ddh(j,1) * rlv(3,1) + ddh(j,2) * rlv(3,2) &
                 & + ddh(j,3) * rlv(3,3)

                ddh(j,1) = x1
                ddh(j,2) = x2
                ddh(j,3) = x3

              ENDIF

            ENDDO

            IF ( ips == 1 ) THEN

              arg = ddh(1,1) + ddh(2,2) + ddh(3,3)
              arg = arg / dsq(1)

              DO j=1,3

                ddh(1,j) = dlv(1,j) * arg
                ddh(2,j) = dlv(2,j) * arg
                ddh(3,j) = dlv(3,j) * arg

              ENDDO

            ELSEIF ( ips == 2 ) THEN

              x1 = dlv(1,1) * ddh(1,1) + dlv(2,1) * ddh(2,1) &
               & + dlv(3,1) * ddh(3,1)
              x2 = dlv(1,2) * ddh(1,2) + dlv(2,2) * ddh(2,2) &
               & + dlv(3,2) * ddh(3,2)
              x3 = dlv(1,3) * ddh(1,3) + dlv(2,3) * ddh(2,3) &
               & + dlv(3,3) * ddh(3,3)

              x1 = x1 / dsq(1)
              x2 = x2 / dsq(2)
              x3 = x3 / dsq(3)

              DO j=1,3

                ddh(j,1) = dlv(j,1) * x1
                ddh(j,2) = dlv(j,2) * x2
                ddh(j,3) = dlv(j,3) * x3

              ENDDO

            ENDIF

            arg = 1.0d0 - ct * xin

            DO j=1,3

              ddh(1,j) = s * s * ddh(1,j)
              ddh(2,j) = s * s * ddh(2,j)
              ddh(3,j) = s * s * ddh(3,j)

              dhn(1,j) = dh(1,j) + ct * ddh(1,j)
              dhn(2,j) = dh(2,j) + ct * ddh(2,j)
              dhn(3,j) = dh(3,j) + ct * ddh(3,j)

              dhn(1,j) = dhn(1,j) / arg
              dhn(2,j) = dhn(2,j) / arg
              dhn(3,j) = dhn(3,j) / arg

              ddh(1,j) = ddh(1,j) + xin * dhn(1,j)
              ddh(2,j) = ddh(2,j) + xin * dhn(2,j)
              ddh(3,j) = ddh(3,j) + xin * dhn(3,j)

            ENDDO

            arg = 1.0d0 + ct * xin

            DO j=1,3

              a(1,j) = dhn(1,1) * rlv(1,j)
              a(1,j) = a(1,j) + dhn(1,2) * rlv(2,j)
              a(1,j) = a(1,j) + dhn(1,3) * rlv(3,j)

              a(2,j) = dhn(2,1) * rlv(1,j)
              a(2,j) = a(2,j) + dhn(2,2) * rlv(2,j)
              a(2,j) = a(2,j) + dhn(2,3) * rlv(3,j)

              a(3,j) = dhn(3,1) * rlv(1,j)
              a(3,j) = a(3,j) + dhn(3,2) * rlv(2,j)
              a(3,j) = a(3,j) + dhn(3,3) * rlv(3,j)

              b(j,1) = ct * a(1,j)
              b(j,2) = ct * a(2,j)
              b(j,3) = ct * a(3,j)

              b(j,j) = b(j,j) + arg

            ENDDO

            CALL MATINV(b)

            DO j=1,3

              x1 = t(j,1) * b(1,1) + t(j,2) * b(1,2) &
               & + t(j,3) * b(1,3)
              x2 = t(j,1) * b(2,1) + t(j,2) * b(2,2) &
               & + t(j,3) * b(2,3)
              x3 = t(j,1) * b(3,1) + t(j,2) * b(3,2) &
               & + t(j,3) * b(3,3)

              tn(j,1) = x1
              tn(j,2) = x2
              tn(j,3) = x3

            ENDDO

            DO j=1,3

              x1 = b(1,1) * tn(1,j) + b(1,2) * tn(2,j) &
               & + b(1,3) * tn(3,j)
              x2 = b(2,1) * tn(1,j) + b(2,2) * tn(2,j) &
               & + b(2,3) * tn(3,j)
              x3 = b(3,1) * tn(1,j) + b(3,2) * tn(2,j) &
               & + b(3,3) * tn(3,j)

              tn(1,j) = x1
              tn(2,j) = x2
              tn(3,j) = x3

            ENDDO

            arg = tn(1,1) + tn(2,2) + tn(3,3)
            arg = arg / cfac1

            IF ( DABS(arg-ekn) < tol ) THEN

              ichk = 0
              ekn = arg

              IF ( itc == 1 ) xi = xin

              DO j=1,3

                dh(1,j) = dhn(1,j)
                dh(2,j) = dhn(2,j)
                dh(3,j) = dhn(3,j)

                t(1,j) = tn(1,j)
                t(2,j) = tn(2,j)
                t(3,j) = tn(3,j)

              ENDDO

              EXIT

            ENDIF

            ekn = arg

          ENDDO

          IF ( ichk == 1 .and. myproc == 0 ) THEN

            WRITE(4,*)'ERROR: Nose-Klein integration failed in scalev.'
            WRITE(4,*)'Number of iterations -- maxit = ',maxit

            IF ( mpijob ) CALL CLOSEMPI

            STOP

          ENDIF

          DO i=1,nat

            c = amass(i) / cfac1

            x1 = b(1,1) * vel(1,i) + b(1,2) * vel(2,i) &
             & + b(1,3) * vel(3,i)
            x2 = b(2,1) * vel(1,i) + b(2,2) * vel(2,i) &
             & + b(2,3) * vel(3,i)
            x3 = b(3,1) * vel(1,i) + b(3,2) * vel(2,i) &
             & + b(3,3) * vel(3,i)

            vel(1,i) = x1
            vel(2,i) = x2
            vel(3,i) = x3

            x1 = a(1,1) * vel(1,i) + a(2,1) * vel(2,i) &
             & + a(3,1) * vel(3,i)
            x2 = a(1,2) * vel(1,i) + a(2,2) * vel(2,i) &
             & + a(3,2) * vel(3,i)
            x3 = a(1,3) * vel(1,i) + a(2,3) * vel(2,i) &
             & + a(3,3) * vel(3,i)

            x1 = x1 + xi * vel(1,i)
            x2 = x2 + xi * vel(2,i)
            x3 = x3 + xi * vel(3,i)

            fold(1,i) = fold(1,i) - c * x1
            fold(2,i) = fold(2,i) - c * x2
            fold(3,i) = fold(3,i) - c * x3

            x1 = x1 - a(1,1) * vel(1,i) - a(1,2) * vel(2,i) &
             & - a(1,3) * vel(3,i)
            x2 = x2 - a(2,1) * vel(1,i) - a(2,2) * vel(2,i) &
             & - a(2,3) * vel(3,i)
            x3 = x3 - a(3,1) * vel(1,i) - a(3,2) * vel(2,i) &
             & - a(3,3) * vel(3,i)

            fnew(1,i) = fnew(1,i) - c * x1
            fnew(2,i) = fnew(2,i) - c * x2
            fnew(3,i) = fnew(3,i) - c * x3

          ENDDO

          temp = temp * ekn / ek
          ek = ekn

        ENDIF


        !=== Berendsen Pressure Coupling ===!

        IF ( ipbc == 2 .and. ipc == 2 ) THEN

          IF ( itc == 2 ) THEN

            xi = 1.0d0 - btemp / temp
            xi = 0.5d0 * xi / taut

            arg = 1.0d0 + ct * xi
            arg = arg * arg

            DO j=1,3

              t(1,j) = t(1,j) / arg
              t(2,j) = t(2,j) / arg
              t(3,j) = t(3,j) / arg

            ENDDO

          ENDIF

          arg = vol * taup * cfac2
          arg = comp / ( 3.0d0 * arg )

          DO j=1,3

            dh(j,1) = arg * ( 2.0d0 * t(j,1) + wrk(j,1) )
            dh(j,2) = arg * ( 2.0d0 * t(j,2) + wrk(j,2) )
            dh(j,3) = arg * ( 2.0d0 * t(j,3) + wrk(j,3) )

            IF ( ips /= 1 ) THEN

              x1 = dh(j,1) * dlv(1,1) + dh(j,2) * dlv(2,1) &
               & + dh(j,3) * dlv(3,1)
              x2 = dh(j,1) * dlv(1,2) + dh(j,2) * dlv(2,2) &
               & + dh(j,3) * dlv(3,2)
              x3 = dh(j,1) * dlv(1,3) + dh(j,2) * dlv(2,3) &
               & + dh(j,3) * dlv(3,3)

              dh(j,1) = x1
              dh(j,2) = x2
              dh(j,3) = x3

            ENDIF

          ENDDO

          IF ( ips == 1 ) THEN

            arg = dh(1,1) + dh(2,2) + dh(3,3)
            arg = arg / 3.0d0

            DO j=1,3

              dh(1,j) = dlv(1,j) * arg
              dh(2,j) = dlv(2,j) * arg
              dh(3,j) = dlv(3,j) * arg

            ENDDO

          ELSEIF ( ips == 2 ) THEN

            x1 = dlv(1,1) * dh(1,1) + dlv(2,1) * dh(2,1) &
             & + dlv(3,1) * dh(3,1)
            x2 = dlv(1,2) * dh(1,2) + dlv(2,2) * dh(2,2) &
             & + dlv(3,2) * dh(3,2)
            x3 = dlv(1,3) * dh(1,3) + dlv(2,3) * dh(2,3) &
             & + dlv(3,3) * dh(3,3)

            x1 = x1 / dsq(1)
            x2 = x2 / dsq(2)
            x3 = x3 / dsq(3)

            DO j=1,3

              dh(j,1) = dlv(j,1) * x1
              dh(j,2) = dlv(j,2) * x2
              dh(j,3) = dlv(j,3) * x3

            ENDDO

          ENDIF

        ENDIF

        RETURN

      END SUBROUTINE SCALEV
