! ==============================================================================
! Subroutine: SHAKE (FOLD,FNEW)
! 
! Purpose: Calculates the forces of constraint on a set of rigid bonds
!          and updates the force and atom positions accordingly.
!
! Method: See Ryckaert, et al., J. Comp. Phys. 23, 327 (1977).
!
! Arguments:
!
!           FOLD - Array of dimension (3,NAT) containing the total
!                  force on each atom at time step t. On OUTPUT
!                  contains the total force plus the additional
!                  force required to enforce the bond constraints.
!           FNEW - Array of dimension (3,NAT) containing the total
!                  force on each atom at time step t. On OUTPUT
!                  contains the total force plus the additional
!                  force required to enforce the bond constraints.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE SHAKE (FOLD,FNEW)

        USE SystemParam, ONLY : rat,vel,amass,tstep,tols,dcbd,lcbd,&
                              & nat,ncbd,maxsi,mpijob,myproc

        IMPLICIT NONE

        !=== ARGUMENTS ==!

        DOUBLE PRECISION, INTENT(INOUT) :: fold(3,nat),fnew(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,ishk,ichk

        DOUBLE PRECISION :: x1,x2,x3,r1,r2,r3
        DOUBLE PRECISION :: a,b,c,ct,deq,rm

        DOUBLE PRECISION, PARAMETER :: cfac = 9.6485339840d3


        ct = tstep * tstep
        ct = 0.50d0 * ct * cfac


        !=== Shake Itteration ===!

        DO ishk=1,maxsi

          ichk = 0

          DO i=1,ncbd

            j = lcbd(1,i)
            k = lcbd(2,i)

            deq = dcbd(2,i)

            a = ct / amass(j)
            b = ct / amass(k)

            rm = 1.0d0 / ( a + b )

            r1 = rat(1,j) - rat(1,k)
            r2 = rat(2,j) - rat(2,k)
            r3 = rat(3,j) - rat(3,k)

            x1 = r1 + tstep * ( vel(1,j) - vel(1,k) )
            x2 = r2 + tstep * ( vel(2,j) - vel(2,k) )
            x3 = r3 + tstep * ( vel(3,j) - vel(3,k) )

            x1 = x1 + a * fnew(1,j) - b * fnew(1,k)
            x2 = x2 + a * fnew(2,j) - b * fnew(2,k)
            x3 = x3 + a * fnew(3,j) - b * fnew(3,k)

            c = x1 * x1 + x2 * x2 + x3 * x3

            a = DSQRT(c) - deq

            IF ( DABS(a) < tols ) CYCLE

            ichk = 1

            a = r1 * r1 + r2 * r2 + r3 * r3
            b = x1 * r1 + x2 * r2 + x3 * r3

            b = b / a
            c = c - deq * deq

            c = b * b - c / a
            c = DSQRT(c)

            a = b + c
            b = b - c

            IF ( DABS(a) < DABS(b) ) THEN
              c = a * rm
            ELSE
              c = b * rm
            ENDIF

            x1 = c * r1
            x2 = c * r2
            x3 = c * r3

            fold(1,j) = fold(1,j) - x1
            fold(2,j) = fold(2,j) - x2
            fold(3,j) = fold(3,j) - x3
 
            fnew(1,j) = fnew(1,j) - x1
            fnew(2,j) = fnew(2,j) - x2
            fnew(3,j) = fnew(3,j) - x3

            fold(1,k) = fold(1,k) + x1
            fold(2,k) = fold(2,k) + x2
            fold(3,k) = fold(3,k) + x3

            fnew(1,k) = fnew(1,k) + x1
            fnew(2,k) = fnew(2,k) + x2
            fnew(3,k) = fnew(3,k) + x3

          ENDDO

          IF ( ichk == 0 ) EXIT

        ENDDO

        IF ( ichk == 1 ) THEN

          IF ( myproc == 0 ) THEN

            WRITE(4,*)'ERROR: shake failed to converge.'
            WRITE(4,*)'Number of iterations -- maxsi = ',maxsi

          ENDIF

          IF ( mpijob ) CALL CLOSEMPI

          STOP

        ENDIF

        RETURN

      END SUBROUTINE SHAKE
