! ==============================================================================
! Subroutine: ANGLE (EANGL,FTOT)
! 
! Purpose: Computes the energy, force and stress tensor due to angle bending
!          between two covalent bonds.
!
! Method:  Cornell et al., J. Am. Chem. Soc. 117, 5179 (1995).
!
!          The energy due to angle bending is:
!
!          E = E0 * (T - TEQ)^2
!
!          where E0 = 1/2 the Spring Constant, and TEQ is the
!          equilibrium angle between the two covalent bonds.
!
! Arguments:
!
!           EANGL - Energy due to angle bending between two covalent
!                   bonds.
!           FTOT  - Array of dimension (3,NAT) containing the total
!                   force on each atom.
!           VIR   - Array of dimension (3,3) containing the virial
!                   tensor with elements given by:
!                   S_ab = SUM_i Fia * Rib where Fia = ath
!                   component of the force on atom i.
!
! History:
!
! Version    Date         Comment
! --------   ----------   -----------------------
!            10/01/2010   Original Code
!
! Dependancies:
!
! Modules - SystemParam
! Functions -
! Subroutines -
!
! Author(s): Eric Dykeman
!
! ==============================================================================

      SUBROUTINE ANGLE (EANGL,FTOT)

        USE SystemParam, ONLY : rat,vir,dangl,langl,nat,nag,ipbc,iag

        IMPLICIT NONE

        !=== ARGUMENTS ===!

        DOUBLE PRECISION, INTENT(OUT) :: eangl
        DOUBLE PRECISION, INTENT(INOUT) :: ftot(3,nat)

        !=== VARIABLES ===!

        INTEGER :: i,j,k,m

        DOUBLE PRECISION :: x1,x2,x3,y1,y2,y3,a1,a2,a3,d1,d2
        DOUBLE PRECISION :: b1,b2,b3,s11,s12,s13,s22,s23,s33
        DOUBLE PRECISION :: c,d,t,teq,spr,arg,xsq,ysq

        DOUBLE PRECISION, PARAMETER :: tm = 0.999d0


        eangl = 0.0d0

        IF ( iag == 0 .or. nag == 0 ) RETURN


        DO i=1,nag

          j = langl(1,i)
          k = langl(2,i)
          m = langl(3,i)

          spr = dangl(1,i)
          teq = dangl(2,i)

          x1 = rat(1,j) - rat(1,k)
          x2 = rat(2,j) - rat(2,k)
          x3 = rat(3,j) - rat(3,k)

          y1 = rat(1,m) - rat(1,k)
          y2 = rat(2,m) - rat(2,k)
          y3 = rat(3,m) - rat(3,k)

          xsq = x1 * x1 + x2 * x2 + x3 * x3
          ysq = y1 * y1 + y2 * y2 + y3 * y3

          d = x1 * y1 + x2 * y2 + x3 * y3

          d1 = xsq * ysq
          d1 = DSQRT(d1)

          t = d / d1

          t = MIN(+tm,t)
          t = MAX(-tm,t)

          t = DACOS(t)

          arg = t - teq
          c = spr * arg

          eangl = eangl + c * arg

          d1 = d1 * DSIN(t)
          c = 2.0d0 * c / d1

          d1 = d / xsq
          d2 = d / ysq

          a1 = c * ( y1 - d1 * x1 )
          a2 = c * ( y2 - d1 * x2 )
          a3 = c * ( y3 - d1 * x3 )

          b1 = c * ( x1 - d2 * y1 )
          b2 = c * ( x2 - d2 * y2 )
          b3 = c * ( x3 - d2 * y3 )

          ftot(1,j) = ftot(1,j) + a1
          ftot(2,j) = ftot(2,j) + a2
          ftot(3,j) = ftot(3,j) + a3

          ftot(1,k) = ftot(1,k) - a1 - b1
          ftot(2,k) = ftot(2,k) - a2 - b2
          ftot(3,k) = ftot(3,k) - a3 - b3

          ftot(1,m) = ftot(1,m) + b1
          ftot(2,m) = ftot(2,m) + b2
          ftot(3,m) = ftot(3,m) + b3

          IF ( ipbc /= 2 ) CYCLE

          s11 = a1 * x1 + b1 * y1
          s12 = a1 * x2 + b1 * y2
          s13 = a1 * x3 + b1 * y3
          s22 = a2 * x2 + b2 * y2
          s23 = a2 * x3 + b2 * y3
          s33 = a3 * x3 + b3 * y3

          vir(1,1) = vir(1,1) + s11
          vir(2,1) = vir(2,1) + s12
          vir(3,1) = vir(3,1) + s13
          vir(1,2) = vir(1,2) + s12
          vir(2,2) = vir(2,2) + s22
          vir(3,2) = vir(3,2) + s23
          vir(1,3) = vir(1,3) + s13
          vir(2,3) = vir(2,3) + s23
          vir(3,3) = vir(3,3) + s33

        ENDDO

        RETURN

      END SUBROUTINE ANGLE
